/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* slow EPSP dynamics */


#include "ndl.h"

typedef struct sepsptag {
  struct sepsptag * next;
  double starttime;
  double strength;
  int    cell_id;
  bool   deleteme;
  int    window_id;
} sepspQ;

typedef struct {
    DYNAMICS
  double Gmax;
  double Er;
  double alpha;
  double beta1;
  double beta2;
  double beta3;
  bool   export_current;
} slowEPSP;

static PyMemberDef slowEPSP_members[] = {
    {"Gmax", T_DOUBLE, offsetof(slowEPSP, Gmax),
     0, "maximum conductance"},
    {"Er", T_DOUBLE, offsetof(slowEPSP, Er),
     0, "reversal potential"},
    {"alpha", T_DOUBLE, offsetof(slowEPSP, alpha),
     0, "coupling constant"},
    {"beta1", T_DOUBLE, offsetof(slowEPSP, beta1),
     0, "decay rate of stage 1"},
    {"beta2", T_DOUBLE, offsetof(slowEPSP, beta2),
     0, "decay rate of stage 2"},
    {"beta3", T_DOUBLE, offsetof(slowEPSP, beta3),
     0, "decay rate of stage 3"},
    {"export_current", T_INT, offsetof(slowEPSP, export_current),
     0, "if set to True current is calculated by this Dynamics"},
    {NULL},
};

static void update(slowEPSP *self, double strength) {
  double dsm1   = GETSTATE_DYN(self, 0);
  SETSTATE_DYN(self, 0, dsm1+strength/100);
}

static void enq_sepsp(slowEPSP *d, double t, double strength) {
	update(d, strength);
}

static void sepsp_accepter(slowEPSP *d, Synapse *s,
			   double strength, int window_id) {
	update(d, strength);
}

static void Isepsp_derivs(slowEPSP *self, double t) {
  double dsm1   = GETSTATE_DYN(self, 0);
  double dsm2   = GETSTATE_DYN(self, 1);
  double dsm3   = GETSTATE_DYN(self, 2);
  
  SETDERIV_DYN(self, 0, -self->beta1 * dsm1);
  SETDERIV_DYN(self, 1, dsm1*dsm1 - self->beta2 * dsm2);
  SETDERIV_DYN(self, 2, - dsm2 * dsm3 + self->beta3 * (1.0 - dsm3));
}

static double Isepsp_current(slowEPSP *self, double t) {
  double E = GETEM_DYN(self, t);
  double Er;
  double Gmax;
  /* Second state variable is the one we want. */
  double y;
  double I;
  
  if( !self->export_current ) return 0;

  Er      = self->Er;
  Gmax    = self->Gmax;
  y       = GETSTATE_DYN(self,1);
  I = y * Gmax * (Er - E);

  return I;
}

static int init(slowEPSP *self) {
	printf("Warning: slowEPSP has not been tested "
		"since asynchornous messaging support was removed.\n");
	self->export_current = true;
	return 0;
}

static char *stateVars[] = {"dsm1", "dsm2", "dsm3"};
static char *derivVars[] = {"ddsm1dt", "ddsm2dt", "ddsm3dt"};
static char *traceVars[] = {"dsm1Trace", "dsm2Trace", "dsm3Trace"};

DynamicsDescriptor slowEPSPDescriptor = {
    "slowEPSP",
    "slow EPSP current. members: Gmax, Er, alpha, beta1, beta2, beta3",
    slowEPSP_members,
    0,
    3,
    stateVars,
    derivVars,
    traceVars,
    (derivsfcn*)Isepsp_derivs,
    (currentfcn*)Isepsp_current,
    (accepterfcn*)sepsp_accepter,
    (enqfcn*)enq_sepsp,
    0,
    sizeof(slowEPSP),
    0,
    (userinitfcn*)init,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(slowEPSPDescriptor)

static DynamicsDescriptor *userDynamics[] = {
    &slowEPSPDescriptor
};

static initproc LuserInitDynamics[] = {
    initslowEPSPDescriptor
};


MAKE_P3_MODEL(spsp, "Myenteric slow EPSP response")
