"""
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
"""

"""
Test voltage clamp mode
"""

import os, sys, time
sys.path = ['C:\Documents and Settings\evan\My Documents\Visual Studio Projects\parplex']+sys.path
from math import pi, exp
from p3 import *
from py2mat import Matwrap
from time import clock

####################################################
# Implement the mutant Na channel models in Python #
####################################################

class Passive(PyDynamics):
    def __init__(self, cell, Gleak, Vleak):
        PyDynamics.__init__(self, cell)
        self.Gleak = Gleak
        self.Vleak = Vleak
        
    def current(self, t):
        Em = self.owner.Em
        return self.Gleak * (self.Vleak - Em)
        
class Current(PyDynamics):
    def __init__(self, cell, start, end, Iinject):
        PyDynamics.__init__(self, cell)
        self.start   = start
        self.end     = end
        self.Iinject = Iinject
        
    def current(self, t):
        return self.Iinject * (self.start<=t and t<=self.end)

class K(PyDynamics):
    stateVars   = ['n']
    stateDerivs = ['dndt']
    
    def __init__(self, cell, GK, EK):
        PyDynamics.__init__(self, cell)
        self.GK = GK
        self.EK = EK
        
    def derivs(self, t):
        cell = self.owner
        Em   = cell.Em
        n    = self.n
        V = Em + 65;
        alpha = 0.07 * (V-47) / (1 - exp((V-47)/-6))
        beta  = 0.264 / exp((V-22)/40)
        self.dndt = alpha*(1-n) - beta*n
        return
        
    def current(self, t):
        cell = self.owner
        Em   = cell.Em
        GK   = self.GK
        EK   = self.EK
        n    = self.n
        return n**4 * GK * (EK - Em)

class Na_wild(PyDynamics):
    stateVars   = ['m', 'h', 's']
    stateDerivs = ['dmdt', 'dhdt', 'dsdt']
    
    def __init__(self, cell, GNa, ENa):
        PyDynamics.__init__(self, cell)
        self.GNa = GNa
        self.ENa = ENa

    def derivs(self, t):
        cell = self.owner
        V    = cell.Em
    
        # m gate
        m = self.m
        m_inf = 1/(1 + exp(-(V+21.2)/4.9))
        tau_m = 0.15
        self.dmdt  = (m_inf - m)/tau_m
    
        # h gate
        h = self.h
        h_inf = 1/(1 + exp((V+39.7)/7.7))
        tau_h = 20.1 * exp(-0.5*((V+61.4)/32.7)**2)
        self.dhdt  = (h_inf - h)/tau_h
    
        # s gate
        s = self.s
        s_inf = 1/(1 + exp((V+46.1)/5.4))
        tau_s = 1000*106.7*exp(-0.5*((V+52.7)/18.3)**2)
        self.dsdt  = (s_inf - s)/tau_s

    def current(self, t):
        cell = self.owner
        Em   = cell.Em
        m    = self.m
        h    = self.h
        s    = self.s
        GNa  = self.GNa
        ENa  = self.ENa
        return m**3 * h * s * GNa * (ENa - Em)

class VStep(PyDynamics):
    def __init__(self, cell):
        PyDynamics.__init__(self, cell);
        self.Vstep = -100
        self.start =   10
        self.hold  = -100
        self.end   =   60

    def voltage(self, time):
        if self.start<=time and time<=self.end:
            return self.Vstep
        else:
            return self.hold
        
def makecell():
    #################
    # Make the cell #
    #################
    cell = Cell()
    cell.surface_area = (1/100e6)/0.0005
    cell.capacitance = 10
    cell.APthreshold = -30
    cell.Em = -60
    cell.emtrace = True
    cell.soma.ClampMode = VoltageClamp

    cell.method = hines
    cell.timeSample = 0


    ########################################
    # Add dynamical properties to the cell #
    ########################################
    # Leak
    Gleak = 0.5
    Eleak = -60
    #Passive(cell, Gleak, Eleak)
    # Sodium
    GNa = 200
    ENa = 50
    d = Na_wild(cell, GNa, ENa)
    d.m, d.h, d.s = 0, 1, 1 # Initial conditions

    #####################
    # Add voltage clamp #
    #####################
    cell.clamp = VStep(cell)
    
    return cell
#

##################
# Make a network #
##################
cell = makecell()
cell.clamp.Vstep = -100

###############
# Run options #
###############
setTRfilename('trace')
gd = GD()
gd.duration  = 100
gd.tolerance = 1e-3
gd.network = [cell]

#####################
# Initialise matlab #
#####################
m = Matwrap()
m.closeOnDel = False
m.write('cd \'' + os.getcwd() + '\'')
print m.read()

#######
# Run #
#######
start = clock()
parplex(gd)
set_message_option('info')
message_print(info, ('made it in %gs\n' % (clock()-start)))
tracefile_close()
apfile_close()

m.write('figure(2);traceplot(\'trace.dat\')\n')
print m.read()
