// K-M Ion Channel Dynamics (Baker 2003)
//
// Copyright 2007 John L Baker. All rights reserved.
//
// This software is provided AS IS under the terms of the Open Source
// MIT License. See http://www.opensource.org/licenses/mit-license.php.
//
// File: ionchan_k_m_baker_2003.h
//
// Release:		1.0.0
// Author:		John Baker
// Updated:		14 July 2006
//
// Description:
//
// This header file contains the classes used to implement
// the K-M (muscarinic) channel. The model is derived using the
// kinetic scheme in Selyanko & Brown adapted to voltage sensitivity
// found in CA1 along with time constants from Halliwell and Adams.
//
// The Q10 value for Halliwell and Adams is fairly large (5) and
// even the change from 30 to 37 degrees is significant as far as
// speed of deactivation is concerned.
//
// Selyanko and Brown show a [Ca++]-external sensitivity, especially 
// in slope (k). For the [Ca++]=0 case, the voltage response is similar.
// to that reported for CA1. However, this results in significant channel
// open probability near rest, which is not reported. Voltage response
// for [Ca++]-ext = 2mM in Selyanko and Brown is used here instead.
//
// References:
//
// Chen X and Johnston D (2004).Properties of single voltage-dependent
// K+ channels in dendrites of CA1 pyramidal neurones of rat hippocampus.
// J. Physiology 559, 187-203.
// 
// Halliwell JV, Adams PR (1982). Voltage-clamp analysis of muscarinic
// excitation in hippocmapal neurons. Brain Res. 250:71-92.
//
// Hu H, Vervaeke K, and Storm J. (2002). Two forms of electrical 
// resonance at theta frequencies, generated by M-current, h-current
// and persistent Na+ current in rat hippocampal pyramidal cells.
// J. Physiology 545, 783-805.
//
// Selyanko AA and Brown DA (1999). M-Channel gating and simulation.
// Biophysical Journal 77, 701-713.
//
// Yamada WM, Koch C, and Adams PR (1999). Multiple channels and 
// calcium dynamics, in Methods in Neuronal Modeling, 2nd edition,
// ed. by C. Koch and I Segev. Cambridge: MIT Press.

 
// Only include this header once
#ifndef __IONCHAN_K_M_BAKER_2003_H_
#define __IONCHAN_K_M_BAKER_2003_H_

#include "bnsf.h"

using namespace std;
using namespace BNSF;

namespace BAKER_2003 {

	// K-M channel

	class K_M_channel : public Order1EnergyBarrierTabChannel {
	
	public:
		// Constructors and destructor
		K_M_channel(Number gSpVal=0) : Order1EnergyBarrierTabChannel(gSpVal) {}
		virtual ~K_M_channel() {}

		// Temperature parameters from Halliwell and Adams.
		virtual Number	ratedTempC()	{return  30; }
		virtual Number	Q10()			{return  5; }

		// Voltage sensitivity from Selyanko & Brown ([Ca++]=2mM)
		virtual Number	Vhalf()			{return	-39.8*UOM::mV; }
		virtual Number	slope()			{return  5.2*UOM::mV; }

		// Rough fit starting with voltage sensitivity of CL and CS
		// states in Selyanko & Brown. kalpha and kbeta are taken
		// from the slopes in fig 5 for these states (19 and 46
		// in log(ms)/mV). The values in Halliwell and Adams at
		// -55 and -45 mV (47 and 92 msec) are used to set tauMax
		// and simultaneously adjust kbeta to achieve a fit.

		virtual Number	tauMax()		{return  150*UOM::msec; }
		virtual Number	tauMin()		{return  0*UOM::msec; }
		virtual Number	kalpha()		{return	 45*UOM::mV; }
		virtual Number	kbeta()			{return	 10*UOM::mV; }

		// Special time constant computation
		virtual Number	tauForTable(Number v);

		// State vector label functions
		virtual const char*	componentName() {return "K_M"; }
		virtual const char** stateLabels()	{
			static const char* sl[] = { "x" }; return sl; }

		// Reversal potential for K
		virtual Number Vrev()			{ return _Vrev; }
		
	protected:
		static const Number			_Vrev;
		virtual AlphaBetaEntry**	pAlphaBetaTable() { return &_abTable; }

	private:
		static AlphaBetaEntry*		_abTable;
	};
};

#endif // #ifndef __IONCHAN_K_M_BAKER_2003_H_
