#include "neuron.h"

void Neuron::setHeterogeneity(double sd) {
    int i;
    for(i=0;i<numberOfCells;i++)
      Q(x[i]) = normalrnd()*sd;
}

void Neuron::init() {
    int i,j;
    for(i=0;i<numberOfCells;i++)
      for(j=0;j<MAXDIM;j++) {
	x[i][j]=0;
	v_rep[i]=0;
	spike_flag[i]=0;
	spike_time[i]=0;
      }
}

void Neuron::correct_voltage(double x[]) {
    if(V(x)>VMax) {V(x)=VMax;}
    if(V(x)<VMin) {V(x)=VMin;}
    for(int i=1;i<MAXDIM;i++) {
        if(x[i]<0) x[i]=0;
        if(x[i]>1) x[i]=1;
    }                 
}

// tack spikes in the i-th cell
void Neuron::track_spikes(int i, double time) {
    if(V(x[i])>SPIKE_VOLTAGE && spike_flag[i]==0) {
       spike_time[i] = time;
       spike_flag[i] = 1;
    }
    if(V(x[i])<SPIKE_VOLTAGE && spike_flag[i]>0) {
       spike_time[i] = (spike_time[i] + time)/2;
       spike_flag[i] = 0;
    }
}

void Neuron::f(double x[], double y[]) {
    // here x are variables and y is the right hand side of dx/dt=f(x);
    double I_syn_int, I_syn_ext;
    double NT;
    double vthup,vthdown,vsl;
    I_syn_int = (g_int_syn*(total_s - S(x)) + g_int_slf*S(x))*(V(x)-E_rev) + g_int_elc*(total_v - numberOfCells*V(x));
    I_syn_ext = V(x)*sum_gsyn_totals - sum_gsyn_totals_erev + sum_gelc_totalv - sum_gelc_nofcells*V(x);
    V(y) = 1/C*(I_app - I_ion - I_syn_int - I_syn_ext);
    NT = 0.5*(1+TANH((V(x)-Vth)/Vsl));
    S(y) = als*(1-S(x))*NT-bes*S(x);
}



void Neuron::cycle(double time, double dt) {
    int i;
    total_s = total_v = 0;
    for(i=0; i<numberOfCells; i++) {
        total_s+= S(x[i]);
        total_v+= V(x[i]);
    }

    sum_gsyn_totals = sum_gsyn_totals_erev = 0;
    sum_gelc_totalv = sum_gelc_nofcells = 0;

    for(i=0; i<numberOfInputs; i++) {
      	sum_gsyn_totals+= 	g_ext_syn[i] * inputs[i]->total_s;
      	sum_gsyn_totals_erev+= 	g_ext_syn[i] * inputs[i]->total_s * inputs[i]->E_rev;
      	sum_gelc_totalv+= 	g_ext_elc[i] * inputs[i]->total_v;
      	sum_gelc_nofcells+=	g_ext_elc[i] * inputs[i]->numberOfCells;
    }

    for(i=0; i<numberOfCells; i++) {
      	RungeKutta(x[i], dt);
	    if(fabs(V(x[i])-v_rep[i])> DVC) need_report = 1;
        track_spikes(i,time);
    }
}

void Neuron::report(FILE *outfile, int report_mode) {
    int i;
    for(i=0; i<numberOfCells; i++) {
	    if(report_mode>=REPORT_VOLTAGE) {
	        fprintf(outfile, "%2.4lf\t", V(x[i]));
	        v_rep[i] = V(x[i]);
            if(report_mode>REPORT_VOLTAGE) {
                for(int j=1;j<MAXDIM;j++) fprintf(outfile, "%2.4lf\t", x[i][j]);
            }
	    }
    }
    need_report = 0;
}

void Neuron::reset() {
    int i,j;
    Neuron::init();
}

void Neuron::save(char *filename) {
    FILE *f;
    int i,j;
    f = fopen(filename, "w");
    fprintf(f,"%i\n",numberOfCells);
    for(i=0; i<numberOfCells; i++) {
        for(j=0; j<MAXDIM; j++) {
            fprintf(f,"%lf\t", x[i][j]);
        }
        fprintf(f,"\n");
    }
    fclose(f);        
}

void Neuron::load(char *filename) {
    FILE *f;
    int i,j;
    f = fopen(filename, "r");
    fscanf(f,"%i\n",&numberOfCells);
    for(i=0; i<numberOfCells; i++) {
        for(j=0; j<MAXDIM; j++) {
            fscanf(f,"%lf", &x[i][j]);
        }
    }
    fclose(f);        
}



void Neuron::letitgo(int i, int m, double dt) {
    // lets the i-th cell run without any synaptic inputs for m cycles
    int j;
    total_s = total_v = 0;
    sum_gsyn_totals = sum_gsyn_totals_erev = 0;
    sum_gelc_totalv = sum_gelc_nofcells = 0;

    for(j=0;j<m;j++) {
        RungeKutta(x[i], dt);
    }
}
  
void Neuron::desynchronize(int n, double dt) {
    int i;
    // lets all cells go free, each goes random number of cycles from 0 to n
    for(i=0;i<numberOfCells;i++) letitgo(i,rand()%n,dt);
}
  
  
void Neuron::RungeKutta(double x[], double dt) {
    double k1[MAXDIM], k2[MAXDIM], k3[MAXDIM], k4[MAXDIM];
    double x1[MAXDIM], x2[MAXDIM], x3[MAXDIM], x4[MAXDIM];
    int i;
    f(x, k1);
    for(i=0;i<MAXDIM;i++) x1[i]=x[i]+k1[i]*dt/2;
    f(x1, k2);
    for(i=0;i<MAXDIM;i++) x2[i]=x[i]+k2[i]*dt/2;
    f(x2, k3);
    for(i=0;i<MAXDIM;i++) x3[i]=x[i]+k3[i]*dt;
    f(x3, k4);
    for(i=0;i<MAXDIM;i++) x[i]+=(k1[i]+2*k2[i]+2*k3[i]+k4[i])*dt/6;

    correct_voltage(x);
}

void Neuron::addSynapseFrom(Neuron *presynaptic, double g_syn) {
    inputs[numberOfInputs]= presynaptic;
    g_ext_syn[numberOfInputs] = g_syn;
    g_ext_elc[numberOfInputs] = 0;
    numberOfInputs++;
};

void Neuron::addSynapseFrom(Neuron *presynaptic, double g_syn, double g_elc) {
    inputs[numberOfInputs]= presynaptic;
    g_ext_syn[numberOfInputs] = g_syn;
    g_ext_elc[numberOfInputs] = g_elc;
    numberOfInputs++;
};


// A freely running cell spikes, and the routine reports time it takes to get from S_max to 37% of that value
double Neuron::getdecaytime(double dt) {
    int i,k;
    long time = 0;
    long t1=0;
    long t2=0;
    double smax=0;

    total_s = total_v = 0;
    sum_gsyn_totals = sum_gsyn_totals_erev = 0;
    sum_gelc_totalv = sum_gelc_nofcells = 0;

    reset();
    while(1) {
    	RungeKutta(x[0], dt);

        if(S(x[0])>smax) {
	    smax = S(x[0]);
 	    t1 = time;
      	}
      	if(fabs(S(x[0])-0.37*smax)<1E-4) {
	    t2 = time;
      	}
      	time++;
      	if(t2>0) break;
    }
    return((t2-t1)*dt);
};

// Returns the period of a freely running i-th cell
double Neuron::getperiod(int i, double dt) {
    int k;
    double time=0;
    double t1;

    total_s = total_v = 0;
    sum_gsyn_totals = sum_gsyn_totals_erev = 0;
    sum_gelc_totalv = sum_gelc_nofcells = 0;

    spike_time[i]=0;
    spike_flag[i]=0;
    while(1) {
      	RungeKutta(x[i], dt);
	    track_spikes(i,time);
        time+=dt;
	    if(spike_time[i]>0 && spike_flag[i]==0) break;
    }
    t1 = spike_time[i];
    spike_time[i]=0;
    while(1) {
      	RungeKutta(x[i], dt);
	    track_spikes(i,time);
        time+=dt;
	    if(spike_time[i]>0 && spike_flag[i]==0) break;
    }
    return(time-t1);
}


void Network::addElement(Neuron *neuron) {
    neurons[networkSize]=neuron;
    networkSize++;
};
  
// This subroutine does one cycle of the entire network with time increment dt
void Network::cycle(double dt) {
    int i;
    for(i=0;i<networkSize;i++) neurons[i]->cycle(time, dt);
    report();
    time+=dt;
};


// This is just a reporting procedure; if at least one neuron needs report, the entire network reports
void Network::report() {
    int i;
    int flag=0;
    for(i=0;i<networkSize;i++) flag = flag | neurons[i]->need_report;
    if(!flag) return; 

    if(report_mode) fprintf(outfile, "%2.2lf\t",time);
    for(i=0;i<networkSize;i++) neurons[i]->report(outfile, report_mode);
    if(report_mode) fprintf(outfile, "\n");
}
  
void Network::reset() {
    int i;
    time = 0;
    for(i=0;i<networkSize;i++) neurons[i]->reset();
}


void Network::getprc(Neuron *presynaptic, int i, Neuron *postsynaptic, int j, double dt, double tau, double *delta, double *phi) {
    int k=0;
    double spike[3];
    time = 0;
    presynaptic->spike_time[i]=postsynaptic->spike_time[j]=0;
    presynaptic->spike_flag[i]=postsynaptic->spike_flag[j]=0;
    while(1) {
      if(fabs(tau-time)<dt/2) {
          presynaptic->I_app=UP;
      }
      if(fabs(tau-time+4)<dt/2) presynaptic->I_app=DOWN;
      cycle(dt);
      if(postsynaptic->spike_time[j]>0 && postsynaptic->spike_flag[j]==0  ) {
          spike[k++] = postsynaptic->spike_time[j];
          postsynaptic->spike_time[j]=0;
      }
      if(k>1) break;
    }
    *delta = presynaptic->spike_time[i]-spike[0];
    *phi = spike[1]-spike[0];
    
}


void Network::getprc1(Neuron *presynaptic, int i, Neuron *postsynaptic, int j, double dt, double tau, double *delta, double *phi) {
    int k=0;
    double spike[3];
    time = 0;
    presynaptic->spike_time[i]=postsynaptic->spike_time[j]=0;
    presynaptic->spike_flag[i]=postsynaptic->spike_flag[j]=0;
    while(1) {
      if(fabs(tau-time)<dt/2) {
          presynaptic->I_app=4;
      }
      if(fabs(tau-time+4)<dt/2) presynaptic->I_app=-20;
      cycle(dt);
      if(postsynaptic->spike_time[j]>0 && postsynaptic->spike_flag[j]==0  ) {
          spike[k++] = postsynaptic->spike_time[j];
          postsynaptic->spike_time[j]=0;
      }
      if(k>1) break;
    }
    *delta = presynaptic->spike_time[i]-spike[0];
    *phi = spike[1]-spike[0];
    
}

void Network::getprc2(Neuron *presynaptic, int i, Neuron *postsynaptic, int j, double dt, double tau, double *delta, double *phi) {
    int k=0;
    double spike[3];
    time = 0;
    presynaptic->spike_time[i]=postsynaptic->spike_time[j]=0;
    presynaptic->spike_flag[i]=postsynaptic->spike_flag[j]=0;
    while(1) {
      if(fabs(tau-time)<dt/2) {
          presynaptic->I_app=5;
      }
      if(fabs(tau-time+4)<dt/2) presynaptic->I_app=DOWN;
      cycle(dt);
      if(postsynaptic->spike_time[j]>0 && postsynaptic->spike_flag[j]==0  ) {
          spike[k++] = postsynaptic->spike_time[j];
          postsynaptic->spike_time[j]=0;
      }
      if(k>1) break;
    }
    *delta = presynaptic->spike_time[i]-spike[0];
    *phi = spike[1]-spike[0];
    
}

/*

void Network::gettrace(double dt) {
    int i,k;
    double f;
    time=0;
    while(1) {
      cycle(dt);
      if(neurons[0]->spike<1) continue;
      printf("%lf\t",time-neurons[0]->spiketime[0]);
      for(k=0;k<MAXDIM;k++) {
	if(k>0) {f=50;} else {f=1;}
	printf("%lf\t",f*neurons[0]->x[0][k]);
      }
      printf("\n");
      if(neurons[0]->spike>1) break;
    }
}

*/


void Network::run(int n, double dt) {
    int i,k;
    for(i=0;i<n;i++) {
      cycle(dt);
    }
}


void Network::set_outfile(char *filename) {
    outfile = fopen(filename,"w");
    if(outfile==NULL) {
        outfile = stdout;
        fprintf(stderr,"Can't open the file %s\n",filename);     
    }
    else {
        fprintf(stderr,"Output set to %s\n",filename);
    }
}
