/*********************************************************************
 * An implementation of a list of events that can vary in size.
 * Mike True, 8/20/06
 *********************************************************************/

#ifndef LISTH
    #include "list.h"
    #define LISTH 1
#endif

#include <stdlib.h>

/* Writes the contents of the List to a file */
void exportList(FILE *fp, List *list)
{
     int i;
     /* Dump the capacity, size, and time of the list to the file */
     fprintf(fp, "%d %d %f\n", list->capacity, list->size, list->time);
     /* Dump each event to the file in order */
     if(list->size > 0)
         for(i = list->front; i <= list->back; i = (i + 1)%(list->capacity))
         {
             exportEvent(fp, list->list[i]);
         }
}

/* Loads the contents of a List from a file */
List* importList(FILE *fp, Cell ***cells)
{
     int i, c, s;
     double t;
     List *list;
     /* Get the capacity, size, and time from the file */
     fscanf(fp, "%d %d %lf\n", &c, &s, &t);     
     list = createList(c);
     list->time = t; 
     /* Read in each event from the file */
     for(i=0; i<s; i++)
     {
         addToTail(list, importEvent(fp, cells));
     }            
     return list;
}

/* Adds an event to the end of the list */
void addToTail(List *list, Event *event)
{
    int pos;
    /* Check capacity */
    if(list->size >= list->capacity)
    {
        expandList(list, list->capacity * 2);
    }
    /* NOTE: This list is circular, so when empty start from 0 */
    if(list->size == 0)
    {
        list->front = 0;
        list->back = 0;
        list->list[list->back] = event;
        list->time = event->time;        
    }
    /* In general, add to one past the back of the list */
    else
    {
        list->back = (list->back + 1) % list->capacity;
        list->list[list->back] = event;
    }
    list->size = list->size + 1;
}

/* Removes the first event from the list */
Event* removeFromHead(List *list)
{
    Event *next = NULL;
    if(list->size == 0)
        return NULL;
    /* Get the next event to return */
    next = list->list[list->front];
    list->size = list->size - 1;
    /* Fix the position of the front of the list */
    if(list->size == 0)
    {
        list->front = -1;
        list->back = -1;
    }
    else
        list->front = (list->front + 1) % list->capacity;
    return next;
}

/* Get the time associated with events on this list */
double getTime(List *list)
{
    return list->time;
}

/* Allocate memory for a list with a specified capacity >= 1 */
List* createList(int initCapacity)
{
    List *newList;
    if(initCapacity <= 0)
        return NULL;
    /* Initialize the properties of the list */
    newList = (List *)malloc(sizeof(List));
    newList->list = (Event **)malloc(sizeof(Event *) * initCapacity);
    newList->size = 0;
    newList->front = -1;
    newList->back = -1;
    newList->time = -1.0;
    newList->capacity = initCapacity;
    return newList;
}

/* Destroy the contents of this list, including all events in it */
void destroyList(List *list)
{
    int i;
    for(i=0; i < list->size; i++)
    {
        if(list->list[i] != NULL)
            destroyEvent(list->list[i]);
    }
    free(list->list);
    free(list);
}

/* Allow more room for this list */
void expandList(List *list, int newCapacity)
{
    Event **newList;
    int i, j;
    if(newCapacity <= list->capacity)
	    return;
	/* Copy over the contents of the array */
    newList = (Event **)malloc(sizeof(Event *) * newCapacity);
    for(i=0, j=list->front; i < list->size; i++, j = (j+1 % list->capacity))
    {
        newList[i] = list->list[j];
    }
    /* Set the appropriate values for front and back */
    if(list->size == 0)
    {
        list->front = -1;
        list->back = -1;
    }
    else
    {
        list->front = 0;
        list->back = list->size -1;
    }
    /* Swap out the list */
    free(list->list);
    list->list = newList;
    list->capacity = newCapacity;
}
