/*********************************************************************
 * A set of operations which are used to determine and update the
 * voltage for a given cell at a given time.
 * Mike True, 8/20/06
 *********************************************************************/

#ifndef CELLH 
    #include "cell.h"
    #define CELLH 1 
#endif
#ifndef PARAMH
    #include "param.h"
    #define PARAMH 1
#endif

#include <math.h>
#include <stdio.h>   

/* Create a new cell in resting state*/
Cell* createCell(double creationTime, int x, int y, Param *p)
{
    Cell *newCell;
    newCell = (Cell *)malloc(sizeof(Cell));
    newCell->state = RESTING_STATE;
    newCell->xPos = x;
    newCell->yPos = y;
    newCell->Vx_old = p->initVX;
    newCell->Vx_cur = p->initVX;
    newCell->Vy_old = p->initVY;
    newCell->Vy_cur = p->initVY;
    newCell->Vz_old = p->initVZ;
    newCell->Vz_cur = p->initVZ;
    newCell->V_old = p->rp;
    newCell->V_cur = p->rp;
    newCell->Vn = INIT_Vn_VAL;
    newCell->time_cur = creationTime;
    newCell->time_old = creationTime;
    newCell->time_enter = creationTime;
    newCell->time_exit = creationTime;
    newCell->I_stim = 0.00000;  
    newCell->awake = 0;
    return newCell;
}

/* Populates a parameter structure with default values */
void initializeParamWithDefaults(Param *p)
{
    p->alphaX0 = ALPHA_X0;
    p->alphaY0 = ALPHA_Y0;
    p->alphaZ0 = ALPHA_Z0;
    
    p->alphaY1 = ALPHA_Y1;
    p->alphaZ1 = ALPHA_Z1;
    
    p->alphaX2 = ALPHA_X2;
    p->alphaY2 = ALPHA_Y2;
    p->alphaZ2 = ALPHA_Z2;
    
    p->alphaX3 = ALPHA_X3;
    p->alphaY3 = ALPHA_Y3;
    p->alphaZ3 = ALPHA_Z3;
    
    p->vO = VO;
    p->vT = VT;
    p->vS = VS;
    p->vR = VR;
    p->vW = VW;
    p->rp = RP;
    
    p->initVX = INIT_Vx_VAL;
    p->initVY = INIT_Vy_VAL;
    p->initVZ = INIT_Vz_VAL;
    
    p->guessUpstroke = GUESS_2;
    p->guessPlateau = GUESS_3;
}

/* Dump a cell to file */
void exportCell(FILE *fp, Cell *cell)
{
     fprintf(fp, "%d %d %d %f %f %f %f %f %f %f %f %f %f %f %f %f %f %d\n", 
       cell->state, cell->xPos, cell->yPos, cell->Vx_old, cell->Vx_cur,
       cell->Vy_old, cell->Vy_cur, cell->Vz_old, cell->Vz_cur, 
       cell->V_old, cell->V_cur, cell->Vn, cell->time_cur, cell->time_old,
       cell->time_enter, cell->time_exit, cell->I_stim, cell->awake);
}

/* Restore a cell from file */
Cell* importCell(FILE *fp)
{
     Cell* cell = (Cell *)malloc(sizeof(Cell));
     fscanf(fp, 
       "%d %d %d %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %d\n", 
       &(cell->state), &(cell->xPos), &(cell->yPos), &(cell->Vx_old), 
       &(cell->Vx_cur), &(cell->Vy_old), &(cell->Vy_cur), &(cell->Vz_old), 
       &(cell->Vz_cur), &(cell->V_old), &(cell->V_cur), &(cell->Vn), 
       &(cell->time_cur), &(cell->time_old), &(cell->time_enter), 
       &(cell->time_exit), &(cell->I_stim), &(cell->awake));
     return cell;
}

/* Destroy this cell */
void destroyCell(Cell *cell)
{
    free(cell);
}

/* Wakes a cell up, if necessary */
int notifyCell(Cell *cell, double time, Param *p)
{
    #if ALLOW_CELLS_TO_SLEEP
    if(cell->awake == 1)
	    return NO_ALERT;
    if((cell->xPos == 0) || (cell->xPos == p->size - 1) || (cell->yPos == 0) || 
      (cell->yPos == p->size - 1))
	    return NO_ALERT;
    cell->state = RESTING_STATE;
    cell->Vx_old = p->initVX;
    cell->Vx_cur = p->initVX;
    cell->Vy_old = p->initVY;
    cell->Vy_cur = p->initVY;
    cell->Vz_old = p->initVZ;
    cell->Vz_cur = p->initVZ;
    cell->V_old = p->rp;
    cell->V_cur = p->rp;
    cell->time_cur = time;
    cell->time_old = time;
    cell->time_enter = time;
    cell->time_exit = time;
    cell->awake = 1;
    return ALERT_NEIGHBORS;
    #else
    cell->awake = 1;
    return NO_ALERT;
    #endif
}

/* Puts a cell to sleep */
void putCellToSleep(Cell *cell)
{
    cell->awake = 0;
}

/* A function which reports the voltage of this cell at a given time */
double getVoltage(Cell *cell, double time, Param *p)
{
    double factor;
    #if ALLOW_CELLS_TO_SLEEP
    if(cell->awake == 0)
	    return p->rp;
    #endif
    switch(cell->state)
    {
        /* In first two states, find the appropriate time */
        case RESTING_STATE:
	        if(cell->time_cur == time)
	        {
                return cell->V_cur;
            }
            else
	        {
		        return cell->V_old;
            }
	        break;
        case STIMULATED_STATE: 
            if(cell->time_cur == time)
            {  
                return cell->V_cur;
            }
            else
            {
                return cell->V_old;
            }
            break;
        /* In second two states, use the appropriate explicit equation */
        case UPSTROKE_STATE:
            time = time - cell->time_enter;
            return (cell->Vx_old*exp(p->alphaX2 *time) -
              cell->Vy_old*exp(p->alphaY2*time) + cell->Vz_old*exp(p->alphaZ2*time)) 
              * 1e-3 + p->rp;
            break;
        case PLATEAU_STATE:  
            time = time - cell->time_enter;
            factor = 1 + 2*(cell->Vn / p->vR);
            return (cell->Vx_old*exp(p->alphaX3*time*factor) - 
              cell->Vy_old*exp(p->alphaY3*time) + cell->Vz_old*exp(p->alphaZ3*time)) 
              * 1e-3 + p->rp;
            break;
    }
    /* Should not occur */
    return 0.0;
}

/* A function which reports the approx current of this cell*/
double getCurrent(Cell *cell, Param *param)
{
    double time = param->dt;    
    if (time == 0.0)
        return 0.0;
    return (cell->V_cur - (((cell->Vx_cur - cell->Vy_cur + cell->Vz_cur)*1e-3) 
      + param->rp)) / time * param->Cm;
    //Note: Doesn't work for upstroke/plateau, but never is needed
}

/* Updates the voltage of a cell based upon its neighbors, returns next 
   query time */
int updateVoltage(Cell *cell, Cell *left, Cell *right, Cell *up, Cell *down,
  Param *p, double time, FILE *log)
{
    double newV, laplace, factor, current, stim;
    int alertCode = NO_ALERT;
    /* Get the external stimulation amount */
    stim = cell->I_stim;
    /* Calculate laplacian operatior */
    laplace = (getVoltage(left, time - p->dt, p) + 
      getVoltage(right, time - p->dt, p) + getVoltage(up, time - p->dt, p) + 
      getVoltage(down, time - p->dt, p) - 4.0 * cell->V_cur) / (p->dd * p->dd);
    /* Calculate the current */     
    current = getCurrent(cell, p);
 
    switch(cell->state)
    {
        case RESTING_STATE:   
            /* Get the three components together to find the new voltage */
            newV = cell->V_cur - p->dt *
	          ((1/p->Cm) * (current + stim) - (p->D * laplace));
	        if(newV < p->rp)
		        newV = p->rp;
	        /* Check for a transition to stimulated state */
            if((((newV - p->rp) * 1e3) - (cell->Vx_cur - cell->Vy_cur + 
              cell->Vz_cur)) > p->vS)
            {
                cell->V_old = cell->V_cur;
                cell->V_cur = newV;
                if(cell->Vx_cur - cell->Vy_cur + cell->Vz_cur <= p->vR)
		            cell->Vn = cell->Vx_cur - cell->Vy_cur + cell->Vz_cur;
                if(cell->Vn < 0.0)
		            cell->Vn = 0.0;
		        cell->state = STIMULATED_STATE;
		        cell->time_cur = time;
		        #if ALLOW_CELLS_TO_SLEEP
		        /* Check if sleep/alert is necessary */
		        if(cell->V_old < p->vW)
		        {
                  if(cell->V_cur >= p->vW)
                        alertCode = alertCode | ALERT_NEIGHBORS;
                }
		        else
		        {
                    if(cell->V_cur < p->vW)      
                        alertCode = alertCode | PUT_TO_SLEEP;             
                }
                #endif
                return alertCode;
            }
	        /* Case for staying in the resting state */            
            else
            {
                cell->V_old = cell->V_cur;
                cell->Vx_old = cell->Vx_cur;
                cell->V_cur = newV;
                factor = 1 + 2*(cell->Vn / p->vR);
		        cell->Vx_cur = cell->Vx_cur * 
                  (1 + ((p->alphaX0 * factor) * p->dt));
		        cell->Vy_cur = cell->Vy_cur * (1 + (p->alphaY0 * p->dt));
		        cell->Vz_cur = cell->Vz_cur * (1 + (p->alphaZ0 * p->dt));
                cell->time_cur = time;     
                #if ALLOW_CELLS_TO_SLEEP          
		        /* Check if sleep/alert is necessary */                   
		        if(cell->V_old < p->vW)
		        {
                    if(cell->V_cur >= p->vW)
                        alertCode = alertCode | ALERT_NEIGHBORS;
                }
		        else
		        {
                    if(cell->V_cur < p->vW)      
                        alertCode = alertCode | PUT_TO_SLEEP;             
                }
                #endif
                return alertCode;
            }
            break;
        case STIMULATED_STATE:
            /* Get the three components together to find the new voltage */            
            newV = cell->V_cur - p->dt *
              ((1/p->Cm) * (current + stim) - (p->D * laplace));
            if(newV < p->rp)
                newV = p->rp;
            /* Check to see if a change to upstroke state is necessary */
	        if((cell->Vx_cur - cell->Vy_cur + cell->Vz_cur) > 
              (p->vT*(1+1.45*(pow((cell->Vn / p->vR), (1.00/6.00))))))
            {
                cell->V_old = cell->V_cur;
                cell->V_cur = newV;       
		        cell->time_cur = time;
                cell->state = UPSTROKE_STATE;
                #if ALLOW_CELLS_TO_SLEEP
		        /* Alert that a change to upstroke is necessary */                
                return (ALERT_NEIGHBORS | UPSTROKE_ALERT);
                #else
                return (UPSTROKE_ALERT);
                #endif
            }
	        else
	        {
                cell->V_old = cell->V_cur;
                /* Case for staying in the stimulated state */
                if((((newV - p->rp) * 1e3) - 
                  (cell->Vx_cur - cell->Vy_cur + cell->Vz_cur)) > 0)
                {
                    cell->V_cur = newV;
                    cell->Vx_old = cell->Vx_cur;                    
		            cell->Vx_cur = ((newV - p->rp) * 1e3);
		            cell->Vy_cur = cell->Vy_cur * (1 + (p->alphaY1 * p->dt));
		            cell->Vz_cur = cell->Vz_cur * (1 + (p->alphaZ1 * p->dt));
                    cell->time_cur = time;   
                    #if ALLOW_CELLS_TO_SLEEP               
		            /* Check if sleep/alert is necessary */ 
                	if(cell->V_old < p->vW)
		            {
                        if(cell->V_cur >= p->vW)
                            alertCode = alertCode | ALERT_NEIGHBORS;
                    }
		            else
		            {
                        if(cell->V_cur < p->vW)      
                            alertCode = alertCode | PUT_TO_SLEEP;             
                    }
                    return alertCode | ALERT_NEIGHBORS;
                    #else
                    return alertCode;
                    #endif
                }
                /* Case for going back to the resting state */
		        else
		        {
                    cell->V_cur = newV;
		            cell->Vy_cur = cell->Vy_cur * (1 + (p->alphaY1 * p->dt));
		            cell->Vz_cur = cell->Vz_cur * (1 + (p->alphaZ1 * p->dt));
                    cell->state = RESTING_STATE;      
                    #if ALLOW_CELLS_TO_SLEEP      
		            /* Check if sleep/alert is necessary */                             
		            if(cell->V_old < p->vW)
		            {
                        if(cell->V_cur >= p->vW)
                           alertCode = alertCode | ALERT_NEIGHBORS;
                    }
		            else
		            {
                        if(cell->V_cur < p->vW)      
                            alertCode = alertCode | PUT_TO_SLEEP;             
                    }
                    #endif
                    return alertCode;
		        }
            }
            break;
    }
    cell->time_cur = time;
    /* Should be unreachable */
    #if ALLOW_CELLS_TO_SLEEP
	if(cell->V_old < p->vW)
	{
        if(cell->V_cur >= p->vW)
          alertCode = alertCode | ALERT_NEIGHBORS;
    }
	else
	{
        if(cell->V_cur < p->vW)      
          alertCode = alertCode | PUT_TO_SLEEP;             
    }
    #endif
    return alertCode;
}

/* Handles the first transition from stimulated to upstroke */
double beginUpstroke(Cell *cell, Param *p)
{
    if(cell->state == UPSTROKE_STATE)
    {
        /* Calculate the exit time for upstroke state */
        cell->time_enter = cell->time_cur;
        cell->time_exit = newton(p->alphaX2, p->alphaY2, p->alphaZ2, cell->Vx_cur,
          cell->Vy_cur, cell->Vz_cur, p->vO - (40 * sqrt(cell->Vn / p->vR)), 
          p->guessUpstroke, ACCURACY, p->dt) + cell->time_enter;
        /* Prepare variables for upstroke state */
        cell->Vx_old = cell->Vx_cur;    
        cell->Vy_old = cell->Vy_cur;
        cell->Vz_old = cell->Vz_cur;
        /* Calculate exit values */
        cell->Vx_cur = cell->Vx_cur * exp(p->alphaX2 * (cell->time_exit -
          cell->time_enter));
        cell->Vy_cur = cell->Vy_cur * exp(p->alphaY2 * (cell->time_exit -
          cell->time_enter));
        cell->Vz_cur = cell->Vz_cur * exp(p->alphaZ2 * (cell->time_exit -
          cell->time_enter));
        /* Return the next time of transition (to plateau) */
        return cell->time_exit;
    }
    /* Should be unreachable */
    return cell->time_exit;
}

/* Handle all state updates.  Return the time of the next event. */
double updateState(Cell *cell, Param *p)
{
    double factor;
    switch(cell->state)
    {
        /* Going to upstroke state */                       
	    case STIMULATED_STATE:
            /* Calculate the exit time for upstroke state */             
            cell->time_enter = cell->time_cur;
            cell->time_exit = newton(p->alphaX2, p->alphaY2, p->alphaZ2, cell->Vx_cur,
              cell->Vy_cur, cell->Vz_cur, p->vO - (40 * sqrt(cell->Vn / p->vR)), 
              p->guessUpstroke, ACCURACY, p->dt) + cell->time_enter;
            /* Prepare variables for upstroke state */              
	        cell->Vx_old = cell->Vx_cur;
	        cell->Vy_old = cell->Vy_cur;
	        cell->Vz_old = cell->Vz_cur;
            /* Calculate exit values */	        
            cell->Vx_cur = cell->Vx_cur * exp(p->alphaX2 * (cell->time_exit - 
              cell->time_enter));
            cell->Vy_cur = cell->Vy_cur * exp(p->alphaY2 * (cell->time_exit -
              cell->time_enter));
            cell->Vz_cur = cell->Vz_cur * exp(p->alphaZ2 * (cell->time_exit -
              cell->time_enter));              
            cell->state = UPSTROKE_STATE;
            /* Return the next time of transition (to plateau) */            
	        return cell->time_exit;
	        break;
        /* Going to plateau state */
        case UPSTROKE_STATE:
            /* Calculate the exit time for plateau state */                   
            cell->Vx_old = cell->Vx_cur;
            cell->Vy_old = cell->Vy_cur;
            cell->Vz_old = cell->Vz_cur;
            factor = 1 + 2*(cell->Vn / p->vR);
            cell->time_enter = cell->time_exit;
            cell->time_exit = newton(p->alphaX3 * factor, p->alphaY3, p->alphaZ3, 
              cell->Vx_cur, cell->Vy_cur, cell->Vz_cur, p->vR, p->guessPlateau, 
              ACCURACY, p->dt) + cell->time_enter;
            /* Calculate exit values */              
            cell->Vx_cur = cell->Vx_cur * exp(p->alphaX3 * factor * 
	          (cell->time_exit - cell->time_enter));
            cell->Vy_cur = cell->Vy_cur * exp(p->alphaY3 * (cell->time_exit -
              cell->time_enter));
            cell->Vz_cur = cell->Vz_cur * exp(p->alphaZ3 * (cell->time_exit -
              cell->time_enter));
            cell->state = PLATEAU_STATE;
            /* Return the next time of transition (to resting) */            
            return cell->time_exit;
	        break;
        /* Going to resting state */	        
	    case PLATEAU_STATE:
            /* Fix cell variables */ 
            cell->Vx_old = cell->Vx_cur;
            cell->Vy_old = cell->Vy_cur;
            cell->Vz_old = cell->Vz_cur;
            cell->V_old = ((cell->Vx_old - cell->Vy_old + cell->Vz_old)*1e-3) 
              + p->rp;
            cell->V_cur = ((cell->Vx_cur - cell->Vy_cur + cell->Vz_cur)*1e-3) 
              + p->rp;
            cell->time_cur = cell->time_exit;
	        cell->time_old = cell->time_exit - p->dt;
	        cell->state = RESTING_STATE;
            /* Return the next time of query neighbors */	        
            return cell->time_exit + p->dt;
	    break;
    }
    return 0.0;
}

/******************************************************************
 * Newton's algorithm for solving for the exit time
 * a1: alpha for Vx
 * a2: alpha for Vy
 * a3: alpha for Vz
 * c1: initial value for Vx
 * c2: initial value for Vy
 * c3: initial value for Vz
 * c4: target value for Vx - Vy + Vz
 * guess: initial guess for time
 * accuracy: when a new approx changes the value by less than this,
 *   stop the algorithm.
 ******************************************************************/
double newton(double a1, double a2, double a3, double c1, double c2, double c3,
  double c4, double guess, double accuracy, double dt)
{
    double f, df;
    double t_n = guess, t_n1, delta = accuracy, dt_n;
    double b1 = a1*c1, b2 = a2*c2, b3 = a3*c3;
    int n = 0;
    f = c1*exp(a1*t_n) - c2*exp(a2*t_n) + c3*exp(a3*t_n) - c4;
    df = b1*exp(a1*t_n) - b2*exp(a2*t_n) + b3*exp(a3*t_n);
    t_n1 = t_n - (f / df);
    dt_n = t_n1 - t_n;
    while(delta < (dt_n > 0.0000? dt_n : dt_n * -1.0000))
    {
        n++;
        t_n = t_n1;
        f = c1*exp(a1*t_n) - c2*exp(a2*t_n) + c3*exp(a3*t_n) - c4;
        df = b1*exp(a1*t_n) - b2*exp(a2*t_n) + b3*exp(a3*t_n);
        t_n1 = t_n - (f / df);
        dt_n = t_n1 - t_n;
    }
    t_n = t_n1;
    return ((int)(t_n / dt) + 1) * dt;
}
