/*********************************************************************
 * An implementation of a min-heap priority queue where a lower time 
 * for an event corresponds to a higher priority.
 * Mike True, 8/20/06
 *********************************************************************/

#ifndef HEAPH
    #include "heap.h"
    #define HEAPH 1
#endif

#include <stdlib.h>

/* A function used to save the Queue to a file */
void exportHeap(FILE *fp, Heap *heap)
{
     int i, s = 0;
     /* Get the number (s) of non-stimulus events */
     for(i = 0; i < heap->size; i++)
         if(heap->heap[i]->type == STATE_TRANSITION || 
           heap->heap[i]->type == DUMP_AP_TO_FILE)
             s++;
     /* Dump the size and capacity to a file */
     fprintf(fp, "%d %d\n", heap->capacity, s);
     /* Dump all non-stimulus events to a file */     
     for(i = 0; i < heap->size; i = (i + 1))
         if(heap->heap[i]->type == STATE_TRANSITION || 
           heap->heap[i]->type == DUMP_AP_TO_FILE)
             exportEvent(fp, heap->heap[i]);
}

/* A function used to save the Queue from a file */
Heap* importHeap(FILE *fp, Cell ***cells)
{
     int i, c, s;
     Heap *heap;
     fscanf(fp, "%d %d\n", &c, &s);
     /* Get the size and capacity of the heap */
     heap = createHeap(c);
     /* Read in each event */
     for(i=0; i<s; i++)
         insert(heap, importEvent(fp, cells));
     return heap;
}

/* A method for inserting an event into the heap */
void insert(Heap *heap, Event *event)
{
    int pos;
    Event *temp;
    /* Check the heap size */
    if(heap->size >= heap->capacity)
    {
        expandHeap(heap, heap->capacity * 2);
    }
    pos = heap->size;
    heap->size = pos + 1;
    heap->heap[pos] = event;
    /* Fix the heap */
    while((pos > 0) && 
      (heap->heap[(pos-1)/2]->time > heap->heap[pos]->time) ||
      ((heap->heap[(pos-1)/2]->time == heap->heap[pos]->time) &&
      (heap->heap[(pos-1)/2]->priority < heap->heap[pos]->priority)))
    {
        /* Perform the swap */
        temp = heap->heap[(pos-1)/2];
	    heap->heap[(pos-1)/2] = heap->heap[pos];
	    heap->heap[pos] = temp;
	    /* Move to the next position */
	    pos = (pos-1)/2;
    }
}

/* A method for removing the next event from the heap */
Event* removeNext(Heap *heap)
{
    Event *next, *temp;
    int pos, best, resume = 1;
    if(heap->size == 0)
        return NULL;
    /* Get the next event to be returned */
    next = heap->heap[0];
    heap->size = heap->size - 1;
    heap->heap[0] = heap->heap[heap->size];
    pos = 0;
    /* Fix the heap */
    while(((pos * 2 + 1) < heap->size) && (resume == 1))
    {
        best = pos;
        /* Determine if the left child has higher priority */
        if((heap->heap[pos * 2 + 1]->time < heap->heap[pos]->time)
          || ((heap->heap[pos * 2 + 1]->time == heap->heap[pos]->time)
          && (heap->heap[pos * 2 + 1]->priority >= 
          heap->heap[pos]->priority)))
            best = pos * 2 + 1;
        /* Determine if the right child has higher priority */
        if(((pos * 2 + 2) < heap->size) &&
          ((heap->heap[pos * 2 + 2]->time < heap->heap[best]->time)
          || ((heap->heap[pos * 2 + 2]->time == heap->heap[best]->time) 
          && (heap->heap[pos * 2 + 2]->priority >= 
          heap->heap[best]->priority))))
	        best = pos * 2 + 2;
        if(best == pos)
	        resume = 0;
	    else
	    {
            /* Perform the swap */
            temp = heap->heap[best];
            heap->heap[best] = heap->heap[pos];
            heap->heap[pos] = temp;
            /* Move to the next position */
            pos = best;
	    }
    }
    return next;
}

/* Return the next time an event will occur */
double getNextTime(Heap *heap)
{
    if(heap->size == 0)
        return -1.0;
    return heap->heap[0]->time;
}

/* Allocate memory for a min-heap with a specified capacity >= 1 */
Heap* createHeap(int initCapacity)
{
    Heap *newHeap;
    if(initCapacity <= 0)
        return NULL;
    newHeap = (Heap *)malloc(sizeof(Heap));
    newHeap->heap = (Event **)malloc(sizeof(Event *) * initCapacity);
    newHeap->size = 0;
    newHeap->capacity = initCapacity;
    return newHeap;
}

/* Destroy the contents of this min-heap, including all events in it */
void destroyHeap(Heap *heap)
{
    int i;
    for(i=0; i < heap->size; i++)
    {
        if(heap->heap[i] != NULL)
            destroyEvent(heap->heap[i]);
    }
    free(heap->heap);
    free(heap);
}

/* Allow more room for this heap */
void expandHeap(Heap *heap, int newCapacity)
{
    Event **newHeap;
    int i;
    if(newCapacity <= heap->capacity)
	    return;
    /* Perform a simple array copy procedure */
    newHeap = (Event **)malloc(sizeof(Event *) * newCapacity);
    for(i=0; i < heap->size; i++)
    {
        newHeap[i] = heap->heap[i];
    }
    free(heap->heap);
    heap->heap = newHeap;
    heap->capacity = newCapacity;
}

