/*********************************************************************
 * The driver program for our two-dimensional simulation implemented
 * using the event-driven model.
 * Mike True, 8/20/06
 *********************************************************************/

#ifndef PARAMH
    #include "param.h"
    #define PARAMH 1  
#endif
#ifndef CELLH
    #include "cell.h"
    #define CELLH 1
#endif
#ifndef EVENTH
    #include "event.h"
    #define EVENTH 1
#endif
#ifndef QUEUEH
    #include "queue.h"
    #define QUEUEH 1
#endif

#include <stdio.h>
#include <math.h>  
#include <stdlib.h>
#include <time.h>

#define OUTPUT_DEBUG_INFO 0
#define STATE_SNAPSHOTS_ENABLED 0
#define STATE_SNAPSHOTS_INTERVAL 10
#define true 1
#define false 0

/* Function for saving the cell array to file */
void exportCells(FILE *fp, Cell ***cells, int size)
{
    int i, j;
    fprintf(fp, "%d\n", size);
    for(i=0; i<size; i++)
        for(j=0; j<size; j++)
            exportCell(fp, cells[i][j]);
}

/* Function for reading the cell array from a file */
Cell*** importCells(FILE *fp)
{
    int i, j, size;
    Cell ***cells;
    fscanf(fp, "%d\n", &size);
    cells = (Cell***)malloc(sizeof(Cell**)*size);
    for(i=0;i<size;i++)
    {
        cells[i] = (Cell**)malloc(sizeof(Cell*)*size);
    }
    for(i=0; i<size; i++)
        for(j=0; j<size; j++)
        {
            cells[i][j] = importCell(fp);
        }
    return cells;
}

#if STATE_SNAPSHOTS_ENABLED
/* Procedure for generating file names for snapshot files */
char* fileStr(int num)
{
    char *result = (char *)malloc(sizeof(char) * 30);
    if(num == 0)
        sprintf(result, "Output\\Dump\\snap0000.ppm", num);  
    else if(num < 10)
        sprintf(result, "Output\\Dump\\snap000%d.ppm", num);          
    else if(num < 100)
        sprintf(result, "Output\\Dump\\snap00%d.ppm", num);
    else if(num < 1000)
        sprintf(result, "Output\\Dump\\snap0%d.ppm", num);        
    else
        sprintf(result, "Output\\Dump\\snap%d.ppm", num);    
    printf("%s\n", result);
    return result;
}
#endif

int main(int argc, char *argv[])
{
    /*Create all handles on files and read in simulation input*/
    FILE *fp, *AP, *log, *state;

    Param *p = (Param *) malloc(sizeof(Param));
    initializeParamWithDefaults(p);
    /* Give some default names for getting input/output files */
    p->inputFileName = "testdata.txt";
    p->outputFileName = "AP.dat";
    p->logFileName = "log.txt";

    #if CELL_REPORTS_ENABLED
    p->cellReportFileName = "report.txt";
    p->cellReportsOn = 0;
    #endif

    /* Handle all special command line arguments here */
    processCommandLineArgs(p, argv, argc);

    /* Try to open the input file */
    fp = fopen(p->inputFileName, "r");        
    if(fp == NULL)
    {
        printf("Error while opening input file...\n");
        exit(1);
    }
    readParam(p,fp);

    #if CELL_REPORTS_ENABLED
    if(p->cellReportsOn)
        readCellReportInfo(p, fopen(p->cellReportFileName,"r"));
    int fileCounter;
    #endif

    /* Load state file information if used */
    if(p->useStateFile)
    {
        readStateInfo(p, fopen(p->stateFileName,"r"));
    }

    /* Prepare the output file */
    AP = fopen(p->outputFileName,"w");
    if(AP == NULL)
    {
        printf("Could not create the data file\n");
	    exit(1);
    }

    /* Prepare the log file */
    log = fopen(p->logFileName,"w");
    if(log == NULL)
    {
        printf("Could not create the log file\n");
	    exit(1);
    }

    fclose(fp);

    //Initialize our array of cells
    Cell ***cells;
    double t = p->simBegin;
    int i, j, alert, wake;

    if(p->loadFromState)
    {
        /* Import old cell states if specified */
        state = fopen(p->loadStateFileName, "r");
        if(state == NULL)
        {
            printf("Could not load the state file\n");
	        exit(1);
        }
        fscanf(state, "%lf\n", &t); 
        cells = importCells(state);
    }
    else
    {
        /* Otherwise, create a new cell array */
        cells = (Cell***)malloc(sizeof(Cell**)*p->size);
        for(i=0;i<p->size;i++)
        {
            cells[i] = (Cell**)malloc(sizeof(Cell*)*p->size);
        }
        for(j=0;j<p->size;j++)
        {
            for(i=0;i<p->size;i++)
            {
	            cells[i][j] = createCell(p->simBegin, i, j, p); //Resting
            }
        }
        fprintf(AP,"\n");
    }
    //Carry out the actual simulation	
    time_t t_begin,t_end, t_mid;
    time(&t_begin);

    int left, right, top, bottom;  //Retangle boundaries for stimulus

    double eventTime, nextTime, stimCurrent;
    int position, posX, posY;
    Cell *eventCell;
    Event *event, *tempEvent;
    Queue *queue;
    
    if(p->loadFromState)
    {                   
        /* Import the queue if necessary */
        queue = importQueue(state, cells); 
    }
    else
    {
        /* Otherwise, initialize a new queue */
        queue = createQueue((p->size + 1) * (p->size + 1));
        enqueue(queue, createEvent(DUMP_AP_TO_FILE, p->simBegin, NULL));
    }

    for(i=0; i < p->stimNum; i++)
    {
        /* Create all stimulus events */
        tempEvent = createEvent(STIMULUS_ON, p->stimuli[i].begin, NULL);
        provideStimulusInfo(tempEvent, p->stimuli[i].left, 
          p->stimuli[i].right, p->stimuli[i].upper, p->stimuli[i].lower, 
          p->stimuli[i].amp);
        enqueue(queue, tempEvent);
        tempEvent = createEvent(STIMULUS_OFF, p->stimuli[i].end, NULL);
        provideStimulusInfo(tempEvent, p->stimuli[i].left, 
          p->stimuli[i].right, p->stimuli[i].upper, p->stimuli[i].lower, 
          p->stimuli[i].amp);
        enqueue(queue, tempEvent);
    }
    /* Add all state events if necessary */
    if(p->useStateFile)
    {
        for(i=0; i<p->stateNum; i++)
        {
            tempEvent = createEvent(SAVE_SYSTEM_STATE, p->stateTimes[i], NULL);
            tempEvent->stateNum = i;
            enqueue(queue, tempEvent);
        }
    }    
    /* Create initial events for cells if not loaded from a state */
    if(!p->loadFromState)
    {
        #if ALLOW_CELLS_TO_SLEEP
        #else
        for(i=1; i < p->size - 1; i++)
            for(j=1; j < p->size - 1; j++)
            {
                tempEvent = createEvent(QUERY_NEIGHBORS, p->simBegin + p->dt,
                  cells[i][j]);
                enqueue(queue, tempEvent);         
            }
        #endif
    }    
    #if STATE_SNAPSHOTS_ENABLED
    int fileCtr = 0;            
    FILE* snap;
    #endif

    event = dequeue(queue);
    eventTime = event->time;
    eventCell = event->cell;
    /* Remove events occurring before simulation begins */
    while (eventTime < t)
    {
  	    destroyEvent(event);	
        event = dequeue(queue);
        eventTime = event->time;
        eventCell = event->cell;          
    }
    /* Run until there the next event is after the simulation */
    while (eventTime < p->simEnd)
    {
        switch(event->type)
        {
            /* Handle the Query Neighbor event */
	        case QUERY_NEIGHBORS:
    		    posX = eventCell->xPos;
	    	    posY = eventCell->yPos;
	    	    /* Update the cell's voltage, see what happens */
		        alert = updateVoltage(eventCell, cells[posX - 1][posY], 
                  cells[posX + 1][posY], cells[posX][posY - 1], 
                  cells[posX][posY + 1], p, eventTime, log);
		        nextTime = eventCell->time_cur + p->dt;
		        /* Stay in this state, add new Query Neighbor event */
		        if(alert == NO_ALERT)
		        {
                    enqueue(queue, createEvent(QUERY_NEIGHBORS, nextTime, 
                      eventCell));                 
                }
                #if ALLOW_CELLS_TO_SLEEP
                /* Wake neighbors if necessary */
		        else if((alert & ALERT_NEIGHBORS) != 0)
		        {
		            wake = notifyCell(cells[posX - 1][posY - 1], eventTime, p);
		            if(wake != NO_ALERT)
                        enqueue(queue, createEvent(QUERY_NEIGHBORS, eventTime + 
                          p->dt, cells[posX - 1][posY - 1]));
		            wake = notifyCell(cells[posX - 1][posY + 1], eventTime, p);
		            if(wake != NO_ALERT)
                        enqueue(queue, createEvent(QUERY_NEIGHBORS, eventTime + 
                          p->dt, cells[posX - 1][posY + 1]));
		            wake = notifyCell(cells[posX + 1][posY + 1], eventTime, p);
		            if(wake != NO_ALERT)
                        enqueue(queue, createEvent(QUERY_NEIGHBORS, eventTime + 
                          p->dt, cells[posX + 1][posY + 1]));
		            wake = notifyCell(cells[posX + 1][posY - 1], eventTime, p);
		            if(wake != NO_ALERT)
                        enqueue(queue, createEvent(QUERY_NEIGHBORS, eventTime + 
                          p->dt, cells[posX + 1][posY - 1]));
                    if((alert & UPSTROKE_ALERT) != 0)
		            {
		                nextTime = beginUpstroke(eventCell, p);
                        enqueue(queue, createEvent(STATE_TRANSITION, nextTime,
		                  eventCell));
	 	            }
                    else
                        enqueue(queue, createEvent(QUERY_NEIGHBORS, nextTime, 
                          eventCell));
		        }
                #endif
                /* Move to upstroke state */
                else if((alert & UPSTROKE_ALERT) != 0)
		        {               
		            nextTime = beginUpstroke(eventCell, p);
                    enqueue(queue, createEvent(STATE_TRANSITION, nextTime, 
                      eventCell));
	 	        }
	 	        #if ALLOW_CELLS_TO_SLEEP
	 	        /* Put cell to sleep */
		        else if((alert & PUT_TO_SLEEP) != 0)
		        {
		            putCellToSleep(eventCell);
		        }
                #endif
                break;
            /* Handles a switch from one state to another */
            case STATE_TRANSITION:
                /* Get the next event time and create the appropriate event */
                nextTime = updateState(eventCell, p);
                if(eventCell->state == RESTING_STATE)
		        {
                    enqueue(queue, createEvent(QUERY_NEIGHBORS, nextTime,
                      eventCell));
		        }
		        else
		        {
                    enqueue(queue, createEvent(STATE_TRANSITION, nextTime,
                      eventCell));
		        }
		        break;
	        /* Handle the output to file, create a new event */
            case DUMP_AP_TO_FILE:
                for(j=0;j<p->size;j++)
                {   
                    for(i=0;i<p->size;i++)
                    {
                    #if OUTPUT_DEBUG_INFO
                        fprintf(AP,"((%d,%d):%d)%f ", i, j, cells[i][j]->state,
                        getVoltage(cells[i][j], eventTime));
                    #else
                        fprintf(AP,"%f ",getVoltage(cells[i][j], eventTime, p));                    
                    #endif
                    }
                }
                fprintf(AP,"\n");          
                printf("Time: %f\n", eventTime);

                #if STATE_SNAPSHOTS_ENABLED
                //This will dump periodic images to an output file in ppm format
                if(fileCtr % STATE_SNAPSHOTS_INTERVAL == 0)
                {
	                snap = fopen(fileStr(fileCtr), "w");
                    if(snap == NULL)
	                {
		                printf("Invalid file name");
		                exit(1);
                    }
	                else
	                {	
		                fprintf(snap, "P6\n%d %d\n255%c", p->size, p->size, 
                          '\r');
		                for(int i=0; i < p->size; i++)
			                for(int j=0; j < p->size; j++)
			                {
				                for(int k=0; k <=2; k++)
				                    switch(cells[j][i]->state)
				                    {
                                        case RESTING_STATE:
                                            fprintf(snap, "%c", 0); 
                                            break;
                                        case STIMULATED_STATE:
                                            fprintf(snap, "%c", 64); 
                                            break;
                                        case UPSTROKE_STATE:
                                            fprintf(snap, "%c", 255); 
                                            break;
                                        case PLATEAU_STATE:
                                            fprintf(snap, "%c", 191); 
                                            break;                   
                                    }		
			                }
		                    fclose(snap);
                    }                           
                }
                fileCtr++;
                #endif
                
                #if CELL_REPORTS_ENABLED
                for(fileCounter = 0; fileCounter < p->reportNum; fileCounter++)
		            fprintf(p->cellReportFiles[fileCounter], "%f %f\n",
		              eventTime, getVoltage(cells[p->cellReportsX[fileCounter]]
                      [p->cellReportsY[fileCounter]], eventTime, p));
                #endif

                enqueue(queue, createEvent(DUMP_AP_TO_FILE, eventTime+
                  (p->iter * p->dt), NULL));
	            break;
	        /* Handles the beginning of an external stimulus */    
            case STIMULUS_ON:
                for(i=event->left; i<=event->right; i++)
                {
                    for(j=event->top; j<=event->bottom; j++)
                    {
                        /* Apply the stimulus to the cells affected */
                        cells[i][j]->I_stim = cells[i][j]->I_stim + 
                          event->amount;
                        #if ALLOW_CELLS_TO_SLEEP
                        /* Wake cells if necessary */
                        wake = notifyCell(cells[i][j], eventTime, p);                  
                        if(wake != NO_ALERT)
                            enqueue(queue, createEvent(QUERY_NEIGHBORS, 
                              eventTime, cells[i][j]));
                        #endif
                    }
                }
		        break;	        
             /* Handles the end of an external stimulus */    
	        case STIMULUS_OFF:               
                for(i=event->left; i<=event->right; i++)
                {
                    for(j=event->top; j<=event->bottom; j++)
                    {
                        /* Remove the stimulus from the cells affected */                                      
                        cells[i][j]->I_stim = cells[i][j]->I_stim - 
                          event->amount;
                    }
                }
		        break;
	        case SAVE_SYSTEM_STATE:
                 /* Perform a state dump by writing the time, cells, and queue*/
                 fprintf(p->stateFiles[event->stateNum], "%f\n", eventTime);
                 exportCells(p->stateFiles[event->stateNum], cells, p->size);
                 exportQueue(p->stateFiles[event->stateNum], queue);
                 break;
        }
        /* Get the next event */
  	    destroyEvent(event);	
        event = dequeue(queue);
        eventTime = event->time;
        eventCell = event->cell;
    }
    
    //Final printout
    for(j=0;j<p->size;j++)
    {
        for(i=0;i<p->size;i++)
        {
             #if OUTPUT_DEBUG_INFO
                 fprintf(AP,"((%d,%d):%d)%f ", i, j, cells[i][j]->state,
                   getVoltage(cells[i][j], p->simEnd));
             #else
                 fprintf(AP,"%f ",getVoltage(cells[i][j], p->simEnd, p));                    
             #endif
        }
    }
    fprintf(AP,"\n");    
    
    time(&t_end);

    /* Print the final time in the log */
    fprintf(log, "Time Full: %d\n", t_end - t_begin);

    /* Clean up resources, close file handles */
    for(i=0; i<p->size; i++)
        for(j=0; j<p->size; j++)
            destroyCell(cells[i][j]);
    free(cells);
    destroyQueue(queue);

    #if CELL_REPORTS_ENABLED
    for(fileCounter = 0; fileCounter < p->reportNum; fileCounter++)
        fclose(p->cellReportFiles[fileCounter]);
    #endif    

    if(p->useStateFile)
    {
        for(i=0; i<p->stateNum; i++)
        {
            fclose(p->stateFiles[i]);
        }
    }   

    return 0;
}
