/*********************************************************************
 * A set of operations which are used to determine and update the
 * voltage for a given cell at a given time.
 * Mike True, 8/20/06
 *********************************************************************/

#ifndef CELLH 
    #include "cell.h"
    #define CELLH 1 
#endif
#ifndef PARAMH
    #include "param.h"
    #define PARAMH 1
#endif

#include <math.h>
#include <stdio.h>   

/* Create a new cell in resting state*/
Cell* createCell(double creationTime, int x, int y, Param *p)
{
    Cell *newCell;
    newCell = (Cell *)malloc(sizeof(Cell));
    newCell->state = RESTING_STATE;
    newCell->xPos = x;
    newCell->yPos = y;
    newCell->Vx_old = p->initVX;
    newCell->Vx_cur = p->initVX;
    newCell->Vy_old = p->initVY;
    newCell->Vy_cur = p->initVY;
    newCell->Vz_old = p->initVZ;
    newCell->Vz_cur = p->initVZ;
    newCell->V_old = p->rp;
    newCell->V_cur = p->rp;
    newCell->Vn = INIT_Vn_VAL;
    newCell->time_cur = creationTime;
    newCell->time_old = creationTime;
    newCell->time_enter = creationTime;
    newCell->time_exit = creationTime;
    newCell->I_stim = 0.00000;
    newCell->awake = 0;
    return newCell;
}

/* Populates a parameter structure with default values */
void initializeParamWithDefaults(Param *p)
{
    p->alphaX0 = ALPHA_X0;
    p->alphaY0 = ALPHA_Y0;
    p->alphaZ0 = ALPHA_Z0;
    
    p->alphaY1 = ALPHA_Y1;
    p->alphaZ1 = ALPHA_Z1;
    
    p->alphaX2 = ALPHA_X2;
    p->alphaY2 = ALPHA_Y2;
    p->alphaZ2 = ALPHA_Z2;
    
    p->alphaX3 = ALPHA_X3;
    p->alphaY3 = ALPHA_Y3;
    p->alphaZ3 = ALPHA_Z3;
    
    p->vO = VO;
    p->vT = VT;
    p->vS = VS;
    p->vR = VR;
    p->rp = RP;
    
    p->initVX = INIT_Vx_VAL;
    p->initVY = INIT_Vy_VAL;
    p->initVZ = INIT_Vz_VAL;
}

/* Destroy this cell */
void destroyCell(Cell *cell)
{
    free(cell);
}

/* Wakes a cell up, if necessary */
int notifyCell(Cell *cell, double time, Param *p)
{
    if(cell->awake == 1)
	return NO_ALERT;
    if((cell->xPos == 0) || (cell->xPos == p->size - 1))
	return NO_ALERT;
    cell->state = RESTING_STATE;
    cell->Vx_old = p->initVX;
    cell->Vx_cur = p->initVX;
    cell->Vy_old = p->initVY;
    cell->Vy_cur = p->initVY;
    cell->Vz_old = p->initVZ;
    cell->Vz_cur = p->initVZ;    
    cell->V_old = p->rp;
    cell->V_cur = p->rp;
    cell->time_cur = time;
    cell->time_old = time;
    cell->time_enter = time;
    cell->time_exit = time;
    cell->awake = 1;
    return ALERT_NEIGHBORS;
}

/* Puts a cell to sleep */
void putCellToSleep(Cell *cell)
{
    cell->awake = 0;
}

/* A function which reports the voltage of this cell at a given time */
double getVoltage(Cell *cell, double time)
{
    if(cell->time_cur == time)
    {
        return cell->V_cur;
    }
    return cell->V_old;
}

/* A function which reports the approx current of this cell*/
double getCurrent(Cell *cell, Param *param)
{
    double time = cell->time_cur - cell->time_old;
    double cur;
    if(time == 0.0)
    {
        return 0.0;
    }
    cur = (cell->V_cur - (((cell->Vx_cur - cell->Vy_cur + cell->Vz_cur)*1e-3) 
      + param->rp)) / param->dt * param->Cm;
    return cur;
}

/* Updates the voltage of a cell based upon its neighbors, returns next 
   query time */
int updateVoltage(Cell *cell, Cell *left, Cell *right, Cell *up, Cell *down,
  Param *p, double time, FILE *log)
{
    double newV, laplace, factor, current, stim;
    int alertCode = NO_ALERT;
    stim = cell->I_stim;
    laplace = (getVoltage(left, time - p->dt) + getVoltage(right, time - p->dt)
      + getVoltage(up, time - p->dt) + getVoltage(down, time - p->dt) - 
      4.0 * cell->V_cur) / (p->dd * p->dd);
    current = getCurrent(cell, p);
    switch(cell->state)
    {
        case RESTING_STATE:   
            newV = cell->V_cur - p->dt *
	          ((1/p->Cm) * (current + stim) - (p->D * laplace));
	        if(newV < p->rp)
            {
                cell->Vx_old = p->initVX;
    		    cell->Vx_cur = p->initVX;
    		    cell->Vy_old = p->initVY;
    		    cell->Vy_cur = p->initVY;
    		    cell->Vz_old = p->initVZ;
    		    cell->Vz_cur = p->initVZ;
    		    cell->V_old = p->rp;
    		    cell->V_cur = p->rp;
		        newV = p->rp;
            }
            if((((newV - p->rp) * 1e3) - 
              (cell->Vx_cur - cell->Vy_cur + cell->Vz_cur)) > p->vS)
            {
                cell->V_old = cell->V_cur;
                cell->Vx_old = cell->Vx_cur;
                cell->Vy_old = cell->Vy_cur;
                cell->Vz_old = cell->Vz_cur;
		        cell->time_old = cell->time_cur;
                cell->V_cur = newV;
                if(cell->Vx_cur - cell->Vy_cur + cell->Vz_cur <= p->vR)                
		            cell->Vn = cell->Vx_cur - cell->Vy_cur + cell->Vz_cur;
                if(cell->Vn < 0.0)
		            cell->Vn = 0.0;
		        cell->state = STIMULATED_STATE;
                cell->time_cur = time;
                return alertCode;
            }
            else
            {
                cell->V_old = cell->V_cur;
                cell->Vx_old = cell->Vx_cur;
                cell->Vy_old = cell->Vy_cur;
                cell->Vz_old = cell->Vz_cur;
                cell->time_old = cell->time_cur;
                cell->V_cur = newV;
                factor = 1 + 2 * (cell->Vn / p->vR);
		        cell->Vx_cur = cell->Vx_cur * (1 + ((p->alphaX0 * factor) * p->dt));
		        cell->Vy_cur = cell->Vy_cur * (1 + (p->alphaY0 * p->dt));
		        cell->Vz_cur = cell->Vz_cur * (1 + (p->alphaZ0 * p->dt));
                cell->time_cur = time;                  
                return alertCode;
            }
            break;
        case STIMULATED_STATE:
            newV = cell->V_cur - p->dt *
              ((1/p->Cm) * (current + stim) - (p->D * laplace));
            if(newV < p->rp)
            {
                cell->Vx_old = p->initVX;
    		    cell->Vx_cur = p->initVX;
    		    cell->Vy_old = p->initVY;
    		    cell->Vy_cur = p->initVY;
    		    cell->Vz_old = p->initVZ;
    		    cell->Vz_cur = p->initVZ;
    		    cell->V_old = p->rp;
    		    cell->V_cur = p->rp;
		        newV = p->rp;
            }
	        if((cell->Vx_cur - cell->Vy_cur + cell->Vz_cur) > 
              (p->vT * (1 + (1.45 * pow((cell->Vn / p->vR), 1.00/6.00)))))
            {
                cell->V_old = cell->V_cur;
                cell->Vx_old = cell->Vx_cur;
                cell->Vy_old = cell->Vy_cur;
                cell->Vz_old = cell->Vz_cur;
                cell->time_old = cell->time_cur;
                cell->V_cur = newV;
                cell->Vx_cur = (1e3 * (cell->V_cur - p->rp));
		        cell->Vy_old = 0.001;
		        cell->Vy_cur = 0.001;
		        cell->Vz_old = 0.001;
		        cell->Vz_cur = 0.001;
		        cell->time_cur = time;
                cell->state = UPSTROKE_STATE;
                return UPSTROKE_ALERT;
            }
	        else
	        {
                cell->V_old = cell->V_cur;
                cell->Vx_old = cell->Vx_cur;
                cell->Vy_old = cell->Vy_cur;
                cell->Vz_old = cell->Vz_cur;
                cell->time_old = cell->time_cur;
                if((((newV - p->rp) * 1e3) - 
                  (cell->Vx_cur - cell->Vy_cur + cell->Vz_cur)) > 0)
                {
		            cell->Vy_cur = cell->Vy_cur * (1 + (p->alphaY1 * p->dt));
		            cell->Vz_cur = cell->Vz_cur * (1 + (p->alphaZ1 * p->dt));
                    cell->V_cur = newV;
		            cell->Vx_cur = ((newV - p->rp) * 1e3);
                    cell->time_cur = time;                  
                    return alertCode;
                }
		        else
		        {
		            cell->Vy_cur = cell->Vy_cur * (1 + (p->alphaY1 * p->dt));
		            cell->Vz_cur = cell->Vz_cur * (1 + (p->alphaZ1 * p->dt));
                    cell->V_cur = newV;
                    cell->state = RESTING_STATE;                    
                    cell->time_cur = time;
                    return alertCode;
		        }
	        }
            break;
        case UPSTROKE_STATE:
            if((cell->Vx_cur - cell->Vy_cur + cell->Vz_cur) > 
              (p->vO - 40 * sqrt(cell->Vn / p->vR)))
            {
                cell->V_old = cell->V_cur;
                cell->Vx_old = cell->Vx_cur;
                cell->Vy_old = cell->Vy_cur;
                cell->Vz_old = cell->Vz_cur;
                cell->time_old = cell->time_cur;  
                cell->V_cur = ((cell->Vx_cur - cell->Vy_cur + cell->Vz_cur)/1e3) 
                  + p->rp;
                cell->time_cur = time;            
                cell->state = PLATEAU_STATE;
                return alertCode;
            }
            else
            {
                cell->V_old = cell->V_cur;
                cell->Vx_old = cell->Vx_cur;
                cell->Vy_old = cell->Vy_cur;
                cell->Vz_old = cell->Vz_cur;
                cell->time_old = cell->time_cur;
                cell->Vx_cur = cell->Vx_cur * (1 + (p->alphaX2 * p->dt));           
                cell->Vy_cur = cell->Vy_cur * (1 + (p->alphaY2 * p->dt));
                cell->Vz_cur = cell->Vz_cur * (1 + (p->alphaZ2 * p->dt));
                cell->V_cur = ((cell->Vx_cur - cell->Vy_cur + cell->Vz_cur)/1e3) 
                  + p->rp;
                cell->time_cur = time;                
		        return alertCode;
            }
	        break;
        case PLATEAU_STATE:
            if((cell->Vx_cur - cell->Vy_cur + cell->Vz_cur) < p->vR)
            {
                cell->V_old = cell->V_cur;
                cell->Vx_old = cell->Vx_cur;
                cell->Vy_old = cell->Vy_cur;
                cell->Vz_old = cell->Vz_cur;
                cell->time_old = cell->time_cur;
                cell->V_cur = ((cell->Vx_cur - cell->Vy_cur + cell->Vz_cur)/1e3) 
                  + p->rp;
                cell->time_cur = time;
                cell->state = RESTING_STATE;
                return alertCode;
            }
            else
            {
                cell->V_old = cell->V_cur;
                cell->Vx_old = cell->Vx_cur;
                cell->Vy_old = cell->Vy_cur;
                cell->Vz_old = cell->Vz_cur;
                cell->time_old = cell->time_cur;
                factor = 1 + 2 * (cell->Vn / p->vR);
                cell->Vx_cur = cell->Vx_cur * (1 + (p->alphaX3 * factor * p->dt));
                cell->Vy_cur = cell->Vy_cur * (1 + (p->alphaY3 * p->dt));
                cell->Vz_cur = cell->Vz_cur * (1 + (p->alphaZ3 * p->dt));
                cell->V_cur = ((cell->Vx_cur - cell->Vy_cur + cell->Vz_cur)/1e3) 
                  + p->rp;
                cell->time_cur = time;
                return alertCode;
            }
            break;
    }
    cell->time_old = cell->time_cur;
    cell->time_cur = time;
    return alertCode;
}
