/*********************************************************************
 * The driver program for our two-dimensional simulation implemented
 * using the time-step integration model.
 * Mike True, 8/20/06
 *********************************************************************/

#ifndef PARAMH
    #include "param.h"
    #define PARAMH 1  
#endif
#ifndef CELLH
    #include "cell.h"
    #define CELLH 1
#endif

#include <stdio.h>
#include <math.h>  
#include <stdlib.h>
#include <time.h>

#define OUTPUT_DEBUG_INFO 0
#define STATE_SNAPSHOTS_ENABLED 0
#define STATE_SNAPSHOTS_INTERVAL 10
#define true 1
#define false 0

#if STATE_SNAPSHOTS_ENABLED
char* fileStr(int num)
{
    char *result = (char *)malloc(sizeof(char) * 30);
    if(num == 0)
        sprintf(result, "Output\\Dump\\snap0000.ppm", num);  
    else if(num < 10)
        sprintf(result, "Output\\Dump\\snap000%d.ppm", num);          
    else if(num < 100)
        sprintf(result, "Output\\Dump\\snap00%d.ppm", num);
    else if(num < 1000)
        sprintf(result, "Output\\Dump\\snap0%d.ppm", num);        
    else
        sprintf(result, "Output\\Dump\\snap%d.ppm", num);    
    printf("%s\n", result);
    return result;
}
#endif

int main(int argc, char *argv[])
{
    //Create all handles on files and read in simulation input
    FILE *fp, *AP, *log, *snap;
    int fileCtr = 0;
    
    Param *p = (Param *) malloc(sizeof(Param));
    initializeParamWithDefaults(p);    
    p->inputFileName = "testdata.txt";
    p->outputFileName = "AP.dat";
    p->logFileName = "log.txt";

    #if CELL_REPORTS_ENABLED
    p->cellReportFileName = "report.txt";
    p->cellReportsOn = 0;
    #endif

    processCommandLineArgs(p, argv, argc);

    fp = fopen(p->inputFileName, "r");        
    if(fp == NULL)
    {
        printf("Error while opening input file...\n");
        exit(1);
    }
    readParam(p,fp);

    #if CELL_REPORTS_ENABLED
    if(p->cellReportsOn)
        readCellReportInfo(p, fopen(p->cellReportFileName,"r"));
    int fileCounter;
    #endif

    AP = fopen(p->outputFileName,"w");
    if(AP == NULL)
    {
        printf("Could not create the data file\n");
	    exit(1);
    }

    log = fopen(p->logFileName,"w");
    if(log == NULL)
    {
        printf("Could not create the log file\n");
	    exit(1);
    }

    fclose(fp);

    //Initialize our array of cells
    Cell ***cells;
    double t = p->simBegin;
    int i, j, alert, wake;

    cells = (Cell***)malloc(sizeof(Cell**)*p->size);
    for(i=0;i<p->size;i++)
    {
        cells[i] = (Cell**)malloc(sizeof(Cell*)*p->size);
    }

    for(j=0;j<p->size;j++)
    {
        for(i=0;i<p->size;i++)
        {
	        cells[i][j] = createCell(p->simBegin, i, j, p);
             #if OUTPUT_DEBUG_INFO
                 fprintf(AP,"((%d,%d):%d)%f ", i, j, cells[i][j]->state,
                   getVoltage(cells[i][j], p->simBegin));
             #else
                 fprintf(AP,"%f ",getVoltage(cells[i][j], p->simBegin));                    
             #endif
        }
    }
    fprintf(AP,"\n");

    //Carry out the actual simulation	
    int midTimeSet = 0;
    time_t t_begin,t_end, t_mid;
    time(&t_begin);

    int left, right, top, bottom;  //Retangle boundaries for stimulus

    double nextTime, stimCurrent;
    int position, posX, posY;
    Cell *eventCell;

    int s=0;                              //The number of stimuli
    int a=0, b=0, c=0, d=0;               //Counters controling the output
    int l=0, k=0, ctr=0;
    int sti_on=false, sti_init = false;

    t = p->simBegin;
    while (t < p->simEnd)
    {
        //Test if there is stimulus
        if (sti_on == false && s<p->stimNum &&
          t-p->stimuli[s].begin>=0 &&
          t-p->stimuli[s].begin<=p->dt)
        {                        
            sti_on=true;
            if(!(sti_init))
	        {
                a = p->stimuli[s].left;
                b = p->stimuli[s].right;
                c = p->stimuli[s].upper;
                d = p->stimuli[s].lower;
                for(l=a; l<=b; l++)
                    for(k=c; k<=d; k++)
                        cells[l][k]->I_stim = cells[l][k]->I_stim + 
                          p->stimuli[s].amp;
		        sti_init = true;
	        }
        }    

        if(sti_on==true && t-p->stimuli[s].end <=p->dt &&
          t-p->stimuli[s].end>0)
        {   
            a = p->stimuli[s].left;
            b = p->stimuli[s].right;
            c = p->stimuli[s].upper;
            d = p->stimuli[s].lower;
            for(l=a; l<=b; l++)
                for(k=c; k<=d; k++)
                    cells[l][k]->I_stim = cells[l][k]->I_stim - 
                      p->stimuli[s].amp;
	        sti_init = false;
	        sti_on = false;
	        s++;
        }
        
        int alert;    
	    for(i=1; i<(p->size - 1); i++)
	    {
            for(j=1; j<(p->size -1);j++)
                alert = updateVoltage(cells[i][j], cells[i - 1][j], 
                  cells[i + 1][j], cells[i][j-1], cells[i][j+1], p, t, log);
        }
	    if(ctr != 0 && ctr % p->iter == 0)
        {            
            printf("%f\n", t);
            for(j=0;j<p->size;j++)
            {   
                for(i=0;i<p->size;i++)
                #if OUTPUT_DEBUG_INFO
                    fprintf(AP,"((%d,%d):%d)%f ", i, j, cells[i][j]->state,
                    getVoltage(cells[i][j], t));
                #else
                fprintf(AP,"%f ",getVoltage(cells[i][j], t));                    
                #endif
            }
            fprintf(AP,"\n");

                #if STATE_SNAPSHOTS_ENABLED
                //This will dump periodic images to an output file in ppm format
                if(fileCtr % STATE_SNAPSHOTS_INTERVAL == 0)
                {
	                snap = fopen(fileStr(fileCtr), "w");
                    if(snap == NULL)
	                {
		                printf("Invalid file name");
		                exit(1);
                    }
	                else
	                {	
		                fprintf(snap, "P6\n%d %d\n255%c", p->size, p->size, 
                          '\r');
		                for(int i=0; i < p->size; i++)
			                for(int j=0; j < p->size; j++)
			                {
				                for(int k=0; k <=2; k++)
				                    switch(cells[j][i]->state)
				                    {
                                        case RESTING_STATE:
                                            fprintf(snap, "%c", 0); 
                                            break;
                                        case STIMULATED_STATE:
                                            fprintf(snap, "%c", 64); 
                                            break;
                                        case UPSTROKE_STATE:
                                            fprintf(snap, "%c", 255); 
                                            break;
                                        case PLATEAU_STATE:
                                            fprintf(snap, "%c", 191); 
                                            break;                   
                                    }		
			                }
		                    fclose(snap);
                    }                           
                }
                fileCtr++;
                #endif

            #if CELL_REPORTS_ENABLED
            for(fileCounter = 0; fileCounter < p->reportNum; fileCounter++)
                fprintf(p->cellReportFiles[fileCounter], "%f %f\n",
                  t, getVoltage(cells[p->cellReportsX[fileCounter]]
                  [p->cellReportsY[fileCounter]], t));
            #endif
        }
	    t = p->dt + t;
        ctr++;
    }
       
    //Final printout
    for(j=0;j<p->size;j++)
    {
        for(i=0;i<p->size;i++)
        {
             #if OUTPUT_DEBUG_INFO
                 fprintf(AP,"((%d,%d):%d)%f ", i, j, cells[i][j]->state,
                   getVoltage(cells[i][j], p->simEnd));
             #else
                 fprintf(AP,"%f ",getVoltage(cells[i][j], p->simEnd));                    
             #endif
        }
    }
    fprintf(AP,"\n");    
    
    time(&t_end);

    fprintf(log, "Time: %d\n", t_end - t_begin);

    //Clean up resources
    for(i=0; i<p->size; i++)
        for(j=0; j<p->size; j++)
            destroyCell(cells[i][j]);
    free(cells);

    #if CELL_REPORTS_ENABLED
    for(fileCounter = 0; fileCounter < p->reportNum; fileCounter++)
        fclose(p->cellReportFiles[fileCounter]);
    #endif    

    return 0;
}
