/*********************************************************************
 * An specialized version of a priority queue with one min-heap 
 * component and two linked-list components.  A lower time for an
 * event corresponds to a higher priority.
 * Mike True, 8/20/06
 *********************************************************************/

#ifndef QUEUEH
    #include "queue.h"
    #define QUEUEH 1
#endif

#define true 1
#define false 0

#include <stdlib.h>

/* A function used to save the Queue to a file */
void exportQueue(FILE *fp, Queue *queue)
{
    /* Save each of the three Queue components to a file */
    exportList(fp, queue->now);
    exportList(fp, queue->later);   
    exportHeap(fp, queue->heap);
}

/* A function used to load the Queue from a file */
Queue* importQueue(FILE *fp, Cell ***cells)
{
    Queue *newQueue;
    newQueue = (Queue *)malloc(sizeof(Queue));
    /* Load in the three Queue components */
    newQueue->now = importList(fp, cells);
    newQueue->later = importList(fp, cells);
    newQueue->heap = importHeap(fp, cells);
    /* Initialize all of the helper state variables of the Queue */
    newQueue->listsEmpty = 
      ((newQueue->now->size == 0) && (newQueue->later->size == 0));
    if(newQueue->listsEmpty)
        newQueue->nextSource = SOURCE_HEAP;
    else if((newQueue->heap->size == 0) ||
      (getNextTime(newQueue->heap) > getTime(newQueue->now)))
        newQueue->nextSource = SOURCE_LIST;
    else
        newQueue->nextSource = SOURCE_HEAP;    
    return newQueue;
}

/* A function that adds an event to the appropriate part of the queue */
void enqueue(Queue *queue, Event *event)
{
    /* QUERY_NEIGHBOR events go on one of the lists */
    if(event->type == QUERY_NEIGHBORS)
    {
        if(queue->listsEmpty)
        {
            addToTail(queue->now, event);
            queue->listsEmpty = false;
            if(event->time < getNextTime(queue->heap))
                queue->nextSource = SOURCE_LIST;
        }
        else if(getTime(queue->now) == event->time)
            addToTail(queue->now, event);
        else
        {
            addToTail(queue->later, event);
        }
    }
    /* All other events go on the heap */
    else
    {
        if(event->time <= queue->now->time || queue->listsEmpty)
            queue->nextSource == SOURCE_HEAP;
        insert(queue->heap, event);
    }
}

/* A function that removes an event from the appropriate part of the queue */
Event* dequeue(Queue *queue)
{
    Event *res;
    List *temp;
    /* Get the next event from the heap if that is the next source */
    if(queue->nextSource == SOURCE_HEAP)
    {
        res = removeNext(queue->heap);
        if(((queue->listsEmpty == false) && 
          (queue->now->time < getNextTime(queue->heap))) || 
          queue->heap->size ==0)
        {
            if(queue->listsEmpty != true)
                queue->nextSource = SOURCE_LIST;
        }
        return res;
    } 
    /* Get the next event from the current time's list */
    res = removeFromHead(queue->now);
    /* Switch lists, if necessary */
    if(queue->now->size == 0)
    {
        temp = queue->now;
        queue->now = queue->later;
        queue->later = temp;
        if(queue->now->time >= getNextTime(queue->heap))
            if(queue->heap->size != 0)
                queue->nextSource = SOURCE_HEAP;
        if(queue->now->size == 0)
        {
            queue->nextSource = SOURCE_HEAP;
            queue->listsEmpty = true;
        }
    }
    return res;
}

/* Return the next time an event will occur */
double peekNextTime(Queue *queue)
{
    if(queue->nextSource == SOURCE_HEAP)
    {
        return getNextTime(queue->heap);
    }
    return queue->now->time;
}

/* Allocate memory for a queue with a specified capacity >= 1 */
Queue* createQueue(int initCapacity)
{
    Queue *newQueue;
    if(initCapacity <= 0)
        return NULL;
    newQueue = (Queue *)malloc(sizeof(Queue));
    /* Create all components of the Queue */
    newQueue->heap = createHeap(initCapacity);
    newQueue->now = createList(initCapacity);
    newQueue->later = createList(initCapacity);
    /* Initialize all of the helper state variables of the Queue */    
    newQueue->nextSource = SOURCE_HEAP;
    newQueue->listsEmpty = true;
    return newQueue;
}

/* Destroy the contents of this queue, including all events in it */
void destroyQueue(Queue *queue)
{
    destroyHeap(queue->heap);
    destroyList(queue->now);
    destroyList(queue->later);
    free(queue);
}
