/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/**********************************************************
  This file implements the Dynamics base class for Python.
***********************************************************/

#define P3_MODULE
#include "ndl.h"

static int clear(Dynamics *self) {
    Py_XDECREF((PyObject*)self->owner);
    Py_XDECREF((PyObject*)self->owningCell);
    self->owner      = 0;
    self->owningCell = 0;
    return 0;
}

static int traverse(Dynamics *self, visitproc v, void *a) {
    if( self->owner && v((PyObject*)self->owner, a)<0 ) return -1;
    if( self->owningCell && v((PyObject*)self->owningCell, a) < 0 ) return -1;
    return 0;
}

static void del(Dynamics *self) {
    int i;

     /* Remove from cell list */
    for(i=0; self->owner && i<il_length(self->owner->dynamics); i++) {
        if( self == il_get(self->owner->dynamics, i) ) {
            il_remove(self->owner->dynamics, i);
            break;
        }
    }
    
    clear(self);

    /* Release Dynamics and Dynamics objects */
    self->ob_type->tp_free((PyObject*)self);
}

static int init(Dynamics *self, PyObject *args) {
    PyObject *in=0;
    Compartment *owner=0;

    if( !PyArg_ParseTuple(args, "O", &in) )
        return -1;

    if( PyObject_TypeCheck(in, &p3_CompartmentType) ) {
        owner = (Compartment*)in;
    } else if( PyObject_TypeCheck(in, &p3_CellType) ) {
        Cell *c = (Cell*)in;
        if( il_length(c->compartments) >= 1 ) {
            owner = il_get(c->compartments, 0);
        }
    }
    if( !owner ) {
        PyErr_SetString(PyExc_TypeError,
            "The first argument must be a Compartment or a Cell with a Compartment");
        return -1;
    }

    Py_INCREF(owner);
    Py_INCREF(owner->owner);
    self->owner      = owner;
    self->owningCell = owner->owner;

    il_append(owner->dynamics, (PyObject*)self);

    return 0;
}

static PyObject *repr(Dynamics *self) {
    return PyString_FromFormat("C Dynamics object at %p owned by Cell at %p, refcnt=%d",
        self, self->owner, ((PyObject*)self)->ob_refcnt);
}

static PyObject *route_enq(Dynamics *self, PyObject *args) {
    double strength=1, time;

    if( !self->enq ) {
        PyErr_SetString(PyExc_AttributeError,
            "Dynamics object does not have an enq method defined");
        return 0;
    }

    if( !PyArg_ParseTuple(args, "d|d", &time, &strength) )
        return 0;

   stepOnInt(self->owningCell, time, self, 0, strength, -1, enqmsg);

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *route_derivs(Dynamics *self, PyObject *args) {
    double time;

    if( !self->derivs ) {
        PyErr_SetString(PyExc_AttributeError,
            "Dynamics object does not have an derivs method defined");
        return 0;
    }

    if( !PyArg_ParseTuple(args, "d", &time) )
        return 0;

    self->derivs(self, time);

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *route_accepter(Dynamics *self, PyObject *args) {
    double strength;
    int windowid;
    Synapse *synapse=0;
	Cell *cell;
	double arrvtime;

    if( !self->accepter ) {
        PyErr_SetString(PyExc_AttributeError,
            "Dynamics object does not have an accepter method defined");
        return 0;
    }

    if( !PyArg_ParseTuple(args, "Oii", &synapse, &strength, &windowid) )
        return 0;
    if( !PyObject_TypeCheck(synapse, &p3_SynapseType) ) {
      PyErr_SetString(PyExc_TypeError,
		      "The first argument must be a Synapse object");
      return 0;
    }

   cell = synapse->owningCell;
   arrvtime = cell->time + synapse->trans_time;
   stepOnInt(cell, arrvtime, self, synapse, strength, windowid, acceptermsg);
 
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *route_cleanup(Dynamics *self, PyObject *args) {
  GD *gd=0;

    if( !self->cleanup ) {
        PyErr_SetString(PyExc_AttributeError,
            "Dynamics object does not have an cleanup method defined");
        return 0;
    }

    if( !PyArg_ParseTuple(args, "O", &gd) )
        return 0;
    if( !PyObject_TypeCheck(gd, &p3_GDType) ) {
      PyErr_SetString(PyExc_TypeError,
		      "The first argument must be a GD object");
      return 0;
    }

    self->cleanup(self, gd);

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *route_current(Dynamics *self, PyObject *args) {
    double time;
    PyObject *i;

    if( !self->current ) {
        PyErr_SetString(PyExc_AttributeError,
            "Dynamics object does not have a current method defined");
        return 0;
    }

    if( !PyArg_ParseTuple(args, "d", &time) )
        return 0;

    i = PyFloat_FromDouble(self->current(self, time));

    Py_INCREF(i);
    return i;
}

static PyObject *getTrace(Dynamics *self) {
    return PyInt_FromLong(self->trace);
}

static int setTrace(Dynamics *self, PyObject *val) {
    int i;
    double_array *x;

    self->trace = PyObject_IsTrue(val);
    
    if( self->trace && !self->traceTimes ) {
        if( (x=da_new()) == NULL ) return -1; 
        else self->traceTimes = x;
        self->traceData = getmain(self->n*sizeof(double_array*));
        if( !self->traceData ) return -1;
        for(i=0; i<self->n; i++) {
            if( (x=da_new()) == NULL ) return -1; 
            else self->traceData[i] = x;
        }
    }

    return 0;
}

void resetTrace(Dynamics *self) {
    int i;
    for(i=0; i<self->n; i++)
        da_clear(self->traceData[i]);
    da_clear(self->traceTimes);
}

void addTraceEntry(Dynamics *self) {
    int i;
    Cell *cell = self->owningCell;
    if( !self->trace ) return;
    for(i=0; i<self->n; i++) {
        if( cell->timeSample > 0 ) {
            da_append(self->traceTimes, cell->lasttime);
            da_append(self->traceData[i],  self->owningCell->Yinterp->data[self->y+i]);
        } else {
            da_append(self->traceTimes, cell->time);
            da_append(self->traceData[i],  GETSTATE_DYN(self,i));
        }
    }
}

static PyMethodDef methods[] = {
    {"enq", (PyCFunction)route_enq, METH_VARARGS, "enq(time, strength): add an exogenous event"},
    {"current", (PyCFunction)route_current, METH_VARARGS, "current(time): current passing through this Dynamics"},
    {"derivs", (PyCFunction)route_derivs, METH_VARARGS, "derivs(time): derivative function"},
    {"accepter", (PyCFunction)route_accepter, METH_VARARGS, "accepter(fromSynapse, strength, windowid): accept a synaptic event"},
    {"cleanup", (PyCFunction)route_cleanup, METH_VARARGS, "cleanup(gd): perform end of window resource management"},
    {NULL, NULL, 0, NULL}  /* sentinel */
};

static PyMemberDef members[] = {
    {"owner", T_OBJECT_EX, offsetof(Dynamics, owner),
    READONLY, "the Compartment owning this Dynamics"},
    {"HinesIntegrable", T_INT, offsetof(Dynamics, HinesIntegrable),
    0, "Force Dynamics to accepted by the Hines method."},
	{NULL}		/* sentinel */
};

static PyGetSetDef getseters[] = {
    {"trace", (getter)getTrace, (setter)setTrace, "flag to turn tracing on for this dynamics", NULL},
    {NULL}  /* Sentinel */
};

PyTypeObject p3_DynamicsType = {
    PyObject_HEAD_INIT(NULL)
    0,                         /*ob_size*/
    "_p3.Dynamics",             /*tp_name*/
    sizeof(Dynamics),      /*tp_basicsize*/
    0,                         /*tp_itemsize*/
    (destructor)del,                         /*tp_dealloc*/
    0,                         /*tp_print*/
    0,                         /*tp_getattr*/
    0,                         /*tp_setattr*/
    0,                         /*tp_compare*/
    (reprfunc)repr,                         /*tp_repr*/
    0,                         /*tp_as_number*/
    0,                         /*tp_as_sequence*/
    0,                         /*tp_as_mapping*/
    0,                         /*tp_hash */
    0,                         /*tp_call*/
    0,                         /*tp_str*/
    0,                         /*tp_getattro*/
    0,                         /*tp_setattro*/
    0,                         /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,        /*tp_flags*/
    "Parplex Dynamics object",   /* tp_doc */
    (traverseproc)traverse,		               /* tp_traverse */
    (inquiry)clear,		               /* tp_clear */
    0,		               /* tp_richcompare */
    0,		               /* tp_weaklistoffset */
    0,		               /* tp_iter */
    0,		               /* tp_iternext */
    methods,             /* tp_methods */
    members,             /* tp_members */
    getseters,          /* tp_getset */
    0,                         /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    (initproc)init,      /* tp_init */
    0,                         /* tp_alloc */
    PyType_GenericNew                 /* tp_new */
};

