/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/****************************************************
  This file implements the Synapse class for Python.
*****************************************************/

#define P3_MODULE
#include "ndl.h"

static int clear(Synapse *self) {
    Py_XDECREF(self->owner);
    self->owner = 0;

    Py_XDECREF(self->owningCell);
    self->owningCell = 0;

    Py_XDECREF(self->target_dynamics);
    self->target_dynamics = 0;

    Py_XDECREF(self->target);
    self->target = 0;

    return 0;
}

static int traverse(Synapse *self, visitproc v, void *a) {
    if( self->owner && v((PyObject*)self->owner, a)<0 ) 
        return -1;

    if( self->owningCell && v((PyObject*)self->owningCell, a)<0 )
        return -1;

    if( self->target && v((PyObject*)self->target, a)<0 ) 
        return -1;

    if( self->target_dynamics && v((PyObject*)self->target_dynamics, a)<0 ) 
        return -1;

    return 0;
}

static void del(Synapse *self) {
    int i;

    /* Remove from Compartment list */
    for(i=0; self->owner && i<il_length(self->owner->synapse); i++) {
        if( self == il_get(self->owner->synapse, i) ) {
            il_remove(self->owner->synapse, i);
            break;
        }
    }

    clear(self);

    /* Release Dynamics and Synapse objects */
    self->ob_type->tp_free((PyObject*)self);
}

#define SYNINDXBUFINC 20000
Synapse **synapseIndex;
static int synBufEnd;

static int init(Synapse *self, PyObject *args) {
    PyObject *in1=0, *target_dynamics=0;
    Compartment *owner=0;
    static int synapse_id;

    if( !PyArg_ParseTuple(args, "OO", &in1, &target_dynamics) )
        return -1;

    if( PyObject_TypeCheck(in1, &p3_CompartmentType) ) {
        owner = (Compartment*)in1;
    } else if( PyObject_TypeCheck(in1, &p3_CellType) ) {
        Cell *c = (Cell*)in1;
        if( il_length(c->compartments) >= 1 ) {
            owner = il_get(c->compartments, 0);
        }
    }
    if( !owner ) {
        PyErr_SetString(PyExc_TypeError,
            "The first argument must be a Compartment or a Cell with a Compartment");
        return -1;
    }      

    if( (!PyObject_TypeCheck(target_dynamics, &p3_DynamicsType)) ) {
        PyErr_SetString(PyExc_TypeError, 
            "the second argument must be of type Dynamics");
        return -1;
    }

    Py_INCREF(owner);
    self->owner  = (owner);

    Py_INCREF(owner->owner);
    self->owningCell  = owner->owner;

    il_append(owner->synapse, (PyObject*)self);       

    Py_INCREF(target_dynamics);
    self->target_dynamics = (Dynamics*)target_dynamics;

    Py_INCREF(((Dynamics*)target_dynamics)->owningCell);
    self->target = ((Dynamics*)target_dynamics)->owningCell;

	self->tgtrank = ((Dynamics*)target_dynamics)->owningCell->mpi_rank;
	
	/* Add the synapse into the master list */
	if( synapse_id >= synBufEnd ) {
		synBufEnd += SYNINDXBUFINC;
		synapseIndex = PyMem_Realloc(synapseIndex, synBufEnd*sizeof(**synapseIndex));
		if( !synapseIndex ) 
			ABEND("Could not allocate %d bytes of memory.\n", self->owningCell);
    }
    synapseIndex[synapse_id] = self;
    self->synapse_id = synapse_id++;

	return 0;
}

static PyObject *repr(Synapse *self) {
    return PyString_FromFormat("C Synapse object at %p connecting Compartment at %p to Cell at %p refcnt=%d",
        self, self->owner, self->target, ((PyObject*)self)->ob_refcnt);
}

static PyMethodDef methods[] = {
 	{NULL, NULL, 0, NULL}		/* sentinel */
};

static PyMemberDef members[] = {
    {"trans_time", T_DOUBLE, offsetof(Synapse, trans_time), 
        0, "action potential transit time to target"},
    {"nominal_strength", T_DOUBLE, offsetof(Synapse, nominal_strength), 
    0, "basal synaptic strength"},
    {"owner", T_OBJECT_EX, offsetof(Synapse, owner),
    READONLY, "the Cell owning this synapse"},
    {"target", T_OBJECT_EX, offsetof(Synapse, target),
    READONLY, "the Cell that this synapse connects to"},
    {"target_dynamics", T_OBJECT_EX, offsetof(Synapse, target_dynamics),
    READONLY, "the dynamics object that implements the post synaptic response"},
    {NULL}		/* sentinel */
};

PyTypeObject p3_SynapseType = {
    PyObject_HEAD_INIT(NULL)
    0,                         /*ob_size*/
    "_p3.Synapse",             /*tp_name*/
    sizeof(Synapse),           /*tp_basicsize*/
    0,                         /*tp_itemsize*/
    (destructor)del,           /*tp_dealloc*/
    0,                         /*tp_print*/
    0,                         /*tp_getattr*/
    0,                         /*tp_setattr*/
    0,                         /*tp_compare*/
    (reprfunc)repr,            /*tp_repr*/
    0,                         /*tp_as_number*/
    0,                         /*tp_as_sequence*/
    0,                         /*tp_as_mapping*/
    0,                         /*tp_hash */
    0,                         /*tp_call*/
    0,                         /*tp_str*/
    0,                         /*tp_getattro*/
    0,                         /*tp_setattro*/
    0,                         /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,  /*tp_flags*/
    "Parplex Synapse object",   /* tp_doc */
    (traverseproc)traverse,     /* tp_traverse */
    (inquiry)clear,             /* tp_clear */
    0,		               /* tp_richcompare */
    0,		               /* tp_weaklistoffset */
    0,		               /* tp_iter */
    0,		               /* tp_iternext */
    methods,               /* tp_methods */
    members,               /* tp_members */
    0,                     /* tp_getset */
    0,                     /* tp_base */
    0,                     /* tp_dict */
    0,                     /* tp_descr_get */
    0,                     /* tp_descr_set */
    0,                     /* tp_dictoffset */
    (initproc)init,        /* tp_init */
    0,                     /* tp_alloc */
    PyType_GenericNew                 /* tp_new */
};
