"""
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
"""

"""
Use the Rinsky and Pinzel reduced model of a pyramidal
cell as a test for parplex. The RP model is available 
as code from modelDB.

This is pretty much a cut 'n' paste job from booth_bose.ode
"""


import sys
from math import exp
from p3 import *

def heav(x): return x>=0

class Na(PyDynamics):
    stateVars   = ['h']
    stateDerivs = ['dhdt']
    stateTrace  = ['hTrace']
    
    def __init__(self, cell):
        PyDynamics.__init__(self, cell)
        self.h = 0
        
    def derivs(self, t):
        v = self.owner.Em
        alphahs = 0.128*exp((-43.0-v)/18.0)
        betahs  = 4.0/(1.0+exp((-20.0-v)/5.0))
        self.dhdt = alphahs - (alphahs+betahs)*self.h
        
    def current(self, t):
        v       = self.owner.Em
        alphams =  0.32*(-46.9-v)/(exp((-46.9-v)/4.0)-1.0)
        betams  =  0.28*(v+19.9)/(exp((v+19.9)/5.0)-1.0)
        Minfs   =  alphams/(alphams+betams)
        return -self.gNa * (Minfs*Minfs) * self.h * (v-self.VNa)
        
class Kdr(PyDynamics):
    stateVars   = ['n']
    stateDerivs = ['dndt']
    stateTrace  = ['nTrace']
    
    def __init__(self, cell):
        PyDynamics.__init__(self, cell)
        self.n = 0
        
    def derivs(self, t):
        v = self.owner.Em
        alphans = 0.016*(-24.9-v)/(exp((-24.9-v)/5.0)-1.0)
        betans  = 0.25*exp(-1.0-0.025*v)
        self.dndt = alphans-(alphans+betans)*self.n
        
    def current(self, t):
        v = self.owner.Em
        return -self.gKdr * self.n * (v-self.VK)
        
class Ca(PyDynamics):
    stateVars   = ['s']
    stateDerivs = ['dsdt']
    stateTrace  = ['sTrace']
    
    def __init__(self, cell):
        PyDynamics.__init__(self, cell)
        self.s = 0
                   
    def derivs(self, t):
        v       = self.owner.Em
        alphasd = 1.6/(1.0+exp(-0.072*(v-5.0)))
        betasd  = 0.02*(v+8.9)/(exp((v+8.9)/5.0)-1.0)
        self.dsdt = alphasd-(alphasd+betasd)*self.s
        
    def current(self, t):
        v = self.owner.Em
        return -self.gCa*self.s*self.s*(v-self.VCa)
        
class Kahp(PyDynamics):
    stateVars   = ['q', 'Ca']
    stateDerivs = ['dqdt', 'dCadt']
    stateTrace  = ['qTrace', 'CaTrace']
    
    def __init__(self, cell, Ca):
        PyDynamics.__init__(self, cell)
        self.Cadyn = Ca
        self.q = 0
        self.Ca = 0
    
    def derivs(self, t):
        v = self.owner.Em
        alphaqd = min(0.00002*self.Ca, 0.01)
        betaqd  = 0.001
        self.dqdt = alphaqd-(alphaqd+betaqd)*self.q
        self.dCadt = 0.13*self.Cadyn.current(t)-0.075*self.Ca
    
    def current(self, t):
        v = self.owner.Em
        return -self.gKahp*self.q*(v-self.VK)
        

class KC(PyDynamics):
    stateVars   = ['c']
    stateDerivs = ['dcdt']
    stateTrace  = ['cTrace']
    
    def __init__(self, cell, Kahp):
        PyDynamics.__init__(self, cell)
        self.Kahp = Kahp
        self.c = 0
    
    def derivs(self, t):
        v         = self.owner.Em
        alphacd   = (1.0-heav(v+10.0))*exp((v+50.0)/11-(v+53.5)/27)/18.975+heav(v+10.0)*2.0*exp((-53.5-v)/27.0) 
        betacd    = (1.0-heav(v+10.0))*(2.0*exp((-53.5-v)/27.0)-alphacd)
        self.dcdt = alphacd-(alphacd+betacd)*self.c
        
    def current(self, t):
        v = self.owner.Em
        chid = min(self.Kahp.Ca/250.0,1.0)
        return -self.gKC*self.c*chid*(v-self.VK)

class Passive(PyDynamics):
    def current(self, t):
        Em = self.owner.Em
        return self.Gleak * (self.Vleak - Em)
        
class Current(PyDynamics):
    def current(self, t):
        return self.Iinject * (self.start<=t and t<=self.end)
      
Ip0 = 0.75
gc = 2.1
pp = 0.5

cell = mCell()

soma = Compartment(cell)
soma.capacitance = 3
soma.Em = -60
soma.emtrace = True

d = Passive(soma)
d.Vleak = -60
d.Gleak = 0.1

d = Na(soma)
d.VNa = 60
d.gNa = 30

d = Kdr(soma)
d.VK = -75
d.gKdr = 15

d = Current(soma)
d.start = 0
d.end   = 100000000
d.Iinject = Ip0/pp

dendrite = Compartment(cell)
dendrite.capacitance = 3
dendrite.Em = -60
dendrite.emtrace = True

d = Passive(dendrite)
d.Vleak = -60
d.Gleak = 0.1

d = Ca(dendrite)
d.gCa = 10
d.VCa = 80

d = Kahp(dendrite, d)
d.gKahp = 0.8
d.VK = -75

d = KC(dendrite, d)
d.gKC = 15
d.VK = -75

soma.connect(dendrite, gc/pp)


###############
# Run options #
###############
setTRfilename('trace')
gd = GD()
gd.duration  = 150
gd.tolerance = 1e-4
gd.network = [cell]

#######
# Run #
#######
parplex(gd)
print '-------------------------------------------------------------'
parplex(gd)
message_print(info, 'made it')

for cell in gd.network:
    message_print(info, 'id=%d %s: %d total steps, %d succesful, %d function evals, %d Jacobian evals, %d Newton iters\n' % (cell.id, cell.method, cell.stepTotal, cell.stepAccepts, cell.functionCnt, cell.jacobianCnt, cell.newtonCnt))
