/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* Gaussian noise current */

#include "ndl.h"

/* Generate noise at 500Hz */
#define SAMPLE_RATE 100

typedef struct {
    DYNAMICS
  double Imean;
  double Istd;
  double *I;
  bool  datadone;
} Inoise;


static PyMemberDef NoiseCurrent_members[] = {
    {"mean", T_DOUBLE, offsetof(Inoise, Imean),
     0, "mean of noise current"},
    {"std", T_DOUBLE, offsetof(Inoise, Istd),
     0, "standard deviation of noise current"},
    {NULL},
};

static void generate_noise(Inoise *i, GD *gd) {
  /* Generate the noise */
  int j;
  /* Number of samples required in the window */
  int num_samples = (int)floor(gd->window*SAMPLE_RATE/1000+0.5);
  
  if( i->I == 0 )
    i->I = getmain(sizeof(*i->I)*(num_samples+1));

  for(j=0; j<=num_samples; j++)
    i->I[j] = rand_norm(i->Imean, i->Istd);
}

static void Inoise_cleanup(Inoise *self, GD *gd) {
  /* Allocate a new set of random numbers for the next window
   */
  generate_noise(self, gd);
}

static double Inoise_current(Inoise *self, double t) {
  /* Linearly interpolate current from adjacent samples */
  
  double nt;
  int j;
  double extra;
  double current;
  GD *gd = self->owningCell->gd;
  double windowsize = gd->window;
  double num_samples = windowsize*SAMPLE_RATE/1000;

  if( !self->datadone ) {
    generate_noise(self, gd);
    self->datadone = true;
  }
  
  /* Get the time into the current window */
  nt = t - floor(t/windowsize)*windowsize;

  /* Get previous sample number */
  j = (int)floor(nt/windowsize*num_samples+0.5);

  /* Fractional extra bit into the sample */
  extra = nt-(double)j/num_samples*windowsize;

  /* Linearly interpolates between previous and next sample */
  current = self->I[j] + extra*(self->I[j+1]-self->I[j]);

  return current;  
}

DynamicsDescriptor NoiseCurrentDescriptor = {
    "Inoise",
    "Inoise, a gaussian noise current. members: mean, std",
    NoiseCurrent_members, /* Members */
    0,
    0,  /* number of state variables */
    0,  /* State variable names */
    0,  /* deriv variable names */
    0,  /* trace variable names */
    0,  /* Derivs */
    (currentfcn*)Inoise_current,
    0,  /* acceptor */
    0,  /* enqueue */
    (cleanupfcn*)Inoise_cleanup,  /* cleanup */
    sizeof(Inoise),
    0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	HIdynamics
};
REGISTER_DESCRIPTOR(NoiseCurrentDescriptor)
  
static DynamicsDescriptor *userDynamics[] = {
    &NoiseCurrentDescriptor
};
  
static initproc LuserInitDynamics[] = {
    initNoiseCurrentDescriptor
};


MAKE_P3_MODEL(noise, "Noise current")
