/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


/**********************************************************************
   A subclass of Dynamics to manage internal Ca. The objects needs
   pointers Ca currents, ie other Dynamicss and provides Caconc for
   Ca dependent channels. Also provided is GHK calculated Ca driving
   force. The class does not provide a current. The instantiator needs
   to provide the dCadt RHS function. This class needs to maintian
   Hines integrability.
***********************************************************************/

#define P3_MODULE
#include "ndl.h"

static double rhs(CaDynamics *self, double ICa, double Ca) {
	return self->B*ICa - (Ca-self->Ca0)/self->Catau;
}

static void Caderivs(CaDynamics *self, double t) {
    double ICa = 0;
	Dynamics *d;
	double dCadt;
	int seq;

	seq = 0;
    while( (d=il_seq_next(self->Cadynamics, &seq)) ) 
		ICa += d->current(d, t);

	dCadt = self->Caupdate(self, ICa, GETSTATE_DYN(self, 0));
	SETDERIV_DYN(self, 0, dCadt);
}

static void HIupdate(CaDynamics *Cad, double t, double h) {
	double Ca = GETSTATE_DYN(Cad, 0);
	double ICa = Cad->Cacurrent(Cad, t);
	double tau = Cad->Catau;
	/* Ca update rule */
	Ca  = Ca + Cad->Caupdate(Cad, ICa, Ca)/(1/h+1/2/tau);
	SETSTATE_DYN(Cad, 0, Ca);
}

#define celsius (6.3)
#define KTF (((25./293.15)*(celsius + 273.15))/2)
static double ghk(CaDynamics *self) {
	//double V = GETEM_DYN(self);
	double ci = GETSTATE_DYN(self, 0);

	//double ghk, nu, efun;
	//nu   = V/KTF;
	//efun = smooth(nu, 1);
	//ghk  = KTF*(1. - (ci/co)*exp(nu))*efun;
	//return ghk;

	return 24*log(self->co/ci) / self->charge;
}

static double Cacurrent(CaDynamics *self, double t) {
	Dynamics *d;
	double ICa = 0;
	int seq = 0;
	while( (d=il_seq_next(self->Cadynamics, &seq)) )
		ICa += d->current(d, t);
	return ICa;
}

static int initCaDynamics(CaDynamics *self, PyObject *args, PyObject *kw) {
    int rc = p3_HinesIntegrableDynamicsType.tp_init((PyObject*)self, args, kw);
    internal_list *l;

    if( rc ) return rc;

    self->current   = 0;
    self->derivs    = (derivsfcn*)Caderivs;
	self->HIupdate  = (HIupdatefcn*)HIupdate;
    self->accepter  = 0;
    self->cleanup   = 0;
	self->type      = Cadynamics;
	self->GHK       = ghk;
	self->Caupdate  = (Caupdatefcn*)rhs;
	self->charge    = 1;
	self->Cacurrent = Cacurrent;

    if( (l=il_new("Dynamics")) == NULL ) return -1;
    else self->Cadynamics = l;

    self->y = addState(self->owner->owner, 1);
    self->n = 1;

    return 0;
}

static int setCa(CaDynamics *self, PyObject *CaPy) {
    if( PyFloat_Check(CaPy) ) {
        SETSTATE_DYN(self, 0, PyFloat_AsDouble(CaPy));
        return 0;
    }

    if( PyInt_Check(CaPy) ) {
        SETSTATE_DYN(self, 0, PyInt_AsLong(CaPy));
        return 0;
    }

    PyErr_SetString(PyExc_TypeError, 
        "Ca must be an integer or a float");
    return -1;
}

static int setdCadt(CaDynamics *self, PyObject *dCadtPy) {
    if( PyFloat_Check(dCadtPy) ) {
        SETDERIV_DYN(self, 0, PyFloat_AsDouble(dCadtPy));
        return 0;
    }

    if( PyInt_Check(dCadtPy) ) {
        SETDERIV_DYN(self, 0, PyInt_AsLong(dCadtPy));
        return 0;
    }

    PyErr_SetString(PyExc_TypeError, 
        "dCadt must be an integer or a float");
    return -1;
}

static PyObject *getCa(CaDynamics *self) {
    return PyFloat_FromDouble(GETSTATE_DYN(self, 0));
}

static PyObject *getdCadt(CaDynamics *self) {
    return PyFloat_FromDouble(GETDERIV_DYN(self, 0));
}

PyObject *getCaTrace(CaDynamics *self) {
    PyObject *t;

    if( !self->trace ) {
        PyErr_SetString(PyExc_AttributeError,
            "Trace variable requested for a Dynamics that is not being trace.");
        return NULL;
    }

    t = (PyObject*)self->traceData[0];
    Py_INCREF(t);

    return t;
}

static PyGetSetDef getseters[] = {
    {"Ca", (getter)getCa, (setter)setCa, "Ca concentration", NULL},
    {"dCadt", (getter)getdCadt, (setter)setdCadt, "Ca concentration rate", NULL},
    {"CaTrace", (getter)getCaTrace, 0, "Ca concentration trace", NULL},
    {NULL}  /* Sentinel */
};

static PyMemberDef members[] = {
    {"B", T_DOUBLE, offsetof(CaDynamics, B), 
     0, "Conversion from current to rate of concentration change"},
    {"Ca0", T_DOUBLE, offsetof(CaDynamics, Ca0), 
     0, "Basal Ca concentration"},
    {"Catau", T_DOUBLE, offsetof(CaDynamics, Catau), 
     0, "Time constant of Ca uptake"},
    {"CaProviders", T_OBJECT_EX, offsetof(CaDynamics, Cadynamics), 
     0, "list of Ca current providers"},
    {"charge", T_DOUBLE, offsetof(CaDynamics, charge), 
     0, "Valence of the charge carrier"},
    {"co", T_DOUBLE, offsetof(CaDynamics, co), 
     0, "external concentration of charge carrier"},
 	{NULL}		/* sentinel */
};

PyTypeObject p3_CaDynamicsType = {
    PyObject_HEAD_INIT(NULL)
    0,                         /*ob_size*/
    "_p3.CaDynamics",             /*tp_name*/
    sizeof(CaDynamics),      /*tp_basicsize*/
    0,                         /*tp_itemsize*/
    0,                         /*tp_dealloc*/
    0,                         /*tp_print*/
    0,                         /*tp_getattr*/
    0,                         /*tp_setattr*/
    0,                         /*tp_compare*/
    0,                         /*tp_repr*/
    0,                         /*tp_as_number*/
    0,                         /*tp_as_sequence*/
    0,                         /*tp_as_mapping*/
    0,                         /*tp_hash */
    0,                         /*tp_call*/
    0,                         /*tp_str*/
    0,                         /*tp_getattro*/
    0,                         /*tp_setattro*/
    0,                         /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,        /*tp_flags*/
    "Parplex internal Ca Dynamics object",   /* tp_doc */
    0,		               /* tp_traverse */
    0,		               /* tp_clear */
    0,		               /* tp_richcompare */
    0,		               /* tp_weaklistoffset */
    0,		               /* tp_iter */
    0,		               /* tp_iternext */
    0,					   /* tp_methods */
    members,                     /* tp_members */
    getseters,					   /* tp_getset */
    &p3_HinesIntegrableDynamicsType, /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    (initproc)initCaDynamics,      /* tp_init */
    0,                         /* tp_alloc */
    PyType_GenericNew                 /* tp_new */
};
