/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


/*********************************************************************
 Define the structures classic Hodgkin-Huxely dynamics.
 These are a leak current, fast Na current and the delayed rectifier.
**********************************************************************/

#include "ndl.h"

typedef struct {
    DYNAMICS
  double Gmax;    /* The maximum conductance */
  double Er;      /* The driving potential */
}  hh_cond;

static PyMemberDef p3_hh_members[] = {
    {"Gmax", T_DOUBLE, offsetof(hh_cond, Gmax),
     0, "maximum conductance"},
    {"Er", T_DOUBLE, offsetof(hh_cond, Er),
     0, "reversal potential"},
    {NULL},
};

/*************
  Na currents
**************/
static void dNadt(hh_cond *self, double t) {
  double E        = GETEM_DYN(self, t);
  double Na_act   = GETSTATE_DYN(self, 0);
  double Na_inact = GETSTATE_DYN(self, 1);
  double dact = 0.1*(-E+25.0)/(exp(-0.1*E+2.5)-1.0)*(1.0-Na_act)-4.0*exp(-E/18.0)*Na_act;
  double dinact = 0.07*exp(-E/20.0)*(1.0- Na_inact)-Na_inact/(exp(3.0-0.1*E)+1.0);

  SETDERIV_DYN(self, 0, dact);
  SETDERIV_DYN(self, 1, dinact);
  
  return;
}

static double IhhNa_current(hh_cond* self, double t) {
  double E = GETEM_DYN(self, t);
  double g = self->Gmax;
  double ENa = self->Er;
  double Na_act   = GETSTATE_DYN(self, 0);
  double Na_inact = GETSTATE_DYN(self, 1);

  return - g * Na_act * Na_act * Na_act * Na_inact * (E - ENa);
}

/*************
  K currents
**************/
static void dKdt(hh_cond *self, double t) {
  double E     = GETEM_DYN(self, t);
  double K_act = GETSTATE_DYN(self, 0);
  double dact  = 0.01*(-E+10.0)/(exp(-0.1*E+1.0)-1.0)*(1.0-K_act)-0.125*exp(E/80.0)*K_act;
  
  SETDERIV_DYN(self, 0, dact);
   
  return;
}

static double IhhK_current(hh_cond* self, double t) {
  double E = GETEM_DYN(self, t);
  double g = self->Gmax;
  double EK    = self->Er;
  double K_act = GETSTATE_DYN(self, 0);

  return - g * K_act * K_act * K_act * K_act * (E - EK);
}

/***************
  Leak currents
****************/
static double IhhLeak_current(hh_cond* d, double t) {
  double E = GETEM_DYN(d, t);
  double g = d->Gmax;
  double ELeak = d->Er;

  return - g * (E - ELeak);
}

static char *NaStateVars[] = {"m", "h"};
static char *NaDerivVars[] = {"dmdt", "dhdt"};
static char *NaTraceVars[] = {"mTrace", "hTrace"};
DynamicsDescriptor hhNaDescriptor = {
    "hhNa",
    "hhNa Hodgkin-Huxely Na current. members: Gmax, Er",
    p3_hh_members,
    0,
    2,
    NaStateVars,
    NaDerivVars,
    NaTraceVars,
    (derivsfcn*)dNadt,
    (currentfcn*)IhhNa_current,
    0,
    0,
    0,
    sizeof(hh_cond),
    0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(hhNaDescriptor)

static char *KStateVars[] = {"n"};
static char *KDerivVars[] = {"dndt"};
static char *KTraceVars[] = {"nTrace"};
DynamicsDescriptor hhKDescriptor = {
    "hhK",
    "hhNa Hodgkin-Huxely K current. members: Gmax, Er",
    p3_hh_members,
    0,
    1,
    KStateVars,
    KDerivVars,
    KTraceVars,
    (derivsfcn*)dKdt,
    (currentfcn*)IhhK_current,
    0,
    0,
    0,
    sizeof(hh_cond),
    0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(hhKDescriptor)

DynamicsDescriptor hhLeakDescriptor = {
    "hhLeak",
    "hhLeak Hodgkin-Huxely leak current. members: Gmax, Er",
    p3_hh_members,
    0,
    0,
    0,
    0,
    0,
    0,
    (currentfcn*)IhhLeak_current,
    0,
    0,
    0,
    sizeof(hh_cond),
    0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(hhLeakDescriptor)

  
static initproc LuserInitDynamics[] = {
  inithhNaDescriptor,
  inithhKDescriptor,
  inithhLeakDescriptor
};
  
static DynamicsDescriptor *userDynamics[] = {
  &hhNaDescriptor,
  &hhKDescriptor,
  &hhLeakDescriptor
};

MAKE_P3_MODEL(hh, "The Hodgkin-Huxely model.")
