/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* Declarations needed by parplex. */

#ifndef NDL_H
#define NDL_H

/* This kludge is required to compile with debug against a 
   standard python distribution. */
#ifdef _DEBUG
#undef _DEBUG
#include <Python.h>
#include "structmember.h"
#define _DEBUG
#else
#include <Python.h>
#include "structmember.h"
#endif

#include <setjmp.h>
#include <float.h>

/* Throw execptions with this jump point */
extern jmp_buf excpt_exit;

/* Parallel information */
extern int mpi_rank;
extern int mpi_size;

/* Base types we are providing */
extern PyTypeObject p3_DoubleArrayType;
extern PyTypeObject p3_CellType;
extern PyTypeObject p3_CompartmentType;
extern PyTypeObject p3_SynapseType;
extern PyTypeObject p3_InternalListType;
extern PyTypeObject p3_GDType;
extern PyTypeObject p3_DynamicsType;
extern PyTypeObject p3_PyDynamicsType;
extern PyTypeObject p3_HinesIntegrableDynamicsType;
extern PyTypeObject p3_VGCDynamicsType;
extern PyTypeObject p3_CaDynamicsType;

/*********************************************/
/*  The double_array structure and its C API */
/********************************************/
typedef struct {
    PyObject_HEAD
    int len;
    int buffer_len;
    int seq;
    double *data;
} double_array;
void          da_set(double_array*, double, int);
double        da_get(double_array*, int);
void          da_clear(double_array*);
int           da_length(double_array*);
void          da_append(double_array*, double);
void          da_copy(double_array*, double_array*);
double_array *da_new(void);
void          da_seq_start(double_array*);
int           da_seq_next(double_array*, double*);
void          da_seq_set(double_array*, double);
void          da_slice_del(double_array*, int, int);

/*********************************************/
/*  The internal_list structure and its C API */
/*********************************************/
typedef enum {
    il_Synapse, il_Cell, il_Dynamics, il_Compartment
} il_type;

typedef struct {
    PyObject_HEAD
    int len;
    int buffer_len;
    PyObject **data;
    il_type type;
} internal_list;
void           il_set(internal_list*, PyObject*, int);
void          *il_get(internal_list*, int);
void           il_clear(internal_list*);
int            il_length(internal_list*);
int            il_append(internal_list*, PyObject*);
void           il_copy(internal_list*, internal_list*);
void           il_remove(internal_list*, int);
internal_list *il_new(char*);
void          *il_seq_next(internal_list*, int*);

/* The name 'bool' causes SWIG to seriously choke.
   But ... we are no longer using SWIG. YAY!!!*/
typedef enum {false, true}  bool;

/* Message levels */  
typedef enum {
  debug, info, warn, fatal
} msglevel;


/* method names */
#define rk32  "rk32"
#define rw23  "rw23"
#define rdIIA "rdIIA"
#define hines "hines"
#define hines_fixed_step "hines_fixed_step"

struct celltag;
struct gdtag;
struct cmpttag;
struct dynamicstag;
struct synapsetag;
typedef double currentfcn(struct dynamicstag*, double);
typedef void   derivsfcn(struct dynamicstag*, double);
typedef void   accepterfcn(struct  dynamicstag*, struct synapsetag*, double, int);
typedef void   enqfcn(struct  dynamicstag*, double, double);
typedef void   cleanupfcn(struct  dynamicstag*, struct gdtag*);
typedef int    userinitfcn(struct dynamicstag*);
typedef void   HIupdatefcn(struct dynamicstag*, double, double);
typedef void   HIcurrentfcn(struct dynamicstag*, double, double*, double*);
typedef bool solverfcn(struct celltag*, double, double, double*);
typedef bool interpfcn(int, double, double*, double*);
typedef void solverInitfcn(struct gdtag*, struct celltag*);
typedef void jacobfcn(double, double*, double*, double*, int,
		      double**, int*, struct celltag*);
typedef void sparse_jacobfcn(double, double*, double*, double*,
			     int, double**, int**, int*, int*, struct celltag*);

typedef enum {nomsg, enqmsg, acceptermsg} MsgType;
typedef struct {
	double   time;
	struct dynamicstag *dynamics;
	struct synapsetag  *synapse;
	double   strength;
	int      window_id;
	MsgType msgtype;
} mustStepObject;

typedef struct celltag { /* contains run time information need per cell */
    PyObject_HEAD
  int queue_magic;    /* used to quickly requeue or remove cell */
  int id;             /* Unique within partition for x-partition id. */

  internal_list *compartments;   /* list of compartments */
  
  bool reDo;          /* Cell needs to be redone as part of a relaxtion. */

  /* State variables, derivs and copies, 
     interpolation work area, jacobian work areas */
  double_array *Y;
  double_array *DYDT;
  double_array *Y_copy;
  double_array *DYDT_copy;
  double_array *Yinterp;
  double_array *fac;
  double_array *fac_copy;
  struct partialsDependencyTag *dependencies;
  int *sparsePattern;
  int  spCnt;

  void *gd;
  
  double  timeSample;  /* Sample interval for tracing */
  double  time;       /* current time of cell solution (time of y) */
  double  lasttime;   /* Time at sample points */
  double  last_copy;
  double  step;       /* next integration step */
  double  step_copy;
  mustStepObject *mustStep;  /* Time points where the cell must step */
  int mustStepStart;
  int     mustStepLen; /* Length of mustStep buffer */
  bool    laststepaccept;

  bool hasDynTrace;   /* At least 1 dynamics is traced */
  bool hasEmTrace;    /* At least 1 compartment is Em traced */
  bool stepTrace;     /* Call stepTrace_handler at every step */

  /* Parallel environment */
  int mpi_rank;  /* Where is the cell located? */
  bool isLocal;

  /* Numerical methods */
  solverfcn *solver;
  interpfcn *interpolator;
  solverInitfcn *solverInit;
  void *solverData;   /* per cell data used by the solver */
  bool sparseLinearAlgebra;
  int stepAccepts;
  int stepTotal;
  int functionCnt; /* Number of function evaluations */
  int newtonCnt;   /* Number of Newton iterations in implicit methods */
  int jacobianCnt; /* Number of Jacobian evaluations in implicit methods */
  int maxStepCnt; /* number of times maxStep was encounted */
  int badStepCnt; /* number of times a bad step size was encounted */
} Cell;
#define MUSTSTEP_BUFSIZE 10
#define MUSTSTEP_SENTINEL DBL_MAX

#define SETEM_CMPT(c,v)     c->owner->Y->data[c->Emindx] = v
#define GETDEMDT_CMPT(c)    c->owner->DYDT->data[c->Emindx]
#define SETDEMDT_CMPT(c,v)  c->owner->DYDT->data[c->Emindx] = v
#define GETEMINTERP_CMPT(c) c->owner->Yinterp->data[c->Emindx]
typedef enum {
	CurrentClamp, VoltageClamp
} clampmode;

typedef struct cmpttag {
    PyObject_HEAD

  bool emtrace;       /* Is this compartment Em traced? */
  
  Cell *owner;   /* Owning cell */
  int   ownerindx; /* Indx into cell compartment list of this cmpt */

  double capacitance;

  int     Emindx;         /* Location of compartment Em in the state vectors */
  double  lastEm;     /* Previous potential, required for AP detection. */
  double  lastEm_copy;

  clampmode ClampMode; /* CurrentClamp or VoltageClamp */
  struct dynamicstag *Vclamper;  /* Supply a voltage clamp wave form */
  double    clampEm;   /* If Vclamper==0 this is the clamped voltage. */

  double APthreshold;  /* AP threshold for this cmpt */

  internal_list *compartments;   /* list of compartments connected to this one */
  double_array *axial_conductance;
  int *treeOrderCol;  /* Used by the Hines solver */
  int  treeOrderRow;  /* Used by the Hines solver */
  
  internal_list *synapse;   /* synapses outgoing from this cmpt */

  internal_list *dynamics; /* List of compartment dynamics */
  
  /* Action potential history */
  double_array   *currentAP;
  double_array   *previousAP;
  
  /* Trace history */
  double_array *traceTimes;
  double_array *traceData;

  struct qmsgtag *msgQ;
} Compartment;

/* Global data */
typedef struct gdtag {
    PyObject_HEAD

  /* Integration & run details */
  double duration;
  double tolerance;
  double maxStep; /* Maximum step size */
  double minStep; /* Minimum step size */
  double window;
  double windowStart;
  int windowID;
  int cellsToGo;

  /* Parallel environment and WR data */
  bool firstRound;

  /* The network of neurons to be solved */
  PyObject *network;

  /* Messaging callback functions */
  PyObject* message_handler;  /* Normal user targeted messages */
  PyObject* debug_handler;    /* Internal debug messages */
  PyObject* trace_handler;    /* cell state variable trace */
  PyObject* ap_handler;       /* cell fired an AP */
  PyObject* endWindow_handler;
  PyObject* startWindow_handler;  
  PyObject* stepTrace_handler; /* Called at every step, failed and true */
  PyObject* dumpCell_handler;

  /* Parallel performance counters */
  int reDoCnt;
  int roundReDoCnt;
  int sendMsgCnt;
  int recvMsgCnt;
  int roundCnt;
} GD;


/* Access state data easily (?!?!?!?) */
//#define SETEM_DYN(c,v)      d->owningCell->Y->data[c->Emindx] = v
//#define GETEM_DYN(d)        d->owningCell->Y->data[d->owner->Emindx]
#define GETEM_DYN(d, t)        GETEM_CMPT(d->owner, t)
//#define GETDEMDT_DYN(d)     d->owningCell->DYDT->data[d->owner->Emindx]
#define GETEMINTERP_DYN(d)  d->owningCell->Yinterp->data[d->owner->Emindx]
#define GETSTATE_DYN(d,i)   d->owningCell->Y->data[d->y+i]
#define SETSTATE_DYN(d,i,v) d->owningCell->Y->data[d->y+i] = v
#define GETDERIV_DYN(d,i)   d->owningCell->DYDT->data[d->y+i]
#define SETDERIV_DYN(d,i,v) d->owningCell->DYDT->data[d->y+i] = v

/* C language inheritable base classes */
typedef enum {
	dynamics, VGCdynamics, HIdynamics, PYdynamics, Cadynamics
} dynamics_type;
/* Information about the dynamic state of the system, including the
   derivative function */
#define DYNAMICS \
        PyObject_HEAD \
\
  int n;              /* Number of state variables */ \
  int y;              /* Index into state variable vector */ \
\
  bool trace;     /* Are state variables being traced? */ \
  double_array *traceTimes;  /* Trace time points */ \
  double_array **traceData; /* State variable trace points */ \
\
   /* Update function in C */ \
  derivsfcn *derivs; \
\
  currentfcn *current; \
\
  currentfcn *voltage; \
\
   /* Accept remote (ie synaptic) events */ \
  accepterfcn *accepter; \
\
   /* Accept exogenous (ie synaptic) events */ \
  enqfcn *enq; \
\
   /* Cleanup unwanted memory at end of window */ \
  cleanupfcn *cleanup; \
\
  Compartment *owner; \
  Cell        *owningCell; \
  \
  dynamics_type type; \
  \
  bool HinesIntegrable; /* User can inist that this HI */ \
  HIupdatefcn *HIupdate; \
  HIcurrentfcn *HIcurrent;

#define HIDYNAMICS \
    DYNAMICS \

#define VGCDYNAMICS \
    HIDYNAMICS \
    gatefcn **inf; \
    gatefcn **tau; \
    int *exponents; \
	bool useGHKrectification; \
	double ki; \
	struct cadynamicstag *Cadynamics; \
    double Gmax; \
    double Er;

#define CADYNAMICS \
	HIDYNAMICS \
	double B; \
    double Ca0; \
    double Catau; \
	double charge; \
	double co; \
	internal_list *Cadynamics; \
	Caupdatefcn *Caupdate; \
	double (*GHK)(struct cadynamicstag*); \
	double (*Cacurrent)(struct cadynamicstag*, double);

/* Base Dynamics class */
typedef struct dynamicstag {
    DYNAMICS
} Dynamics;

/* Voltage gated channels */
typedef double gatefcn(void*, double, double);
typedef double Caupdatefcn(void*, double, double);
typedef struct vgcdynamicstag {
    VGCDYNAMICS
} VGCDynamics;
typedef struct hidynamicstag {
    HIDYNAMICS
} HIDynamics;

typedef struct cadynamicstag {
    CADYNAMICS
} CaDynamics;

typedef struct synapsetag {
        PyObject_HEAD
  int synapse_id;
  double trans_time;
  double nominal_strength;
  Compartment *owner;
  Cell *owningCell;
  int  tgtrank;     /* MPI rank of remote target */
  Cell *target;
  Dynamics *target_dynamics;
} Synapse;

/* This describes the data structures needed by the 
   module initialisation to add user C dynamics into
   the module */
typedef struct {
    char *name;  /* Warning: This must be the first member!
		    (See initUserDynamics in p3.c) */
    char *docstring;
    PyMemberDef *members;
    PyMethodDef *methods;
    int varcnt;
    char **stateVars;
    char **derivVars;
    char **traceVars;
    derivsfcn *derivs;
    currentfcn *current;
    accepterfcn *accepter;
    enqfcn *enq;
    cleanupfcn *cleanup;
    int basicsize;
    PyTypeObject *type;
    userinitfcn *userInit;
    int *exponents;
    char **exponentNames;
    gatefcn **tau;
    char **tauNames;
    gatefcn **inf;
    char **infNames;
	dynamics_type basetype;
	Caupdatefcn  *Caupdate;
	HIupdatefcn  *HIupdate;
	HIcurrentfcn *HIcurrent;
	currentfcn *voltage;
} DynamicsDescriptor;


/********************************
  Interface to the priority queue
 ********************************/
int   pq_isEmpty(void);
void  pq_insert(Cell*);
void  pq_lazy_insert(Cell*);
void  rr_insert(Cell*);
void *pq_lazy_dequeue(void);
void *rr_dequeue(void);
void  pq_init(int);
void  pq_requeue(Cell*, double, double);
void  pq_remove(Cell*);
double pq_nextTime(void);
int pq_size(void);
void pq_seq_start(void);
void *pq_seq_next(void);


/*************************************
  Solver/stepper interface
 *************************************/
PyObject* parplex(void);
void stepNext(Cell*, double);
void stepOnInt(Cell*, double, Dynamics*, Synapse*, double, int, MsgType);
void removeStep(Synapse*, int);
void stepCommit(Cell*);
void stepRewind(Cell*);

/*************************************
  Random numbers
 *************************************/
void  set_random_seed(int);
int  get_random_seed(void);
/* See also _p3 exports section below */

/*************************************
  Miscellaneous helper functions
 *************************************/
int addState(Cell*, int);
void remState(Cell*, int);
void addTraceEntry(Dynamics*);

/*********************************************
  Generic state/derivative variable handlers
**********************************************/
PyObject *p3_generic_get_statevariable(Dynamics*, void*);
int       p3_generic_set_statevariable(Dynamics*, PyObject*, void*);
PyObject *p3_generic_get_derivvariable(Dynamics*, void*);
int       p3_generic_set_derivvariable(Dynamics*, PyObject*, void*);
PyObject *p3_generic_get_tracevariable(Dynamics*, void*);
PyObject *p3_generic_tau_0(VGCDynamics*, PyObject*, PyObject*);
PyObject *p3_generic_tau_1(VGCDynamics*, PyObject*, PyObject*);
PyObject *p3_generic_tau_2(VGCDynamics*, PyObject*, PyObject*);
PyObject *p3_generic_tau_3(VGCDynamics*, PyObject*, PyObject*);
PyObject *p3_generic_tau_4(VGCDynamics*, PyObject*, PyObject*);
PyObject *p3_generic_inf_0(VGCDynamics*, PyObject*, PyObject*);
PyObject *p3_generic_inf_1(VGCDynamics*, PyObject*, PyObject*);
PyObject *p3_generic_inf_2(VGCDynamics*, PyObject*, PyObject*);
PyObject *p3_generic_inf_3(VGCDynamics*, PyObject*, PyObject*);
PyObject *p3_generic_inf_4(VGCDynamics*, PyObject*, PyObject*);

/***************************************************
  Generic get exponent handlers for the HH subclass
****************************************************/
PyObject *p3_generic_get_exponent(VGCDynamics*, void*);

/*************************************
  Messaging interface
 *************************************/
extern void abend(int, char*, char*, Cell*);
extern void traceout(Cell*);
extern void apout(Cell*);
extern void do_void_handler(PyObject*, GD*);
extern void do_cell_handler(Cell*, PyObject*);
extern double do_dynamics_handler(PyObject*, Dynamics*, double);
extern void mycallobject(PyObject*, PyObject*);
#define ABEND(c, o) abend(__LINE__, __FILE__, c, o)
void dumpCell(Cell*, msglevel);


/******************************
  Numerical methods interfaces
*******************************/
void matrix_init(const GD*, int);
solverfcn rk32_solve;
interpfcn rk32_interp;
solverInitfcn rk32_init;
solverfcn rw23_solve;
interpfcn rw23_interp;
solverInitfcn rw23_init;
solverfcn hines_solve;
solverfcn hines_fixed_step_solve;
interpfcn hines_interp;
solverInitfcn hines_init;
solverfcn rd5_solve;
interpfcn rd5_interp;
solverInitfcn rd5_init;
void derivs(double, double*, double*);
void sparse_derivs(double, double*, double*, double*, int);
sparse_jacobfcn sparse_jacobian;
jacobfcn jacobian;
jacobfcn quality_jacobian;
void linsolve(double**, double*, int*, double*, int, double, Cell*, int, int);
void freevec(double*);
void freemat(double**, int);
double  *vector(int);
double **matrix(int);

/***************************************
  Synaptic messages and wr convergence
****************************************/
void detectAP(GD*, Cell*);
bool exchangeMessages(GD*);
void initMPI(GD*);
void sendLocalMsgNow(Dynamics*, Synapse*, double, int, MsgType);

/**********************
  User module services
**********************/
#define addStateVars_NUM 0
#define addStateVars_RETURN int
#define addStateVars_PROTO (PyObject*, DynamicsDescriptor*)

#define initUserDynamics_NUM 1
#define initUserDynamics_RETURN int
#define initUserDynamics_PROTO (Dynamics*, PyObject*, PyObject*, DynamicsDescriptor*)

#define initUserVGCDynamics_NUM 2
#define initUserVGCDynamics_RETURN int
#define initUserVGCDynamics_PROTO (VGCDynamics*, PyObject*, PyObject*, DynamicsDescriptor*)

#define p3_DynamicsType_NUM 3

#define p3_VGCDynamicsType_NUM 4

#define getmain_NUM 5
#define getmain_RETURN void *
#define getmain_PROTO (int)

#define rand_flat_NUM 6
#define rand_flat_RETURN double
#define rand_flat_PROTO (void)

#define rand_norm_NUM 7
#define rand_norm_RETURN double
#define rand_norm_PROTO (double, double)

#define stepOn_NUM 8
#define stepOn_RETURN void
#define stepOn_PROTO (Cell*, double)

#define p3_SynapseType_NUM 9

#define message_NUM 10
#define message_RETURN void
#define message_PROTO (msglevel, char*, ...)

#define p3_HinesIntegrableDynamicsType_NUM 11

#define p3_CaDynamicsType_NUM 12

#define makeUserClass_NUM 13
#define makeUserClass_PROTO (DynamicsDescriptor**, initproc*, int, PyObject*)
#define makeUserClass_RETURN void

#define il_set_NUM 14
#define il_set_PROTO (internal_list*, PyObject*, int)
#define il_set_RETURN void

#define il_get_NUM 15
#define il_get_PROTO (internal_list*, int)
#define il_get_RETURN void*

#define il_clear_NUM 16
#define il_clear_PROTO (internal_list*)
#define il_clear_RETURN void

#define il_length_NUM 17
#define il_length_PROTO (internal_list*)
#define il_length_RETURN int

#define il_append_NUM 18
#define il_append_PROTO (internal_list*, PyObject*)
#define il_append_RETURN int

#define il_copy_NUM 19
#define il_copy_PROTO (internal_list*, internal_list*)
#define il_copy_RETURN void

#define il_remove_NUM 20
#define il_remove_PROTO (internal_list*, int)
#define il_remove_RETURN void

#define il_new_NUM 21
#define il_new_PROTO (char*)
#define il_new_RETURN internal_list*

#define il_seq_next_NUM 22
#define il_seq_next_PROTO (internal_list*, int*)
#define il_seq_next_RETURN void*

#define stepOnInt_NUM 23
#define stepOnInt_PROTO (Cell*, double, Dynamics*, Synapse*, double, int, MsgType)
#define stepOnInt_RETURN void

#define debug_msg_NUM 24
#define debug_msg_PROTO (char*, ...)
#define debug_msg_RETURN void

#define GETEM_CMPT_NUM 25
#define GETEM_CMPT_PROTO (Compartment*, double)
#define GETEM_CMPT_RETURN double

/* Total number of API pointers */
#define p3_API_pointers 26

#ifdef P3_MODULE
addStateVars_RETURN addStateVars addStateVars_PROTO;
initUserDynamics_RETURN initUserDynamics initUserDynamics_PROTO;
initUserVGCDynamics_RETURN initUserVGCDynamics initUserVGCDynamics_PROTO;
getmain_RETURN getmain getmain_PROTO;
rand_norm_RETURN rand_norm rand_norm_PROTO;
rand_flat_RETURN rand_flat rand_flat_PROTO;
stepOn_RETURN stepOn stepOn_PROTO;
message_RETURN message message_PROTO;
makeUserClass_RETURN makeUserClass makeUserClass_PROTO;
il_set_RETURN il_set il_set_PROTO;
il_get_RETURN il_get il_get_PROTO;
il_clear_RETURN il_clear il_clear_PROTO;
il_length_RETURN il_length il_length_PROTO;
il_append_RETURN il_append il_append_PROTO;
il_copy_RETURN il_copy il_copy_PROTO;
il_remove_RETURN il_remove il_remove_PROTO;
il_new_RETURN il_new il_new_PROTO;
il_seq_next_RETURN il_seq_next il_seq_next_PROTO;
stepOnInt_RETURN stepOnInt stepOnInt_PROTO;
debug_msg_RETURN debug_msg debug_msg_PROTO;
GETEM_CMPT_RETURN GETEM_CMPT GETEM_CMPT_PROTO;
#else
static void **p3_API;

#define addStateVars \
(*(addStateVars_RETURN (*)addStateVars_PROTO) p3_API[addStateVars_NUM])
#define p3_DynamicsType \
(PyTypeObject*) p3_API[p3_DynamicsType_NUM]
#define p3_VGCDynamicsType \
(PyTypeObject*) p3_API[p3_DynamicsType_NUM]
#define p3_CaDynamicsType \
(PyTypeObject*) p3_API[p3_CaDynamicsType_NUM]
#define p3_HinesIntegrableDynamicsType \
(PyTypeObject*) p3_API[p3_HinesIntegrableDynamicsType_NUM]
#define initUserDynamics \
(*(initUserDynamics_RETURN (*)initUserDynamics_PROTO) p3_API[initUserDynamics_NUM])
#define initUserVGCDynamics \
(*(initUserVGCDynamics_RETURN (*)initUserVGCDynamics_PROTO) p3_API[initUserVGCDynamics_NUM])
#define getmain \
(*(getmain_RETURN (*)getmain_PROTO) p3_API[getmain_NUM])
#define rand_norm \
(*(rand_norm_RETURN (*)rand_norm_PROTO) p3_API[rand_norm_NUM])
#define rand_flat \
(*(rand_flat_RETURN (*)rand_flat_PROTO) p3_API[rand_flat_NUM])
#define stepOn \
(*(stepOn_RETURN (*)stepOn_PROTO) p3_API[stepOn_NUM])
#define p3_SynapseType \
(PyTypeObject*) p3_API[p3_SynapseType_NUM]
#define message \
(*(message_RETURN (*)message_PROTO) p3_API[message_NUM])

#define makeUserClass \
(*(makeUserClass_RETURN (*)makeUserClass_PROTO) p3_API[makeUserClass_NUM])

#define il_set \
(*(il_set_RETURN (*)il_set_PROTO) p3_API[il_set_NUM])
#define il_get \
(*(il_get_RETURN (*)il_get_PROTO) p3_API[il_get_NUM])
#define il_clear \
(*(il_clear_RETURN (*)il_clear_PROTO) p3_API[il_clear_NUM])
#define il_length \
(*(il_length_RETURN (*)il_length_PROTO) p3_API[il_length_NUM])
#define il_append \
(*(il_append_RETURN (*)il_append_PROTO) p3_API[il_append_NUM])
#define il_copy \
(*(il_copy_RETURN (*)il_copy_PROTO) p3_API[il_copy_NUM])
#define il_remove \
(*(il_remove_RETURN (*)il_remove_PROTO) p3_API[il_remove_NUM])
#define il_new \
(*(il_new_RETURN (*)il_new_PROTO) p3_API[il_new_NUM])

#define il_seq_next \
(*(il_seq_next_RETURN (*)il_seq_next_PROTO) p3_API[il_seq_next_NUM])
#define stepOnInt \
(*(stepOnInt_RETURN (*)stepOnInt_PROTO) p3_API[stepOnInt_NUM])

#define debug_msg \
(*(debug_msg_RETURN (*)debug_msg_PROTO) p3_API[debug_msg_NUM])

#define GETEM_CMPT \
(*(GETEM_CMPT_RETURN (*)GETEM_CMPT_PROTO) p3_API[GETEM_CMPT_NUM])

static int import_p3(void) {
  PyObject *m = PyImport_ImportModule("parplex._p3");

  if( m ) {
    PyObject *c_api_object = PyObject_GetAttrString(m, "_C_API");
    if( !c_api_object ) return -1;
    if( PyCObject_Check(c_api_object) )
      p3_API = (void**)PyCObject_AsVoidPtr(c_api_object);
    Py_DECREF(c_api_object);
    return 0;
  } else
    return -1;
}
#endif

  
#define REGISTER_DESCRIPTOR(descriptor) \
static  int init##descriptor(PyObject *self, \
PyObject *args, PyObject *kw) { \
	return initUserDynamics((Dynamics*)self, args, kw, &descriptor); \
}

#define REGISTER_VGC_DESCRIPTOR(descriptor) \
static  int init##descriptor(PyObject *self, \
PyObject *args, PyObject *kw) { \
return initUserVGCDynamics((VGCDynamics*)self, args, kw, &descriptor); \
}

#ifndef PyMODINIT_FUNC	/* declarations for DLL import/export */
#define PyMODINIT_FUNC void
#endif

#define MAKE_P3_MODEL(NAME, DOC) \
PyMODINIT_FUNC init##NAME(void) { \
    PyObject* m; \
	m = Py_InitModule3(#NAME, 0, DOC); \
    if( import_p3()<0 ) return; \
	makeUserClass(userDynamics, \
	LuserInitDynamics, \
	sizeof(LuserInitDynamics)/sizeof(initproc*), \
	m);\
}

extern GD *gd;

/* Kludges to compile in Microsoft development thingy */
#ifndef min
#define min(a,b) ((a)<(b)?(a):(b))
#endif
#ifndef max
#define max(a,b) ((a)>(b)?(a):(b))
#endif
#ifdef _WIN32
#define  isnan(X)   _isnan(X) 
#define  finite(X)   _finite(X)
#endif

#endif

/* Changes:
   EAT 23Jun08
   Start to remove dependence on libc
   */
