/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


/***********************************************************
    The _p3 python module definitions and initialisation.
************************************************************/


#ifdef _DEBUG
#undef _DEBUG
#include <Python.h>
#include "structmember.h"
#define _DEBUG
#else
#include <Python.h>
#include "structmember.h"
#endif

#ifdef MPI
#include <mpi.h>
#endif

#define P3_MODULE
#include "ndl.h"

int mpi_rank;
int mpi_size;

void makeUserClass(DynamicsDescriptor **userDynamics, 
				   initproc *LuserInitDynamics, int N, PyObject *m) {
	/* Make the user type object for this class */
	int i, n;

    for(i=0; i<N; i++) {
        PyTypeObject *o = PyMem_Malloc(sizeof(*o));
        DynamicsDescriptor *d = userDynamics[i];

        if( !o ) return;
        memset(o, 0, sizeof(*o));
        o->tp_name      = d->name;
        o->tp_basicsize = d->basicsize;
        o->tp_doc       = d->docstring;
		n = 0;
		if( d->members ) {while( d->members[n++].name );n--;}		switch( d->basetype ) {
			case dynamics:
				o->tp_base = &p3_DynamicsType;
				o->tp_methods   = d->methods;
				o->tp_members   = d->members;
				break;

			case VGCdynamics:
				o->tp_base = &p3_VGCDynamicsType;
				o->tp_members = d->members;
				break;

			case HIdynamics:
				o->tp_base = &p3_HinesIntegrableDynamicsType;
				o->tp_members   = d->members;
				break;

			case Cadynamics:
				o->tp_base = &p3_CaDynamicsType;
				o->tp_members = d->members;
				break;

			default:
				message(fatal, "Invalid base specified in user module\n");
				exit(0);
		}
        o->tp_init      = (initproc)LuserInitDynamics[i];
        o->tp_flags     = Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE;

        d->type = o;

        if( PyType_Ready(o) < 0) return;

	/* Add the user C classes */
        Py_INCREF(d->type);
        PyModule_AddObject(m, d->name, (PyObject *)d->type);
    }
}


static PyObject *route_parplex(PyObject *self, PyObject *ingd) {
    if( !PyObject_TypeCheck(ingd, &p3_GDType) ) {
        PyErr_SetString(PyExc_TypeError,
            "The parameter to parplex must be of type GD");
        return NULL;
    }
    gd = (GD*)ingd;
    return parplex();
}

static PyObject *route_set_random_seed(PyObject *self, PyObject *seed) {
    if( !PyInt_Check(seed) ) {
        PyErr_SetString(PyExc_TypeError,
            "The seed must be an integer");
        return NULL;
    }
    set_random_seed(PyInt_AsLong(seed));
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *route_get_random_seed(PyObject *self, PyObject *args) {
    return PyInt_FromLong(get_random_seed());
}

static PyObject *route_rand_flat(PyObject *self, PyObject *args) {
    return PyFloat_FromDouble(rand_flat());
}

static PyObject *route_rand_norm(PyObject *self, PyObject *args) {
    double mean, std;
    if( !PyArg_ParseTuple(args, "dd", &mean, &std) ) return NULL;
    return PyFloat_FromDouble(rand_norm(mean, std));
}

static PyObject *route_stepOn(PyObject *self, PyObject *args) {
    double t;
    PyObject *cell;
    if( !PyArg_ParseTuple(args, "Od", &cell, &t) ) return NULL;
    if( PyObject_TypeCheck(cell, &p3_CellType) ) {
        stepOn((Cell*)cell, t);
        Py_INCREF(Py_None);
        return Py_None;
    }
    if( PyObject_TypeCheck(cell, &p3_CompartmentType) ) {
        stepOn(((Compartment*)cell)->owner, t);
        Py_INCREF(Py_None);
        return Py_None;
    }
    PyErr_SetString(PyExc_TypeError,
        "The first argument to stepOn() must be a Cell or a Compartment");
    return NULL;
}

static PyObject *p3_MPI_Finalize(PyObject *self, PyObject *args) {
#ifdef MPI
	MPI_Finalize();
#endif
	return Py_None;
}

static PyObject *p3_MPI_Abort(PyObject *self, PyObject *args) {
#ifdef MPI
	MPI_Abort(MPI_COMM_WORLD, 0);
#endif
	return Py_None;
}

static PyMethodDef p3_methods[] = {
    {"parplex", route_parplex, METH_O, "Perform a network simulation: parplex(GD*)"},
    {"set_random_seed", route_set_random_seed, METH_O, "set_random_seed(int): Set the random number generator seed"},
    {"get_random_seed", route_get_random_seed, METH_NOARGS, "get_random_seed(): Get the random number generator seed"},
    {"rand_flat", route_rand_flat, METH_NOARGS, "rand_flat(): Random numbers uniformly distributed in [0, 1]"},
    {"rand_norm", route_rand_norm, METH_VARARGS, "rand_norm(mean, std): Uniformly distributed random numbers"},
    {"stepOn", route_stepOn, METH_VARARGS, "stepOn(cell, time): Ensure that an integration step occurs at this time"},
	{"MPI_Finalize", p3_MPI_Finalize, METH_NOARGS, "call MPI_Finalize."},
	{"MPI_Abort", p3_MPI_Abort, METH_NOARGS, "call MPI_Abort(MPI_COMM_WORLD,0)."},
    {NULL}  /* Sentinel */
};

int addStateVars(PyObject *self, DynamicsDescriptor *dyndescr) {
    /* The 'interesting' feature of this initialiser is that it looks
       for a class variable "stateVars" which is a list of the state 
       variables.  It then adds PyGetSetDefs for them and sets flag
       in the type dictionary so that it is only done once. */
    char **state, **dxdt, **trace;
    char **exponentNames, **tauNames, **infNames;
    int *exponents;
    gatefcn **tau, **inf;
    int firstTimeFlag;
    Dynamics *d = (Dynamics*)self;
    int i, size;

    /* Have we already done this */
    firstTimeFlag = 
        PyDict_GetItemString(self->ob_type->tp_dict, "stateVarsFlag")==0;

    /* Set flag in the class dictionary so we don't do this again */
    if( firstTimeFlag ) {
        Py_INCREF(Py_None);
        if( PyDict_SetItemString(self->ob_type->tp_dict, "stateVarsFlag", Py_None)!=0 )
            return -1;
    }

    size  = dyndescr->varcnt;
    state = dyndescr->stateVars;
    dxdt  = dyndescr->derivVars;
    trace = dyndescr->traceVars;

    exponents      = dyndescr->exponents;
    exponentNames  = dyndescr->exponentNames;
    tau            = dyndescr->tau;
    tauNames       = dyndescr->tauNames;
    inf            = dyndescr->inf;
    infNames       = dyndescr->infNames;

    if( state && !dxdt ) {
        PyErr_SetString(PyExc_SystemError,
            "Inconsistent DynamicsDescriptor detected during module initialisation");
        return -1;
    }

    if( (exponents&&exponentNames&&tau&&tauNames&&inf&&infNames)
        != (exponents!=0) ) {
        PyErr_SetString(PyExc_SystemError,
            "Inconsistent DynamicsDescriptor detected during module initialisation");
        return -1;
    }

    /* Add the GetSetDefs for the state variables and their derivatives */
    if( state ) {
        d->y = addState(d->owningCell, size);
        d->n = size;
    } else {
        d->y = -1;
        d->n = 0;
    }

    if( firstTimeFlag && state ) {

        for(i=0; i<size; i++) {
            PyGetSetDef *gs;
            PyObject *descr;

            /* ith state variable */
            gs = PyMem_Malloc(sizeof(*gs));
            if( !gs ) {
                PyErr_NoMemory();
                return -1;
            }
            gs->name = state[i];
            gs->doc  = "State variable";
            gs->get  = (getter)p3_generic_get_statevariable;
            gs->set  = (setter)p3_generic_set_statevariable;
            gs->closure = (void*)i; /* Legal and portable methinks! */
            descr = PyDescr_NewGetSet(self->ob_type, gs);
            if( !descr ) return -1;
            if( PyDict_SetItemString(self->ob_type->tp_dict, state[i], descr)!=0 )
                return -1;

            /* ith derivative variable */
            gs = PyMem_Malloc(sizeof(*gs));
            if( !gs ) {
                PyErr_NoMemory();
                return -1;
            }
            gs->name = dxdt[i];
            gs->doc  = "Derivative of a state variable";
            gs->get  = (getter)p3_generic_get_derivvariable;
            gs->set  = (setter)p3_generic_set_derivvariable;
            gs->closure = (void*)i; /* Legal and portable methinks! */
            descr = PyDescr_NewGetSet(self->ob_type, gs);
            if( !descr ) return -1;
            if( PyDict_SetItemString(self->ob_type->tp_dict, dxdt[i], descr)!=0 )
                return -1;
        }
    }

    if( firstTimeFlag && state && trace) {
        for(i=0; i<size; i++) {
            PyGetSetDef *gs;
            PyObject *descr;

            /* ith trace variable */
            gs = PyMem_Malloc(sizeof(*gs));
            if( !gs ) {
                PyErr_NoMemory();
                return -1;
            }
            gs->name = trace[i];
            gs->doc  = "Trace variable";
            gs->get  = (getter)p3_generic_get_tracevariable;
            gs->set  = 0;
            gs->closure = (void*)i; /* Legal and portable methinks! */
            descr = PyDescr_NewGetSet(self->ob_type, gs);
            if( !descr ) return -1;
            if( PyDict_SetItemString(self->ob_type->tp_dict, trace[i], descr)!=0 )
                return -1;
        }
    }

    /* Add the special variables and methods for HH channels */
    if( firstTimeFlag && exponents ) {
        for(i=0; i<size; i++) {
            PyGetSetDef *gs;
            PyMethodDef *method;
            PyObject *descr;

            /* ith exponent */
            gs = PyMem_Malloc(sizeof(*gs));
            if( !gs ) {
                PyErr_NoMemory();
                return -1;
            }
            gs->name = exponentNames[i];
            gs->doc  = "gating exponent";
            gs->get  = (getter)p3_generic_get_exponent;
            gs->set  = 0;
            gs->closure = (void*)i; /* Legal and portable methinks! */
            descr = PyDescr_NewGetSet(self->ob_type, gs);
            if( !descr ) return -1;
            if( PyDict_SetItemString(self->ob_type->tp_dict, exponentNames[i], descr)!=0 )
                return -1;

            /* ith tau method */
            method = PyMem_Malloc(sizeof(*method));
            if( !method ) {
                PyErr_NoMemory();
                return -1;
            }
            method->ml_name = tauNames[i];
            method->ml_doc = "gating rate (tau)";
            method->ml_flags = METH_VARARGS;
            switch(i) {
                case 0:
                    method->ml_meth = (PyCFunction)p3_generic_tau_0;
                    break;
                case 1:
                    method->ml_meth = (PyCFunction)p3_generic_tau_1;
                    break;
                case 2:
                    method->ml_meth = (PyCFunction)p3_generic_tau_2;
                    break;
                case 3:
                    method->ml_meth = (PyCFunction)p3_generic_tau_3;
                    break;
                case 4:
                    method->ml_meth = (PyCFunction)p3_generic_tau_4;
                    break;
                default:
                    PyErr_SetString(PyExc_SystemError,
                        "More than 5 gating variables not supported at the moment");
                    return -1;
            }
            descr = PyDescr_NewMethod(self->ob_type, method);
            if( !descr ) return -1;
            if( PyDict_SetItemString(self->ob_type->tp_dict, tauNames[i], descr)!=0 )
                return -1;

            /* ith inf method */
            method = PyMem_Malloc(sizeof(*method));
            if( !method ) {
                PyErr_NoMemory();
                return -1;
            }
            method->ml_name = infNames[i];
            method->ml_doc = "gating activation (inf)";
            method->ml_flags = METH_VARARGS;
            switch(i) {
                case 0:
                    method->ml_meth = (PyCFunction)p3_generic_inf_0;
                    break;
                case 1:
                    method->ml_meth = (PyCFunction)p3_generic_inf_1;
                    break;
                case 2:
                    method->ml_meth = (PyCFunction)p3_generic_inf_2;
                    break;
                case 3:
                    method->ml_meth = (PyCFunction)p3_generic_inf_3;
                    break;
                case 4:
                    method->ml_meth = (PyCFunction)p3_generic_inf_4;
                    break;
                default:
                    PyErr_SetString(PyExc_SystemError,
                        "More than 5 gating variables not supported at the moment");
                    return -1;
            }
            descr = PyDescr_NewMethod(self->ob_type, method);
            if( !descr ) return -1;
            if( PyDict_SetItemString(self->ob_type->tp_dict, infNames[i], descr)!=0 )
                return -1;
        }
    }

    if( !state ) {
        d->y    = -1;
        d->n    = 0;
    }

    return 0;
 }

int initUserVGCDynamics(VGCDynamics *self, PyObject *args, 
                              PyObject *kw, DynamicsDescriptor *d) {
	/* Superclass initialiser */
	int rc = self->ob_type->tp_base->tp_init((PyObject*)self, args, kw);
	message(warn, "Fix the superclass initialiaser call!\n");

	if( rc ) return rc;

    self->exponents = d->exponents;
    self->tau       = d->tau;
    self->inf       = d->inf;

    /* Add interface elements for the state variables */
    rc = addStateVars((PyObject*)self, d);
    if( rc ) return rc;
    
    if( d->userInit ) return d->userInit((Dynamics*)self);
    else return 0;
}

int initUserDynamics(Dynamics *self, PyObject *args, 
                            PyObject *kw, DynamicsDescriptor *d) {
    int rc = p3_DynamicsType.tp_init((PyObject*)self, args, kw);

    /* Add interface elements for the state variables */
    rc = addStateVars((PyObject*)self, d);
    if( rc ) return rc;

	if( d->derivs )   self->derivs   = d->derivs;
    if( d->accepter ) self->accepter = d->accepter;
	if( d->cleanup )  self->cleanup  = d->cleanup;
	if( d->current )  self->current  = d->current;
	if( d->voltage )  self->voltage  = d->voltage;
	if( d->enq )      self->enq      = d->enq;
	if( d->Caupdate ) ((CaDynamics*)self)->Caupdate = d->Caupdate;
	if( d->HIcurrent ) self->HIcurrent = d->HIcurrent;
	if( d->HIupdate )  self->HIupdate  = d->HIupdate;

    if( d->userInit ) return d->userInit(self);
    else return 0;
}

#ifdef MPI
static void do_MPI() {
  int argc, rc, i;
  char **argv;
  PyObject *args;
  PyObject *sys;

  sys = PyImport_ImportModule("sys");
  if( !sys ) ABEND("Could not import sys.\n", (Cell*)Py_None);
  args = PyObject_GetAttrString(sys, "argv");
  if( !args ) ABEND("Could not get sys.argv.\n", (Cell*)Py_None);

  /************************************************************/
  /* This code fragment courtesy of Ole Neilsen and his pypar */
  /************************************************************/
  argc = PyList_Size(args); /*Number of commandline arguments*/
  argv = (char**) malloc((argc+1)*sizeof(char*)); 

  for (i=0; i<argc; i++)
    argv[i] = PyString_AsString(PyList_GetItem(args, i));
  argv[i] = NULL; /*Lam 7.0 requires last arg to be NULL  */

  Py_DECREF(args);
  Py_DECREF(sys);

  rc = MPI_Init(&argc, &argv); 
  if( rc!=0 ) {
	  printf("MPI initialisation failed rc=%d.\n", rc);
	  longjmp(excpt_exit, 1);
  }
}
#endif

static PyObject *ParplexRuntimeError;

#ifndef PyMODINIT_FUNC	/* declarations for DLL import/export */
#define PyMODINIT_FUNC void
#endif
PyMODINIT_FUNC init_p3(void) {
    PyObject* m;
    static void *p3_API[p3_API_pointers];
    PyObject *c_api_object;

    /* Ready the base classes */
    if( PyType_Ready(&p3_DoubleArrayType) < 0) return;
    if( PyType_Ready(&p3_CellType) < 0) return;
    if( PyType_Ready(&p3_CompartmentType) < 0) return;
    if( PyType_Ready(&p3_SynapseType) < 0) return;
    if( PyType_Ready(&p3_InternalListType) < 0) return;
    if( PyType_Ready(&p3_GDType) < 0) return;
    if( PyType_Ready(&p3_DynamicsType) < 0) return;
    if( PyType_Ready(&p3_PyDynamicsType) < 0) return;
    if( PyType_Ready(&p3_HinesIntegrableDynamicsType) < 0) return;
    if( PyType_Ready(&p3_VGCDynamicsType) < 0) return;
    if( PyType_Ready(&p3_CaDynamicsType) < 0) return;

    m = Py_InitModule3("_p3", p3_methods,
                       "The parplex neural network simulation extension.");

    /* Add the base classes */
    Py_INCREF(&p3_DoubleArrayType);
    PyModule_AddObject(m, "DoubleArray", (PyObject *)&p3_DoubleArrayType);
    Py_INCREF(&p3_CellType);
    PyModule_AddObject(m, "Cell_C", (PyObject *)&p3_CellType);
    Py_INCREF(&p3_CompartmentType);
    PyModule_AddObject(m, "Compartment_C", (PyObject *)&p3_CompartmentType);
    Py_INCREF(&p3_SynapseType);
    PyModule_AddObject(m, "Synapse_C", (PyObject *)&p3_SynapseType);
    Py_INCREF(&p3_InternalListType);
    PyModule_AddObject(m, "InternalList", (PyObject *)&p3_InternalListType);
    Py_INCREF(&p3_GDType);
    PyModule_AddObject(m, "GD_C", (PyObject *)&p3_GDType);
    Py_INCREF(&p3_DynamicsType);
    PyModule_AddObject(m, "Dynamics", (PyObject *)&p3_DynamicsType);
    Py_INCREF(&p3_PyDynamicsType);
    PyModule_AddObject(m, "PyDynamics", (PyObject *)&p3_PyDynamicsType);
    Py_INCREF(&p3_HinesIntegrableDynamicsType);
    PyModule_AddObject(m, "HinesIntegrableDynamics", (PyObject *)&p3_HinesIntegrableDynamicsType);
    Py_INCREF(&p3_VGCDynamicsType);
    PyModule_AddObject(m, "VGCDynamics", (PyObject *)&p3_VGCDynamicsType);
    Py_INCREF(&p3_CaDynamicsType);
    PyModule_AddObject(m, "CaDynamics", (PyObject *)&p3_CaDynamicsType);

    /* Add constants */
    PyModule_AddObject(m, "debug", PyInt_FromLong(debug));
    PyModule_AddObject(m, "info",  PyInt_FromLong(info));
    PyModule_AddObject(m, "warn",  PyInt_FromLong(warn));
    PyModule_AddObject(m, "fatal", PyInt_FromLong(fatal));
    PyModule_AddObject(m, "CurrentClamp", PyString_FromString("CurrentClamp"));
    PyModule_AddObject(m, "VoltageClamp", PyString_FromString("VoltageClamp"));
    PyModule_AddObject(m, rk32,    PyString_FromString(rk32));
    PyModule_AddObject(m, rw23,    PyString_FromString(rw23));
    PyModule_AddObject(m, rdIIA,   PyString_FromString(rdIIA));
    PyModule_AddObject(m, hines,   PyString_FromString(hines));
    PyModule_AddObject(m, hines_fixed_step, PyString_FromString(hines_fixed_step));

    /* Add exception(s) */
    ParplexRuntimeError = PyErr_NewException("_p3.ParplexRuntimeError",
					     NULL, NULL);
    if( !ParplexRuntimeError ) return; /* Huh? */
    Py_INCREF(ParplexRuntimeError);
    PyModule_AddObject(m, "ParplexRuntimeError", ParplexRuntimeError);

    /* Initialise the C API */
    p3_API[addStateVars_NUM]        = (void*)addStateVars;
    p3_API[initUserDynamics_NUM]    = (void*)initUserDynamics;
    p3_API[initUserVGCDynamics_NUM] = (void*)initUserVGCDynamics;
    p3_API[p3_DynamicsType_NUM]     = (void*)&p3_DynamicsType;
    p3_API[p3_VGCDynamicsType_NUM]  = (void*)&p3_VGCDynamicsType;
    p3_API[p3_CaDynamicsType_NUM]   = (void*)&p3_CaDynamicsType;
    p3_API[p3_HinesIntegrableDynamicsType_NUM]  = (void*)&p3_HinesIntegrableDynamicsType;
    p3_API[getmain_NUM]             = (void*)getmain;
    p3_API[rand_flat_NUM]           = (void*)rand_flat;
    p3_API[rand_norm_NUM]           = (void*)rand_norm;
    p3_API[stepOn_NUM]              = (void*)stepOn;
    p3_API[p3_SynapseType_NUM]      = (void*)&p3_SynapseType;
    p3_API[message_NUM]             = (void*)message;
    p3_API[makeUserClass_NUM]       = (void*)makeUserClass;
    p3_API[il_set_NUM]              = (void*)il_set;
    p3_API[il_get_NUM]              = (void*)il_get;
    p3_API[il_clear_NUM]            = (void*)il_clear;
    p3_API[il_length_NUM]           = (void*)il_length;
    p3_API[il_append_NUM]           = (void*)il_append;
    p3_API[il_copy_NUM]             = (void*)il_copy;
    p3_API[il_remove_NUM]           = (void*)il_remove;
    p3_API[il_new_NUM]              = (void*)il_new;
    p3_API[il_seq_next_NUM]         = (void*)il_seq_next;
    p3_API[stepOnInt_NUM]           = (void*)stepOnInt;
    p3_API[debug_msg_NUM]           = (void*)debug_msg;
    p3_API[GETEM_CMPT_NUM]          = (void*)GETEM_CMPT;
    
    c_api_object = PyCObject_FromVoidPtr((void *)p3_API, NULL);
    if( c_api_object != NULL ) PyModule_AddObject(m, "_C_API", c_api_object);

#ifdef MPI
    do_MPI();
    MPI_Comm_size(MPI_COMM_WORLD, &mpi_size);
    MPI_Comm_rank(MPI_COMM_WORLD, &mpi_rank);
#else
	mpi_size = 1;
	mpi_rank = 0;
#endif

    PyModule_AddObject(m, "mpi_rank",  PyInt_FromLong(mpi_rank));
    PyModule_AddObject(m, "mpi_size",  PyInt_FromLong(mpi_size));
}

/*************************************
  Miscellaneous helper functions
 *************************************/
void *getmain(int size) {
    char buf[200];
    void *rc = PyMem_Malloc(size);
    if( rc ) {
        memset(rc, 0, size);
        return rc;
    }
    sprintf(buf, "Unable to allocate %d bytes of memory", size);
    ABEND(buf, (Cell*)Py_None);
    return 0;
}

 void do_cell_handler(Cell *c, PyObject *handler) {
  PyObject *arglist;

  if( !handler )
      ABEND("A cell required a handler that has not been set.\n", c);
  arglist = Py_BuildValue("(O)", c);
  if( !arglist ) longjmp(excpt_exit, 1);
  mycallobject(handler, arglist);
  Py_DECREF(arglist);
}

 void do_void_handler(PyObject *handler, GD *gd) {
   PyObject *arglist = Py_BuildValue("(O)", gd);
   if( !arglist ) longjmp(excpt_exit, 1);
   mycallobject(handler, arglist);
}
 
void traceout(Cell *c) {
  if( gd->trace_handler ) do_cell_handler(c, gd->trace_handler);
}

void apout(Cell *c) {
  if( gd->ap_handler ) do_cell_handler(c, gd->ap_handler);
}

extern jmp_buf excpt_exit;

void mycallobject(PyObject *handler, PyObject *arglist) {
  PyObject *rc;
  int i;

  if( !PyList_Check(handler) ) {
    rc = PyObject_CallObject(handler, arglist);
    if( rc == NULL ) longjmp(excpt_exit, 1);
    Py_DECREF(rc);
    return;
  }

  for(i=0; i<PyList_Size(handler); i++) {
    rc = PyObject_CallObject(PyList_GetItem(handler, i), arglist);
    if( rc == NULL ) longjmp(excpt_exit, 1);
    Py_DECREF(rc);
  }
}

void abend(int lineno, char *file, char *msg, Cell *o) {
    char buf[1000];
    sprintf(buf, "%s:%d %s", file, lineno, msg);
    PyErr_SetString(ParplexRuntimeError, buf);
    PyObject_SetAttrString(ParplexRuntimeError, "currentCell", (PyObject*)o);
    longjmp(excpt_exit, 1);
}

