/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*****************************************************************
   Routines to control integration step size.  The integrator
   recommends a step size but this must be reconsiled with brief,
   discontinious events that the solver doesn't know about.
******************************************************************/


#define P3_MODULE
#include "ndl.h"

void stepOn(Cell *cell, double time) {
	stepOnInt(cell, time, 0, 0, 0, -1, nomsg);
}

void stepOnInt(Cell *cell, double time, Dynamics *d, Synapse *s,
			   double strength, int window_id, MsgType msgtype) {
  /* Set a time that the cell must step at for example brief event or
     end of window */
  int i;

  /* find where the new value must be inserted ...*/
  i=0;
  while( cell->mustStep[i].time < time ) i++;
  
  /* ... and insert */
  if( cell->mustStep[i].time != MUSTSTEP_SENTINEL ) {
    memmove(&cell->mustStep[i+1], &cell->mustStep[i],
	    (cell->mustStepLen-i-1)*sizeof(*cell->mustStep));
  }

  cell->mustStep[i].time      = time;
  cell->mustStep[i].dynamics  = d;
  cell->mustStep[i].synapse   = s;
  cell->mustStep[i].strength  = strength;
  cell->mustStep[i].window_id = window_id;
  cell->mustStep[i].msgtype   = msgtype;

  /* The cell might need an immediate stepsize adjustment */
  if( i==cell->mustStepStart && cell->time+cell->step>cell->mustStep[cell->mustStepStart].time ) {
    double newstep = cell->mustStep[cell->mustStepStart].time-cell->time;
    cell->laststepaccept = false;
    stepNext(cell, newstep);
    pq_requeue(cell, cell->time, newstep);
  }

  /* If the mustStep buffer needs to be increased */
  if( cell->mustStep[cell->mustStepLen-1].time != MUSTSTEP_SENTINEL ) {
    int n = cell->mustStepLen + MUSTSTEP_BUFSIZE;
    cell->mustStep      = PyMem_Realloc(cell->mustStep, n*sizeof(*cell->mustStep));
    for(i=cell->mustStepLen; i<n; i++) {
      cell->mustStep[i].time = MUSTSTEP_SENTINEL;
    }
    cell->mustStepLen = n;
  }

  return;  
}

extern GD *gd;

void stepNext(Cell *cell, double stepnext) {
  /* Determine next step size based on upcoming events and the solver's
     recommendation. */

	int mustStepStart = cell->mustStepStart;

  if( stepnext==0 )
    ABEND("Zero step detected in step control", cell);

  if( gd->maxStep>0 && stepnext>gd->maxStep ) {
	  stepnext = gd->maxStep;
	  cell->maxStepCnt++;
  }
  
  /* Does the cell need to receive any syncrhonous messages */
  while( cell->laststepaccept && cell->time == cell->mustStep[mustStepStart].time ) {
	  if( cell->mustStep[mustStepStart].msgtype!=nomsg )
		sendLocalMsgNow(cell->mustStep[mustStepStart].dynamics,
						cell->mustStep[mustStepStart].synapse, 
						cell->mustStep[mustStepStart].strength, 
						cell->mustStep[mustStepStart].window_id, 
						cell->mustStep[mustStepStart].msgtype);

	  mustStepStart++;
  }

  if( cell->time+stepnext > cell->mustStep[mustStepStart].time ) {
	  if( cell->time == cell->mustStep[mustStepStart].time ) {
		cell->step = stepnext;
	  } else {
	      cell->step = cell->mustStep[mustStepStart].time - cell->time;
	  }
  } else {
	cell->step = stepnext;
  }

  if( cell->step<0 ) {
	  message(warn, "Bad step: cell.id=%d cell.time=%g cell.step=%g laststepaccept=%d\n",
		  cell->id, cell->time, cell->step, cell->laststepaccept);
	  cell->step = 1e-3;
	  cell->badStepCnt++;
  }

  cell->mustStepStart = mustStepStart;
  
  return;
}

void stepCommit(Cell *cell) {
	/* The window has ended, so now we can overwrite all the old messages */
	mustStepObject *ms = cell->mustStep;
	int n = cell->mustStepStart;
	int l = cell->mustStepLen;
	int i;
	if( n==0 ) return;
	for(i=0; i<l-n; i++) ms[i] = ms[i+n];
	for( ; i<l; i++) ms[i].time = MUSTSTEP_SENTINEL;
	cell->mustStepStart = 0;

	for(i=0; i<l; i++) {
		if( ms[i].time == gd->window )
			ABEND("mustStep buffer corruption.\n", cell);
		if( i>0 && ms[i].time<ms[i-1].time )
			ABEND("mustStep buffer corruption.\n", cell);
	}
}

void stepRewind(Cell *cell) {
	/* Cell needs to be reset back to the beginning of the window */
	cell->mustStepStart = 0;
}

void removeStep(Synapse *s, int window_id) {
	/* Remove queued messages from the mustStep buffer */
	Cell *cell = s->target;
	int i = 0;
	mustStepObject *ms;

	/* There could be more than one message from the synapse
	   to the dynamics in this window. */
	while( cell->mustStep[i].window_id == window_id && cell->mustStep[i].time!=MUSTSTEP_SENTINEL ) {
		ms = &cell->mustStep[i];
		if( ms->synapse==s ) {
			memmove(ms, ms+1, (cell->mustStepLen-i-1)*sizeof(*ms));
		} else {
			i++;
		}
	}

	return;
}

/* CHANGELOG
   EAT 27/11/02
   fixed a bug in the realloc where the mustStep_copy was being
   realloced from mustStep.

   EAT 28/11/02
   Moved the mustStep buffer realloc down in the function to prevent
   the last entry from being a non-sentinel.

   EAT 04/09/07
   Added removeStep function.

   EAT 22/11/07
   Added maxStep and badStep handling as a work around for the dreaded
   "step reject leads to loss of queue integrity" bug

   EAT 01/05/08
   Fixed the lost messages bug. Previously, every time a message was
   removed from the buffer the buffer was moved down over the top.
   Now, a pointer is maintained to the current position in the buffer
   and this and the buffer are not reset until the window is committed.

*/
