/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


/**********************************************************************
   A subclass of Dynamics to implement HH like voltage gated channels.
   This class is designed to be used from C code. Another class will
   implement a Python version.
***********************************************************************/

#define P3_MODULE
#include "ndl.h"

PyObject *p3_generic_get_exponent(VGCDynamics *d, void *y) {
    return PyInt_FromLong(d->exponents[(int)y]);
}

static PyObject *p3_generic_rate(VGCDynamics *self, PyObject *args, PyObject *kw, int i, int type) {
    double V, Ca=0;
    PyObject *t;

    if( !PyArg_ParseTuple(args, "d|d", &V, &Ca) )
        return 0;

    t = Py_None;
    if( type==0 )
        t = PyFloat_FromDouble(self->tau[i](self, V, Ca));
    if( type==1)
        t = PyFloat_FromDouble(self->inf[i](self, V, Ca));

    Py_INCREF(t);
    return t;
}

PyObject *p3_generic_tau_0(VGCDynamics *self, PyObject *args, PyObject *kw) {
    return p3_generic_rate(self, args, kw, 0, 0);
}
PyObject *p3_generic_tau_1(VGCDynamics *self, PyObject *args, PyObject *kw) {
    return p3_generic_rate(self, args, kw, 1, 0);
}
PyObject *p3_generic_tau_2(VGCDynamics *self, PyObject *args, PyObject *kw) {
    return p3_generic_rate(self, args, kw, 2, 0);
}
PyObject *p3_generic_tau_3(VGCDynamics *self, PyObject *args, PyObject *kw) {
    return p3_generic_rate(self, args, kw, 3, 0);
}
PyObject *p3_generic_tau_4(VGCDynamics *self, PyObject *args, PyObject *kw) {
    return p3_generic_rate(self, args, kw, 4, 0);
}
PyObject *p3_generic_inf_0(VGCDynamics *self, PyObject *args, PyObject *kw) {
    return p3_generic_rate(self, args, kw, 0, 1);
}
PyObject *p3_generic_inf_1(VGCDynamics *self, PyObject *args, PyObject *kw) {
    return p3_generic_rate(self, args, kw, 1, 1);
}
PyObject *p3_generic_inf_2(VGCDynamics *self, PyObject *args, PyObject *kw) {
    return p3_generic_rate(self, args, kw, 2, 1);
}
PyObject *p3_generic_inf_3(VGCDynamics *self, PyObject *args, PyObject *kw) {
    return p3_generic_rate(self, args, kw, 3, 1);
}
PyObject *p3_generic_inf_4(VGCDynamics *self, PyObject *args, PyObject *kw) {
    return p3_generic_rate(self, args, kw, 4, 1);
}

static double HHcurrent(VGCDynamics *self, double t) {
    int i, p;
    double x, I;
    double V = GETEM_DYN(self, t);
    double Gmax = self->Gmax;
    double Er;
	double ki = self->ki;
	double ci, h2;

	if( ki ) {
		ci = GETSTATE_DYN(self->Cadynamics, 0);
		h2 = ki/(ki+ci);
	} else {
		h2 = 1;
	}

	if( self->useGHKrectification ) {
		Er = self->Cadynamics->GHK(self->Cadynamics);
	} else {
		Er = self->Er;
	}

	I = Gmax * h2 * (Er - V);

    for(i=0; i<self->n; i++) {
        x = GETSTATE_DYN(self, i);
        p = self->exponents[i];
        I *= pow(x, p);
    }
 
    return I;
}

static void HHderivs(VGCDynamics *self, double t) {
    double x, dxdt;
    int i;
    double V = GETEM_DYN(self, t);
    double tau;
    double inf;
	double Ca;

	if( self->Cadynamics ) {
		Ca = GETSTATE_DYN(self->Cadynamics, 0);
	} else {
		Ca = 0;
	}

    for(i=0; i<self->n; i++) {
        tau = self->tau[i](self, V, Ca);
        inf = self->inf[i](self, V, Ca);
        x = GETSTATE_DYN(self, i);
        dxdt = (inf - x) / tau;
        SETDERIV_DYN(self, i, dxdt);
    }
}

static void HIupdate(VGCDynamics *d, double t, double h) {
	int k;
	double Ca, m, tau, inf;
	double V = GETEM_DYN(d, t);

	if( d->Cadynamics ) {
		Ca = GETSTATE_DYN(d->Cadynamics, 0);
	} else {
		Ca = 0;
	}
	for(k=0; k<d->n; k++) {
		tau  = d->tau[k](d, V, Ca);
		inf  = d->inf[k](d, V, Ca);
		m  = GETSTATE_DYN(d, k);
		/* Update rule in tau_m/m_inf language */
		m = (2*h*inf - h*m + 2*tau*m) / (h + 2*tau);
		SETSTATE_DYN(d, k, m);
	}
}

static void HIcurrent(VGCDynamics *d, double t, double *I, double *Er) {
	double m, p;
	int k;
	*I = d->Gmax;
	for(k=0; k<d->n; k++) {
		p  = d->exponents[k];
		m  = GETSTATE_DYN(d, k);
		*I *= pow(m, p);
	}
	if( d->useGHKrectification ) {
		*Er = d->Cadynamics->GHK(d->Cadynamics);
	} else {
		*Er = d->Er;
	}
}

static int initVGCDynamics(VGCDynamics *self, PyObject *args, PyObject *kw) {
    int rc = p3_HinesIntegrableDynamicsType.tp_init((PyObject*)self, args, kw);

    if( rc ) return rc;

    self->current    = (currentfcn*)HHcurrent;
    self->derivs     = (derivsfcn*)HHderivs;
	self->HIupdate   = (HIupdatefcn*)HIupdate;
	self->HIcurrent  = (HIcurrentfcn*)HIcurrent;
    self->accepter   = 0;
    self->cleanup    = 0;
	self->type       = VGCdynamics;
	self->Cadynamics = 0;

    return 0;
}

static PyMemberDef members[] = {
    {"Gmax", T_DOUBLE, offsetof(VGCDynamics, Gmax), 
     0, "maximum conductance"},
    {"Er", T_DOUBLE, offsetof(VGCDynamics, Er), 
     0, "reversal potential"},
    {"Cadynamics", T_OBJECT_EX, offsetof(VGCDynamics, Cadynamics), 
     0, "internal Ca provider"},
    {"useGHKrectification", T_INT, offsetof(VGCDynamics, useGHKrectification), 
     0, "use GHK rectification to calculate driving force"},
    {"ki", T_DOUBLE, offsetof(VGCDynamics, ki), 
     0, "Ca dependent inactivation"},
 	{NULL}		/* sentinel */
};

PyTypeObject p3_VGCDynamicsType = {
    PyObject_HEAD_INIT(NULL)
    0,                         /*ob_size*/
    "_p3.VGCDynamics",             /*tp_name*/
    sizeof(VGCDynamics),      /*tp_basicsize*/
    0,                         /*tp_itemsize*/
    0,                         /*tp_dealloc*/
    0,                         /*tp_print*/
    0,                         /*tp_getattr*/
    0,                         /*tp_setattr*/
    0,                         /*tp_compare*/
    0,                         /*tp_repr*/
    0,                         /*tp_as_number*/
    0,                         /*tp_as_sequence*/
    0,                         /*tp_as_mapping*/
    0,                         /*tp_hash */
    0,                         /*tp_call*/
    0,                         /*tp_str*/
    0,                         /*tp_getattro*/
    0,                         /*tp_setattro*/
    0,                         /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,        /*tp_flags*/
    "Parplex voltage gated channel Dynamics object",   /* tp_doc */
    0,		               /* tp_traverse */
    0,		               /* tp_clear */
    0,		               /* tp_richcompare */
    0,		               /* tp_weaklistoffset */
    0,		               /* tp_iter */
    0,		               /* tp_iternext */
    0,             /* tp_methods */
    members,                   /* tp_members */
    0,          /* tp_getset */
    &p3_HinesIntegrableDynamicsType, /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    (initproc)initVGCDynamics,      /* tp_init */
    0,                         /* tp_alloc */
    PyType_GenericNew                 /* tp_new */
};
