/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


/* fast EPSP dynamics */


#include "ndl.h"

typedef struct {
    DYNAMICS
  double Gmax;
  double Er;
  double k;
} fastEPSP;

static PyMemberDef p3_fastEPSP_members[] = {
    {"Gmax", T_DOUBLE, offsetof(fastEPSP, Gmax),
     0, "maximum conductance"},
    {"Er", T_DOUBLE, offsetof(fastEPSP, Er),
     0, "reversal potential"},
    {"k", T_DOUBLE, offsetof(fastEPSP, k),
     0, "time constant"},
    {NULL},
};

static void update(fastEPSP *self, double strength) {
	double r = GETSTATE_DYN(self, 0);
	r += strength * 1e-3;
	SETSTATE_DYN(self, 0, r);
}

static void fepsp_accepter(fastEPSP *d, Synapse *s,
			   double strength, int window_id) {
				   update(d, strength);
}

void enq_fepsp(fastEPSP *d, double t, double strength) {
  update(d, strength);
}

static int init(fastEPSP *self) {    
	printf("Warning: fEPSP is completely untested "
		"since asynchornous messaging support was removed.\n");
	SETSTATE_DYN(self, 0, 0);
	SETSTATE_DYN(self, 1, 0);
    return 0;
}

static void Fderivs(fastEPSP *self, double t) {
	double r = GETSTATE_DYN(self, 0); 
	double s = GETSTATE_DYN(self, 1);
	double drdt = -r*self->k*1e-3;
	double dsdt = r - s*self->k*1e-3;
    SETDERIV_DYN(self, 0, drdt);
    SETDERIV_DYN(self, 1, dsdt);
}

static double Ifepsp_current(fastEPSP *self, double time) {
  double E = GETEM_DYN(self, time);
  double Gmax = self->Gmax;
  double Er = self->Er;
  double s = GETSTATE_DYN(self, 1);

  return Gmax * (Er - E) * s;
}


static char *fastEPSPStateVars[] = {"r", "s"};
static char *fastEPSPDerivVars[] = {"drdt", "dsdt"};
static char *fastEPSPTraceVars[] = {"sTrace", "sTrace"}; 
DynamicsDescriptor fastEPSPDescriptor = {
    "fastEPSP",
    "fastEPSP, members: Gmax, Er, k",
    p3_fastEPSP_members, /* Members */
    0, /* Methods */
    2,  /* number of state variables */
    fastEPSPStateVars,  /* State variable names */
    fastEPSPDerivVars,  /* deriv variable names */
    fastEPSPTraceVars,  /* trace variable names */
    (derivsfcn*)Fderivs,  /* Derivs */
    (currentfcn*)Ifepsp_current,
    (accepterfcn*)fepsp_accepter,  /* accepter */
    (enqfcn*)enq_fepsp,
	0,
    sizeof(fastEPSP),
    0,
	(userinitfcn*)init,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(fastEPSPDescriptor)

static initproc LuserInitDynamics[] = {
    initfastEPSPDescriptor
};

static DynamicsDescriptor *userDynamics[] = {
    &fastEPSPDescriptor
};

MAKE_P3_MODEL(fpsp, "Stereotyped fast synaptic response")

/* Changelog
   EAT 10Dec02
   see entry in synmsg.c
*/
