/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/***************************************************************
  The RK32 method of Bogacki and Shampine.

  Ref: Numerical Solution of Ordinary Differential Equations
       LF Shampine
       Solving ODEs, Hairer, et al for step size control
****************************************************************/

#define P3_MODULE
#include "ndl.h"

#define PGROW -0.33
#define PSHRNK -0.33
#define SAFETY 0.8
#define FACTOR 5.0

double *ytemp, *yout, *k1, *k2, *k3, *k4, *lasty;
static double lastx, laststep;

void rk32_init(GD *gd, Cell *cell) {
	static int nMax;
	int n = da_length(cell->Y);
	int s = n*sizeof(double);

	if( n>nMax ) {
		if( ytemp ) PyMem_Free(ytemp);
		if( yout  ) PyMem_Free(yout );
		if( k1    ) PyMem_Free(k1   );
		if( k2    ) PyMem_Free(k2   );
		if( k3    ) PyMem_Free(k3   );
		if( k4    ) PyMem_Free(k4   );
		if( lasty ) PyMem_Free(lasty);

		ytemp = getmain(s);
		yout  = getmain(s);
		k1    = getmain(s);
		k2    = getmain(s);
		k3    = getmain(s);
		k4    = getmain(s);
		lasty = getmain(s);

		nMax = n;
	}

	cell->step = 0.1;
}


bool rk32_interp(int n, double xinterp, double *yinterp, double *fsal) { 
  int i;
  double h = laststep;
  double s = (xinterp - lastx)/h;
  
  for(i=0; i<n; i++) {
      ytemp[i] = ((5.0/9.0)*k1[i]-(2.0/3.0)*k2[i]-(8.0/9.0)*k3[i]+k4[i])*s;
      ytemp[i] = (ytemp[i]-(4.0/3.0)*k1[i]+k2[i]+(4.0/3.0)*k3[i]-k4[i])*s;
      ytemp[i] = (ytemp[i]+k1[i])*s;
      yinterp[i] = lasty[i] + ytemp[i]*h;
  }
  
  return true;
}

/* I think I deserve a prize for this function declaration! */
bool rk32_solve(Cell *cell, double eps, double minStep, double *hnext) {
  int i;
  double temp, errmax;

    double *y = cell->Y->data;
    double *fsal = cell->DYDT->data;
    double *x = &cell->time;
    double h = cell->step;
    int n = da_length(cell->Y);
    int *functionCnt = &cell->functionCnt;

  if( *x + h == *x ) {
      char buf[200];
      sprintf(buf, "Step size, %g, too small in routine rk32().\n", h);
      ABEND(buf, cell);
  }

  *functionCnt += 3;

  lastx = *x;
  for(i=0; i<n; i++) lasty[i] = y[i];
  laststep = h;

  /* Calculate */
  for(i=0; i<n; i++) ytemp[i] = y[i] + fsal[i]*h/2.0;
  derivs(*x + h/2.0, ytemp, k2);
  
  for(i=0; i<n; i++) ytemp[i] = y[i] + k2[i]*h*3.0/4.0;
  derivs(*x + h*3.0/4.0, ytemp, k3);
  
  for(i=0; i<n; i++)
    yout[i] = y[i]+ h * (fsal[i]*2.0/9.0 + k2[i]/3.0 + k3[i]*4.0/9.0);
  derivs(*x + h, yout, k4);
  
  errmax = 0.0;
  for(i=0; i<n; i++) {
      double scale = max(max(fabs(y[i]), fabs(ytemp[i])), eps);
    /* ytemp now contains error estimate divided by h */
    ytemp[i] = -fsal[i]*5.0/72.0 + k2[i]/12.0 + k3[i]/9.0 - k4[i]/8.0;
    temp = fabs(ytemp[i] / scale);
    if( errmax < temp ) errmax = temp;
  }
  
  errmax = errmax*h/eps;
  if( errmax <= 1.0 ) {
    /********
      Accept
    *********/
    if( errmax == 0 ) {
      *hnext = h * FACTOR;
    } else {
      temp = SAFETY * pow(errmax, PGROW);
      *hnext = temp < FACTOR ? temp * h : FACTOR * h; /* Don't grow too much! */
    }
    *x += h;
    for(i=0; i<n; i++)
      y[i] = yout[i], k1[i] = fsal[i], fsal[i] = k4[i];
    return true;
  } else {
    /********
      Reject
    *********/
    *hnext = SAFETY * h * pow(errmax, PSHRNK); /* truncation error too large,
						  reduce stepsize */
    *hnext = *hnext < h/FACTOR ? h/FACTOR : *hnext; /* Don't shrink too much! */
    return false;
  }
}
