/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
  This provides a python class to wrap internal lists.  For example
  cell.synapses, cell.dynamics, etc.
  It behaves like a Python list.
*/


#define P3_MODULE
#include "ndl.h"


PyTypeObject p3_InternalListType;

static int puff_buffer(internal_list *a, int newsize) {
    /* Increase the buffer size */
    void *newdata;
    newsize = newsize ? newsize : (int)((float)a->buffer_len*1.1)+1;
    newdata = PyMem_Realloc(a->data, newsize*sizeof(*a->data));
    if( !newdata ) {
        PyErr_NoMemory();
        return -1;
    }
    a->buffer_len = newsize;
    a->data = newdata;
    return 0;
}

static il_type get_il_type(PyObject *o) {
    if( PyObject_TypeCheck(o, &p3_SynapseType) ) return il_Synapse;
    if( PyObject_TypeCheck(o, &p3_CellType) )    return il_Cell;
    if( PyObject_TypeCheck(o, &p3_DynamicsType) )    return il_Dynamics;
    if( PyObject_TypeCheck(o, &p3_CompartmentType) )    return il_Compartment;
    return -1;
}
static il_type str2type(char *str) {
    if( strcmp(str, "Synapse")==0 ) return il_Synapse;
    if( strcmp(str, "Cell")==0 ) return il_Cell;
    if( strcmp(str, "Dynamics")==0 ) return il_Dynamics;
    if( strcmp(str, "Compartment")==0 ) return il_Compartment;
    return -1;
}
static char *type2str(il_type t) {
    static char *names[] = {"Synapse", "Cell", "Dynamics", "Compartment"};
    if( 0<=t && t<=2 ) return names[t];
    else return "--bad it_type--";
}

/*********/
/* C API */
/********/
void il_set(internal_list *a, PyObject *x, int i) {
    il_type t = get_il_type(x);

    if( t!=a->type ) {
        PyErr_SetString(PyExc_RuntimeError, 
            "type mismatch in il_set()");
        longjmp(excpt_exit, 1);
    }
    if( i>=a->buffer_len && puff_buffer(a, 0)==-1 )
        /* No memory - couldn't increase the buffer */
        longjmp(excpt_exit, 1);

    a->data[i] = x;
    Py_INCREF(x);
    a->len = a->len>i ? a->len : ++i;

    return;
}

void il_remove(internal_list *a, int i) {
    if( i>=a->len ) {
        PyErr_SetString(PyExc_RuntimeError,
            "Internal error removing item from an InternalList");
        longjmp(excpt_exit, 1);
    }
    Py_DECREF(a->data[i]);
    memmove(&a->data[i], &a->data[i+1], (a->len-1-i)*sizeof(a->data));
    a->len--;
}

void *il_get(internal_list *a, int i) { return a->data[i]; }

void il_clear(internal_list *a) {
    int i;
    if( !a ) return;
    for(i=0; i<a->len; i++) Py_DECREF(a->data[i]);
    a->len = 0; 
    return;
}

internal_list *il_new(char* type) {
    internal_list *x;
    PyObject *args = Py_BuildValue("(s)", type);

    x = (internal_list*)PyObject_New(internal_list, &p3_InternalListType);
    if( x==NULL ) return NULL;
    /* Is this what I am supposed to do? */
    if( p3_InternalListType.tp_init((PyObject*)x, args, args) != 0 ) {
        PyObject_Del((PyObject*)x);
        return NULL;
    }
    return x;
}

int il_length(internal_list *a) { if(a) return a->len; return 0; }

int il_append(internal_list *a, PyObject *x) {
	int rc = a->len;
    il_set(a, x, rc);
	return rc;
}

void il_copy(internal_list *tgt, internal_list *src) {
    int i;
    if( tgt->len != src->len || tgt->buffer_len != src->buffer_len ) {
        PyErr_SetString(PyExc_RuntimeError, 
            "internal_list size mismatch in da_copy()");
        longjmp(excpt_exit, 1);
    }
    for(i=0; i<tgt->len; i++) {
        tgt->data[i] = src->data[i];
        Py_INCREF(tgt->data[i]);
    }
}


static void delInternalList(internal_list *self) {
    int i;
    for(i=0; i<self->len; i++) 
        Py_XDECREF(self->data[i]);    
    if( self->data ) 
        PyMem_Free(self->data);
    self->ob_type->tp_free((PyObject*)self);
}

static int initInternalList(internal_list *self, PyObject *args) {
    /* The klunky interface is because this would not normally
       be used from within Python. */
    PyObject *data=0, *item;
    int i, size;
    char *str;
    il_type type;

    if( !PyArg_ParseTuple(args, "s|O", &str, &data) )
        return -1;

    type = str2type(str);
    if( type<0 ) {
        PyErr_SetString(PyExc_TypeError,
            "InternalList type must be one of the strings {\"Cell\", \"Synapse\", \"Dynamics\"}");
        return -1;
    }
    self->type = type;

    if( !data ) {
        self->buffer_len = self->len = 0;
        self->data = 0;
        return 0;
    }

    size = PyList_Size(data);
    if( size < 0 ) {
        PyErr_SetString(PyExc_TypeError, 
            "InternalList can only be initialised from a list");
        return -1;
    }

    self->buffer_len = self->len = size;
    if( size ) {
        self->data = PyMem_Malloc(size*sizeof(self->data[0]));
        for(i=0; i<self->len; i++) self->data[i] = 0;
        if( !self->data ) {
            PyErr_NoMemory();
            return -1;
        }

        for(i=0; i<size; i++) {
            item = PyList_GetItem(data, i);

            if( get_il_type(item)==self->type ) {
                Py_INCREF(item);
                self->data[i] = item;
            } else {
                PyErr_SetString(PyExc_TypeError,
                    "List items must be consistenly one of {Cell, Synapse, Dynamics}");
                return -1;
            }
        }
    }
    
    return 0;
}

void *il_seq_next(internal_list *self, int *seq) {
    if( *seq == self->len ) {
        *seq = -1;
        return NULL;
    }
    return self->data[(*seq)++];
}

static PyObject *reprInternalList(internal_list *v) {
    /* This is slightly modified from listobject.c in the Python source */
	int i;
	PyObject *s, *temp;
	PyObject *pieces = NULL, *result = NULL;

	i = Py_ReprEnter((PyObject*)v);
	if (i != 0) {
		return i > 0 ? PyString_FromString("[...]") : NULL;
	}

	if (v->len == 0) {
		result = PyString_FromString("[]");
		goto Done;
	}

	pieces = PyList_New(0);
	if (pieces == NULL)
		goto Done;

	/* Do repr() on each element.  Note that this may mutate the list,
	   so must refetch the list size on each iteration. */
    /* ????: Is this true for InternalLists? */
	for (i = 0; i < v->len; ++i) {
		int status;
		s = PyObject_Repr(v->data[i]);
		if (s == NULL)
			goto Done;
		status = PyList_Append(pieces, s);
		Py_DECREF(s);  /* append created a new ref */
		if (status < 0)
			goto Done;
	}

	/* Add "[]" decorations to the first and last items. */
	assert(PyList_GET_SIZE(pieces) > 0);
	s = PyString_FromString("[");
	if (s == NULL)
		goto Done;
	temp = PyList_GET_ITEM(pieces, 0);
	PyString_ConcatAndDel(&s, temp);
	PyList_SET_ITEM(pieces, 0, s);
	if (s == NULL)
		goto Done;

	s = PyString_FromString("]");
	if (s == NULL)
		goto Done;
	temp = PyList_GET_ITEM(pieces, PyList_GET_SIZE(pieces) - 1);
	PyString_ConcatAndDel(&temp, s);
	PyList_SET_ITEM(pieces, PyList_GET_SIZE(pieces) - 1, temp);
	if (temp == NULL)
		goto Done;

	/* Paste them all together with ", " between. */
	s = PyString_FromString(", ");
	if (s == NULL)
		goto Done;
	result = _PyString_Join(s, pieces);
	Py_DECREF(s);

Done:
	Py_XDECREF(pieces);
	Py_ReprLeave((PyObject *)v);
	return result;
}

static PyObject *InternalList_append(internal_list *op, PyObject *item) {

	if( !PyObject_TypeCheck(op, &p3_InternalListType) ) {
		PyErr_BadInternalCall();
		return NULL;
	}
    
    if( get_il_type(item)!=op->type ) {
        PyErr_SetString(PyExc_TypeError,
            "Types within an InternalList must be consistently {Cell, Synapse, Dynamics}");
        return NULL;
    }

    Py_INCREF(item);
    il_append(op, item);

    Py_INCREF(Py_None);
	return Py_None;
}

/*********************************************
               Sequence methods
**********************************************/
static int InternalList_length(internal_list *a) {
	return a->len;
}

static PyObject *InternalList_concat(internal_list *a, PyObject *b) {
	int size;
	int i, j;
	internal_list *np;

    /* Concatenate an ordinary list (type checking will be done
       on each item) or on two InternalLists of the same type. */
	if( !PyList_Check(b) && !PyObject_TypeCheck(b, &p3_InternalListType) ) {
		PyErr_Format(PyExc_TypeError,
			  "can only concatenate list or InternalList (not \"%.200s\") to InternalList",
			  b->ob_type->tp_name);
		return NULL;
	}
    if( PyObject_TypeCheck(b, &p3_InternalListType) && ((internal_list*)b)->type!=a->type ) {
		PyErr_Format(PyExc_TypeError,
			  "Both InternalLists must be of the same type");
		return NULL;
	}

    np = (internal_list*)a->ob_type->tp_alloc(&p3_InternalListType, 0);
    if( np==NULL ) return PyErr_NoMemory();

    size = a->len;
    if( PyObject_TypeCheck(b, &p3_InternalListType) )
        size += ((internal_list*)b)->len;
    else
        size += PyList_Size(b);

    np->data = PyMem_Malloc(size*sizeof(np->data[0]));
    if( !np->data ) {
        PyMem_Free(np);
        return PyErr_NoMemory();
    }
    np->len = np->buffer_len = size;

	for (i = 0; i < a->len; i++) {
        Py_INCREF(a->data[i]);
		np->data[i] = a->data[i];
	}

    if( PyObject_TypeCheck(b, &p3_InternalListType) ) {
	    for (i = 0; i < ((internal_list*)b)->len; i++) {
            Py_INCREF(((internal_list*)b)->data[i]);
		    np->data[i + a->len] = ((internal_list*)b)->data[i];
        }
    } else {
	    for (i = 0; i < PyList_Size(b); i++) {
            PyObject *item;
            item = PyList_GetItem(b, i);
            if( get_il_type(item)==a->type ) {
                Py_INCREF(item);
                np->data[i+a->len] = item;
            } else {
                PyErr_SetString(PyExc_TypeError,
                    "List item types must match those in the InternalList");
                for(j=0; j<a->len+i; j++) Py_DECREF(np->data[j]);
                PyMem_Free(np->data);
                PyMem_Free(np);
                return NULL;
            }
        }
    }

	return (PyObject *)np;
}


static PyObject * InternalList_item(internal_list *a, int i) {
	if (i < 0 || i >= a->len) {
		PyErr_SetString(PyExc_IndexError, 
                        "InternalList index out of range");
		return NULL;
	}
    Py_INCREF(a->data[i]);
	return a->data[i];
}

static PyObject *InternalList_slice(internal_list *a, int ilow, int ihigh) {
	internal_list *np;
	int i;

	if (ilow < 0)
		ilow = 0;
	else if (ilow > a->len)
		ilow = a->len;
	if (ihigh < ilow)
		ihigh = ilow;
	else if (ihigh > a->len)
		ihigh = a->len;

	np = (internal_list*)a->ob_type->tp_alloc(&p3_InternalListType, 0);
	if (np == NULL) return NULL;

    np->type = a->type;
    np->len = np->buffer_len = ihigh - ilow;
    np->data = PyMem_Malloc(np->len*sizeof(np->data[0]));
    if( np->data == NULL ) {
        PyMem_Free(np);
        return PyErr_NoMemory();
    }

	for (i = ilow; i < ihigh; i++) {
        Py_INCREF(a->data[i]);
		np->data[i - ilow] = a->data[i];
	}

	return (PyObject *)np;
}

static int InternalList_ass_slice(internal_list *a, int ilow, int ihigh, PyObject *v) {
	int size; /* Size of replacement list */
    int i;
    PyObject **temp;

    if( v==NULL ) 
        size = 0;
    else {
        size = PySequence_Size(v);
        if( size < 0 ) {
            PyErr_SetString(PyExc_TypeError, 
                "Only a sequence can be slice assigned to an InternalList");
            return -1;
        }
    }

	if( a == (internal_list*)v ) {
		/* Special case "a[i:j] = a" -- copy b first */
		int ret;
		v = InternalList_slice((internal_list*)v, 0, ((internal_list*)v)->len);
		if (v == NULL)
			return -1;
		ret = InternalList_ass_slice(a, ilow, ihigh, v);
		Py_DECREF(v);
		return ret;
	}

	if (ilow < 0)
		ilow = 0;
	else if (ilow > a->len)
		ilow = a->len;
	if (ihigh < ilow)
		ihigh = ilow;
	else if (ihigh > a->len)
		ihigh = a->len;

    /* Allocate new array */
    temp = PyMem_Malloc((a->len-ihigh+ilow+size)*sizeof(*temp));
    if( !temp ) {
        PyErr_NoMemory();
        return -1;
    }

    /* Copy over stuff from the low part of target */
    for(i=0; i<ilow; i++) {
        temp[i] = a->data[i];
    }

    /* Copy new data */
    for(i=ilow; i<ilow+size; i++) {
	    PyObject *item = PySequence_GetItem(v, i-ilow);
        if( get_il_type(item)==a->type ) {
            /* not sure why I don't need to increment this */
            temp[i] = item;
        } else {
            PyErr_SetString(PyExc_TypeError,
                "All list items must be the same type as the InternalList");
            for(i--; i>=0; i--) Py_DECREF(temp[i]);
            PyMem_Free(temp);
            return -1;
        }
    }

    /* Delete stuff being replaced in the slice */
    for(i=ilow; i<ihigh; i++) {
        Py_DECREF(a->data[i]);
    }

    /* Copy high part of target */
    for(i=ilow+size; i<a->len-ihigh+ilow+size; i++) {
        temp[i] = a->data[ihigh+i-ilow-size];
    }

    PyMem_Free(a->data);
    a->data = temp;
    a->buffer_len = a->len = a->len-ihigh+ilow+size;

	return 0;
}

static int InternalList_ass_item(internal_list *a, int i, PyObject *v) {

	if (i < 0 || i >= a->len) {
		PyErr_SetString(PyExc_IndexError,
				"InternalList assignment index out of range");
		return -1;
	}

    /* This might be del a[i] */
	if (v == NULL)
		return InternalList_ass_slice(a, i, i+1, v);

    if( get_il_type(v)==a->type ) {
        Py_XDECREF(a->data[i]);
        Py_INCREF(v);
        a->data[i] = v;
    } else {
        PyErr_SetString(PyExc_TypeError,
            "Assigned item must be of the same type as the InteralList");
        return -1;
    }

	return 0;
}

static int InternalList_contains(internal_list *a, PyObject *el) {
	int i, cmp;

	for (i = 0, cmp = 0 ; cmp == 0 && i < a->len; ++i)
		cmp = PyObject_RichCompareBool(el, a->data[i], Py_EQ);

	return cmp;
}

static PySequenceMethods InternalList_as_sequence = {
	(inquiry)InternalList_length,			/* sq_length */
	(binaryfunc)InternalList_concat,		/* sq_concat */
	(intargfunc)NULL,		/* sq_repeat - doesn't make sense here */
	(intargfunc)InternalList_item,			/* sq_item */
	(intintargfunc)InternalList_slice,		/* sq_slice */
	(intobjargproc)InternalList_ass_item,		/* sq_ass_item */
	(intintobjargproc)InternalList_ass_slice,	/* sq_ass_slice */
	(objobjproc)InternalList_contains,		/* sq_contains */
    NULL,                                   /* sq_inplace_concat */
    NULL	                                /* sq_inplace_repeat */
};

static PyMethodDef p3_InternalList_methods[] = {
	{"append",	(PyCFunction)InternalList_append,
        METH_O, "DA.append(number) -- append one of {Cell, Synapse, Dynamics} to InternalList"},
 	{NULL,		NULL}		/* sentinel */
};

PyTypeObject p3_InternalListType = {
    PyObject_HEAD_INIT(NULL)
    0,                         /*ob_size*/
    "_p3.InternalList",             /*tp_name*/
    sizeof(internal_list),      /*tp_basicsize*/
    0,                         /*tp_itemsize*/
    (destructor)delInternalList,                         /*tp_dealloc*/
    0,                         /*tp_print*/
    0,                         /*tp_getattr*/
    0,                         /*tp_setattr*/
    0,                         /*tp_compare*/
    (reprfunc)reprInternalList,                         /*tp_repr*/
    0,                         /*tp_as_number*/
    &InternalList_as_sequence,                         /*tp_as_sequence*/
    0,                         /*tp_as_mapping*/
    0,                         /*tp_hash */
    0,                         /*tp_call*/
    0,                         /*tp_str*/
    0,                         /*tp_getattro*/
    0,                         /*tp_setattro*/
    0,                         /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,        /*tp_flags*/
    "Parplex internal list object",   /* tp_doc */
    0,		               /* tp_traverse */
    0,		               /* tp_clear */
    0,		               /* tp_richcompare */
    0,		               /* tp_weaklistoffset */
    0,		               /* tp_iter */
    0,		               /* tp_iternext */
    p3_InternalList_methods,             /* tp_methods */
    0,             /* tp_members */
    0,                         /* tp_getset */
    0,                         /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    (initproc)initInternalList,      /* tp_init */
    0,                         /* tp_alloc */
    PyType_GenericNew                 /* tp_new */
};

/* Changes:

EAT 20Sep07
- Changed the sequence interface from mainting a single
  internal sequeance state to having the user provide the
  state
  */
