#!/usr/bin/python
import os, sys, time
from math import pi, exp
from p3 import *

#########################################
# Test the state variable trace system: #
# both python and C dynamics            #
#########################################

class Passive(PyDynamics):
    def __init__(self, cell, Gleak, Vleak):
        PyDynamics.__init__(self, cell)
        self.Gleak = Gleak
        self.Vleak = Vleak
        
    def current(self, t):
        Em = self.owner.Em
        return self.Gleak * (self.Vleak - Em)
        
class Current(PyDynamics):
    def __init__(self, cell, start, end, Iinject):
        PyDynamics.__init__(self, cell)
        self.start   = start
        self.end     = end
        self.Iinject = Iinject
        
    def current(self, t):
        return self.Iinject * (self.start<=t and t<=self.end)

class K(PyDynamics):
    stateVars   = ['n']
    stateDerivs = ['dndt']
    stateTrace  = ['nTrace']
    
    def __init__(self, cell, GK, EK):
        PyDynamics.__init__(self, cell)
        self.GK = GK
        self.EK = EK
        
    def derivs(self, t):
        cell = self.owner
        Em   = cell.Em
        n    = self.n
        V = Em + 65;
        alpha = 0.07 * (V-47) / (1 - exp((V-47)/-6))
        beta  = 0.264 / exp((V-22)/40)
        self.dndt = alpha*(1-n) - beta*n
        return
        
    def current(self, t):
        cell = self.owner
        Em   = cell.Em
        GK   = self.GK
        EK   = self.EK
        n    = self.n
        return n**4 * GK * (EK - Em)

class Na_wild(PyDynamics):
    stateVars   = ['m', 'h', 's']
    stateDerivs = ['dmdt', 'dhdt', 'dsdt']
    stateTrace  = ['mTrace', 'hTrace', 'sTrace']
    
    def __init__(self, cell, GNa, ENa):
        PyDynamics.__init__(self, cell)
        self.GNa = GNa
        self.ENa = ENa

    def derivs(self, t):
        cell = self.owner
        V    = cell.Em
    
        # m gate
        m = self.m
        m_inf = 1/(1 + exp(-(V+21.2)/4.9))
        tau_m = 0.15
        self.dmdt  = (m_inf - m)/tau_m
    
        # h gate
        h = self.h
        h_inf = 1/(1 + exp((V+39.7)/7.7))
        tau_h = 20.1 * exp(-0.5*((V+61.4)/32.7)**2)
        self.dhdt  = (h_inf - h)/tau_h
    
        # s gate
        s = self.s
        s_inf = 1/(1 + exp((V+46.1)/5.4))
        tau_s = 1000*106.7*exp(-0.5*((V+52.7)/18.3)**2)
        self.dsdt  = (s_inf - s)/tau_s

    def current(self, t):
        cell = self.owner
        Em   = cell.Em
        m    = self.m
        h    = self.h
        s    = self.s
        GNa  = self.GNa
        ENa  = self.ENa
        return m**3 * h * s * GNa * (ENa - Em)

class fEPSP(PyDynamics):
    # Simple class to hold event data    
    class event: pass
    
    def __init__(self, cell, Gsyn, Esyn, k):
        PyDynamics.__init__(self, cell)
        self.Gsyn = Gsyn
        self.Esyn = Esyn
        self.k    = k*1e-3
        self.eventlist = []
        
    def current(self, t):
        """Current provided by this synapse"""
        Em = self.owner.Em
        time = self.owner.time
        Gsyn = self.Gsyn
        Esyn = self.Esyn
        current = 0
        for f in self.eventlist:
            t = time - f.starttime
            if t>0:
                current = current + Gsyn * (Esyn - Em) * self.alpha(t) * f.strength
        return current
        
    def acceptor(self, synapse, strength, window_id):
        """Add the incoming event to the pending event list"""
        e = self.event() # New event
        tgt = synapse.target
        e.starttime = tgt.time + synapse.trans_time
        e.strength = strength
        e.cell_id = synapse.owner.id
        e.window_id = window_id
        stepOn(tgt, e.starttime+1e-3)
        stepOn(tgt, e.starttime+1e-2)
        self.eventlist.append(e)
        
    def cleanup(self, gd):
        """Remove any events no longer required"""
        rlist = []
        time = gd.windowStart + gd.window
        eps = gd.tolerance
        for e in self.eventlist:
            t = time - e.starttime
            if t>0 and self.alpha(t)<eps:
                rlist.append(e)
        for e in rlist:
            self.eventlist.remove(e)
            
    def enq(self, starttime, strength):
        """This function allows a fEPSP to be "exogenously" triggered"""
        e = self.event() # New event
        e.starttime = starttime
        e.strength = strength
        e.cell_id = -1
        stepOn(self.owner, e.starttime+1e-3)
        stepOn(self.owner, e.starttime+1e-2)
        self.eventlist.append(e)
        
    def alpha(self, t):
        k = self.k
        return t * k * exp( 1 - t * k )
        
def makePycell():
    #################
    # Make the cell #
    #################
    cell = Cell()
    cell.surface_area = (1/100e6)/0.0005
    cell.capacitance = 10
    cell.APthreshold = -30
    cell.Em = -60
    #cell.emtrace = True


    ########################################
    # Add dynamical properties to the cell #
    ########################################
    # Leak
    Gleak = 0.5
    Eleak = -60
    Passive(cell, Gleak, Eleak)
    # Potassium
    GK  = 30
    EK = -80
    d = K(cell, GK, EK)
    d.n = 0 # Initial conditions
    cell.K = d
    # Sodium
    GNa = 200
    ENa = 50
    d = Na_wild(cell, GNa, ENa)
    d.m, d.h, d.s = 0, 1, 1 # Initial conditions
    cell.Na = d
    # Synaptic current
    Gsyn = 30
    Esyn = 0
    cell.fepsp = fEPSP(cell, Gsyn, Esyn, 8000)
    
    return cell
#

##################
# Make a network #
##################
   
def makeHHcell():
    cell = Cell()
    cell.capacitance = 1
    cell.APthreshold = -50
    cell.Em = 0
    cell.emtrace = True
    cell.timeSample = 0.1

    # Leak
    hh = hhLeak(cell)
    hh.Gmax = 0.3
    hh.Er   = 10.163

    # Na
    Na = hhNa(cell)
    Na.Gmax = 115
    Na.Er   = 120
    Na.m = 0.0519918603202
    Na.h = 0.601419324468
    cell.Na = Na

    # K
    K = hhK(cell)
    K.Gmax = 36
    K.Er   = -12
    K.n = 0.316173233528
    cell.K = K
    
    # fast EPSP
    fEPSP = fastEPSP(cell)
    fEPSP.Gmax = 36
    fEPSP.Er   = 50
    fEPSP.k    = 8000
    cell.fEPSP = fEPSP

    cell.fEPSP.enq(5, 1.1)
    
    # Noise
    #c = Inoise(cell)
    #c.mean = 0
    #c.std  = 0.2
    
    return cell


# Change this to set the cell type you want
makecell = makeHHcell

cell          = makecell()
cell.emtrace  = True
cell.K.trace  = True
cell.Na.trace = True
#cell.method   = rdIIA
Current(cell, 20, 220, 4) # current injection

##############################
# Special trace capture file #
##############################
trout = None
def traceout(cell):
    global trout
    if not trout:
        trout = open('trace.dat', 'w')
    l = len(cell.traceTimes)
    for i in range(l):
        s = '%d %g %g %g %g %g\n' % (cell.id, cell.traceTimes[i], cell.traceData[i], 
                                   cell.Na.mTrace[i], cell.Na.hTrace[i], cell.K.nTrace[i])
        trout.write(s)

###############
# Run options #
###############
gd = GD()
gd.duration  = 300
gd.tolerance = 1e-4
gd.network = [cell]
gd.trace_handler = traceout

#######
# Run #
#######
parplex(gd)
message_print(info, 'made it')

#from py2mat import *
#mw = Matwrap()
#mw.write('C:\\Documents and Settings\\evan\\My Documents\\Visual Studio Projects\\parplex')
#mw.write('load trace.dat;\n')
#mw.write('figure(12)')
#mw.write('plot(trace(:,2), trace(:,[4 5 6]))\n')
#mw.write('figure(13)')
#mw.write('plot(trace(:,2), trace(:,[3]))\n')