/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/********************************************************
  This file implements the Compartment class for Python.
*********************************************************/

#define P3_MODULE
#include "ndl.h"

static int clear(Compartment *self) {
    il_clear(self->synapse);
    Py_XDECREF(self->synapse);
    self->synapse = 0;

    il_clear(self->dynamics);
    Py_XDECREF(self->dynamics);
    self->dynamics = 0;

    il_clear(self->compartments);
    Py_XDECREF(self->compartments);
    self->compartments = 0;

    return 0;
}


static int traverse(Compartment *self, visitproc v, void *a) {
    int i;

    for(i=0; i<il_length(self->dynamics); i++)
        if( v(il_get(self->dynamics, i), a) < 0 ) 
            return -1;

    for(i=0; i<il_length(self->synapse); i++)
        if( v(il_get(self->synapse, i), a) < 0 )
            return -1;

    for(i=0; i<il_length(self->compartments); i++)
        if( v(il_get(self->compartments, i), a) < 0 )
            return -1;

    return 0;
}

static void del(Compartment *self) {
    Py_XDECREF(self->currentAP);
    Py_XDECREF(self->previousAP);
    Py_XDECREF(self->traceTimes);
    Py_XDECREF(self->traceData);

    clear(self);

    /* Release Dynamics and Synapse objects */
    self->ob_type->tp_free((PyObject*)self);
}

static int init(Compartment *self, PyObject *args) {
    double_array *x;
    internal_list *l;
    PyObject *owner=0;

    if( !PyArg_ParseTuple(args, "O", &owner) )
        return -1;

    if( (!PyObject_TypeCheck(owner, &p3_CellType)) ) { 
        PyErr_SetString(PyExc_TypeError, 
            "the first argument must be of type Cell");
        return -1;
    }

    self->owner = (Cell*)owner;
	if( self->owner->compartments == 0 ) { 
        PyErr_SetString(PyExc_TypeError, 
            "cell not initialised (parent initializer not called?)");
        return -1;
    }
    self->ownerindx = il_append(self->owner->compartments, (PyObject*)self);

    if( (x=da_new()) == NULL ) return -1; else self->currentAP = x; 
    if( (x=da_new()) == NULL ) return -1; else self->previousAP = x; 
    if( (x=da_new()) == NULL ) return -1; else self->traceTimes = x;
    if( (x=da_new()) == NULL ) return -1; else self->traceData = x; 
    if( (x=da_new()) == NULL ) return -1; else self->axial_conductance = x; 

    if( (l=il_new("Synapse")) == NULL ) return -1;
    else self->synapse = l;
    if( (l=il_new("Dynamics")) == NULL ) return -1;
    else self->dynamics = l;
    if( (l=il_new("Compartment")) == NULL ) return -1;
    else self->compartments = l;

	self->Vclamper = 0;

	/* By default the compartment can't fire action potentials */
	self->APthreshold = 500;

    /* The first state variable is Em and is always present */
    self->Emindx = addState(self->owner, 1);

    return 0;
}

static PyObject *repr(Compartment *self) {
    return PyString_FromFormat("C Compartment object at %p, refcnt=%d",
        self, ((PyObject*)self)->ob_refcnt);
}

static PyObject *connect(Compartment *self, PyObject *args) {
    Compartment *tgt=0;
    double g;

    if( !PyArg_ParseTuple(args, "Od", &tgt, &g) )
        return NULL;

    if( (!PyObject_TypeCheck((PyObject*)tgt, &p3_CompartmentType)) ) { 
        PyErr_SetString(PyExc_TypeError, 
            "the first argument must be of type Compartment");
        return NULL;
    }

    il_append(tgt->compartments, (PyObject*)self);
    da_append(tgt->axial_conductance, g);

    il_append(self->compartments, (PyObject*)tgt);
    da_append(self->axial_conductance, g);

    Py_INCREF(Py_None);
    return Py_None;
}

static PyMethodDef methods[] = {
    {"connect", (PyCFunction)connect, METH_VARARGS, "connect(compatment, conductance): connect a compartment to this compartment"},
 	{NULL, NULL, 0, NULL}		/* sentinel */
};

static PyMemberDef members[] = {
    {"emtrace", T_INT, offsetof(Compartment, emtrace),
     0, "mark compartment for Em tracing"},

    {"capacitance", T_DOUBLE, offsetof(Compartment, capacitance), 
     0, "membrane capacitance"},

    {"APthreshold", T_DOUBLE, offsetof(Compartment, APthreshold), 0,
     "action potential detection (or triggering) threshold"},

    {"APtimes", T_OBJECT_EX, offsetof(Compartment, currentAP), 
     READONLY, "Time stamps of action potentials for the window just ending"},

    {"axial_conductance", T_OBJECT_EX, offsetof(Compartment, axial_conductance), 
     READONLY, "Conductance between two compartments"},

    {"traceTimes", T_OBJECT_EX, offsetof(Compartment, traceTimes), 
     READONLY, "times of trace samples for the current window"},

    {"traceData", T_OBJECT_EX, offsetof(Compartment, traceData), 
     READONLY, "trace samples for the current window"},

    {"dynamics", T_OBJECT_EX, offsetof(Compartment, dynamics), 
     READONLY, "list of Dynamics objects attached to this compartment"},

    {"synapse", T_OBJECT_EX, offsetof(Compartment, synapse), 
     READONLY, "list of Synapse objects attached to this compartment"},

    {"owner", T_OBJECT_EX, offsetof(Compartment, owner), 
     READONLY, "Cell object owning this compartment"},

    {"compartments", T_OBJECT_EX, offsetof(Compartment, compartments), 
     READONLY, "list of Comparment objects connected to this compartment"},
 	{NULL}		/* sentinel */
};

double GETEM_CMPT(Compartment *self, double t) {
	if( self->ClampMode==CurrentClamp )
		return self->owner->Y->data[self->Emindx];
	else {
		if( self->Vclamper )
			return self->Vclamper->voltage(self->Vclamper, t);
		else
			return self->clampEm;
	}
}

static PyObject *getEm(Compartment *self) {
	return PyFloat_FromDouble(GETEM_CMPT(self, self->owner->time));
}

static int setEm(Compartment *self, PyObject *EmPy) {
	double Em;
	bool bad = true;

	if( PyFloat_Check(EmPy) ) {
		Em = PyFloat_AsDouble(EmPy);
		bad = false;
	}

	if( PyInt_Check(EmPy) ) {
		Em = PyInt_AsLong(EmPy);
		bad = false;
	}

	if( bad ) {
	    PyErr_SetString(PyExc_TypeError, 
		    "Em must be an integer or a float");
		return -1;
	}

    if( self->ClampMode==CurrentClamp )
		SETEM_CMPT(self, Em);
	else
		self->clampEm = Em;

	return 0;
}

static PyObject *getCM(Compartment *self) {
	char *str;

	if( self->ClampMode==VoltageClamp )
		str = "VoltageClamp";
	else
		str = "CurrentClamp";

    return PyString_FromString(str);
}

static int setCM(Compartment *self, PyObject *cmpy) {
	char *cm = PyString_AsString(cmpy);
	clampmode currentclampmode = self->ClampMode;
	clampmode newclampmode;

	if( !cm ) return -1;

	if( strcmp(cm, "VoltageClamp")==0 )
		newclampmode = VoltageClamp;

	if( strcmp(cm, "CurrentClamp")==0 )
		newclampmode = CurrentClamp;

	if( newclampmode==currentclampmode ) return 0;

	if( newclampmode==CurrentClamp )
		self->Emindx = addState(self->owner, 1);
	else {
		self->clampEm = GETEM_CMPT(self, 0);
		remState(self->owner, self->Emindx);
		self->Emindx = -1;
	}

	self->ClampMode = newclampmode;


	return 0;
}

static PyGetSetDef getseters[] = {
    {"Em", (getter)getEm, (setter)setEm, "membrane potential", NULL},
    {"ClampMode", (getter)getCM, (setter)setCM, "clamp mode", NULL},
    {NULL}  /* Sentinel */
};

PyTypeObject p3_CompartmentType = {
    PyObject_HEAD_INIT(NULL)
    0,                         /*ob_size*/
    "_p3.Compartment",             /*tp_name*/
    sizeof(Compartment),      /*tp_basicsize*/
    0,                         /*tp_itemsize*/
    (destructor)del,                         /*tp_dealloc*/
    0,                         /*tp_print*/
    0,                         /*tp_getattr*/
    0,                         /*tp_setattr*/
    0,                         /*tp_compare*/
    (reprfunc)repr,                         /*tp_repr*/
    0,                         /*tp_as_number*/
    0,                         /*tp_as_sequence*/
    0,                         /*tp_as_mapping*/
    0,                         /*tp_hash */
    0,                         /*tp_call*/
    0,                         /*tp_str*/
    0,                         /*tp_getattro*/
    0,                         /*tp_setattro*/
    0,                         /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,        /*tp_flags*/
    "Parplex Compartment object",   /* tp_doc */
    (traverseproc)traverse,		               /* tp_traverse */
    (inquiry)clear,		               /* tp_clear */
    0,		               /* tp_richcompare */
    0,		               /* tp_weaklistoffset */
    0,		               /* tp_iter */
    0,		               /* tp_iternext */
    methods,               /* tp_methods */
    members,               /* tp_members */
    getseters,             /* tp_getset */
    0,                         /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    (initproc)init,      /* tp_init */
    0,                         /* tp_alloc */
    PyType_GenericNew                 /* tp_new */
};

/* Changes
EAT 21Sep07
Changed default APthreshold to 500mV to prevent unecassary AP
processing

EAT 16May08
Added VoltageClamp mode.
*/
