"""
 * Copyright (C) 2007 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
"""

"""
Support for generating C language DLLs
"""

import re
from math import exp

# Help functions
def fcnrewrite(variables, constants, code):
    for v in variables:
        code = re.sub('\\b'+v.name+'\\b', 'self->'+v.name, code)
    for c in constants:
        code = re.sub('\\b'+c.name+'\\b', '('+c.value+')', code)
    return code

def makeTableFcn(fcn):
    Vmin = -100
    Vmax = 100
    dV = 0.1
    cnt = int((Vmax-Vmin)/dV) + 1
    V = Vmin
    s = ''
    while V<=Vmax:
        s = s + '%.12f,\n' % fcn(V)
        V = V + dV
    tl = """
   double x[%d] = {
%s
};
   int i;

   if( V<=%f ) return x[0];
   if( V>=%f ) return x[%d];

   i = (int)floor((V-(%f))/%f);

   return x[i] + (x[i+1]-x[i])*(V-(%f)-i*%f)/%f;
""" % (cnt, s, Vmin, Vmax, cnt-1, Vmin, dV, Vmin, dV, dV)

    return tl

def makeGateFcns(d, g):
    ItsAStringWereDealingWithBoyo = False
    if g.tau and isinstance(g.tau, str):
        tau = g.tau
        inf = g.inf
        tau = '  return %s;' % fcnrewrite(d.variables, d.constants, tau)
        inf = '  return %s;' % fcnrewrite(d.variables, d.constants, inf)
        
    if g.alpha and isinstance(g.alpha, str):
        tau = '1/((%s)+(%s))' % (g.alpha, g.beta)
        inf = '(%s)/((%s)+(%s))' % (g.alpha, g.alpha, g.beta)    
        tau = '  return %s;' % fcnrewrite(d.variables, d.constants, tau)
        inf = '  return %s;' % fcnrewrite(d.variables, d.constants,  inf)
        
    if g.tau and callable(g.tau):
        tau = makeTableFcn(g.tau)
        inf = makeTableFcn(g.inf)
    if g.alpha and callable(g.alpha):
        tau = makeTableFcn(lambda V: 1/(g.alpha(V)+g.beta(V)))
        inf = makeTableFcn(lambda V: g.alpha(V)/(g.alpha(V)+g.beta(V)))

    fstr = 'static double %s_%s_%s(%s *self, double V, double Ca) {\n%s\n}\n' 
    inf = fstr % ('inf', d.name, g.name, d.name, inf)
    tau = fstr % ('tau', d.name, g.name, d.name, tau)
    return (tau, inf)

class Gate:
    def __init__(self):
        self.tau   = None
        self.inf   = None
        self.alpha = None
        self.beta  = None
        self.exponent = 1
        self.name = 'x'
        self.description = ''


class variable:
    def __init__(self, name, init, description):
        self.name = name
        self.init = init
        self.description = description

class constant:
    def __init__(self, name, value):
        self.name  = name
        self.value = str(value)
        
class DynamicsBuilder:
    DynamicsList = []
    
    def __init__(self):
        self.variables = []
        self.constants = []
        self.name = 'unnamed'
        self.description = ''
        self.type = 'dynamics'
        DynamicsBuilder.DynamicsList.append(self)

    def variable(self, name, init, description=''):
        self.variables.append(variable(name, init, description))

    def constant(self, name, value):
        self.constants.append(constant(name, value))
        
class VGCDynamicsBuilder(DynamicsBuilder):
    
    def __init__(self):
        DynamicsBuilder.__init__(self)
        self.gates = []
        self.type = 'VGCdynamics'

    def writer(self, modf):
        sep = ''
        infFcns  = ''
        tauFcns  = ''
        stateVars = ''
        derivVars = ''
        traceVars = ''
        exponents = ''
        expNames= ''
        tauFcnList = ''
        tauNameList = ''
        infFcnList = ''
        infNameList = ''
        init = 'static int init%s(%s *self) {return 0;}\n\n' % (self.name, self.name)
        for g in self.gates:
            (tau, inf) = makeGateFcns(self, g)
            infFcns = infFcns + inf
            tauFcns = tauFcns + tau

            stateVars   = stateVars   + sep + '"' + g.name + '"'
            derivVars   = derivVars   + sep + '"d' + g.name + 'dt"'
            traceVars   = traceVars   + sep + '"' + g.name + 'Trace"'
            exponents   = exponents   + sep + str(g.exponent)
            expNames    = expNames    + sep + '"' + g.name + 'exp"'
            tauFcnList  = tauFcnList  + sep + '(gatefcn*)tau_' + self.name + '_' + g.name
            infFcnList  = infFcnList  + sep + '(gatefcn*)inf_' + self.name + '_' + g.name
            tauNameList = tauNameList + sep + '"tau_' + g.name + '"'
            infNameList = infNameList + sep + '"' + g.name + '_inf"'            

            sep = ', '

        memberdef = 'static PyMemberDef %s_members[] = {\n' % self.name
        struct    = 'typedef struct {\n   VGCDYNAMICS\n'
        for v in self.variables:
            memberdef = memberdef + '{"' + v.name + '", T_DOUBLE, offsetof(' + \
            self.name + ', ' + v.name + '), 0, "' + v.description + '"},\n'
            struct = struct + '   double ' + v.name + ';\n'
        memberdef = memberdef + '{NULL}\n};\n\n'
        struct = struct + '}  %s;\n\n' % self.name

        if self.variables:
            init = 'static int init%s(%s *self) {\n' % (self.name, self.name)
            for v in self.variables:
                init = init + '   self->%s=%g;\n' % (v.name, v.init)
            init = init + '   return 0;\n}\n\n'
            
        modf.write(struct)
        modf.write(infFcns)
        modf.write(tauFcns)
        modf.write(init)
        modf.write(memberdef)
        modf.write('static char *%sStateVars[] = {%s};\n'    % (self.name, stateVars))
        modf.write('static char *%sDerivVars[] = {%s};\n'    % (self.name, derivVars))
        modf.write('static char *%sTraceVars[] = {%s};\n'    % (self.name, traceVars))
        modf.write('static int   %sexponents[] = {%s};\n'    % (self.name, exponents))
        modf.write('static char *%sexponentNames[] = {%s};\n' % (self.name, expNames))
        modf.write('static gatefcn *%stau[] = {%s};\n'       % (self.name, tauFcnList))
        modf.write('static char *%stauNames[] = {%s};\n'     % (self.name, tauNameList))
        modf.write('static gatefcn *%sinf[] = {%s};\n'       % (self.name, infFcnList))
        modf.write('static char *%sinfNames[] = {%s};\n\n'   % (self.name, infNameList))
        modf.write('DynamicsDescriptor %sDescriptor = {\n' % self.name)
        modf.write('    "%s",\n' % self.name)
        modf.write('    "%s",\n' % self.description)
        modf.write('    %s_members,\n' % self.name)
        modf.write('    0,\n')
        modf.write('    %d,\n' % len(self.gates))
        modf.write('    %sStateVars,\n' % self.name)
        modf.write('    %sDerivVars,\n' % self.name)
        modf.write('    %sTraceVars,\n' % self.name)
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    sizeof(%s),\n' % self.name)
        modf.write('    0,\n')
        modf.write('    (userinitfcn*)init%s,\n' % self.name)
        modf.write('    &%sexponents[0],\n' % self.name)
        modf.write('    %sexponentNames,\n' % self.name)
        modf.write('    %stau,\n' % self.name)
        modf.write('    %stauNames,\n' % self.name)
        modf.write('    %sinf,\n' % self.name)
        modf.write('    %sinfNames,\n' % self.name)
        modf.write('    %s,\n' % self.type)
        modf.write('    0\n')
        modf.write('};\n')
        modf.write('REGISTER_VGC_DESCRIPTOR(%sDescriptor)\n\n' % self.name)


class VerbatimDynamicsBuilder(DynamicsBuilder):

    def __init__(self):
        DynamicsBuilder.__init__(self)
        self.text = ''

    def writer(self, modf):
        modf.write(self.text)

class CaDynamicsBuilder(DynamicsBuilder):
    
    def __init__(self):
        DynamicsBuilder.__init__(self)
        self.Caupdate = ''
        self.type = 'Cadynamics'

    def writer(self, modf):

        memberdef = 'static PyMemberDef %s_members[] = {\n' % self.name
        struct    = 'typedef struct {\n   CADYNAMICS\n'
        for v in self.variables:
            memberdef = memberdef + '{"' + v.name + '", T_DOUBLE, offsetof(' + \
            self.name + ', ' + v.name + '), 0, "' + v.description + '"},\n'
            struct = struct + '   double ' + v.name + ';\n'
        memberdef = memberdef + '{NULL}\n};\n\n'
        struct = struct + '}  %s;\n\n' % self.name

        code = fcnrewrite(self.variables, self.constants, self.Caupdate)
        modf.write(struct)
        modf.write(memberdef)
        modf.write('static double caupdate_%s(%s *self, double ICa, double Ca) {\n' % (self.name, self.name))
        modf.write('    return %s;\n' % code)
        modf.write('}\n\n');

        modf.write('static char *%sStateVars[] = {"Ca"};\n'         % self.name)
        modf.write('static char *%sDerivVars[] = {"dCadt"};\n'      % self.name)
        modf.write('static char *%sTraceVars[] = {"CaTrace"};\n'    % self.name)

        modf.write('DynamicsDescriptor %sDescriptor = {\n' % self.name)
        modf.write('    "%s",\n' % self.name)
        modf.write('    "%s",\n' % self.description)
        modf.write('    %s_members,\n' % self.name)
        modf.write('    0,\n')
        modf.write('    1,\n')
        modf.write('    %sStateVars,\n' % self.name)
        modf.write('    %sDerivVars,\n' % self.name)
        modf.write('    %sTraceVars,\n' % self.name)
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    sizeof(%s),\n' % self.name)
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    0,\n')
        modf.write('    %s,\n' % self.type)
        modf.write('    caupdate_%s,\n' % self.name)
        modf.write('};\n')
        modf.write('REGISTER_DESCRIPTOR(%sDescriptor)\n\n' % self.name)
    

def build(modname, moddescription):
    modf = open(modname+'.c', 'w')
    modf.write('#include <parplex/ndl.h>\n\n')

    dlist1 = ''
    dlist2 = ''
    for d in DynamicsBuilder.DynamicsList:
        d.writer(modf)
        dlist1 = dlist1 + '&' + d.name + 'Descriptor,\n'
        dlist2 = dlist2 + 'init' + d.name + 'Descriptor,\n'

    modf.write('static DynamicsDescriptor *userDynamics[] = {\n%s};\n\n' % dlist1)
    modf.write('static initproc LuserInitDynamics[] = {\n%s};\n\n' % dlist2)

    modf.write('MAKE_P3_MODEL(%s, "%s")\n' % (modname, moddescription))


    modf.close()

    # ... and compile
    from distutils.core import setup, Extension
    import sys
    sys.argv[1:] = ['build_ext', '--inplace']
    E = Extension(modname,  [modname + '.c'])
    setup(ext_modules=[E])

                   

