/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Kahpmbridge, MA 02139, USA.
 *
 */


/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
This file implements:
1) The currents for the Pinksy-Rinzel 2 compartment pyramidal neuron
2) The carbochol extensions of of Tiesinga, et al. Hippocampus 
11:251-274 (2001)
3) The Ih model of Destexhe, et al J Neurophysiology 76(3):2049-2070
(1996)
*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/


#include "ndl.h"

/*---------
     Na
-----------*/
typedef struct {
    DYNAMICS
  double gNa;    /* The maximum conductance */
  double VNa;      /* The driving potential */
}  Na;

static void Naderivs(Na *self, double t) {
  double v       = GETEM_DYN(self, t);
  double h       = GETSTATE_DYN(self, 0);
  double alphahs = 0.128*exp((-43.0-v)/18.0);
  double betahs  = 4.0/(1.0+exp((-20.0-v)/5.0));
  double dhdt    = alphahs - (alphahs+betahs)*h;
  SETDERIV_DYN(self, 0, dhdt);
  return;
}

static double Nacurrent(Na *self, double t) {
  double v   = GETEM_DYN(self, t);
  double gNa = self->gNa;
  double VNa = self->VNa;
  double h   = GETSTATE_DYN(self, 0);
  double alphams =  0.32*(-46.9-v)/(exp((-46.9-v)/4.0)-1.0);
  double betams  = 0.28*(v+19.9)/(exp((v+19.9)/5.0)-1.0);
  double Minfs =  alphams/(alphams+betams);
  return -gNa * (Minfs*Minfs) * h * (v-VNa);
}

static PyMemberDef Na_members[] = {
    {"gNa", T_DOUBLE, offsetof(Na, gNa), 0, "maximum conductance"},
    {"VNa", T_DOUBLE, offsetof(Na, VNa), 0, "reversal potential"},
    {NULL},
};

static char *NaStateVars[] = {"h"};
static char *NaDerivVars[] = {"dhdt"};
static char *NaTraceVars[] = {"hTrace"};
DynamicsDescriptor prNaDescriptor = {
    "prNa_C",
    "prNa Pinksy and Rinzel Na current. members: gNa, VNa",
    Na_members,
    0,
    1,
    NaStateVars,
    NaDerivVars,
    NaTraceVars,
    (derivsfcn*)Naderivs,
    (currentfcn*)Nacurrent,
    0,
    0,
    0,
    sizeof(Na),
    0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(prNaDescriptor)

/*----------
     Kdr
------------*/
typedef struct {
    DYNAMICS
  double gKdr;    /* The maximum conductance */
  double VK;      /* The driving potential */
}  Kdr;


static void Kdrderivs(Kdr *self, double t) {
  double v    = GETEM_DYN(self, t);
  double n    = GETSTATE_DYN(self, 0);
  double alphans = 0.016*(-24.9-v)/(exp((-24.9-v)/5.0)-1.0);
  double betans = 0.25*exp(-1.0-0.025*v);
  double dndt = alphans-(alphans+betans)*n;
  SETDERIV_DYN(self, 0, dndt);
  return;
}

static double Kdrcurrent(Kdr *self, double t) {
  double v    = GETEM_DYN(self, t);
  double gKdr = self->gKdr;
  double VK   = self->VK;
  double n    = GETSTATE_DYN(self, 0);
  return -gKdr*n*(v-VK);
}

static PyMemberDef Kdr_members[] = {
    {"gKdr", T_DOUBLE, offsetof(Kdr, gKdr), 0, "maximum conductance"},
    {"VK", T_DOUBLE, offsetof(Kdr, VK), 0, "reversal potential"},
    {NULL},
};

static char *KdrStateVars[] = {"n"};
static char *KdrDerivVars[] = {"dndt"};
static char *KdrTraceVars[] = {"nTrace"};
DynamicsDescriptor prKdrDescriptor = {
    "prKdr_C",
    "prKdr Pinksy and Rinzel K delayed rectifier current. members: gKdr, VK",
    Kdr_members,
    0,
    1,
    KdrStateVars,
    KdrDerivVars,
    KdrTraceVars,
    (derivsfcn*)Kdrderivs,
    (currentfcn*)Kdrcurrent,
    0,
    0,
    0,
    sizeof(Kdr),
    0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(prKdrDescriptor)

/*----------
     Ca
------------*/
typedef struct {
    DYNAMICS
  double gCa;    /* The maximum conductance */
  double VCa;      /* The driving potential */
}  Cadyn;

static double Cacurrent(Cadyn *self, double t) {
  double v   = GETEM_DYN(self, t);
  double gCa = self->gCa;
  double VCa = self->VCa;
  double s   = GETSTATE_DYN(self, 1);
  return -gCa*s*s*(v-VCa);
}

static void Caderivs(Cadyn *self, double t) {
  double v    = GETEM_DYN(self, t);
  double s    = GETSTATE_DYN(self, 1);
  double Ca   = GETSTATE_DYN(self, 0);
  double alphasd = 1.6/(1.0+exp(-0.072*(v-5.0)));
  double betasd  = 0.02*(v+8.9)/(exp((v+8.9)/5.0)-1.0);
  double dsdt = alphasd-(alphasd+betasd)*s;
  double dCadt = 0.13*Cacurrent(self, t) - 0.075*Ca;
  SETDERIV_DYN(self, 1, dsdt);
  SETDERIV_DYN(self, 0, dCadt);
  return;
}

static PyMemberDef Ca_members[] = {
    {"gCa", T_DOUBLE, offsetof(Cadyn, gCa), 0, "maximum conductance"},
    {"VCa", T_DOUBLE, offsetof(Cadyn, VCa), 0, "reversal potential"},
    {NULL},
};

static char *CaStateVars[] = {"Ca", "s"};
static char *CaDerivVars[] = {"dCadt", "dsdt"};
static char *CaTraceVars[] = {"CaTrace", "sTrace"};
DynamicsDescriptor prCaDescriptor = {
    "prCa_C",
    "prCa Pinksy and Rinzel K calcium current. members: gCa, VCa",
    Ca_members,
    0,
    2,
    CaStateVars,
    CaDerivVars,
    CaTraceVars,
    (derivsfcn*)Caderivs,
    (currentfcn*)Cacurrent,
    0,
    0,
    0,
    sizeof(Cadyn),
    0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(prCaDescriptor)

/*----------
     Kahp
------------*/
typedef struct {
    DYNAMICS
  double gKahp;    /* The maximum conductance */
  double VK;      /* The driving potential */
  Dynamics *Cadyn;
}  Kahp;


static void Kahpderivs(Kahp *self, double t) {
  double q    = GETSTATE_DYN(self, 0);
  double Ca   = GETSTATE_DYN(self->Cadyn, 0);
  double alphaqd = min(0.00002*Ca, 0.01);
  double betaqd  = 0.001;
  double dqdt = alphaqd-(alphaqd+betaqd)*q;
  SETDERIV_DYN(self, 0, dqdt);
  return;
}

static double Kahpcurrent(Kahp *self, double t) {
  double v   = GETEM_DYN(self, t);
  double gKahp = self->gKahp;
  double VK    = self->VK;
  double q   = GETSTATE_DYN(self, 0);
  return -gKahp*q*(v-VK);
}

static PyMemberDef Kahp_members[] = {
    {"gKahp", T_DOUBLE, offsetof(Kahp, gKahp), 0, "maximum conductance"},
    {"VK", T_DOUBLE, offsetof(Kahp, VK), 0, "reversal potential"},
    {"Cadyn", T_OBJECT_EX, offsetof(Kahp, Cadyn), 0, "Internal Ca dynamics"},
    {NULL},
};

static char *KahpStateVars[] = {"q"};
static char *KahpDerivVars[] = {"dqdt"};
static char *KahpTraceVars[] = {"qTrace"};
DynamicsDescriptor prKahpDescriptor = {
    "prKahp_C",
    "prKahp Pinksy and Rinzel after-hyperpolarising potential. members: gKahp, VK, Cadyn",
    Kahp_members,
    0,
    1,
    KahpStateVars,
    KahpDerivVars,
    KahpTraceVars,
    (derivsfcn*)Kahpderivs,
    (currentfcn*)Kahpcurrent,
    0,
    0,
    0,
    sizeof(Kahp),
    0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(prKahpDescriptor)
  
/*----------
     KC
------------*/
typedef struct {
    DYNAMICS
  double gKC;    /* The maximum conductance */
  double VK;      /* The driving potential */
  Dynamics *Cadyn;
}  KC;

#define HEAV(c) ((c)>0)
static void KCderivs(KC *self, double t) {
  double v    = GETEM_DYN(self, t);
  double c    = GETSTATE_DYN(self, 0);
  double alphacd   = (1.0-HEAV(v+10.0))*exp((v+50.0)/11-(v+53.5)/27)/18.975+HEAV(v+10.0)*2.0*exp((-53.5-v)/27.0);
  double betacd    = (1.0-HEAV(v+10.0))*(2.0*exp((-53.5-v)/27.0)-alphacd);
  double dcdt = alphacd-(alphacd+betacd)*c;
  SETDERIV_DYN(self, 0, dcdt);
  return;
}

static double KCcurrent(KC *self, double t) {
  double v   = GETEM_DYN(self, t);
  double gKC = self->gKC;
  double VK = self->VK;
  double c = GETSTATE_DYN(self, 0);
  double Ca = GETSTATE_DYN(self->Cadyn, 0);
  double chid   = min(Ca/250.0,1.0);
  return -gKC*c*chid*(v-VK);
}

static PyMemberDef KC_members[] = {
    {"gKC", T_DOUBLE, offsetof(KC, gKC), 0, "maximum conductance"},
    {"VK", T_DOUBLE, offsetof(KC, VK), 0, "reversal potential"},
    {"Cadyn", T_OBJECT_EX, offsetof(KC, Cadyn), 0, "Internal Ca dynamics"},
    {NULL},
};

static char *KCStateVars[] = {"c"};
static char *KCDerivVars[] = {"dcdt"};
static char *KCTraceVars[] = {"cTrace"};
DynamicsDescriptor prKCDescriptor = {
    "prKC_C",
    "prKC Pinksy and Rinzel calcium activated K current. members: gKC, VK",
    KC_members,
    0,
    1,
    KCStateVars,
    KCDerivVars,
    KCTraceVars,
    (derivsfcn*)KCderivs,
    (currentfcn*)KCcurrent,
    0,
    0,
    0,
    sizeof(KC),
    0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(prKCDescriptor)

/*----------
     KM
------------*/
typedef struct {
    DYNAMICS
  double gKM;    /* The maximum conductance */
  double VK;      /* The driving potential */
}  KM;


static void KMderivs(KM *self, double t) {
  double Em = GETEM_DYN(self, t);
  double r  = GETSTATE_DYN(self, 0);
  double alphar = 0.016*exp((Em+52.7)/23);
  double betar  = 0.016*exp(-(Em+52.7)/18.8);
  double drdt = alphar - (alphar+betar)*r;
  SETDERIV_DYN(self, 0, drdt);
  return;
}

static double KMcurrent(KM *self, double t) {
  double v    = GETEM_DYN(self, t);
  double gKM = self->gKM;
  double VK   = self->VK;
  double r    = GETSTATE_DYN(self, 0);
  return -gKM*r*r*(v-VK);
}

static PyMemberDef KM_members[] = {
    {"gKM", T_DOUBLE, offsetof(KM, gKM), 0, "maximum conductance"},
    {"VK", T_DOUBLE, offsetof(KM, VK), 0, "reversal potential"},
    {NULL},
};

static char *KMStateVars[] = {"r"};
static char *KMDerivVars[] = {"drdt"};
static char *KMTraceVars[] = {"rTrace"};
DynamicsDescriptor prKMDescriptor = {
    "prKM_C",
    "prKM Pinksy and Rinzel M current. members: gKM, VK",
    KM_members,
    0,
    1,
    KMStateVars,
    KMDerivVars,
    KMTraceVars,
    (derivsfcn*)KMderivs,
    (currentfcn*)KMcurrent,
    0,
    0,
    0,
    sizeof(KM),
    0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(prKMDescriptor)

/*----------
     KO
------------*/
typedef struct {
    DYNAMICS
  double gKO;    /* The maximum conductance */
  double VK;      /* The driving potential */
}  KO;


static void KOderivs(KO *self, double t) {
  double v    = GETEM_DYN(self, t);
  double a    = GETSTATE_DYN(self, 0);
  double ainf = 1/(1+exp(-(v+63)/10));
  double taua = 30;
  double dadt = (ainf - a) / taua;
  SETDERIV_DYN(self, 0, dadt);
  return;
}

static double KOcurrent(KO *self, double t) {
  double v    = GETEM_DYN(self, t);
  double gKO = self->gKO;
  double VK   = self->VK;
  double a    = GETSTATE_DYN(self, 0);
  return -gKO*a*(v-VK);
}

static PyMemberDef KO_members[] = {
    {"gKO", T_DOUBLE, offsetof(KO, gKO), 0, "maximum conductance"},
    {"VK", T_DOUBLE, offsetof(KO, VK), 0, "reversal potential"},
    {NULL},
};

static char *KOStateVars[] = {"a"};
static char *KODerivVars[] = {"dadt"};
static char *KOTraceVars[] = {"aTrace"};
DynamicsDescriptor prKODescriptor = {
    "prKO_C",
    "prKO Pinksy and Rinzel peristant K current. members: gKO, VK",
    KO_members,
    0,
    1,
    KOStateVars,
    KODerivVars,
    KOTraceVars,
    (derivsfcn*)KOderivs,
    (currentfcn*)KOcurrent,
    0,
    0,
    0,
    sizeof(KO),
    0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(prKODescriptor)
  
/*----------
     NaO
------------*/
typedef struct {
    DYNAMICS
  double gNaO;    /* The maximum conductance */
  double VNa;      /* The driving potential */
}  NaO;


static void NaOderivs(NaO *self, double t) {
  double v    = GETEM_DYN(self, t);
  double b    = GETSTATE_DYN(self, 0);
  double binf = 1/(1+exp(-(v+60)/5));
  double taub = 0.05;
  double dbdt = (binf - b) / taub;
  SETDERIV_DYN(self, 0, dbdt);
  return;
}

static double NaOcurrent(NaO *self, double t) {
  double v    = GETEM_DYN(self, t);
  double gNaO = self->gNaO;
  double VNa   = self->VNa;
  double b    = GETSTATE_DYN(self, 0);
  return -gNaO*b*(v-VNa);
}

static PyMemberDef NaO_members[] = {
    {"gNaO", T_DOUBLE, offsetof(NaO, gNaO), 0, "maximum conductance"},
    {"VNa", T_DOUBLE, offsetof(NaO, VNa), 0, "reversal potential"},
    {NULL},
};

static char *NaOStateVars[] = {"b"};
static char *NaODerivVars[] = {"dbdt"};
static char *NaOTraceVars[] = {"bTrace"};
DynamicsDescriptor prNaODescriptor = {
    "prNaO_C",
    "prNaO Pinksy and Rinzel peristant Na current. members: gNaO, VNa",
    NaO_members,
    0,
    1,
    NaOStateVars,
    NaODerivVars,
    NaOTraceVars,
    (derivsfcn*)NaOderivs,
    (currentfcn*)NaOcurrent,
    0,
    0,
    0,
    sizeof(NaO),
    0,
	0,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(prNaODescriptor)
  
/*-----------------
    fast EPSP
------------------*/
typedef struct {
    DYNAMICS
  double gSyn;    /* The maximum conductance */
  double VSyn;      /* The driving potential */
  double strength;
  double starttime;
}  Syn;

/* Rate constants */
static double beta  = 0.1;

static int Syninit(Syn *self) {self->starttime = -100; return 0;}

static void Synderivs(Syn *self, double t) {
    double s = GETSTATE_DYN(self, 0);
    double dsdt, f=0;
    double tt = t-self->starttime;
    if( 0<=tt && tt<0.5 ) f = 3.0*tt;
    if( 0.5<=tt && tt<1.0 ) f = 1.5;
    if( 1.0<=tt && tt<=1.5 ) f = 3.0*(1.5-tt);
    dsdt = f*(1-s) - beta*s;
    SETDERIV_DYN(self, 0, dsdt);
    return;
}

static void Synaccepter(Syn *self, Synapse *synapse, double strength, int window_id) {
    Cell *tgt   = synapse->target;
    self->starttime = tgt->time + synapse->trans_time;
    self->strength = strength;
    stepOn(tgt, self->starttime+1e-3);
    return;
}

static void Synenq(Syn *self, double starttime, double strength) {
    self->starttime = starttime;
    self->strength  = strength;
    stepOn(self->owningCell, self->starttime+1e-3);
}

static double Syncurrent(Syn *self, double t) {
  double Em   = GETEM_DYN(self, t);
  double s    = GETSTATE_DYN(self, 0);
  return -self->gSyn*s*(Em-self->VSyn)*self->strength;
}

static PyMemberDef Syn_members[] = {
    {"gSyn", T_DOUBLE, offsetof(Syn, gSyn), 0, "maximum conductance"},
    {"VSyn", T_DOUBLE, offsetof(Syn, VSyn), 0, "reversal potential"},
    {"starttime", T_DOUBLE, offsetof(Syn, starttime), 0, "current event starttime"},
    {"strength", T_DOUBLE, offsetof(Syn, strength), 0, "current event starttime"},
    {NULL},
};

static char *SynStateVars[] = {"s"};
static char *SynDerivVars[] = {"dsdt"};
static char *SynTraceVars[] = {"sTrace"};
DynamicsDescriptor prSynDescriptor = {
    "prEPSP_C",
    "prEPSP Pinksy and Rinzel fast EPSP. members: gSyn, Syn, strenth, starttime",
    Syn_members,
    0,
    1,
    SynStateVars,
    SynDerivVars,
    SynTraceVars,
    (derivsfcn*)Synderivs,
    (currentfcn*)Syncurrent,
    (accepterfcn*)Synaccepter,
    (enqfcn*)Synenq,
    0,
    sizeof(Syn),
    0,
    (userinitfcn*)Syninit,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(prSynDescriptor)

/*-----------
     Ih
------------*/
    /* parameters */
static double celsius = 36;
static double k2      = 0.0004;
static double Pc      = 0.01;
static double k4      = 0.001;
static double nca     = 4;
static double nexp    = 1;
static double ginc    = 2;
static double taum    = 20;
static double shift   = 0;
static double tadj;
    
typedef struct {
    DYNAMICS
  double gIh;     /* The maximum conductance */
  double Vh;      /* The driving potential */
  double Cac;
  Dynamics *Cadyn;
}  Ih;

static void Ihderivs(Ih *self, double t) {
    double v  = GETEM_DYN(self, t);
    double c1 = GETSTATE_DYN(self, 0);
    double o1 = GETSTATE_DYN(self, 1);
    double p0 = GETSTATE_DYN(self, 2);
    double p1 = 1 - p0;
    double o2 = 1 - c1 -o1;
    double Ca = GETSTATE_DYN(self->Cadyn, 0);
    double h_inf = 1 / ( 1 + exp((v+75-shift)/5.5) );
    double tau_s = (taum + 1000 /
        ( exp((v+71.5-shift)/14.2) + exp(-(v+89-shift)/11.6) ) )
        / tadj;
    double alpha = h_inf / tau_s;
    double beta  = ( 1 - h_inf ) / tau_s;
    double k1ca  = k2 * pow(Ca/self->Cac, nca);
    double k3p   = k4 * pow(p1/Pc, nexp);
    double dc1dt = -alpha*c1 + beta*o1;
    double dp0dt = -k1ca*p0  + k2*p1;
    double do1dt = -k3p*o1   + k4*o2;
    SETDERIV_DYN(self, 0, dc1dt);
    SETDERIV_DYN(self, 1, do1dt);
    SETDERIV_DYN(self, 2, dp0dt);
    return;
}

static double Ihcurrent(Ih *self, double t) {
    double v  = GETEM_DYN(self, t);
    double c1 = GETSTATE_DYN(self, 0);
    double o1 = GETSTATE_DYN(self, 1);
    double o2 = 1 - c1 -o1;
    double m  = o1 + ginc * o2;
    return -self->gIh * m * (v - self->Vh);
}

static int IhInit(Ih *self) {
    tadj = pow(3.0, (celsius-36)/10);
    return 0;
}
static PyMemberDef Ih_members[] = {
    {"gIh", T_DOUBLE, offsetof(Ih, gIh), 0, "maximum conductance"},
    {"Eh", T_DOUBLE, offsetof(Ih, Vh), 0, "reversal potential"},
    {"Cadyn", T_OBJECT_EX, offsetof(Ih, Cadyn), 0, "Internal Ca dynamics"},
    {"CaC", T_DOUBLE, offsetof(Ih, Cac), 0, "half activation calcium dependence"},
    {NULL},
};

static char *IhStateVars[] = {"c1", "o1", "p0"};
static char *IhDerivVars[] = {"dc1dt", "do1dt", "dp0dt"};
static char *IhTraceVars[] = {"c1Trace", "o1Trace", "p0Trace"};
DynamicsDescriptor IhDescriptor = {
    "Ih_C",
    "Ih Destexhe et al Ih. members: gIh, Vh, Cadyn",
    Ih_members,
    0,
    3,
    IhStateVars,
    IhDerivVars,
    IhTraceVars,
    (derivsfcn*)Ihderivs,
    (currentfcn*)Ihcurrent,
    0,
    0,
    0,
    sizeof(Ih),
    0,
    (userinitfcn*)IhInit,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(IhDescriptor)

/*-----------
   GABAb
------------*/
    /* parameters */
static double Kd = 100;
static double K1 = 90;
static double K2 = 1.2e-3;
static double K3 = 180e-3;
static double K4 = 34e-3;
static double T  = 0.5e-3 * 0.3;
static double n  = 4;

typedef struct GABAbtag {
  struct GABAbtag * next;
  double starttime;
  double strength;
  int cell_id;
  int window_id;
} GABAbQ;

typedef struct {
    DYNAMICS
  double gSyn;
  double VSyn;
} GABAbIPSP;

static void GABAbderivs(GABAbIPSP *self, double t) {
    double r    = GETSTATE_DYN(self, 0);
    double s    = GETSTATE_DYN(self, 1);
    double dsdt = K3*r - K4*s;
	double drdt = -K2*r;
    SETDERIV_DYN(self, 0, drdt);
    SETDERIV_DYN(self, 1, dsdt);
    return;
}

static double GABAbcurrent(GABAbIPSP *self, double t) {
    double v = GETEM_DYN(self, t);
    double s = GETSTATE_DYN(self, 1);
    double x = pow(s, n);
    return -self->gSyn * (v-self->VSyn) * x / (Kd+x);
}

static int GABAbInit(GABAbIPSP *self) {
  if( K1*T>1 )
    message(warn, "K1*T>1, which will lead to integration errors.");
  printf("Warning: GABAbIPSP has not been tested "
	  "since asynchornous messaging support was removed.\n");
  return 0;
}

static void update(GABAbIPSP *self, double strength) {
	double r = GETSTATE_DYN(self, 0);
	r = strength*K1*T*(1-r);
	SETSTATE_DYN(self, 0, r);
}

static void GABAbaccepter(GABAbIPSP *d, Synapse *s,
			   double strength, int window_id) {
				   update(d, strength);
			   }

void GABAbenq(GABAbIPSP *d, double t, double strength) {
	update(d, strength);
}


static PyMemberDef GABAb_members[] = {
    {"gSyn", T_DOUBLE, offsetof(GABAbIPSP, gSyn), 0, "maximum conductance"},
    {"VSyn", T_DOUBLE, offsetof(GABAbIPSP, VSyn),  0, "reversal potential"},
    {NULL},
};

static char *GABAbStateVars[] = {"r", "s"};
static char *GABAbDerivVars[] = {"drdt", "dsdt"};
static char *GABAbTraceVars[] = {"rTrace", "sTrace"};
DynamicsDescriptor GABAbDescriptor = {
    "GABAb_C",
    "Destexhe and Sejnowski GABAb model. members: gSyn, VSyn",
    GABAb_members,
    0,
    2,
    GABAbStateVars,
    GABAbDerivVars,
    GABAbTraceVars,
    (derivsfcn*)GABAbderivs,
    (currentfcn*)GABAbcurrent,
    (accepterfcn*)GABAbaccepter,  /* accepter */
    (enqfcn*)GABAbenq,
    0,
    sizeof(GABAbIPSP),
    0,
    (userinitfcn*)GABAbInit
};
REGISTER_DESCRIPTOR(GABAbDescriptor)

static DynamicsDescriptor *userDynamics[] = {
  &GABAbDescriptor,
  &IhDescriptor,
  &prNaODescriptor,
  &prKODescriptor,
  &prKMDescriptor,
  &prKCDescriptor,
  &prKahpDescriptor,
  &prCaDescriptor,
  &prKdrDescriptor,
  &prCaDescriptor,
  &prSynDescriptor,
  &prNaDescriptor
};

static initproc LuserInitDynamics[] = {
  initGABAbDescriptor,
  initIhDescriptor,
  initprNaODescriptor,
  initprKODescriptor,
  initprKMDescriptor,
  initprKCDescriptor,
  initprKahpDescriptor,
  initprCaDescriptor,
  initprKdrDescriptor,
  initprCaDescriptor,
  initprSynDescriptor,
  initprNaDescriptor
};


MAKE_P3_MODEL(pr, "The Pinksy and Rinzel Pyramidal neuron model")
