/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


/********************************************************************
   This code does a leaky integrate a fire neuron.  There are not
   state variables as such.  The action potential is implemented by
   stereotyped time course conductance changes, which in turn are
   implemented like post-synaptic events from 'auto-synapses'.
*********************************************************************/


#include "ndl.h"

typedef struct aptag {
  struct aptag * next;
  int cell_id;
  int window_id;
  double starttime;
} AP;

typedef struct componenttag {
  struct componenttag *next;
  double Gmax;
  double Gtotal;
  double Gsum;
  double Er;
  double time_constant;
  double delay;
  double Gbasal;
  double weight;
  Dynamics *modulator;
  double (*shape)(double, double);
} Component;

typedef struct {
    DYNAMICS
  Component *components;
  AP *active;
  double abs_refractory;
  int phasiccount;
  Synapse *synapse;
} ActionPotential;


static double alpha(double t, double k) {
  return t * k * 1e-3 * exp( 1 - t * k * 1e-3 );
}

static double pulse(double t, double duration) {
  return t<=duration;
}

static void accepter(ActionPotential *d, Synapse *s,
		     double strength, int window_id) {
  AP *f;
  Cell *tgt        = s->target;
  double refractory = d->abs_refractory;
  double starttime = tgt->time + s->trans_time;
  AP *q;

  /* First check that this cell is not refractory, that is that an
     action potential started less that the refractory period ago. */
  q = d->active;
  while(q) {
    if( starttime<=q->starttime+refractory && q->starttime<=starttime )
      return;
    q = q->next;
  }

  /* If the neuron is phasic, check that is hasn't fired too many
     action potentials. */
  if( d->phasiccount == 0 ) return;
  
  /* Add this event to the queue */
  f = getmain(sizeof(*f));
  f->next = d->active;
  d->active = f;

  /* Fill in the details */
  f->starttime = starttime;
  f->cell_id   = s->owner->owner->id;
  f->window_id = window_id;

  /* Force the step size down */
  stepOn(tgt, starttime+1e-3);

  /* Decrement allowed number of action potentials */
  if( d->phasiccount > 0 ) d->phasiccount--;
  /* Kludge to supress spurious reporting of action potentials, now
     that the cell is silent. */
  if( d->phasiccount == 0 )
    s->owner->APthreshold = 10000;
}

void enq_AP(ActionPotential *d, double start) {
  /* Provide a method to force and AP on a cell. */
  AP *ap;

  /* First check that this cell is not refractory, that is that an
     action potential started less that the refractory period ago. */
  ap = d->active;
  while(ap) {
    if( start <= ap->starttime + d->abs_refractory )
      return;
    ap = ap->next;
  }
  
  ap = getmain(sizeof(*ap));
  ap->next      = d->active;
  d->active    = ap;
  ap->starttime = start;
  ap->cell_id   = -1;
  ap->window_id = -1;

  /* Force the step size down */
  stepOn(((Dynamics*)d)->owningCell, start+1e-3);

  return;
}

static double current(ActionPotential *d, double t) {
  Component *c;
  AP *ap = d->active;
  double E = GETEM_DYN(d, t);
  double Er;
  double time_constant;
  double dt;
  double (*shape)(double, double);
  double current;

  /* Initialise working storage */
  c = d->components;
  while(c) {
    c->Gsum = 0;
    c = c->next;
  }

  /* Sum up the nominal conductance changes, in each component
     generated by each action potential. */
  while(ap) {
    dt = t - ap->starttime;
    if( dt>0  ) {
      c = d->components;
      while(c) {
	shape         = c->shape;
	time_constant = c->time_constant;
	if( dt>c->delay)
	  c->Gsum += c->Gmax * shape(dt-c->delay, time_constant);
	c = c->next;
      }
    }
    ap = ap->next;
  }

  /* Finally work out the current, taking into account components that
     have maxed out their conductances and/or are modulated by other
     dynamics. */
  c = d->components;
  current = 0;
  while(c) {
    double g;
    Er = c->Er;
    
    if( c->modulator ) {
      double P = GETSTATE_DYN(c->modulator, 1);
      double Ca = c->Gsum < c->Gtotal ? c->Gsum : c->Gtotal;
      double rho = 1/(1+c->weight);
      /* See equation 2 Thomas and Bornstein 2002 */
      g = c->Gbasal * P * (1-Ca) +
	P * Ca * c->Gmax +
	rho * (1-P) * Ca * c->Gmax;
    } else {
      g = c->Gsum < c->Gtotal ? c->Gsum : c->Gtotal;
    }
    current += g * (Er - E);
    c = c->next;
  }

  return current;  
}

static Component *add_component(ActionPotential *d,
                          double Gmax, double Gtotal, double Er,
                          double time_constant, double delay, 
                          double (*shape)(double, double)) {
  /* 1) Search for the dynamics, if it doesn't exist create one and
     create the autosynapse.
     2) Make a Component and enqueue it to the dynamics
  */
  Component *c;

  c = getmain(sizeof(*c));
  c->next          = d->components;
  d->components    = c;
  c->Gmax          = Gmax;
  c->Gtotal        = Gtotal;
  c->Er            = Er;
  c->time_constant = time_constant;
  c->delay         = delay;
  c->shape         = shape;
  c->modulator     = 0;

  return c;
}

static PyObject * addPulse(ActionPotential *self, PyObject *args) {
    double Gmax;
    double Gtotal;
    double Er;
    double duration;
    double delay;

    if( !PyArg_ParseTuple(args, "ddddd", &Gmax, &Gtotal, &Er, &duration, &delay) )
        return NULL;

    if( !add_component(self, Gmax, Gtotal, Er, duration, delay, pulse) )
        return 0;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject * addAlpha(ActionPotential *self, PyObject *args) {
    double Gmax;
    double Gtotal;
    double Er;
    double k;
    double delay;

    if( !PyArg_ParseTuple(args, "ddddd", &Gmax, &Gtotal, &Er, &k, &delay) )
        return NULL;

    if( !add_component(self, Gmax, Gtotal, Er, k, delay, alpha) )
        return 0;

    Py_INCREF(Py_None);
    return Py_None; 
}

static PyObject * addGKCa(ActionPotential *self, PyObject *args) {
    double Gmax;
    double Gtotal;
    double Gbasal;
    double Er;
    double k;
    double delay;
    double weight;
    Dynamics *sEPSP;
    Component *comp;

    if( !PyArg_ParseTuple(args, "dddddddO", &Gmax, &Gtotal, &Gbasal, &Er, &k, &delay, &weight, &sEPSP) )
        return NULL;
    if( !(PyObject_TypeCheck(sEPSP, p3_DynamicsType)) ) {
        PyErr_SetString(PyExc_TypeError, 
            "The eighth argument must be of type slowEPSP");
        return NULL;
    }      

    comp = add_component(self, Gmax, Gtotal, Er, k, delay, alpha);
    if( !comp ) return 0;

    comp->modulator = sEPSP;
    comp->Gbasal    = Gbasal;
    comp->weight    = weight;

    Py_INCREF(Py_None);
    return Py_None; 
}

static PyMemberDef ActionPotential_members[] = {
    {"abs_refractory", T_DOUBLE, offsetof(ActionPotential, abs_refractory),
     0, "absolute refractory period"},
    {"phasiccount", T_INT, offsetof(ActionPotential, phasiccount),
     0, "absolute number of action potentials a cell may fire"},
    {NULL},
};

static int init(ActionPotential *ap) {    
  /* Create the auto synapse and initialise phasiccount*/
    PyObject *x;
    PyObject *args = Py_BuildValue("(OO)", (PyObject*)ap->owner, (PyObject*)ap);

    if( args==NULL ) return -1;

    x = (PyObject*)PyObject_GC_New(Synapse, p3_SynapseType);
    if( x==NULL ) return -1;

    /* Is this what I am supposed to do? */
    if( (p3_SynapseType)->tp_init(x, args, 0) != 0 ) {
        PyObject_GC_Del(x);
        return -1;
    }
    Py_DECREF(args);

    ap->synapse             = (Synapse*)x;
    ap->synapse->trans_time =  0;
    ap->components          =  0;  
    ap->phasiccount         = -1;
    ap->active              =  0;

	printf("Warning: ActionPotential has been completely broken "
		"since asynchornous messaging support was removed.\n");

    return 0;
}

static PyMethodDef ActionPotential_methods[] = {
    {"addPulse", (PyCFunction)addPulse, 
        METH_VARARGS, "addPulse(Gmax, Gtotal, Er, duration, delay)"},
    {"addAlpha", (PyCFunction)addAlpha, 
        METH_VARARGS, "addAlpha(Gmax, Gtotal, Er, k, delay)"},
    {"addGKCa", (PyCFunction)addGKCa, 
        METH_VARARGS, "addGKCa(Gmax, Gtotal, Gbasal, Er, k, delay, weight, slowEPSP)"},
    {NULL} /* Sentinel */
};

DynamicsDescriptor ActionPotentialDescriptor = {
    "ActionPotential",
    "ActionPotential, members: abs_refractory, phasiccount, methods: addPulse, addAlpha, addGKCa",
    ActionPotential_members, /* Members */
    ActionPotential_methods, /* Methods */
    0,  /* number of state variables */
    0,  /* State variable names */
    0,  /* deriv variable names */
    0,  /* trace variable names */
    0,  /* Derivs */
    (currentfcn*)current,
    (accepterfcn*)accepter,  /* accepter */
    (enqfcn*)enq_AP,
    0,
    sizeof(ActionPotential),
    0,
    (userinitfcn*)init,
	0,
	0,
	0,
	0,
	0,
	0,
	dynamics
};
REGISTER_DESCRIPTOR(ActionPotentialDescriptor)

static DynamicsDescriptor *userDynamics[] = {
    &ActionPotentialDescriptor
};

static initproc LuserInitDynamics[] = {
    initActionPotentialDescriptor
};


MAKE_P3_MODEL(lif, "The leaky integrate and fire model")


/* Changelog
   EAT 10Dec02
   see entry in synmsg.c
*/
