/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*******************************************************
  This provides a python class to wrap C double arrays.  
  It behaves like a Python list.
********************************************************/

#define P3_MODULE
#include "ndl.h"


PyTypeObject p3_DoubleArrayType;
static PyObject *indexerr;

static int puff_buffer(double_array *a, int newsize) {
    /* Increase the buffer size */
    double *newdata;
    newsize = newsize ? newsize : (int)(a->buffer_len*1.1)+1;
    newdata = PyMem_Realloc(a->data, newsize*sizeof(*a->data));
    if( !newdata ) {
        PyErr_NoMemory();
        return -1;
    }
    a->buffer_len = newsize;
    a->data = newdata;
    return 0;
}

/*********/
/* C API */
/********/
void da_set(double_array *a, double x, int i) {
    if( i>=a->buffer_len && puff_buffer(a, 0)==-1 )
        /* No memory - couldn't increase the buffer */
        longjmp(excpt_exit, 1);

    a->data[i] = x;
    a->len = a->len>i ? a->len : ++i;

    return;
}

double da_get(double_array *a, int i) { return a->data[i]; }

void da_clear(double_array *a) { if(a) a->len = 0; return; }

int da_length(double_array *a) { if(a) return a->len; else return 0;}

void da_append(double_array *a, double x) {
    da_set(a, x, a->len);
}

void da_copy(double_array *tgt, double_array *src) {
    if( tgt->len != src->len || tgt->buffer_len != src->buffer_len ) {
        PyErr_SetString(PyExc_RuntimeError, 
            "double_array size mismatch in da_copy()");
        longjmp(excpt_exit, 1);
    }
    
    memcpy(tgt->data, src->data, tgt->len*sizeof(double));
}

double_array *da_new(void) {
    double_array *x;
    PyObject *args = Py_BuildValue("()");

    x = (double_array*)PyObject_New(double_array, &p3_DoubleArrayType);
    if( x==NULL ) return NULL;
    /* Is this what I am supposed to do? */
    if( p3_DoubleArrayType.tp_init((PyObject*)x, args, args) != 0 ) {
        PyObject_Del((PyObject*)x);
        return NULL;
    }
    x->seq = -1;
    return x;
}

void da_seq_start(double_array *self) {
    if( self->seq != -1 ) {
        PyErr_SetString(PyExc_SystemError,
            "Attempt to start sequencing an array in which a sequence is already active.");
        longjmp(excpt_exit, 1);
    }
    self->seq = 0;
}

int da_seq_next(double_array *self, double *val) {
    if( self->seq == self->len ) {
        self->seq = -1;
        return 0;
    }
    *val = self->data[self->seq++];
    return 1;
}

void da_seq_set(double_array *self, double val) {
    if( self->seq <= 0 ) {
        PyErr_SetString(PyExc_SystemError,
            "Protocol error using da_seq_set().");
        longjmp(excpt_exit, 1);
    }
    self->data[self->seq-1] = val;
}

static int DoubleArray_ass_slice(double_array*, int, int, PyObject*);
void da_slice_del(double_array *self, int hi, int lo) {
	DoubleArray_ass_slice(self, lo, hi, 0);
}

static void delDoubleArray(double_array *self) {
    if( self->data ) PyMem_Free(self->data);
    self->ob_type->tp_free((PyObject*)self);
}

static int initDoubleArray(double_array *self, PyObject *args) {
    PyObject *data=0, *item;
    int i, size;

    if( !PyArg_ParseTuple(args, "|O", &data) )
        return -1;

    if( !data ) {
        self->buffer_len = self->len = 0;
        self->data = 0;
        return 0;
    }

    size = PyList_Size(data);
    if( size < 0 ) {
        PyErr_SetString(PyExc_TypeError, 
            "Array can only be initialised from a list");
        return -1;
    }

    self->buffer_len = self->len = size;
    if( size ) {
        self->data = PyMem_Malloc(size*sizeof(self->data[0]));
        if( !self->data ) {
            PyErr_NoMemory();
            return -1;
        }

        for(i=0; i<size; i++) {
            item = PyList_GetItem(data, i);

            if( PyFloat_Check(item) )
                self->data[i] = PyFloat_AsDouble(item);
            else if( PyInt_Check(item) )
                self->data[i] = (double)PyInt_AsLong(item);
            else {
                PyErr_SetString(PyExc_TypeError,
                    "List items must be floats or ints");
                return -1;
            }
        }
    }
    
    return 0;
}

static PyObject *reprDoubleArray(double_array *self) {
    int i=1, j=0;
    static char bigbuf[1000];

    bigbuf[0] = '[';

    if( self->len > 0 ) {
        while( i<1000-50 ) {
            i += sprintf(&bigbuf[i], "%g", self->data[j++]);
            if( j!=self->len )
                i += sprintf(&bigbuf[i], ", ");
            else
                break;
        }
    }

    bigbuf[i] = ']';
    bigbuf[i+1] = '\0';

    return PyString_FromString(bigbuf);
}

static PyObject *DoubleArray_append(double_array *op, PyObject *item) {
    double x;

    if( !PyObject_TypeCheck(op, &p3_DoubleArrayType) ) {
		PyErr_BadInternalCall();
		return NULL;
	}
    
    if( PyFloat_Check(item) )
        x = PyFloat_AsDouble(item);
    else if( PyInt_Check(item) )
        x = (double)PyInt_AsLong(item);
    else {
        PyErr_SetString(PyExc_TypeError,
            "DoubleArray items must be floats or ints");
        return NULL;
    }

    da_append(op, x);

    Py_INCREF(Py_None);
	return Py_None;
}

/*********************************************
               Sequence methods
**********************************************/
static int DoubleArray_length(double_array *a) {
	return a->len;
}

static PyObject *DoubleArray_concat(double_array *a, PyObject *b) {
	int size;
	int i;
	double_array *np;


    /* Concatenate any numeric list */
	if (!PyList_Check(b) && !PyObject_TypeCheck(b, &p3_DoubleArrayType) ) {
		PyErr_Format(PyExc_TypeError,
			  "can only concatenate list or DoubleArray (not \"%.200s\") to DoubleArray",
			  b->ob_type->tp_name);
		return NULL;
	}

    np = (double_array*)a->ob_type->tp_alloc(&p3_DoubleArrayType, 0);
    if( np==NULL ) return PyErr_NoMemory();

    size = a->len;
    if( PyObject_TypeCheck(b, &p3_DoubleArrayType) )
        size += ((double_array*)b)->len;
    else
        size += PyList_Size(b);

    np->data = PyMem_Malloc(size*sizeof(np->data[0]));
    if( !np->data ) {
        PyMem_Free(np);
        return PyErr_NoMemory();
    }
    np->len = np->buffer_len = size;

	for (i = 0; i < a->len; i++) {
		np->data[i] = a->data[i];
	}

    if( PyObject_TypeCheck(b, &p3_DoubleArrayType) ) {
	    for (i = 0; i < ((double_array*)b)->len; i++) {
		    np->data[i + a->len] = ((double_array*)b)->data[i];
        }
    } else {
	    for (i = 0; i < PyList_Size(b); i++) {
            PyObject *item;
            item = PyList_GetItem(b, i);
            if( PyFloat_Check(item) )
                np->data[i+a->len] = PyFloat_AsDouble(item);
            else if( PyInt_Check(item) ) {
                np->data[i+a->len] = (double)PyInt_AsLong(item);
            } else {
                PyErr_SetString(PyExc_TypeError,
                    "List items must be floats or ints");
                PyMem_Free(np->data);
                PyMem_Free(np);
                return NULL;
            }
        }
    }

	return (PyObject *)np;
}

static PyObject * DoubleArray_repeat(double_array *a, int n) {
	int i, j;
	int size;
	double_array *np;

    np = (double_array*)a->ob_type->tp_alloc(&p3_DoubleArrayType, 0);
	if (np == NULL)
		return NULL;

	if (n < 0) n = 0;
	size = a->len * n;
    np->len = np->buffer_len = size;
    if (size == 0) {
        np->data = 0;
        return (PyObject*)np;
    }

    np->data = PyMem_Malloc(size*sizeof(np->data[0]));
    if( !np->data ) {
        PyMem_Free(np);
        return PyErr_NoMemory();
    }

	if (a->len == 1) {
		for (i = 0; i < n; i++) {
			np->data[i] = a->data[0];
		}
		return (PyObject *) np;
	}

	for (i = 0; i < n; i++) {
		for (j = 0; j < a->len; j++) {
			np->data[i*a->len+j] = a->data[j];
		}
	}
	return (PyObject *) np;
}

static PyObject * DoubleArray_item(double_array *a, int i) {
	if (i < 0 || i >= a->len) {
		if (indexerr == NULL)
			indexerr = PyString_FromString(
				"list index out of range");
		PyErr_SetObject(PyExc_IndexError, indexerr);
		return NULL;
	}
	return PyFloat_FromDouble(a->data[i]);
}

static PyObject *DoubleArray_slice(double_array *a, int ilow, int ihigh) {
	double_array *np;
	int i;

	if (ilow < 0)
		ilow = 0;
	else if (ilow > a->len)
		ilow = a->len;
	if (ihigh < ilow)
		ihigh = ilow;
	else if (ihigh > a->len)
		ihigh = a->len;

	np = (double_array*)a->ob_type->tp_alloc(&p3_DoubleArrayType, 0);
	if (np == NULL)
		return NULL;
    np->len = np->buffer_len = ihigh - ilow;
    np->data = PyMem_Malloc(np->len*sizeof(np->data[0]));
    if( np->data == NULL ) {
        PyMem_Free(np);
        return PyErr_NoMemory();
    }

	for (i = ilow; i < ihigh; i++) {
		np->data[i - ilow] = a->data[i];
	}

	return (PyObject *)np;
}

static int DoubleArray_ass_slice(double_array *a, int ilow, int ihigh, PyObject *v) {
	int size; /* Size of replacement list */
    int i;
    double *temp;

    if( v==NULL ) 
        size = 0;
    else {
        size = PySequence_Size(v);
        if( size < 0 ) {
            PyErr_SetString(PyExc_TypeError, 
                "Only a sequence can slice assigned to a DoubleArray");
            return -1;
        }
    }

	if (ilow < 0)
		ilow = 0;
	else if (ilow > a->len)
		ilow = a->len;
	if (ihigh < ilow)
		ihigh = ilow;
	else if (ihigh > a->len)
		ihigh = a->len;

    /* Allocate new array */
    temp = PyMem_Malloc((a->len-ihigh+ilow+size)*sizeof(*temp));
    if( !temp ) {
        PyErr_NoMemory();
        return -1;
    }

    for(i=0; i<ilow; i++) temp[i] = a->data[i];

    for(i=ilow; i<ilow+size; i++) {
	    PyObject *item = PySequence_GetItem(v, i-ilow);
        if( PyFloat_Check(item) )
            temp[i] = PyFloat_AsDouble(item);
        else if( PyInt_Check(item) )
            temp[i] = (double)PyInt_AsLong(item);
        else {
            PyErr_SetString(PyExc_TypeError,
                "List items must be floats or ints");
            PyMem_Free(temp);
            return -1;
        }
    }

    for(i=ilow+size; i<a->len-ihigh+ilow+size; i++)
        temp[i] = a->data[ihigh+i-ilow-size];

    PyMem_Free(a->data);
    a->data = temp;
    a->buffer_len = a->len = a->len-ihigh+ilow+size;

	return 0;
}

static int DoubleArray_ass_item(double_array *a, int i, PyObject *v) {

	if (i < 0 || i >= a->len) {
		PyErr_SetString(PyExc_IndexError,
				"list assignment index out of range");
		return -1;
	}

    /* This might be del a[i] */
	if (v == NULL)
		return DoubleArray_ass_slice(a, i, i+1, v);

    if( PyFloat_Check(v) )
        a->data[i] = PyFloat_AsDouble(v);
    else if( PyInt_Check(v) ) {
        a->data[i] = (double)PyInt_AsLong(v);
    } else {
        PyErr_SetString(PyExc_TypeError,
            "List items must be floats or ints");
        return -1;
    }

	return 0;
}

static int DoubleArray_contains(double_array *a, PyObject *el) {
	int i, cmp;
    double x;
    
    if( PyFloat_Check(el) )
        x = PyFloat_AsDouble(el);
    else if( PyInt_Check(el) )
        x = (double)PyInt_AsLong(el);
    else
        return 0;

	for (i = 0, cmp = 0 ; cmp == 0 && i < a->len; ++i)
		cmp = x==a->data[i];

	return cmp;
}

static PySequenceMethods DoubleArray_as_sequence = {
	(inquiry)DoubleArray_length,			/* sq_length */
	(binaryfunc)DoubleArray_concat,		/* sq_concat */
	(intargfunc)DoubleArray_repeat,		/* sq_repeat */
	(intargfunc)DoubleArray_item,			/* sq_item */
	(intintargfunc)DoubleArray_slice,		/* sq_slice */
	(intobjargproc)DoubleArray_ass_item,		/* sq_ass_item */
	(intintobjargproc)DoubleArray_ass_slice,	/* sq_ass_slice */
	(objobjproc)DoubleArray_contains		/* sq_contains */
};

static PyMethodDef p3_DoubleArray_methods[] = {
	{"append",	(PyCFunction)DoubleArray_append,
        METH_O, "DA.append(number) -- append int or float to DoubleArray"},
 	{NULL,		NULL}		/* sentinel */
};

PyTypeObject p3_DoubleArrayType = {
    PyObject_HEAD_INIT(NULL)
    0,                         /*ob_size*/
    "_p3.DoubleArray",             /*tp_name*/
    sizeof(double_array),      /*tp_basicsize*/
    0,                         /*tp_itemsize*/
    (destructor)delDoubleArray,                         /*tp_dealloc*/
    0,                         /*tp_print*/
    0,                         /*tp_getattr*/
    0,                         /*tp_setattr*/
    0,                         /*tp_compare*/
    (reprfunc)reprDoubleArray,                         /*tp_repr*/
    0,                         /*tp_as_number*/
    &DoubleArray_as_sequence,                         /*tp_as_sequence*/
    0,                         /*tp_as_mapping*/
    0,                         /*tp_hash */
    0,                         /*tp_call*/
    0,                         /*tp_str*/
    0,                         /*tp_getattro*/
    0,                         /*tp_setattro*/
    0,                         /*tp_as_buffer*/
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,        /*tp_flags*/
    "Parplex double array object",   /* tp_doc */
    0,		               /* tp_traverse */
    0,		               /* tp_clear */
    0,		               /* tp_richcompare */
    0,		               /* tp_weaklistoffset */
    0,		               /* tp_iter */
    0,		               /* tp_iternext */
    p3_DoubleArray_methods,             /* tp_methods */
    0,             /* tp_members */
    0,                         /* tp_getset */
    0,                         /* tp_base */
    0,                         /* tp_dict */
    0,                         /* tp_descr_get */
    0,                         /* tp_descr_set */
    0,                         /* tp_dictoffset */
    (initproc)initDoubleArray,      /* tp_init */
    0,                         /* tp_alloc */
    PyType_GenericNew                 /* tp_new */
};

/* Changes
EAT 21Sep07
memcpy in da_copy was being based the wrong buffer length.
*/
