/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
   This code implements the Na, K (delayed rectifier) and synaptic
   dynamics used in Wang & Buzsaki, J Neurosci, 6402-6413 (1996)
*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/


#include "ndl.h"

/*---------
     Na
-----------*/
typedef struct {
    DYNAMICS
  double gNa;    /* The maximum conductance */
  double VNa;      /* The driving potential */
  double _theta;
}  Na;

static void Naderivs(Na *self, double t) {
  double Em     = GETEM_DYN(self, t);
  double h      = GETSTATE_DYN(self, 0);
  double alphah = 0.07 * exp(-(Em+58)/20);
  double betah  = 1/(exp(-0.1*(Em+28))+1);
  double dhdt = self->_theta * (alphah*(1-h) - betah*h);
  SETDERIV_DYN(self, 0, dhdt);
  return;
}

static double Nacurrent(Na *self, double t) {
  double Em  = GETEM_DYN(self, t);
  double gNa = self->gNa;
  double VNa = self->VNa;
  double h   = GETSTATE_DYN(self, 0);
  double alpham, betam, minf;
  if( Em==-35 )
      alpham = 1;
  else
      alpham = -0.1*(Em+35)/(exp(-0.1*(Em+35))-1);
  betam = 4*exp(-(Em+60)/18);
  minf = alpham/(alpham+betam);
  return -gNa*pow(minf,3)*h*(Em-VNa);
}
static PyMemberDef Na_members[] = {
    {"gNa", T_DOUBLE, offsetof(Na, gNa), 0, "maximum conductance"},
    {"VNa", T_DOUBLE, offsetof(Na, VNa), 0, "reversal potential"},
    {"_theta", T_DOUBLE, offsetof(Na, _theta), 0, "theta"},
    {NULL},
};

static char *NaStateVars[] = {"h"};
static char *NaDerivVars[] = {"dhdt"};
static char *NaTraceVars[] = {"hTrace"};
DynamicsDescriptor wbNaDescriptor = {
    "wbNa",
    "wbNa Wang and Buzsaki Na current. members: gNa, VNa, _theta",
    Na_members,
    0,
    1,
    NaStateVars,
    NaDerivVars,
    NaTraceVars,
    (derivsfcn*)Naderivs,
    (currentfcn*)Nacurrent,
    0,
    0,
    0,
    sizeof(Na),
    0
};
REGISTER_DESCRIPTOR(wbNaDescriptor)


/*----------
     Kdr
------------*/
typedef struct {
    DYNAMICS
  double gKdr;    /* The maximum conductance */
  double VK;      /* The driving potential */
  double _theta;
}  Kdr;

static void Kdrderivs(Kdr *self, double t) {
  double Em   = GETEM_DYN(self, t);
  double n    = GETSTATE_DYN(self, 0);
  double dndt, alphan, betan;
  if( Em==-34 )
      alphan = 0.1;
  else
      alphan = -0.01*(Em+34)/(exp(-0.1*(Em+34))-1);
  betan = 0.125*exp(-(Em+44)/80);
  dndt = self->_theta * (alphan*(1-n) - betan*n);
  SETDERIV_DYN(self, 0, dndt);
  return;
}

static double Kdrcurrent(Kdr *self, double t) {
  double Em   = GETEM_DYN(self, t);
  double gKdr = self->gKdr;
  double VK   = self->VK;
  double n    = GETSTATE_DYN(self, 0);
  return -gKdr*pow(n,4)*(Em-VK);
}

static PyMemberDef Kdr_members[] = {
    {"gKdr", T_DOUBLE, offsetof(Kdr, gKdr), 0, "maximum conductance"},
    {"VK", T_DOUBLE, offsetof(Kdr, VK), 0, "reversal potential"},
    {"_theta", T_DOUBLE, offsetof(Kdr, _theta), 0, "theta"},
    {NULL},
};

static char *KdrStateVars[] = {"n"};
static char *KdrDerivVars[] = {"dndt"};
static char *KdrTraceVars[] = {"nTrace"};
DynamicsDescriptor wbKdrDescriptor = {
    "wbKdr",
    "wbKdr Wang and Buzsaki K delayed rectifier current. members: gKdr, VK, _theta",
    Kdr_members,
    0,
    1,
    KdrStateVars,
    KdrDerivVars,
    KdrTraceVars,
    (derivsfcn*)Kdrderivs,
    (currentfcn*)Kdrcurrent,
    0,
    0,
    0,
    sizeof(Kdr),
    0
};
REGISTER_DESCRIPTOR(wbKdrDescriptor)

/*-----------------
    fast IPSP
------------------*/
typedef struct {
    DYNAMICS
  double gSyn;    /* The maximum conductance */
  double VSyn;      /* The driving potential */
  double strength;
  double starttime;
}  Syn;

/* Rate constants */
static double alpha = 12;
static double beta  = 0.1;

/* Table lookup for presynaptic voltage */
static double Vpre[] = {
        -2.0680000e+001, -1.6621000e+001, -1.2232600e+001, -7.8104400e+000,
        -3.5422700e+000,  4.4759400e-001,  4.0797800e+000,  7.3201100e+000,
        1.0277600e+001,   1.2917800e+001,  1.5247600e+001,  1.7304200e+001,
        1.9123100e+001,   2.0752600e+001,  2.2025600e+001,  2.3184800e+001,
        2.4243900e+001,   2.4946500e+001,  2.5630000e+001,  2.6056600e+001,
        2.6435600e+001,   2.6566000e+001,  2.6714600e+001,  2.6532000e+001,
        2.6339500e+001,   2.6021400e+001,  2.5561900e+001,  2.5092300e+001,
        2.4356000e+001,   2.3599500e+001,  2.2764300e+001,  2.1800300e+001,
        2.0809000e+001,   1.9656400e+001,  1.8517900e+001,  1.7210500e+001,
        1.5933500e+001,   1.4508400e+001,  1.3108900e+001,  1.1598700e+001,
        1.0099700e+001,   8.5302300e+000,  6.9586100e+000,  5.3488200e+000,
        3.7314400e+000,   2.1012800e+000,  4.5609000e-001, -1.1888500e+000,
        -2.8347900e+000, -4.4754500e+000, -6.1089500e+000, -7.7321900e+000,
        -9.3380000e+000, -1.0934600e+001, -1.2504700e+001,  -1.4062400e+001,
        -1.5604600e+001, -1.7099100e+001, -1.8595200e+001
};
static double dt = 0.01; /* Table sample rate */

static int init(Syn *self) {self->starttime = -100; return 0;}

static double F(double t) {
    /* Linear interpolation from the Vpre table,
    the processed through a sigmoid */
    int i = (int)(t/dt);
    double x;
    if( i>=(sizeof(Vpre)/sizeof(double))-1 ) return 0;
    x = Vpre[i] + (t/dt-i)*Vpre[i+1];
    return 1/(exp(-x/2)+1);
}

static void Synderivs(Syn *self, double t) {
    double s = GETSTATE_DYN(self, 0);
    double dsdt, f;
    if( t>=self->starttime )
        f = F(t-self->starttime);
    else
        f = 0;
    dsdt = f*alpha*(1-s) - beta*s;
  SETDERIV_DYN(self, 0, dsdt);
  return;
}

static void Synaccepter(Syn *self, Synapse *synapse, double strength, int window_id) {
    Cell *tgt   = synapse->target;
    self->starttime = tgt->time + synapse->trans_time;
    self->strength = strength;
    stepOn(tgt, self->starttime+1e-3);
    return;
}

static void Synenq(Syn *self, double starttime, double strength) {
    self->starttime = starttime;
    self->strength  = strength;
    stepOn(self->owningCell, self->starttime+1e-3);
}

static double Syncurrent(Syn *self, double t) {
  double Em   = GETEM_DYN(self, t);
  double s    = GETSTATE_DYN(self, 0);
  return -self->gSyn*s*(Em-self->VSyn)*self->strength;
}

static PyMemberDef Syn_members[] = {
    {"gSyn", T_DOUBLE, offsetof(Syn, gSyn), 0, "maximum conductance"},
    {"VSyn", T_DOUBLE, offsetof(Syn, VSyn), 0, "reversal potential"},
    {"starttime", T_DOUBLE, offsetof(Syn, starttime), 0, "current event starttime"},
    {"strength", T_DOUBLE, offsetof(Syn, strength), 0, "current event strength"},
    {NULL},
};

static char *SynStateVars[] = {"s"};
static char *SynDerivVars[] = {"dsdt"};
static char *SynTraceVars[] = {"sTrace"};
DynamicsDescriptor wbSynDescriptor = {
    "wbIPSP",
    "wbIPSP Wang and Buzsaki fast IPSP. members: gSyn, Syn, strenth, starttime",
    Syn_members,
    0,
    1,
    SynStateVars,
    SynDerivVars,
    SynTraceVars,
    (derivsfcn*)Synderivs,
    (currentfcn*)Syncurrent,
    (accepterfcn*)Synaccepter,
    (enqfcn*)Synenq,
    0,
    sizeof(Syn),
    0,
    (userinitfcn*)init
};
REGISTER_DESCRIPTOR(wbSynDescriptor)

static DynamicsDescriptor *userDynamics[] = {
  &wbKdrDescriptor,
  &wbNaDescriptor,
  &wbSynDescriptor
};

static initproc LuserInitDynamics[] = {
  initwbKdrDescriptor,
  initwbNaDescriptor,
  initwbSynDescriptor
};


MAKE_P3_MODEL(wb, "The Wang and Buzsaki interneuron model")
