/*
 * Copyright (C) 2004 Evan Thomas
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*********************************************************************
   A simple (non-reentrant) priority queue based on a floating point
   priority which represents the time at which the item is ready.
**********************************************************************/


#define P3_MODULE
#include "ndl.h"

#ifndef DBL_MAX
#define DBL_MAX 1e64  // just a really big number!
#endif

static Cell **thequeue;
static int last_el=0;
static int size=0;

static Cell **thelist;

static int next_pend_entry;
static Cell **pend_cell_list;
static double *pend_step_list;
static double *pend_time_list;

static void upheap(int);
static void downheap(int);

static int ge(Cell *a, Cell *b) {
  return (a->time + a->step) >= (b->time + b->step);
}
static int gt(Cell *a, Cell *b) {
  return (a->time + a->step) > (b->time + b->step);
}

void pq_init(int total_size) {
    if( thequeue ) PyMem_Free(thequeue);
    last_el = 0;
    size = total_size+1;
    thequeue = getmain(size * sizeof(*thequeue));
    thequeue[0] = getmain(sizeof(Cell));
    thequeue[0]->time = -DBL_MAX;
    thequeue[0]->step = 0;

    pend_cell_list = getmain(total_size * sizeof(*pend_cell_list));
    pend_time_list = getmain(total_size * sizeof(*pend_time_list));
    pend_step_list = getmain(total_size * sizeof(*pend_step_list));

    thelist  = getmain(size * sizeof(*thequeue));

}

int pq_isEmpty() { return last_el==0; }

void pq_insert(Cell *data) {
  last_el++;
  
  if( last_el >=  size ) {
      char buf[200];
      sprintf(buf, "Queue overflow in pq_insert() - current size %d.\n", size);
      ABEND(buf, (Cell*)Py_None);
  }

  thequeue[last_el] = data;
  thequeue[last_el]->queue_magic = last_el;

  thelist[last_el] = data;

  upheap(last_el);

  return;
}

void pq_lazy_insert(Cell *r) {
  /* Fix up the queue after a pq_lazy_dequeue() */

  downheap(1);

  return;
}

static void upheap(int k) {
  /* Propogate last element up to the correct point in the queue */
  
  Cell *v = thequeue[k];
  int i = k;

  while( ge(thequeue[i / 2], v) ) {
    thequeue[i] = thequeue[i / 2];
    thequeue[i]->queue_magic = i; 
    i /= 2;
    thequeue[i] = v;
    thequeue[i]->queue_magic = i; 
  }

  return;

}

/* This function breaks the pq_seq_* functions and so should not be
  used unless some sort of locking or checking protocol is used.

void *pq_dequeue() {
  Cell *r;

  if( pq_isEmpty() ) return 0;

  r = thequeue[1];
  thequeue[1] = thequeue[last_el--];
  thequeue[1]->queue_magic = 1;

  downheap(1);

  return r;
} */

void *pq_lazy_dequeue() {
  /* The queue is not integral while the cell is "checked out" */
  return thequeue[1];
}

static void downheap(int k) {
  int j;
  Cell *v = thequeue[k];

  while(k <= last_el / 2) {
    j = 2 * k;
    if( j < last_el )
      if( gt(thequeue[j], thequeue[j+1]) )
	j++;
    if( ge(thequeue[j], v) ) break;
    thequeue[k] = thequeue[j];
    thequeue[k]->queue_magic = k;
    k = j;
  }
  thequeue[k] = v;
  thequeue[k]->queue_magic = k;

  return;
    
}

void pq_requeue(Cell *v, double new_time, double new_step) {
  /* Change the priority of an element */
  int k = v->queue_magic;
  double old_priority = v->time + v->step;

  thequeue[k]->time = new_time;
  thequeue[k]->step = new_step;

  if( old_priority > new_time + new_step ) upheap(k);
  else downheap(k);

  return;

}

double pq_nextTime() {
  if( pq_isEmpty() ) return DBL_MAX;
  return thequeue[1]->time + thequeue[1]->step;
}

void pq_remove(Cell *v ) {
  /* Remove and arbitrary element from the Q */
  int k = v->queue_magic;

  thequeue[k] = thequeue[last_el--];
  thequeue[k]->queue_magic = k;
  downheap(k);

  return;
}

int pq_size() { return last_el; }

/**********************************************************
  Methods optimised for fixed time step solvers (eg Euler)
  (ie round robin)
 **********************************************************/
static int next_rr_cell = -1;

void *rr_dequeue() {
  next_rr_cell++;
  next_rr_cell %= last_el;
  return thequeue[next_rr_cell+1];
}

void rr_insert(Cell *data) { return; }

/*************************************************
  Methods to interate over all cells in the queue
**************************************************/
static int seq_next_cell;
void pq_seq_start(void) {seq_next_cell = 1;}
void* pq_seq_next(void) {
  if( seq_next_cell > last_el ) return 0;

  return thelist[seq_next_cell++];
}

/* CHANGELOG
   EAT 27/11/02
   added the pending reqeue feature

   EAT 01Oct07
   got rid of the pend requeue stuff since it didn't provide
   any actual benefit.
*/
