package org.catacomb.interlish.content;

import org.catacomb.interlish.structure.*;
import org.catacomb.report.E;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;


public class KeyedList<V> implements ElementWriter {

   ArrayList<V> items;
   HashMap<String, V> itemHM;

   Class itemClass;

   ArrayList<String> keyCache;

   HashMap<String, String> shortToFullHM;

   HashSet<String> duplicateShorts;
   
   int inewid = 0;

   ArrayList<ListWatcher> listWatchers;
   
   public KeyedList() {
      items = new ArrayList<V>();
      itemHM = new HashMap<String, V>();
      shortToFullHM = new HashMap<String, String>();
   }


   public KeyedList(String s) {
      this();
      try {
         itemClass = Class.forName(s);
      } catch (Exception ex) {
         E.error("cant find class " + s + " " + ex);
      }
   }


   public KeyedList(Class c) {
      this();
      itemClass = c;
   }


   public KeyedList(ArrayList<V> elts) {
      this();
      addAll(elts);
   }
   
   public void add(V v) {
      addItem(v);
   }


   public void addAll(ArrayList<V> elts) {
      for (V v : elts) {
         addItem(v);
      }
   }

   public ArrayList<V> getItems(String[] ks) {
      ArrayList<V> ret = new ArrayList<V>();
      if (ks != null) {
         for (String s : ks) {
            if (hasItem(s)) {
               ret.add(get(s));
            }
         }
      }
      return ret;
   }
   
   public ArrayList<V> getItems(ArrayList<String> ks) {
      ArrayList<V> ret = new ArrayList<V>();
      for (String s : ks) {
         ret.add(get(s));
      }
      return ret;
   }
   
   
   public ArrayList<V> getItems() {
      return items;
   }

   public void silentAddItem(V obj) {
      items.add(obj);
      String sid = addToHM(obj);
      addKey(sid);
      reportChange();
   }
   

   public void addItem(V obj) {
      silentAddItem(obj);
      reportChange();
   }

   private String addToHM(V obj) {
      String sid = "";
      if (obj instanceof IDd) {
         sid = ((IDd)obj).getID();
      } else {
         sid = obj.toString();
      }
      itemHM.put(sid, obj);
      keyCache = null;
      return sid;
   }


   public void remove(V obj) {
      String sid = "";
      if (obj instanceof IDd) {
         sid = ((IDd)obj).getID();
      } else {
         sid = obj.toString();
      }
      items.remove(obj);
      itemHM.remove(sid);
      removeKey(sid);
      reportChange();
   }


   public void put(String s, V v) {
      items.add(v);
      itemHM.put(s, v);
      keyCache = null;
      addKey(s);
   }


   public void putNew(String sid, V psc) {
      if (itemHM.containsKey(sid)) {
         E.warning("put new tried to override existing item " + sid);
      } else {
         put(sid, psc);
      }
   }


   public boolean hasItem(String s) {
      return (shortToFullHM.containsKey(s) || itemHM.containsKey(s));
   }


   public V get(String s) {
      V ret = null;
      if (itemHM.containsKey(s)) {
         ret = itemHM.get(s);
      } else {
            String sf = getFullID(s);
            if (itemHM.containsKey(sf)) {
               ret = itemHM.get(sf);
            } else {
               E.error("cant get " + s + " from keyed list");
            }

      }
      return ret;
   }


   @SuppressWarnings( { "unchecked" })
   public V getOrMake(String s) {
      V ret = null;
      if (hasItem(s)) {
         ret = get(s);

      } else if (itemClass != null) {
         try {
            Object obj = itemClass.newInstance();
            ret = (V)obj;

            if (ret instanceof IDable) {
               ((IDable)ret).setID(s);
            } else {
               E.warning("autogenerated items should be IDable " + itemClass);
            }

            addItem(ret);

         } catch (Exception ex) {
            E.error("cant make item of type " + itemClass + " " + ex);
         }


      } else {
         E.error("cant make new item - null classs");
      }


      return ret;

   }


   @SuppressWarnings( { "unchecked" })
   public void superceded(V old) {
      if (old instanceof Supercedable) {
         V repl = (V)((Supercedable)old).getSupercessor();

         if (items.contains(old)) {
            items.remove(old);
            items.add(repl);
         } else {
            E.error("supercedee isnt in list " + old + " " + old.hashCode());
            for (Object obj : items) {
               E.info("is in: " + obj + " " + obj.hashCode());
            }
         }

        addToHM(repl);

      } else {
         E.error(" cant supercede non supercedable: " + old);
      }
   }

   public void quietSuperceded(V old) {
      if (items.contains(old)) {
         superceded(old);
      }
      // else do nothing, and dont complain;
   }



   public V getFirst() {
      return items.get(0);
   }



   public Element makeElement(ElementFactory ef, Elementizer eltz) {
      Element elt = ef.makeElement("KeyedList");
      for (V obj : getItems()) {

         Element chld = eltz.elementize(obj);
         ef.addElement(elt, chld);
      }
      return elt;
   }


   public ArrayList<String> getKeys() {
      if (keyCache == null) {
         keyCache = new ArrayList<String>();
         keyCache.addAll(itemHM.keySet());
      }
      return keyCache;
   }


   public ArrayList<String> getShortKeys() {
      ArrayList<String> al = new ArrayList<String>();
      al.addAll(shortToFullHM.keySet());
      return al;
   }


   public void remove(String s) {
      if (hasItem(s)) {
         V val = get(s);
         items.remove(val);
         itemHM.remove(s);
         removeKey(s);
         reportChange();
      }
   }

   private void removeKey(String s) {
      String ss = s.substring(s.lastIndexOf(".") + 1, s.length());
      shortToFullHM.remove(ss);
      keyCache = null;
   }



   private void addKey(String s) {
      String ss = s.substring(s.lastIndexOf(".") + 1, s.length());
      if (shortToFullHM.containsKey(ss)) {

         E.shortWarning("duplicate items - deleted " + 
               shortToFullHM.get(ss) + " and " + s);
      
         shortToFullHM.remove(ss);
         if (duplicateShorts == null) {
            duplicateShorts = new HashSet<String>();
         }
         duplicateShorts.add(ss);
         
      } else {
         shortToFullHM.put(ss, s);
      }
   }


   public boolean hasFullID(String sid) {
      boolean ret = false;
      if (sid.indexOf(".") > 0) {
         ret = itemHM.containsKey(sid);
      } else {
         ret = shortToFullHM.containsKey(sid);
      }
      return ret;
    }


   public String getFullID(String sid) {
      String ret = null;
      if (sid == null) {
         // just leave fro null return;
      } else {
      if (sid.indexOf(".") > 0) {
         ret = sid;
      } else {
         if (shortToFullHM.containsKey(sid)) {
            String s = shortToFullHM.get(sid);
            if (s.equals("_duplicate_")) {
               E.shortError("Duplicate short ID - must use full ID " + sid);
            } else {
               ret = s;
            }

         } else {
            E.error("no such key " + sid);
         }
      }
      }
      return ret;
   }


   public String printIDs() {
      StringBuffer sb = new StringBuffer();
      int iel = 0;
      for (String s : itemHM.keySet()) {
         sb.append(s);
         sb.append(", ");
         iel += 1;
         if (iel % 4 == 0) {
            sb.append("\n");
         }
      }
      return sb.toString();
   }


   public void dump() {
      for (String s : itemHM.keySet()) {
         E.info("kl item " + s + " " + itemHM.get(s));
      }
      for (String s : shortToFullHM.keySet()) {
         E.info("short key " + s + " " + shortToFullHM.get(s));
      }
   }
   
   
   public String[] getKeysArray() {
      return getKeys().toArray(new String[0]);
   }

   public String[] getShortKeysArray() {
      return getShortKeys().toArray(new String[0]);
   }


   public ArrayList<V> getDescendants(String rtid) {
      ArrayList<V> ret = new ArrayList<V>();
      String fr = rtid + ".";
      for (String sk : itemHM.keySet()) {
         if (sk.startsWith(fr)) {
            ret.add(itemHM.get(sk));
         }
      }
      return ret;
   }


   public String newName(String root) {
      while (itemHM.containsKey(root + "_" + inewid)) {
         inewid += 1;
      }
      return root + "_" + inewid;
   }


   public void removeListWatcher(ListWatcher lw) {
      listWatchers.remove(lw);
   }


   public void reportChange() {
      if (listWatchers != null) {
         for (ListWatcher lw : listWatchers) {
            lw.listChanged(this);
         }
      }
   }
   
   public void addListWatcher(ListWatcher lw) {
      if (listWatchers == null) {
         listWatchers = new ArrayList<ListWatcher>();
      }
      listWatchers.add(lw);
   }


   public int size() {
      return items.size();
   }




}
