COMMENT
-----------------------------------------------------------------------------

	Kinetic model of GABA-A receptors
	=================================

	7-state gating model from Jones and Westbrook (Neuron 15, 181 - 191, 1995)



	      D1    D2
              |	    |
	C1 -- C2 -- C3
	      |     |
      	      O1    O2

	Or 16-state gating scheme from Haas and MacDonald (J Physiol 514.1, 27 - 45, 1999)

	

		    D2    D3 -- D4
		    |	  | 
	C1 -- C2 -- C3 -- C4
	      |	    |     | 	
	     O1     O2    O3
	      /\    /\    /\
	    C5 C6 C7 C8 C9 C10

Reversal potential Egaba is changing according to [Cl-]i change (due to Cl- influx). Bicarbonate (HCO3) flows through the GABAR too, and therefore Egaba is also [HCO3]i/[HCO3]o -dependent. igaba = icl + ihco3 (we assume icl and ihco3 to be mutually independent)

Based on gabaA_Cl.mod, modified to use GHK current equation.

ENDCOMMENT


TITLE detailed GABAergic conductance with changing Cl- concentration

NEURON {
	POINT_PROCESS gaghk
	USEION cl READ cli, clo WRITE icl VALENCE -1
	USEION hco3 READ hco3i, hco3o WRITE ihco3 VALENCE -1
	RANGE icl, ihco3, i

	RANGE GABAdur, GABA, taugaba, GABAINIT
	RANGE f1, f2
	RANGE kon, koff, koff2, k34, k43, alfa1, beta1, alfa2, beta2, alfa3, beta3 
	RANGE a1o, a1c, b1o, b1c, a2o, a2c, b2o, b2c, a3o, a3c, b3o, b3c
	RANGE d1, r1, d2, r2, d3, r3, d4, r4
	RANGE C1, C2, C3, C4, C5, C6, C7, C8, C9, C1O
	RANGE O1, O2, O3, D1, D2, D3, D4
	RANGE Prel, Pcl, Phco3, Rnumber
	RANGE ecl, ehco3, egaba
	RANGE gcl, ghco3, grel

	
}

UNITS {	
        (mA)    = (milliamp)
        (nA)    = (nanoamp)
	(mV)    = (millivolt)
	(uS)  = (micromho)
	(mM)    = (milli/liter)
	(uM)    = (micro/liter)
	F = (faraday) (coulombs)
	R = (k-mole)  (joule/degC)
}

PARAMETER {

: these must be specified at the hoc level, or through clever use 
: of the INITIAL block
:	hco3o   = 26    (mM)          : extracellular HCO3- concentration
:	hco3i   = 16    (mM)          : intracellular HCO3- concentration			

	Prel  	= 0.18		      : Phco3/Pcl relative permeability
	Rnumber	= 50		      : number of GABAARs in the synaptic compartment

	Pcl    	= 8e-14 (cm3/s)	      : maximum Cl- single channel permeability for GABAAR 

	
: the value assigned here will have no effect; must be specified at the hoc level

	celsius = 37    (degC)
	GABAdur	= 1	(ms)	      : transmitter duration (rising phase)
	taugaba = 0.2   (ms)

: Rates
	kon	= .003   (/uM /ms)	: binding (.007 Haas)				
	koff	= .150	 (/ms)		: unbinding (170 Haas)
	koff2   = .300   (/ms)		: unbinding
	alfa1	= 1.111	 (/ms)		: closing
	beta1	= .200	 (/ms)		: opening
	alfa2	= .142   (/ms)		: closing
	beta2	= 2.500  (/ms)		: opening
	alfa3   = .150	 (/ms)		: closing
	beta3   = .076   (/ms)		: opening
	d1      = .013   (/ms)		: slow desensitizing
	r1	= .00013 (/ms)		: resensitizing
	d2	= .750   (/ms)		: fast desensitizing (750 - 1000), 960 (Haas)
	r2	= .015   (/ms)		: resensitizing (15 - 25), 22 (Haas)
	d3	= .008   (/ms)		: intermediate desensitizing
	r3	= .00081 (/ms)		: resensitizing
	d4      = .00075 (/ms)		: slow desenzitizing
	r4	= .00049 (/ms)		: resenzitizing
	k34	= .710   (/ms)		
	k43	= .058   (/ms)		
	a1o	= .180	 (/ms)
	a2o	= .180	 (/ms)
	a1c	= 5.100  (/ms)
	a2c	= 5.100  (/ms) 
	b1o	= .070   (/ms)
	b2o	= .070   (/ms)
	b1c	= .630   (/ms)
	b2c	= .630   (/ms)
	a3o	= .090   (/ms)
	b3o	= .035   (/ms)
	a3c	= 5.100  (/ms)
	b3c	= .630   (/ms)
}


ASSIGNED {
	v		(mV)		: postsynaptic voltage - we hypothesize that Egaba changes due to increase of [Cl]i

	cli	        (mM)
	clo		(mM)
	icl             (nA)		: chloride current
	ecl		(mV)		: equilibrium potential for Cl-

	hco3i		(mM)
	hco3o		(mM)
	ihco3           (nA)		: bicarb current
	ehco3		(mV)		: equilibrium potential for HCO3-	

	egaba		(mV)		: reversal potential for GABAR	

	i               (nA)    	: total current generated by this mechanism = icl + ihco3
	
	GABA		(mM)		: transmitter concentration
	GABAINIT	(mM) 		: increased transmitter concentration after release
	time0		(ms)
			
        f1		(/ms)    	: binding
	f2		(/ms)   	: binding
	
	
	Phco3  		(cm3/s)		: max Phco3 = 0.18 * max Pcl
	gcl		(uS)		: GABA - induced conductance for chloride
	ghco3		(uS)		: GABA - induced conductance for bicarbonate
	grel				: relative conductance

}

STATE {
        : Channel states (all fractions)

	C1		: unbound
	C2		: single bound
	C3		: double bound
	C4		: double bound
	C5
	C6
	C7
	C8
	C9
	C1O
	O1		: open
	O2		: open
	O3		: open
	D1		: desensitized
	D2		: desensitized
	D3		: desensitized
	D4		: desensitized
}

INITIAL { 
	GABA		= 0
	GABAINIT	= 0
	C1		= 1
	C2		= 0
	C3		= 0
	C4		= 0
	C5		= 0
	C6		= 0
	C7		= 0
	C8		= 0
	C9		= 0
	C1O		= 0
	O1		= 0
	O2		= 0
	O3		= 0
	D1		= 0
	D2		= 0
	D3		= 0
	D4		= 0
	icl   = 0
	ihco3 = 0
	i     = 0

	Phco3 = Prel * Pcl	
}

BREAKPOINT {
	
	if (GABAINIT > 0) 
          
          { GABA = GABAINIT * exp(-(t-time0)/taugaba) }
	
	else {GABA = GABAINIT}
	
	SOLVE kstates METHOD sparse

	icl   = (1e+06)*(O1+O2+O3) * Pcl * Rnumber * ghk(v, cli, clo, -1)          :1e+6 is a factor to convert mA to nA
	ihco3 = (1e+06)*(O1+O2+O3) * Phco3 * Rnumber * ghk(v, hco3i, hco3o, -1) 
	i = icl + ihco3
	
	egaba = ghkvoltage(cli, clo, hco3i, hco3o)
	gcl = (1e+06)*(O1+O2+O3) * Pcl * Rnumber * conduct(v, cli, clo)		:1e+6 is a factor to convert S to uS
	ghco3 = (1e+06)*(O1+O2+O3) * Phco3 * Rnumber *conduct(v, hco3i, hco3o)
	if (gcl>0) {grel = ghco3/gcl} else {grel = 0}
}

KINETIC kstates {
	f1    = 2 * kon * (1e3) * GABA
	f2    = kon * (1e3) * GABA

	~ C1 <-> C2	(f1,koff)
	~ C2 <-> C3	(f2,koff2)
	~ C2 <-> O1	(beta1,alfa1)
	~ C3 <-> O2	(beta2,alfa2)
	~ C2 <-> D1     (d1,r1)
	~ C3 <-> D2     (d2,r2)

	: Haas and MacDonald kinetics
	~ C3 <-> C4     (k34,k43)
	~ C4 <-> O3     (beta3,alfa3)
	~ C4 <-> D3     (d3,r3)
	~ D3 <-> D4     (d4,r4)

	~ O1 <-> C5     (a1c,a1o)
	~ O1 <-> C6     (b1c,b1o)
	~ O2 <-> C7     (a2c,a2o)
	~ O2 <-> C8     (b2c,b2o)
	~ O3 <-> C9     (a3c,a3o)
	~ O3 <-> C1O    (b3c,b3o)

	CONSERVE C1+C2+C3+C4+C5+C6+C7+C8+C9+C1O+O1+O2+O3+D1+D2+D3+D4 = 1
}


NET_RECEIVE(weight(mM), nspike) {

	: an onset event (generated by NetStim) always has an implicit argument called flag which is set to 0
	if (flag == 0) {
		nspike = nspike + 1
		time0 = t
		GABAINIT  = weight
		: come again in Cdur with flag = current value of nspike (selfevent generated with delay Cdur & flag=nspike)
		net_send(GABAdur, nspike)
	}
	
	if (flag == nspike) {
		: if this associated with last spike then turn off
		GABAINIT = 0
	}
}

FUNCTION ghk(v(mV), ci(mM), co(mM), z)  (millicoul/cm3) { 
	LOCAL e, w
        w = v * (.001) * z*F / (R*(celsius+273.15))
        if (fabs(w)>1e-4) 
          { e = w / (exp(w)-1) }
        else 
	: denominator is small -> Taylor series
          { e = 1-w/2 }
        ghk = - (.001) * z* F * (co-ci*exp(w)) * e
}

FUNCTION ghkvoltage(c1i(mM), c1o(mM), c2i(mM), c2o(mM))  (mV) {

	ghkvoltage = - (1000)*(celsius + 273.15)*R/F*log((c1o + Prel*c2o)/(c1i + Prel*c2i))
}

FUNCTION conduct(v(mV), ci(mM), co(mM))  (millicoul/cm3/mV) { 
	LOCAL w
        w = v * (.001) *F / (R*(celsius+273.15))
	conduct = (0.001)*(.001)*F^2/(R*(celsius+273.15))*(ci-(co+ci)*exp(w)+(ci-co)*w*exp(w)+co*(exp(w)^2))/((1-exp(w))^2)
}

