COMMENT

Synaptic GABAergic mechanism 

Reversal potential Egaba is changing according to [Cl-]i change (due to Cl- influx, which we hypothesize to be significant). Bicarbonate (HCO3) flows through the GABAR too, and therefore Egaba is also [HCO3]i/[HCO3]o -dependent
igaba = icl + ihco3 (we assume icl and ihco3 to be mutually independent)

Two state kinetic scheme synapse described by rise time tau1,
and decay time constant tau2. The normalized peak condunductance is 1.
Decay time MUST be greater than rise time.

The solution of A->G->bath with rate constants 1/tau1 and 1/tau2 is
 A = a*exp(-t/tau1) and
 G = a*tau2/(tau2-tau1)*(-exp(-t/tau1) + exp(-t/tau2))
	where tau1 < tau2

If tau2-tau1 -> 0 then we have a alphasynapse.
and if tau1 -> 0 then we have just single exponential decay.

The factor is evaluated in the
initial block such that an event of weight 1 generates a
peak conductance of 1.

Because the solution is a sum of exponentials, the
coupled equations can be solved as a pair of independent equations
by the more efficient cnexp method.

ENDCOMMENT


TITLE GABAergic conductance with changing Cl- concentration

NEURON {
	POINT_PROCESS gaba

	USEION cl READ ecl WRITE icl VALENCE -1

	NONSPECIFIC_CURRENT ihco3
	RANGE tau1, tau2, g 
	RANGE P, HCO3e, HCO3i, i

	RANGE icl, ihco3, ehco3, e
	GLOBAL total
}

UNITS {	
	(mA)    = (milliamp)
	(nA)    = (nanoamp)
	(mV)    = (millivolt)
	(uS)  = (micromho)
	(mM)    = (milli/liter)
	F = (faraday) (coulombs)
	R = (k-mole)  (joule/degC)
}

PARAMETER {
	tau1	=.1	(ms)	<1e-9,1e9>
	tau2	= 10	(ms)	<1e-9,1e9>

	HCO3e   = 26	(mM)	: extracellular HCO3- concentration
	HCO3i   = 16	(mM)	: intracellular HCO3- concentration			
	P       = 0.18		: HCO3/Cl relative permeability

	celsius = 37    (degC)
}

ASSIGNED {
	v	(mV)		: postsynaptic voltage 

	icl	(nA)		: chloride current = (1-P)*g*(v - ecl)
	ihco3	(nA)		: bicarb current = P*g*(v - ehco3)
	i	(nA)		: total current generated by this mechanism 
				: = icl + ihco3
	g 	(uS)		: total conductance, split between bicarb (P*g)
				: and chloride ((1-P)*g)
	factor
	total	(uS)

	ecl	(mV)		: equilibrium potential for Cl-
	ehco3	(mV)		: equilibrium potential for HCO3-
		
	e	(mV)		: reversal potential for GABAR			
}

STATE {
	A (uS)
	B (uS)
}

INITIAL { 
	LOCAL tp
	total = 0
	if (tau1/tau2 > .9999) {
		tau1 = .9999*tau2
	}
	A = 0
	B = 0
	tp = (tau1*tau2)/(tau2 - tau1) * log(tau2/tau1)
	factor = -exp(-tp/tau1) + exp(-tp/tau2)
	factor = 1/factor

	ehco3 = log(HCO3i/HCO3e)*(1000)*(celsius + 273.15)*R/F
	e = P*ehco3 + (1-P)*ecl
}

BREAKPOINT {
	SOLVE state METHOD cnexp

	g = B - A 

	icl = (1-P)*g*(v-ecl)

	ihco3 = P*g*(v-ehco3)
	i = icl + ihco3
	e = P*ehco3 + (1-P)*ecl

}

DERIVATIVE state {
	A' = -A/tau1
	B' = -B/tau2
}

NET_RECEIVE(weight (uS)) {
	A = A + weight*factor
	B = B + weight*factor
	total = total+weight
}
