/*
  © Copyright 2008 Randal A. Koene <randalk@netmorph.org>
  
  With design assistance from J. van Pelt & A. van Ooyen, and support
  from the Netherlands Organization for Scientific Research (NWO)
  Program Computational Life Sciences grant CLS2003 (635.100.005) and
  from the EC Marie Curie Research and Training Network (RTN)
  NEURoVERS-it 019247.

  This file is part of NETMORPH.

  NETMORPH is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  NETMORPH is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with NETMORPH.  If not, see <http://www.gnu.org/licenses/>.
*/
// Catacomb_Object.hh
// Randal A. Koene, 20080316
//
// Classes that define Catacomb_Object and objects that inherit it.

// This format was developed for DIL#20080227165942.1.

/*
  The present implementation of Catacomb output is the most desirable at this time, because it relates directly
  to the level at which Catacomb modeling has traditionally taken place, namely at the level of networks of IF
  neurons. An alternative is to go through the structure generated by NETMORPH and produce individual cells and
  cables, distributed in a 2D transformation of the 3D generated network.
 */

#ifndef __Catacomb_OBJECT_HH
#define __Catacomb_OBJECT_HH

#include <limits.h>
#include "BigString.hh"
#include "templates.hh"

// definitions

#ifndef __NETWORK_HH
class regionslist;
#endif

// functions

// classes

typedef int * ccm_intptr; // [***NOTE] We can use a multi-parameter data structure instead.

class Catacomb_connectiontable {
  // This is not a Catacomb output object.
  // It is a simple matrix object that uses arrays.
  // The matrix contains connection parameter data
  // between pre- and postsynaptic neurons.
protected:
  ccm_intptr * ccmtable;
  unsigned int dim_in, dim_out;
public:
  Catacomb_connectiontable(unsigned int _dim_in, unsigned int _dim_out): dim_in(_dim_in), dim_out(_dim_out) {
    ccmtable = new ccm_intptr[dim_in];
    for (unsigned int i=0; i<dim_in; i++) {
      ccmtable[i] = new int[dim_out];
      for (unsigned int j=0; j<dim_out; j++) ccmtable[i][j] = 0;
    }
  }
  ~Catacomb_connectiontable() {
    for (unsigned int i=0; i<dim_in; i++) delete[] ccmtable[i];
    delete[] ccmtable;
  }
  int Dim_in() { return dim_in; }
  int Dim_out() { return dim_out; }
  int & el(unsigned int i, unsigned int j) { if ((i>=dim_in) || (j>=dim_out)) error("Error: i>=dim_in or j>=dim_out in Catacomb_connectiontable::el()\n"); return ccmtable[i][j]; }
};

typedef Catacomb_connectiontable * ccm_tableptr; // pointer to one table
typedef ccm_tableptr * ccm_tablevec; // pointer to or array of pointers to table
typedef ccm_tablevec * ccm_tablemat; // array of arrays (matrix) of pointers to table

class Catacomb_projectionsmatrix {
  // This is not a Catacomb output object.
  // It is a simple matrix object that uses arrays.
  // The matrix contains pointers, initialized to NULL
  // that can point to Catacomb_connectiontable objects.
protected:
  ccm_tablemat projections;
  unsigned int dim;
public:
  Catacomb_projectionsmatrix(int _dim): dim(_dim) {
    if (dim<1) error("Error: dim<1 in Catacomb_projectionsmatrix constructor.\n");
    projections = new ccm_tablevec[dim];
    for (unsigned int i=0; i<dim; i++) {
      projections[i] = new ccm_tableptr[dim];
      for (unsigned int j=0; j<dim; j++) projections[i][j] = NULL;
    }
  }
  ~Catacomb_projectionsmatrix() {
    for (unsigned int i=0; i<dim; i++) delete[] projections[i];
    delete[] projections;
  }
  // first dereference [i] returns ccm_tablevec, second dereference [j] returns ccm_tableptr
  // function returns a reference to the ccm_tableptr variable location
  ccm_tableptr & el(unsigned int i, unsigned int j) { if ((i>=dim) || (j>=dim)) error("Error: i>=dim_in or j>=dim_out in Catacomb_projectionsmatrix::el()\n"); return projections[i][j]; }
  //ccm_tableptr get_el(int i, int j) { return projections[i][j]; }
  //ccm_tableptr set_el(int i, int j, ccm_tableptr ctp) { projections[i][j] = ctp; return ctp; }
  int Dim() { return dim; }
};

class Catacomb_Object: public PLLHandle<Catacomb_Object> {
public:
  Catacomb_Object() {}
  virtual String str() { return String(""); }
};

class Catacomb_Header: public Catacomb_Object {
protected:
  String text;
public:
  Catacomb_Header();
  virtual String str() { return text; }
};

class Catacomb_Neuron: public Catacomb_Object {
  //protected:
public:
  Catacomb_Neuron() {}
  virtual String str();
};

class Catacomb_Region: public Catacomb_Object {
protected:
  String name;
  int n;
  double x, y;
public:
  Catacomb_Region(String _name, int _n, double _x, double _y): name(_name), n(_n), x(_x), y(_y) {}
  virtual String str();
};

class Catacomb_Projection: public Catacomb_Object {
  // In this version, this object searches for all projections
  // in a projections matrix, and produces corresponding output
  // in str().
protected:
  Catacomb_projectionsmatrix * ccmprojections;
  regionslist * regions;
public:
  Catacomb_Projection(Catacomb_projectionsmatrix & cmp, regionslist & _regions): ccmprojections(&cmp), regions(&_regions) {}
  virtual String str();
};

class Catacomb_ConnectionTable: public Catacomb_Projection {
  // In this version, this object searches for all connections
  // in a projections matrix, and produces corresponding output
  // in str().
public:
  Catacomb_ConnectionTable(Catacomb_projectionsmatrix & cmp, regionslist & _regions): Catacomb_Projection(cmp,_regions) {}
  virtual String str();
};

class Catacomb_Group: public Catacomb_Object {
protected:
  String comment;
  PLLRoot<Catacomb_Object> ccmobjects;
public:
  Catacomb_Group(String c): comment(c) {}
  Catacomb_Group() {}
  virtual String str();
  void Add_Catacomb_Object(Catacomb_Object * ccmobject);
};

extern Catacomb_Group * ccmregionsgroup;
extern Catacomb_Group * ccmprojectionsgroup;
extern Catacomb_Group * ccmconnectionsgroup;
extern long Catacomb_neuronindex;
extern long Catacomb_connectionindex;

#endif
