/* vdep_ligdep_chan.c */

/* Avrama Blackwell, GMU, October 1997
   modified from one-state-recept0.c
*/

/* This function computes voltage-dependent and ligand-dependent 
   (e.g. Calcium dependent) channel currents given the parameters 
   specified as inf (steady state value) and tau (time const).
   This function was designed initially to compute the voltage and calcium 
   dependent potassium channel of Hermissenda, modified from
   Sakakibara et al. 1993, Biophysical J. 
*/

#include "chan_ext.h"

#define	Field(F) (channel->F)
#define sep 0
#define nonsep 1
#define ln10 log(10)

vdep_ligdep_chan (channel,action)

register struct vdep_ligdep_chan_type *channel;
Action		*action;

{
double 	v;			/* Voltage from compartment */
double  ligand;			/* ligand concentration */
double 	dt;			/* time step */
double	A, B;			/* parameters for IntegrateMethod */
float one, zero;        /* dummy rate_constant parameters */
MsgIn	*msg;
int     n;                      /* for counting messages */
double savedata[2];

  if (debug > 1) {
     ActionHeader("vdep_ligdep_chan", channel,action);
  }
  one = 1.0;
  zero = 0.0;

  SELECT_ACTION(action){ 
    case INIT:
      channel->activation = 0;
      break;
    case PROCESS:
    
 /* Get the voltage and ligand concentration */	
	v = 0;
	ligand = 0;
	
	MSGLOOP(channel,msg)
	{
	case 0:	/* 0 = membrane potential */	
		v = MSGVALUE(msg,0);
		Field(Vm) = v; 
		break;
	
	case 1:
		ligand = MSGVALUE(msg,0);
		Field(ligand) = ligand;
		break;
	}

    	dt = Clockrate(channel);

/* activation gate first */

      if(Field(act_power) != 0){
	if (channel->act_ss_type > sep) {

	  /* non-separable, calculate vdep constants as a function of ligand */
	  cadep_rate_const(channel, ligand, channel->act_ss_type,
			   &channel->act_ssv.min, 
			   &channel->act_ssv.max, 
			   &channel->act_ssv.v0, 
			   &channel->act_ssv.slope,
			   channel->act_ssv.min_slp,
			   channel->act_ssv.min_int,
			   channel->act_ssv.max_slp,
			   channel->act_ssv.max_int,
			   channel->act_ssv.v0_slp,
			   channel->act_ssv.v0_int,
			   channel->act_ssv.slope_slp,
			   channel->act_ssv.slope_int);
	if (debug > 0)
	  printf (" vdep_ligdep: act_ssv: min, max, slp, v0 = %f %f %f %f \n",
		  channel->act_ssv.min, channel->act_ssv.max,
		  channel->act_ssv.slope, channel->act_ssv.v0);
	}
	if (channel->act_tau_type > sep) {
	  cadep_rate_const(channel, ligand, channel->act_tau_type, 
			   &channel->act_tauv.min, 
			   &channel->act_tauv.max, 
			   &channel->act_tauv.v0, 
			   &channel->act_tauv.slope, 
			   channel->act_tauv.min_slp,
			   channel->act_tauv.min_int,
			   channel->act_tauv.max_slp,
			   channel->act_tauv.max_int,
			   channel->act_tauv.v0_slp,
			   channel->act_tauv.v0_int,
			   channel->act_tauv.slope_slp,
			   channel->act_tauv.slope_int);
	if (debug > 0)
	  printf (" vdep_ligdep: act_tauv: min, max, slp, v0 = %f %f %f %f \n",
		  channel->act_tauv.min, channel->act_tauv.max,
		  channel->act_tauv.slope, channel->act_tauv.v0);
	}

	/* voltage dependent rate constants */
	 vdep_rate_const (&channel->act_ssv.rate_const, v, 
		channel->act_ssv.min,
		channel->act_ssv.max,
		channel->act_ssv.v0,
		channel->act_ssv.slope,
		one, channel->act_ssv.power,
		zero, channel->act_ssv.offset);
		
	vdep_rate_const (&channel->act_tauv.rate_const, v, 
		channel->act_tauv.min,
		channel->act_tauv.max,
		channel->act_tauv.v0,
		channel->act_tauv.slope,
		one, channel->act_tauv.power,
		zero, channel->act_tauv.offset);

	if (debug > 0)
	  printf (" vdep_ligdep: act_ssv.rate_const = %f act_tauv.rate_const = %f \n", channel->act_ssv.rate_const, channel->act_tauv.rate_const);

	if (channel->act_ss_type == sep) {
	  /* ligand dependent rate constants */
	 vdep_rate_const (&channel->act_ssca.rate_const, log(ligand)/ln10, 
		channel->act_ssca.min,
		channel->act_ssca.max,
		channel->act_ssca.v0,
		channel->act_ssca.slope,
		channel->act_ssca.in_exp_power,
		channel->act_ssca.out_exp_power,
		channel->act_ssca.in_exp_offset,
		channel->act_ssca.out_exp_offset);

	if (debug > 0)
	  printf (" vdep_ligdep: act_ssca.rate_const = %f \n", channel->act_ssca.rate_const);

	  /* rate constants = product of vdep and ligdep */
	  channel->act_ssv.rate_const = 
	    channel->act_ssv.rate_const * channel->act_ssca.rate_const;
	}

	if (channel->act_tau_type == sep) {	
	vdep_rate_const (&channel->act_tauca.rate_const, log(ligand)/ln10, 
		channel->act_tauca.min,
		channel->act_tauca.max,
		channel->act_tauca.v0,
		channel->act_tauca.slope,
		channel->act_tauca.in_exp_power,
		channel->act_tauca.out_exp_power,
		channel->act_tauca.in_exp_offset,
		channel->act_tauca.out_exp_offset);
	
	if (debug > 0)
	  printf (" vdep_ligdep:act_tauca.rate_const = %f \n", channel->act_tauca.rate_const);

	  channel->act_tauv.rate_const = 
	    channel->act_tauv.rate_const * channel->act_tauca.rate_const;
	}

/* Evaluate the derivative using IntegrateMethod, which computes
 *	output = state * exp(-B*dt) + (A/B) * (1 - exp(-B*dt))
 *
 * the explicit integration formula using ss and tau is
 *	output = state * exp(-tau*dt) + inf * (1 - exp(-tau*dt))
 *
 * thus, B = 1 / tau and inf = (A/B), i.e. A = B*inf =  inf / tau
*/

	A = channel->act_ssv.rate_const / channel->act_tauv.rate_const;
	B = 1 / channel->act_tauv.rate_const;
	Field(act) = 
		IntegrateMethod(channel->object->method,channel,Field(act),A,B,dt,"act");
   }

/* inactivation gate next */

      if(Field(inact_power) != 0){
	if (channel->inact_ss_type > sep) {

	  /* non-separable, calculate vdep constants as a function of ligand */
	  cadep_rate_const(channel, ligand, channel->inact_ss_type,
			   &channel->inact_ssv.min, 
			   &channel->inact_ssv.max, 
			   &channel->inact_ssv.v0, 
			   &channel->inact_ssv.slope,
			   channel->inact_ssv.min_slp,
			   channel->inact_ssv.min_int,
			   channel->inact_ssv.max_slp,
			   channel->inact_ssv.max_int,
			   channel->inact_ssv.v0_slp,
			   channel->inact_ssv.v0_int,
			   channel->inact_ssv.slope_slp,
			   channel->inact_ssv.slope_int);
	if (debug > 0)
	  printf (" vdep_ligdep: inact_ssv: min, max, slp, v0 = %f %f %f %f \n",
		  channel->inact_ssv.min, channel->inact_ssv.max,
		  channel->inact_ssv.slope, channel->inact_ssv.v0);
	}

	if (channel->inact_tau_type > sep) {
	  cadep_rate_const(channel, ligand, channel->inact_tau_type, 
			   &channel->inact_tauv.min, 
			   &channel->inact_tauv.max, 
			   &channel->inact_tauv.v0, 
			   &channel->inact_tauv.slope, 
			   channel->inact_tauv.min_slp,
			   channel->inact_tauv.min_int,
			   channel->inact_tauv.max_slp,
			   channel->inact_tauv.max_int,
			   channel->inact_tauv.v0_slp,
			   channel->inact_tauv.v0_int,
			   channel->inact_tauv.slope_slp,
			   channel->inact_tauv.slope_int);
	if (debug > 0)
	  printf (" vdep_ligdep: inact_tauv: min, max, slp, v0 = %f %f %f %f \n",
		  channel->inact_tauv.min, channel->inact_tauv.max,
		  channel->inact_tauv.slope, channel->inact_tauv.v0);
	}

	/* voltage dependent rate constants */
	 vdep_rate_const (&channel->inact_ssv.rate_const, v, 
		channel->inact_ssv.min,
		channel->inact_ssv.max,
		channel->inact_ssv.v0,
		channel->inact_ssv.slope,
		one, channel->inact_ssv.power,
		zero, channel->inact_ssv.offset);
		
	vdep_rate_const (&channel->inact_tauv.rate_const, v, 
		channel->inact_tauv.min,
		channel->inact_tauv.max,
		channel->inact_tauv.v0,
		channel->inact_tauv.slope,
		one, channel->inact_tauv.power,
		zero, channel->inact_tauv.offset);

	if (debug > 0)
	  printf (" vdep_ligdep: inact_ssv.rate_const = %f inact_tauv.rate_const = %f \n", channel->inact_ssv.rate_const, channel->inact_tauv.rate_const);

	if (channel->inact_ss_type == sep) {
	  /* ligand dependent rate constants */
	 vdep_rate_const (&channel->inact_ssca.rate_const, log(ligand)/ln10, 
		channel->inact_ssca.min,
		channel->inact_ssca.max,
		channel->inact_ssca.v0,
		channel->inact_ssca.slope,
		channel->inact_ssca.in_exp_power,
		channel->inact_ssca.out_exp_power,
		channel->inact_ssca.in_exp_offset,
		channel->inact_ssca.out_exp_offset);
		
	if (debug > 0)
	  printf (" vdep_ligdep: inact_ssca.rate_const = %f \n", channel->inact_ssca.rate_const);

	  /* rate constants = product of vdep and ligdep */
	  channel->inact_ssv.rate_const = 
	    channel->inact_ssv.rate_const * channel->inact_ssca.rate_const;
	}

	if (channel->inact_tau_type == sep) {	
	vdep_rate_const (&channel->inact_tauca.rate_const, log(ligand)/ln10, 
		channel->inact_tauca.min,
		channel->inact_tauca.max,
		channel->inact_tauca.v0,
		channel->inact_tauca.slope,
		channel->inact_tauca.in_exp_power,
		channel->inact_tauca.out_exp_power,
		channel->inact_tauca.in_exp_offset,
		channel->inact_tauca.out_exp_offset);

	if (debug > 0)
	  printf (" vdep_ligdep: inact_tauca.rate_const = %f \n", channel->inact_tauca.rate_const);

	  /* rate constants = product of vdep and ligdep */
	  channel->inact_tauv.rate_const = 
	    channel->inact_tauv.rate_const * channel->inact_tauca.rate_const;
	}

	A = channel->inact_ssv.rate_const / channel->inact_tauv.rate_const;
	B = 1 / channel->inact_tauv.rate_const;
	Field(inact) = 
		IntegrateMethod(channel->object->method,channel,Field(inact),A,B,dt,"inact");
      }

/* calculate the conductance and current */

	if(Field(act_power) != 0 && Field(inact_power) != 0){
	    Field(G) = Field(Gbar) * pow(Field(act),Field(act_power)) * 
	    		pow(Field(inact),Field(inact_power));
	} else 
	if(Field(inact_power) == 0) {
	    Field(G) = Field(Gbar) * pow(Field(act),Field(act_power));
	} else {
	    Field(G) = Field(Gbar);
        }

/* Calculate the conductance and current */

	channel->I = (v - channel->Vr) * channel->G;

	break;
    
    case RESET:

 /* Get the current voltage and transmitter concentration */	
	v = 0;
	ligand = 0;
	
	MSGLOOP(channel,msg)
	{
	case 0:		
		v = MSGVALUE(msg,0);
		Field(Vm) = v; 
		break;
	
	case 1:
		ligand = MSGVALUE(msg,0);
		Field(ligand) = ligand;
		break;
	}

/* activation gate first */

      if(Field(act_power) != 0){
	if (channel->act_ss_type > sep) {

	  /* non-separable, calculate vdep constants as a function of ligand */
	  cadep_rate_const(channel, ligand, channel->act_ss_type,
			   &channel->act_ssv.min, 
			   &channel->act_ssv.max, 
			   &channel->act_ssv.v0, 
			   &channel->act_ssv.slope,
			   channel->act_ssv.min_slp,
			   channel->act_ssv.min_int,
			   channel->act_ssv.max_slp,
			   channel->act_ssv.max_int,
			   channel->act_ssv.v0_slp,
			   channel->act_ssv.v0_int,
			   channel->act_ssv.slope_slp,
			   channel->act_ssv.slope_int);
	if (debug > 0)
	  printf (" vdep_ligdep: act_ssv: min, max, slp, v0 = %f %f %f %f \n",
		  channel->act_ssv.min, channel->act_ssv.max,
		  channel->act_ssv.slope, channel->act_ssv.v0);
	}

	/* voltage dependent rate constants */
	 vdep_rate_const (&channel->act_ssv.rate_const, v, 
		channel->act_ssv.min,
		channel->act_ssv.max,
		channel->act_ssv.v0,
		channel->act_ssv.slope,
		one, channel->act_ssv.power,
		zero, channel->act_ssv.offset);
		
	if (debug > 0)
	  printf (" vdep_ligdep: act_ssv.rate_const = %f \n", channel->act_ssv.rate_const);

	if (channel->act_ss_type == sep) {
	  /* ligand dependent rate constants */
	 vdep_rate_const (&channel->act_ssca.rate_const, log(ligand)/ln10, 
		channel->act_ssca.min,
		channel->act_ssca.max,
		channel->act_ssca.v0,
		channel->act_ssca.slope,
		channel->act_ssca.in_exp_power,
		channel->act_ssca.out_exp_power,
		channel->act_ssca.in_exp_offset,
		channel->act_ssca.out_exp_offset);
		
	if (debug > 0)
	  printf (" vdep_ligdep: act_ssca.rate_const = %f \n", channel->act_ssca.rate_const);

	  /* rate constants = product of vdep and ligdep */
	  channel->act_ssv.rate_const = 
	    channel->act_ssv.rate_const * channel->act_ssca.rate_const;
	}

	/* activation = steady state value (no integration with Reset) */
	channel->act = channel->act_ssv.rate_const;
      }

/* inactivation gate next */

      if(Field(inact_power) != 0){
	if (channel->inact_ss_type > sep) {

	  /* non-separable, calculate vdep constants as a function of ligand */
	  cadep_rate_const(channel, ligand, channel->inact_ss_type, 
			   &channel->inact_ssv.min, 
			   &channel->inact_ssv.max, 
			   &channel->inact_ssv.v0, 
			   &channel->inact_ssv.slope,
			   channel->inact_ssv.min_slp,
			   channel->inact_ssv.min_int,
			   channel->inact_ssv.max_slp,
			   channel->inact_ssv.max_int,
			   channel->inact_ssv.v0_slp,
			   channel->inact_ssv.v0_int,
			   channel->inact_ssv.slope_slp,
			   channel->inact_ssv.slope_int);
	if (debug > 0)
	  printf (" vdep_ligdep: inact_ssv: min, max, slp, v0 = %f %f %f %f \n",
		  channel->inact_ssv.min, channel->inact_ssv.max,
		  channel->inact_ssv.slope, channel->inact_ssv.v0);
	}

	/* voltage dependent rate constants */
	 vdep_rate_const (&channel->inact_ssv.rate_const, v, 
		channel->inact_ssv.min,
		channel->inact_ssv.max,
		channel->inact_ssv.v0,
		channel->inact_ssv.slope,
		one, channel->inact_ssv.power,
		zero, channel->inact_ssv.offset);

	if (debug > 0)
	  printf (" vdep_ligdep: inact_ssv.rate_const = %f \n", channel->inact_ssv.rate_const);
		
	if (channel->inact_ss_type == sep) {
	  /* ligand dependent rate constants */
	 vdep_rate_const (&channel->inact_ssca.rate_const, log(ligand)/ln10, 
		channel->inact_ssca.min,
		channel->inact_ssca.max,
		channel->inact_ssca.v0,
		channel->inact_ssca.slope,
		channel->inact_ssca.in_exp_power,
		channel->inact_ssca.out_exp_power,
		channel->inact_ssca.in_exp_offset,
		channel->inact_ssca.out_exp_offset);		

	if (debug > 0)
	  printf (" vdep_ligdep: inact_ssca.rate_const = %f \n", channel->inact_ssca.rate_const);

	  /* rate constants = product of vdep and ligdep */
	  channel->inact_ssv.rate_const = 
	    channel->inact_ssv.rate_const * channel->inact_ssca.rate_const;
	}

	/* inactivation = steady state value (no integration with Reset) */
	channel->inact = channel->inact_ssv.rate_const;
      }

/* Calculate the conductance and current */

	if(Field(act_power) != 0 && Field(inact_power) != 0){
	    Field(G) = Field(Gbar) * pow(Field(act),Field(act_power)) * 
	    		pow(Field(inact),Field(inact_power));
	} else 
	if(Field(inact_power) == 0) {
	    Field(G) = Field(Gbar) * pow(Field(act),Field(act_power));
	} else {
	    Field(G) = Field(Gbar);
	}
	channel->I = (v - channel->Vr) * channel->G;
	break;

    case CHECK:
      n=0;
      MSGLOOP(channel,msg){
        case 0:
	  n +=1;
	  break;
        case 1:
	  n+=1;
	  break;
      }
      if(n != 2) {
	    ErrorMessage(" vdep_ligdep_chan","Missing v or ligand msg.",
			      channel);
       }
       break;
    case SAVE2:
        savedata[0] = channel->act;
        savedata[1] = channel->inact;
        /* action->data contains the file pointer */
        n=2;
        fwrite(&n,sizeof(int),1,(FILE*)action->data);
        fwrite(savedata,sizeof(double),2,(FILE*)action->data);
        break;
    case RESTORE2:
        /* action->data contains the file pointer */
        fread(&n,sizeof(int),1,(FILE*)action->data);
        if (n != 2) {
            ErrorMessage("vdep_ligdep_chan.c", "Invalid savedata length", channel);
            return n;
        }
        fread(savedata,sizeof(double),2,(FILE*)action->data);
        channel->act = savedata[0];
        channel->inact = savedata[1];
        break;

    }
}


