/* ligand1_chan.c */

/* Avrama Blackwell, GMU, October 1997
   modified from one-state-recept0.c
*/

/* This function computes channel conductance of a one state kinetic
   channel which is ligand-gated (and possibly voltage-dependent) according to:
   
   [R] + n[G] <-> [R-G^n]
   
   where n is the number of molecules of ligand G that binds to the 
   receptor.  alpha and beta are forward and backward rate constant.

   This function was designed initially to compute ionotropic GABA induced 
   Potassium currents in the Hermissenda B cell, as measured in 
   Alkon et al., 1992 PNAS.
   Assumptions: Channel protein for activation is ligand-gated
                Channel protein for inactivation (optional) is not ligand-gated
     This assumption is easily eliminated by making inact gate ligand-gated and allowing rxn_ord = 0, but then rate consts must be specified as alpha and beta 
     *** Modified Aug 4, 1998, allow for ligand-gated inact gate ***
*/   
#include "chan_ext.h"

#define	Field(F) (channel->F)
#define ACT 0
#define INACT 1

ligand1_chan (channel,action)

register struct ligand1_chan_type *channel;
Action		*action;

{
double 	v;			/* Voltage from compartment */
double  ligand;			/* ligand concentration */
double 	dt;			/* time step */
double	A, B;			/* parameters for IntegrateMethod */
MsgIn	*msg;
double savedata[2];
int     n,i;                    /* for counting messages */
int	which;			/* which rate constant is modulated */

  if (debug > 1) {
     ActionHeader("ligand1_chan", channel,action);
  }
    SELECT_ACTION(action)
    {
    case INIT:
	channel->activation = 0;
	for (i=0; i<6; i++)
		channel->modulator[i]=1;
	break;
    
    case PROCESS:
    
 /* Get the current voltage and ligand concentration, and modulation */	
	v = 0;
	ligand = 0;
	
	MSGLOOP(channel,msg)
	{
	case 0:	/* 0 = membrane potential */	
		v = MSGVALUE(msg,0);
		Field(Vm) = v; 
		break;
	
	case 1:
		ligand = MSGVALUE(msg,0);
		Field(ligand) = ligand;
		break;

        case 2:
                which = MSGVALUE(msg,1);
		channel->modulator[which]=MSGVALUE(msg,0);
		break;
	}

    	dt = Clockrate(channel);

/* calculate voltage dependent parts of alpha and beta from v	
   For voltage independence, make min - desired value and max = 0*/

      if(Field(act_power) != 0){
	vdep_rate_const (&channel->alpha.rate_const, v, 
		channel->alpha.min,
		channel->alpha.max,
		channel->alpha.v0,
		channel->alpha.slope,
		channel->alpha.in_exp_power,
		channel->alpha.out_exp_power,
		channel->alpha.in_exp_offset,
		channel->alpha.out_exp_offset);
		
	vdep_rate_const (&channel->beta.rate_const, v, 
		channel->beta.min,
		channel->beta.max,
		channel->beta.v0,
		channel->beta.slope,
		channel->beta.in_exp_power,
		channel->beta.out_exp_power,
		channel->beta.in_exp_offset,
		channel->beta.out_exp_offset);

	channel->alpha.rate_const*=channel->modulator[0];
	channel->beta.rate_const*=channel->modulator[1];

  	if (debug > 0) 
   	printf (" ligand1_chan: alpha.rate_const=%f beta=%f \n",
   		channel->alpha.rate_const, channel->beta.rate_const);
   		
/* Compute inf and tau from vdep rate constants and ligand concentration */

	lig_rate_const (channel, 
			&channel->act_inf,
			&channel->act_tau,
			channel->alpha.rate_const,
			channel->beta.rate_const,
			channel->ligand,
			channel->rxn_ord, ACT);

/* Evaluate the derivative using IntegrateMethod, which computes
 *	output = state * exp(-B*dt) + (A/B) * (1 - exp(-B*dt))
 *
 * act_tau computed as alpha + beta => it's really 1/tau
 *
 * the explicit integration formula using ss and tau is
 *	output = state * exp(-tau*dt) + inf * (1 - exp(-tau*dt))
 *
 * thus, B = tau and inf = (A/B), i.e. A = B*inf = tau * inf 
*/

	A = channel->act_inf * channel->act_tau;
	B = channel->act_tau;
	Field(act) = 
		IntegrateMethod(channel->object->method,channel,Field(act),A,B,dt,"act");

	if (debug > 0)
	   printf (" ligand1_chan: A=%f B=%f act=%f\n", 
		A, B, Field(act));
		
      }

      if((Field(inact_power) != 0) && (Field(inact_type == 0))){
	 vdep_rate_const (&channel->inact_ss.rate_const, v, 
		channel->inact_ss.min,
		channel->inact_ss.max,
		channel->inact_ss.v0,
		channel->inact_ss.slope,
		channel->inact_ss.in_exp_power,
		channel->inact_ss.out_exp_power,
		channel->inact_ss.in_exp_offset,
		channel->inact_ss.out_exp_offset);
		
	vdep_rate_const (&channel->inact_tau.rate_const, v, 
		channel->inact_tau.min,
		channel->inact_tau.max,
		channel->inact_tau.v0,
		channel->inact_tau.slope,
		channel->inact_tau.in_exp_power,
		channel->inact_tau.out_exp_power,
		channel->inact_tau.in_exp_offset,
		channel->inact_tau.out_exp_offset);

	A = channel->inact_ss.rate_const / channel->inact_tau.rate_const;
	B = 1 / channel->inact_tau.rate_const;
	Field(inact) = 
		IntegrateMethod(channel->object->method,channel,Field(inact),A,B,dt,"inact");
        }

      if((Field(inact_power) != 0) && (Field(inact_type == 1))){
	vdep_rate_const (&channel->gamma.rate_const, v, 
		channel->gamma.min,
		channel->gamma.max,
		channel->gamma.v0,
		channel->gamma.slope,
		channel->gamma.in_exp_power,
		channel->gamma.out_exp_power,
		channel->gamma.in_exp_offset,
		channel->gamma.out_exp_offset);
		
	vdep_rate_const (&channel->delta.rate_const, v, 
		channel->delta.min,
		channel->delta.max,
		channel->delta.v0,
		channel->delta.slope,
		channel->delta.in_exp_power,
		channel->delta.out_exp_power,
		channel->delta.in_exp_offset,
		channel->delta.out_exp_offset);

	channel->gamma.rate_const*=channel->modulator[2];
	channel->delta.rate_const*=channel->modulator[3];

  	if (debug > 0) 
   	printf (" ligand1_chan: gamma.rate_const=%f delta=%f \n",
   		channel->gamma.rate_const, channel->delta.rate_const);
   		
/* Compute inf and tau from vdep rate constants and ligand concentration */

	lig_rate_const (channel, 
			&channel->inact_inf,
			&channel->inact_rate,
			channel->gamma.rate_const,
			channel->delta.rate_const,
			channel->ligand,
			channel->inact_rxn_ord, INACT);

	A = channel->inact_inf * channel->inact_rate;
	B = channel->inact_rate;
	Field(inact) = 
		IntegrateMethod(channel->object->method,channel,Field(inact),A,B,dt,"inact");
        }

	if (debug > 0)
	   printf (" ligand1_chan: A=%f B=%f inact=%f\n", 
		A, B, Field(inact));

/* calculate the conductance and current */

	if(Field(act_power) != 0 && Field(inact_power) != 0)
	  {
	    Field(G) = Field(Gbar) * pow(Field(act),Field(act_power)) * 
	    		pow(Field(inact),Field(inact_power));
	  }  
	if(Field(inact_power) == 0 && Field(act_power) != 0)
	  {
	    Field(G) = Field(Gbar) * pow(Field(act),Field(act_power));
	  } 
	if(Field(act_power) == 0 && Field(inact_power) != 0) 
	  {
	    Field(G) = Field(Gbar) * pow(Field(inact),Field(inact_power));
	  }
	if(Field(act_power) == 0 && Field(inact_power) == 0)
	  {
	    Field(G) = Field(Gbar);
          }

/* Calculate the conductance and current */

	channel->I = (Field(Vm) - channel->Vr) * channel->G;

	if (debug > 0)
	   printf (" ligand1_chan: G=%f, I=%f act = %f inact= %f\n",
		channel->G, channel->I, channel->act, channel->inact);	

	break;
    
    case RESET:

 /* Get the current voltage and transmitter concentration */	
	v = 0;
	ligand = 0;
	for (i=0; i<6; i++)
		channel->modulator[i]=1;
	
	MSGLOOP(channel,msg)
	{
	case 0:		
		v = MSGVALUE(msg,0);
		Field(Vm) = v; 
		break;
	
	case 1:
		ligand = MSGVALUE(msg,0);
		Field(ligand) = ligand;
		break;

        case 2:
                which = MSGVALUE(msg,1);
		channel->modulator[which]=MSGVALUE(msg,0);
		break;
	}

/* calculate alpha and beta from v	*/

      if(Field(act_power) != 0){
	vdep_rate_const (&channel->alpha.rate_const, v, 
		channel->alpha.min,
		channel->alpha.max,
		channel->alpha.v0,
		channel->alpha.slope,
		channel->alpha.in_exp_power,
		channel->alpha.out_exp_power,
		channel->alpha.in_exp_offset,
		channel->alpha.out_exp_offset);
		
	vdep_rate_const (&channel->beta.rate_const, v, 
		channel->beta.min,
		channel->beta.max,
		channel->beta.v0,
		channel->beta.slope,
		channel->beta.in_exp_power,
		channel->beta.out_exp_power,
		channel->beta.in_exp_offset,
		channel->beta.out_exp_offset);

	channel->alpha.rate_const*=channel->modulator[0];
	channel->beta.rate_const*=channel->modulator[1];

/* Compute inf and tau from vdep rate constants and ligand concentration */

	lig_rate_const (channel, 
			&channel->act_inf,
			&channel->act_tau,
			channel->alpha.rate_const,
			channel->beta.rate_const,
			channel->ligand,
			channel->rxn_ord, ACT);
        Field(act) = channel->act_inf;
      }

      if((Field(inact_power) != 0) && (Field(inact_type == 0))){
	 vdep_rate_const (&channel->inact_ss.rate_const, v, 
		channel->inact_ss.min,
		channel->inact_ss.max,
		channel->inact_ss.v0,
		channel->inact_ss.slope,
		channel->inact_ss.in_exp_power,
		channel->inact_ss.out_exp_power,
		channel->inact_ss.in_exp_offset,
		channel->inact_ss.out_exp_offset);
	  channel->inact = channel->inact_ss.rate_const;
     }

      if((Field(inact_power) != 0) && (Field(inact_type == 1))){
	vdep_rate_const (&channel->gamma.rate_const, v, 
		channel->gamma.min,
		channel->gamma.max,
		channel->gamma.v0,
		channel->gamma.slope,
		channel->gamma.in_exp_power,
		channel->gamma.out_exp_power,
		channel->gamma.in_exp_offset,
		channel->gamma.out_exp_offset);
		
	vdep_rate_const (&channel->delta.rate_const, v, 
		channel->delta.min,
		channel->delta.max,
		channel->delta.v0,
		channel->delta.slope,
		channel->delta.in_exp_power,
		channel->delta.out_exp_power,
		channel->delta.in_exp_offset,
		channel->delta.out_exp_offset);

	channel->gamma.rate_const*=channel->modulator[2];
	channel->delta.rate_const*=channel->modulator[3];

  	if (debug > 0) 
   	printf (" ligand1_chan: gamma.rate_const=%f delta=%f \n",
   		channel->gamma.rate_const, channel->delta.rate_const);
   		
/* Compute inf and tau from vdep rate constants and ligand concentration */

	lig_rate_const (channel, 
			&channel->inact_inf,
			&channel->inact_rate,
			channel->gamma.rate_const,
			channel->delta.rate_const,
			channel->ligand,
			channel->inact_rxn_ord, INACT);

        Field(inact) = channel->inact_inf;
	}

/* Calculate the conductance and current */

	if(Field(act_power) != 0 && Field(inact_power) != 0)
	  {
	    Field(G) = Field(Gbar) * pow(Field(act),Field(act_power)) * 
	    		pow(Field(inact),Field(inact_power));
	  }  
	if(Field(inact_power) == 0 && Field(act_power) != 0) 
	  {
	    Field(G) = Field(Gbar) * pow(Field(act),Field(act_power));
	  } 
	if(Field(act_power) == 0 && Field(inact_power) != 0) 
	  {
	    Field(G) = Field(Gbar) * pow(Field(inact),Field(inact_power));
	  }
	if(Field(act_power) == 0 && Field(inact_power) == 0)  
	  {
	    Field(G) = Field(Gbar);
	  }

	channel->I = (Field(Vm) - channel->Vr) * channel->G;
	break;

    case CHECK:
      n=0;
      MSGLOOP(channel,msg){
        case 0:
	  n +=1;
	  break;
        case 1:
	  n+=1;
	  break;
      }
      if(n != 2)
		{
	   	ErrorMessage(" ligand1_chan","Missing v or ligand msg.",
			      channel);
		}
       break;

    case SAVE2:
        savedata[0] = channel->act;
        savedata[1] = channel->inact;
        /* action->data contains the file pointer */
        n=2;
        fwrite(&n,sizeof(int),1,(FILE*)action->data);
        fwrite(savedata,sizeof(double),2,(FILE*)action->data);
        break;
    case RESTORE2:
        /* action->data contains the file pointer */
        fread(&n,sizeof(int),1,(FILE*)action->data);
        if (n != 2) {
            ErrorMessage("ligand1_chan.c", "Invalid savedata length", channel);
            return n;
        }
        fread(savedata,sizeof(double),2,(FILE*)action->data);
        channel->act = savedata[0];
        channel->inact = savedata[1];
        break;
    }
}


