function SimPar = ConfigIDNet(N1,IN1,Nstripes,RunTime,I,s,p,RState,UniqueNum)
% sets all simulation parameters for IDNetSim.m, except for the (optional)
% connectivity matrix 
%
% INPUT:
% N1:           Number of simulated neurons
% M1:           Number of input neurons (prescribed spike trains)
% Nstripes:     Number of columns/stripes
% RunTime:      Simulation time in ms
% I:            Background input (1x14 vector, one entry for each neuron type)
% s:            Scaling factors for synaptic weights (1x4 vector)
% p:            Scaling factors for synaptic connectvity (1x4 vector)
% RState:       State for Random number generation, optional (pass for reproduction of random numbers)
% UniqueNum:    Unique number to differentiate several simulations running in parallel, optional

% OUTPUT:
% SimPar:       Simulation parameters (structure)


% ------------------------ Set meta data ----------------------------------
SimPar.name = 'PFC';
SimPar.comment = 'Data-based PFC-like circuit with layer 2/3 and 5, synaptic clustering, four types of interneurons and cell-type specific short-term synaptic plasticity; neuron parameters fitted to in vitro data, all other parameters constrained by experimental literature.';
SimPar.ver = '';                                % version number of the code to be used with this parameter set

% ------------------------ Set control parameters -------------------------
if ~exist('RunTime', 'var')
    RunTime = 1000;
end;
CtrPar = [0;                % starting time
          RunTime;          % stopping time
          0.05;             % time step
          2;                % # variables to save for viewlist neurons
          1]';              % record spike times?     
      
SimPar.fnOut = 'PFC_all';          % file name to save simulation details

% ---------------- Set random seed and random state, if needed ------------
if RandStream.getGlobalStream.Seed == 0
    RandStream.setGlobalStream(RandStream('mt19937ar','seed',sum(100*clock)));
end;

if exist('RState', 'var')
    mtstream = RandStream.getGlobalStream;
    mtstream.State = RState;
    RandStream.setGlobalStream(mtstream);
end;
SimPar.RStateInit = RandStream.getGlobalStream.State;

% ------------------------ Set network parameters -------------------------
% Used neuron types (repeated Nstripes times): 
% (1) L23-E  (2) L23-I-L  (3) L23-I-L-d  (4) L23-I-CL   (5) L23-I-CL-AC  (6) L23-I-CS   (7) L23-I-F 
% (8) L5-E   (9) L5-I-L  (10) L5-I-L-d  (11) L5-I-CL   (12) L5-I-CL-AC  (13) L5-I-CS   (14) L5-I-F 

NTypes = round(N1*[47 1.55 1.55 1.3 1.3 2.6 2.1 38 0.25 0.25 0.25 0.25 1.8 1.8]/100);            % number of neurons of each type

%NTypes = ones(1,14); NTypes(1,[1 3 10]) = 2;   % replace the above line by this one before using update_inv_con_PSP
NTypesN = length(NTypes);
N = sum(NTypes);
if (N~=N1)
    disp('Warning: Neurons can not be distributed evenly across cell types. Number of generated neurons differs from specified number.')
end;
InpNTypes = [IN1 IN1];

% ------------------ Set initial condition parameters ---------------------
% [NTypesN x #variables x 5] matrix
% parameters: [mean, std, min, max, distr_flag], see rand_par.m for details
V0Par = zeros(NTypesN,2,5);

V0Par(:,1,:) = repmat([-70 0   0   0 1],NTypesN,1); %[-65 0 -70 -60 1]
V0Par(:,2,:) = repmat([  0 0   0   0 1],NTypesN,1);

% ------------------------- Set neuron parameters -------------------------
CellPar = zeros(10,NTypesN);                                      % (mean) independent neuron parameters

CellPar(1,:)  = I;            % I:    external driving current
CellPar(2,:) = [ 3.0751 1.6902 1.6902 3.0014 3.0014 3.0751 3.3869 ...
		2.2513 1.6902 1.6902 3.0014 3.0014 2.2513 3.3869 ];
CellPar(3,:) = [ 1.9661 1.0353 1.0353 1.4581 1.4581 1.9661 1.0106 ...
		1.0196 1.0353 1.0353 1.4581 1.4581 1.0196 1.0106 ];
CellPar(4,:) = [ 3.5945 2.9528 2.9528 3.0991 3.0991 3.5945 3.8065 ...
		3.4415 2.9528 2.9528 3.0991 3.0991 3.4415 3.8065 ];
CellPar(5,:) = [ 1.0309 3.2163 3.2163 3.1517 3.1517 1.0309 3.0269 ...
		1.5178 3.2163 3.2163 3.1517 3.1517 1.5178 3.0269 ];
CellPar(6,:) = [ 3.1428 2.8230 2.8230 2.9335 2.9335 3.1428 2.3911 ...
		1.0702 2.8230 2.8230 2.9335 2.9335 1.0702 2.3911 ];
CellPar(7,:) = [ 4.4809 1.0542 1.0542 1.0730 1.0730 4.4809 4.1986 ...
		4.5650 1.0542 1.0542 1.0730 1.0730 4.5650 4.1986 ];
CellPar(8,:) = [ 1.0189 2.5959 2.5959 0.6931 0.6931 1.0189 0.8080 ...
		1.1154 2.5959 2.5959 0.6931 0.6931 1.1154 0.8080 ];
CellPar(9,:) = [ 5.0719 4.1321 4.1321 1.9059 1.9059 5.0719 3.0051 ...
		4.3414 4.1321 4.1321 1.9059 1.9059 4.3414 3.0051 ];
CellPar(10,:) = [ 2.9010 3.6925 3.6925 2.9462 2.9462 2.9010 3.0701 ...
		3.3302 3.6925 3.6925 2.9462 2.9462 3.3302 3.0701 ];

% Neuron parameter correlation matrices (one for each cell type)
ParCov{1}=[1.0000 0.1580 -0.5835 0.4011 -0.0561 0.0718 -0.2038 0.2615 -0.2365 
0.1580 1.0000 0.0141 -0.1272 -0.4327 0.1778 -0.0902 -0.0329 -0.3778 
-0.5835 0.0141 1.0000 -0.6295 -0.2949 -0.2008 0.3164 -0.2615 -0.0536 
0.4011 -0.1272 -0.6295 1.0000 0.6960 -0.2587 -0.0988 0.6113 0.5636 
-0.0561 -0.4327 -0.2949 0.6960 1.0000 -0.3370 0.2042 0.3959 0.8581 
0.0718 0.1778 -0.2008 -0.2587 -0.3370 1.0000 -0.0634 -0.5202 -0.3829 
-0.2038 -0.0902 0.3164 -0.0988 0.2042 -0.0634 1.0000 0.0559 0.3322 
0.2615 -0.0329 -0.2615 0.6113 0.3959 -0.5202 0.0559 1.0000 0.3210 
-0.2365 -0.3778 -0.0536 0.5636 0.8581 -0.3829 0.3322 0.3210 1.0000 ];

ParCov{2}=[1.0000 -0.2894 0.0381 0.0664 -0.2418 0.2253 0.2822 -0.2919 0.0581 
-0.2894 1.0000 -0.2259 0.4265 0.1859 -0.6307 -0.0140 0.4944 0.2495 
0.0381 -0.2259 1.0000 -0.2855 0.0724 0.1199 -0.1487 -0.3773 0.1881 
0.0664 0.4265 -0.2855 1.0000 0.2208 -0.3752 0.0660 0.3415 0.7289 
-0.2418 0.1859 0.0724 0.2208 1.0000 0.1412 -0.2931 0.1993 0.4609 
0.2253 -0.6307 0.1199 -0.3752 0.1412 1.0000 -0.2855 -0.2046 -0.1974 
0.2822 -0.0140 -0.1487 0.0660 -0.2931 -0.2855 1.0000 -0.1172 -0.0851 
-0.2919 0.4944 -0.3773 0.3415 0.1993 -0.2046 -0.1172 1.0000 0.0530 
0.0581 0.2495 0.1881 0.7289 0.4609 -0.1974 -0.0851 0.0530 1.0000 ];

ParCov{3}=[1.0000 -0.2894 0.0381 0.0664 -0.2418 0.2253 0.2822 -0.2919 0.0581 
-0.2894 1.0000 -0.2259 0.4265 0.1859 -0.6307 -0.0140 0.4944 0.2495 
0.0381 -0.2259 1.0000 -0.2855 0.0724 0.1199 -0.1487 -0.3773 0.1881 
0.0664 0.4265 -0.2855 1.0000 0.2208 -0.3752 0.0660 0.3415 0.7289 
-0.2418 0.1859 0.0724 0.2208 1.0000 0.1412 -0.2931 0.1993 0.4609 
0.2253 -0.6307 0.1199 -0.3752 0.1412 1.0000 -0.2855 -0.2046 -0.1974 
0.2822 -0.0140 -0.1487 0.0660 -0.2931 -0.2855 1.0000 -0.1172 -0.0851 
-0.2919 0.4944 -0.3773 0.3415 0.1993 -0.2046 -0.1172 1.0000 0.0530 
0.0581 0.2495 0.1881 0.7289 0.4609 -0.1974 -0.0851 0.0530 1.0000 ];

ParCov{4}=[1.0000 -0.2394 -0.6001 0.3114 -0.2367 0.5856 0.2077 0.0171 -0.4079 
-0.2394 1.0000 -0.1764 0.4675 0.1810 -0.4942 -0.4389 0.6950 0.0811 
-0.6001 -0.1764 1.0000 -0.6002 0.2170 -0.0922 0.2129 -0.3566 0.4204 
0.3114 0.4675 -0.6002 1.0000 0.2597 -0.1039 -0.5507 0.7230 0.0775 
-0.2367 0.1810 0.2170 0.2597 1.0000 0.2159 -0.7123 0.0193 0.8494 
0.5856 -0.4942 -0.0922 -0.1039 0.2159 1.0000 0.0587 -0.4724 0.0957 
0.2077 -0.4389 0.2129 -0.5507 -0.7123 0.0587 1.0000 -0.3395 -0.5780 
0.0171 0.6950 -0.3566 0.7230 0.0193 -0.4724 -0.3395 1.0000 -0.1084 
-0.4079 0.0811 0.4204 0.0775 0.8494 0.0957 -0.5780 -0.1084 1.0000 ];

ParCov{5}=[1.0000 -0.2394 -0.6001 0.3114 -0.2367 0.5856 0.2077 0.0171 -0.4079 
-0.2394 1.0000 -0.1764 0.4675 0.1810 -0.4942 -0.4389 0.6950 0.0811 
-0.6001 -0.1764 1.0000 -0.6002 0.2170 -0.0922 0.2129 -0.3566 0.4204 
0.3114 0.4675 -0.6002 1.0000 0.2597 -0.1039 -0.5507 0.7230 0.0775 
-0.2367 0.1810 0.2170 0.2597 1.0000 0.2159 -0.7123 0.0193 0.8494 
0.5856 -0.4942 -0.0922 -0.1039 0.2159 1.0000 0.0587 -0.4724 0.0957 
0.2077 -0.4389 0.2129 -0.5507 -0.7123 0.0587 1.0000 -0.3395 -0.5780 
0.0171 0.6950 -0.3566 0.7230 0.0193 -0.4724 -0.3395 1.0000 -0.1084 
-0.4079 0.0811 0.4204 0.0775 0.8494 0.0957 -0.5780 -0.1084 1.0000 ];

ParCov{6}=[1.0000 0.1580 -0.5835 0.4011 -0.0561 0.0718 -0.2038 0.2615 -0.2365 
0.1580 1.0000 0.0141 -0.1272 -0.4327 0.1778 -0.0902 -0.0329 -0.3778 
-0.5835 0.0141 1.0000 -0.6295 -0.2949 -0.2008 0.3164 -0.2615 -0.0536 
0.4011 -0.1272 -0.6295 1.0000 0.6960 -0.2587 -0.0988 0.6113 0.5636 
-0.0561 -0.4327 -0.2949 0.6960 1.0000 -0.3370 0.2042 0.3959 0.8581 
0.0718 0.1778 -0.2008 -0.2587 -0.3370 1.0000 -0.0634 -0.5202 -0.3829 
-0.2038 -0.0902 0.3164 -0.0988 0.2042 -0.0634 1.0000 0.0559 0.3322 
0.2615 -0.0329 -0.2615 0.6113 0.3959 -0.5202 0.0559 1.0000 0.3210 
-0.2365 -0.3778 -0.0536 0.5636 0.8581 -0.3829 0.3322 0.3210 1.0000 ];

ParCov{7}=[1.0000 -0.1586 0.1817 -0.0195 -0.0884 0.0282 0.0560 -0.1369 0.0099 
-0.1586 1.0000 0.0440 0.1013 -0.2510 -0.0046 -0.1105 0.0738 -0.1152 
0.1817 0.0440 1.0000 -0.5118 0.0414 0.2570 0.0932 0.0961 0.4938 
-0.0195 0.1013 -0.5118 1.0000 0.0480 -0.1155 -0.2463 -0.0754 0.0204 
-0.0884 -0.2510 0.0414 0.0480 1.0000 0.2577 -0.0581 0.3152 0.3151 
0.0282 -0.0046 0.2570 -0.1155 0.2577 1.0000 -0.1598 0.4397 0.1107 
0.0560 -0.1105 0.0932 -0.2463 -0.0581 -0.1598 1.0000 -0.4617 0.1872 
-0.1369 0.0738 0.0961 -0.0754 0.3152 0.4397 -0.4617 1.0000 -0.0114 
0.0099 -0.1152 0.4938 0.0204 0.3151 0.1107 0.1872 -0.0114 1.0000 ];

ParCov{8}=[1.0000 -0.2440 -0.2729 0.2863 -0.0329 0.2925 -0.0588 0.3377 -0.1914 
-0.2440 1.0000 0.0874 -0.1523 -0.2565 -0.1605 0.0874 -0.2895 -0.2125 
-0.2729 0.0874 1.0000 -0.6332 0.2012 -0.0578 0.0283 -0.1100 0.3013 
0.2863 -0.1523 -0.6332 1.0000 0.3140 0.2152 -0.1084 0.4114 0.1732 
-0.0329 -0.2565 0.2012 0.3140 1.0000 0.3184 -0.1923 0.3761 0.8433 
0.2925 -0.1605 -0.0578 0.2152 0.3184 1.0000 0.1246 0.4736 0.2078 
-0.0588 0.0874 0.0283 -0.1084 -0.1923 0.1246 1.0000 0.0752 -0.1578 
0.3377 -0.2895 -0.1100 0.4114 0.3761 0.4736 0.0752 1.0000 0.2114 
-0.1914 -0.2125 0.3013 0.1732 0.8433 0.2078 -0.1578 0.2114 1.0000 ];

ParCov{9}=[1.0000 -0.2894 0.0381 0.0664 -0.2418 0.2253 0.2822 -0.2919 0.0581 
-0.2894 1.0000 -0.2259 0.4265 0.1859 -0.6307 -0.0140 0.4944 0.2495 
0.0381 -0.2259 1.0000 -0.2855 0.0724 0.1199 -0.1487 -0.3773 0.1881 
0.0664 0.4265 -0.2855 1.0000 0.2208 -0.3752 0.0660 0.3415 0.7289 
-0.2418 0.1859 0.0724 0.2208 1.0000 0.1412 -0.2931 0.1993 0.4609 
0.2253 -0.6307 0.1199 -0.3752 0.1412 1.0000 -0.2855 -0.2046 -0.1974 
0.2822 -0.0140 -0.1487 0.0660 -0.2931 -0.2855 1.0000 -0.1172 -0.0851 
-0.2919 0.4944 -0.3773 0.3415 0.1993 -0.2046 -0.1172 1.0000 0.0530 
0.0581 0.2495 0.1881 0.7289 0.4609 -0.1974 -0.0851 0.0530 1.0000 ];

ParCov{10}=[1.0000 -0.2894 0.0381 0.0664 -0.2418 0.2253 0.2822 -0.2919 0.0581 
-0.2894 1.0000 -0.2259 0.4265 0.1859 -0.6307 -0.0140 0.4944 0.2495 
0.0381 -0.2259 1.0000 -0.2855 0.0724 0.1199 -0.1487 -0.3773 0.1881 
0.0664 0.4265 -0.2855 1.0000 0.2208 -0.3752 0.0660 0.3415 0.7289 
-0.2418 0.1859 0.0724 0.2208 1.0000 0.1412 -0.2931 0.1993 0.4609 
0.2253 -0.6307 0.1199 -0.3752 0.1412 1.0000 -0.2855 -0.2046 -0.1974 
0.2822 -0.0140 -0.1487 0.0660 -0.2931 -0.2855 1.0000 -0.1172 -0.0851 
-0.2919 0.4944 -0.3773 0.3415 0.1993 -0.2046 -0.1172 1.0000 0.0530 
0.0581 0.2495 0.1881 0.7289 0.4609 -0.1974 -0.0851 0.0530 1.0000 ];

ParCov{11}=[1.0000 -0.2394 -0.6001 0.3114 -0.2367 0.5856 0.2077 0.0171 -0.4079 
-0.2394 1.0000 -0.1764 0.4675 0.1810 -0.4942 -0.4389 0.6950 0.0811 
-0.6001 -0.1764 1.0000 -0.6002 0.2170 -0.0922 0.2129 -0.3566 0.4204 
0.3114 0.4675 -0.6002 1.0000 0.2597 -0.1039 -0.5507 0.7230 0.0775 
-0.2367 0.1810 0.2170 0.2597 1.0000 0.2159 -0.7123 0.0193 0.8494 
0.5856 -0.4942 -0.0922 -0.1039 0.2159 1.0000 0.0587 -0.4724 0.0957 
0.2077 -0.4389 0.2129 -0.5507 -0.7123 0.0587 1.0000 -0.3395 -0.5780 
0.0171 0.6950 -0.3566 0.7230 0.0193 -0.4724 -0.3395 1.0000 -0.1084 
-0.4079 0.0811 0.4204 0.0775 0.8494 0.0957 -0.5780 -0.1084 1.0000 ];

ParCov{12}=[1.0000 -0.2394 -0.6001 0.3114 -0.2367 0.5856 0.2077 0.0171 -0.4079 
-0.2394 1.0000 -0.1764 0.4675 0.1810 -0.4942 -0.4389 0.6950 0.0811 
-0.6001 -0.1764 1.0000 -0.6002 0.2170 -0.0922 0.2129 -0.3566 0.4204 
0.3114 0.4675 -0.6002 1.0000 0.2597 -0.1039 -0.5507 0.7230 0.0775 
-0.2367 0.1810 0.2170 0.2597 1.0000 0.2159 -0.7123 0.0193 0.8494 
0.5856 -0.4942 -0.0922 -0.1039 0.2159 1.0000 0.0587 -0.4724 0.0957 
0.2077 -0.4389 0.2129 -0.5507 -0.7123 0.0587 1.0000 -0.3395 -0.5780 
0.0171 0.6950 -0.3566 0.7230 0.0193 -0.4724 -0.3395 1.0000 -0.1084 
-0.4079 0.0811 0.4204 0.0775 0.8494 0.0957 -0.5780 -0.1084 1.0000 ];

ParCov{13}=[1.0000 -0.2440 -0.2729 0.2863 -0.0329 0.2925 -0.0588 0.3377 -0.1914 
-0.2440 1.0000 0.0874 -0.1523 -0.2565 -0.1605 0.0874 -0.2895 -0.2125 
-0.2729 0.0874 1.0000 -0.6332 0.2012 -0.0578 0.0283 -0.1100 0.3013 
0.2863 -0.1523 -0.6332 1.0000 0.3140 0.2152 -0.1084 0.4114 0.1732 
-0.0329 -0.2565 0.2012 0.3140 1.0000 0.3184 -0.1923 0.3761 0.8433 
0.2925 -0.1605 -0.0578 0.2152 0.3184 1.0000 0.1246 0.4736 0.2078 
-0.0588 0.0874 0.0283 -0.1084 -0.1923 0.1246 1.0000 0.0752 -0.1578 
0.3377 -0.2895 -0.1100 0.4114 0.3761 0.4736 0.0752 1.0000 0.2114 
-0.1914 -0.2125 0.3013 0.1732 0.8433 0.2078 -0.1578 0.2114 1.0000 ];

ParCov{14}=[1.0000 -0.1586 0.1817 -0.0195 -0.0884 0.0282 0.0560 -0.1369 0.0099 
-0.1586 1.0000 0.0440 0.1013 -0.2510 -0.0046 -0.1105 0.0738 -0.1152 
0.1817 0.0440 1.0000 -0.5118 0.0414 0.2570 0.0932 0.0961 0.4938 
-0.0195 0.1013 -0.5118 1.0000 0.0480 -0.1155 -0.2463 -0.0754 0.0204 
-0.0884 -0.2510 0.0414 0.0480 1.0000 0.2577 -0.0581 0.3152 0.3151 
0.0282 -0.0046 0.2570 -0.1155 0.2577 1.0000 -0.1598 0.4397 0.1107 
0.0560 -0.1105 0.0932 -0.2463 -0.0581 -0.1598 1.0000 -0.4617 0.1872 
-0.1369 0.0738 0.0961 -0.0754 0.3152 0.4397 -0.4617 1.0000 -0.0114 
0.0099 -0.1152 0.4938 0.0204 0.3151 0.1107 0.1872 -0.0114 1.0000 ];

% Box-Cox transformation exponents for neuron parameters
k_trans=[0.3700 0.2200 0.2200 0.0000 0.0000 0.3700 0.0000 0.2300 0.2200 0.2200 0.0000 0.0000 0.2300 0.0000 
0.0000 0.0200 0.0200 0.0000 0.0000 0.0000 0.0100 0.0100 0.0200 0.0200 0.0000 0.0000 0.0100 0.0100 
0.0000 0.3600 0.3600 0.0000 0.0000 0.0000 0.0000 0.0000 0.3600 0.3600 0.0000 0.0000 0.0000 0.0000 
0.0100 0.0000 0.0000 0.0000 0.0000 0.0100 0.0000 0.1300 0.0000 0.0000 0.0000 0.0000 0.1300 0.0000 
0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0200 0.0000 0.0000 0.0000 0.0000 0.0200 0.0000 
0.0000 0.0200 0.0200 0.0200 0.0200 0.0000 0.0000 0.0000 0.0200 0.0200 0.0200 0.0200 0.0000 0.0000 
0.0100 0.3600 0.3600 0.0000 0.0000 0.0100 0.0000 0.0000 0.3600 0.3600 0.0000 0.0000 0.0000 0.0000 
0.0000 0.0000 0.0000 0.1200 0.1200 0.0000 0.2600 0.0000 0.0000 0.0000 0.1200 0.1200 0.0000 0.2600 
0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 ];

% Remaining neuron parameter statistics (SD, min, max)           
N_sig = zeros(9,NTypesN);                                         % SDs of independent neuron parameters
N_min = zeros(12,NTypesN);                                        % minima of all neuron parameters
N_max = zeros(12,NTypesN);                                        % maxima of all neuron parameters

N_sig(2,:) = [ 0.4296 0.0754 0.0754 0.3283 0.3283 0.4296 0.5029 ...
		0.1472 0.0754 0.0754 0.3283 0.3283 0.1472 0.5029 ];
N_sig(3,:) = [ 0.3558 0.0046 0.0046 0.1844 0.1844 0.3558 0.0022 ...
		0.0030 0.0046 0.0046 0.1844 0.1844 0.0030 0.0022 ];
N_sig(4,:) = [ 0.3644 0.3813 0.3813 0.3630 0.3630 0.3644 0.3359 ...
		0.2846 0.3813 0.3813 0.3630 0.3630 0.2846 0.3359 ];
N_sig(5,:) = [ 0.0048 0.7107 0.7107 0.3568 0.3568 0.0048 0.7395 ...
		0.0554 0.7107 0.7107 0.3568 0.3568 0.0554 0.7395 ];
N_sig(6,:) = [ 0.5259 0.5033 0.5033 0.4372 0.4372 0.5259 0.3035 ...
		0.0062 0.5033 0.5033 0.4372 0.4372 0.0062 0.3035 ];
N_sig(7,:) = [ 0.4947 0.0052 0.0052 0.0170 0.0170 0.4947 0.3186 ...
		0.6356 0.0052 0.0052 0.0170 0.0170 0.6356 0.3186 ];
N_sig(8,:) = [ 0.0113 1.9269 1.9269 1.4550 1.4550 0.0113 1.0353 ...
		1.3712 1.9269 1.9269 1.4550 1.4550 1.3712 1.0353 ];
N_sig(9,:) = [ 0.6104 0.4817 0.4817 0.1504 0.1504 0.6104 0.1813 ...
		0.3497 0.4817 0.4817 0.1504 0.1504 0.3497 0.1813 ];
N_sig(10,:) = [ 0.4608 0.4385 0.4385 0.4311 0.4311 0.4608 0.3632 ...
		0.2857 0.4385 0.4385 0.4311 0.4311 0.2857 0.3632 ];


N_min(2,:) = [ 61.4187 42.1156 42.1156 51.8447 51.8447 61.4187 32.3194 ...
		110.7272 42.1156 42.1156 51.8447 51.8447 110.7272 32.3194 ];
N_min(3,:) = [ 3.2940 3.6802 3.6802 2.9852 2.9852 3.2940 2.1462 ...
		3.4510 3.6802 3.6802 2.9852 2.9852 3.4510 2.1462 ];
N_min(4,:) = [ -104.9627 -96.9345 -96.9345 -98.8335 -98.8335 -104.9627 -102.3895 ...
		-101.5624 -96.9345 -96.9345 -98.8335 -98.8335 -101.5624 -102.3895 ];
N_min(5,:) = [ 10.5568 2.1840 2.1840 11.0503 11.0503 10.5568 1.8285 ...
		12.7969 2.1840 2.1840 11.0503 11.0503 12.7969 1.8285 ];
N_min(6,:) = [ -62.5083 -60.6745 -60.6745 -65.4193 -65.4193 -62.5083 -42.8895 ...
		-66.1510 -60.6745 -60.6745 -65.4193 -65.4193 -66.1510 -42.8895 ];
N_min(7,:) = [ 54.0018 10.2826 10.2826 12.2898 12.2898 54.0018 20.0311 ...
		33.1367 10.2826 10.2826 12.2898 12.2898 33.1367 20.0311 ];
N_min(8,:) = [ 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 ...
		0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 ];
N_min(9,:) = [ 1.2406 1.0000 1.0000 1.0000 1.0000 1.2406 1.0000 ...
		1.0000 1.0000 1.0000 1.0000 1.0000 1.0000 1.0000 ];
N_min(10,:) = [ -219.2039 -128.4559 -128.4559 -271.9846 -271.9846 -219.2039 -105.1880 ...
		-124.5158 -128.4559 -128.4559 -271.9846 -271.9846 -124.5158 -105.1880 ];
N_min(11,:) = [ -63.2375 -85.2096 -85.2096 -70.3537 -70.3537 -63.2375 -53.3897 ...
		-69.5922 -85.2096 -85.2096 -70.3537 -70.3537 -69.5922 -53.3897 ];
N_min(12,:) = [ 10.3876 7.3511 7.3511 9.2264 9.2264 10.3876 5.8527 ...
		16.7015 7.3511 7.3511 9.2264 9.2264 16.7015 5.8527 ];


N_max(2,:) = [ 337.9765 94.6939 94.6939 126.2367 126.2367 337.9765 201.3221 ...
		617.2776 94.6939 94.6939 126.2367 126.2367 617.2776 201.3221 ];
N_max(3,:) = [ 10.8106 8.6130 8.6130 5.6192 5.6192 10.8106 5.3460 ...
		15.6329 8.6130 8.6130 5.6192 5.6192 15.6329 5.3460 ];
N_max(4,:) = [ -76.8526 -71.7548 -71.7548 -75.7868 -75.7868 -76.8526 -59.6898 ...
		-66.4770 -71.7548 -71.7548 -75.7868 -75.7868 -66.4770 -59.6898 ];
N_max(5,:) = [ 45.3814 40.4333 40.4333 31.3533 31.3533 45.3814 47.6214 ...
		43.5882 40.4333 40.4333 31.3533 31.3533 43.5882 47.6214 ];
N_max(6,:) = [ -30.0577 -36.5929 -36.5929 -45.6445 -45.6445 -30.0577 -30.7977 ...
		-25.2891 -36.5929 -36.5929 -45.6445 -45.6445 -25.2891 -30.7977 ];
N_max(7,:) = [ 232.8699 21.9964 21.9964 120.5043 120.5043 232.8699 102.4180 ...
		909.5520 21.9964 21.9964 120.5043 120.5043 909.5520 102.4180 ];
N_max(8,:) = [ 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 ...
		0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 0.0000 ];
N_max(9,:) = [ 40.2930 196.7634 196.7634 71.0958 71.0958 40.2930 54.2781 ...
		325.7906 196.7634 196.7634 71.0958 71.0958 325.7906 54.2781 ];
N_max(10,:) = [ -45.0393 -56.5047 -56.5047 -56.8682 -56.8682 -45.0393 -35.7409 ...
		-35.1145 -56.5047 -56.5047 -56.8682 -56.8682 -35.1145 -35.7409 ];
N_max(11,:) = [ -36.8701 -39.1085 -39.1085 -49.0974 -49.0974 -36.8701 -20.6720 ...
		-27.8669 -39.1085 -39.1085 -49.0974 -49.0974 -27.8669 -20.6720 ];
N_max(12,:) = [ 42.7304 15.9128 15.9128 25.9839 25.9839 42.7304 48.7992 ...
		67.7062 15.9128 15.9128 25.9839 25.9839 67.7062 48.7992 ];

NPList=1:N*Nstripes;                                                 % type of each neuron
NeuronGroupsSaveArray = [];


% ------------------------- Set synapse parameters -------------------------
% Used synapse types: (1) AMPA  (2) GABA    (3) NMDA
STypPar=zeros(8,3);                                         % synapse parameters
STypPar(1,:)=[1   1   1.09];     % gmax (relative)
STypPar(2,:)=[1.4 3.0 4.3];      % tc_on
STypPar(3,:)=[10  40  75];       % tc_off
STypPar(4,:)=[0 -70 0];          % Erev (org: -70)
STypPar(5,:)=[0 0 1];            % Mg_gate
STypPar(6,:)=[0 0 0.33];         % Mg_fac
STypPar(7,:)=[0 0 0.0625];       % Mg_slope
STypPar(8,:)=[0 0 0];            % Mg_half

% ConParIdc is currently redundant, apart from the maximum value being used
% to determine the size of the third dimension of SPMtx (i.e. the maximum
% number of synapse types that a connection can have).
ConParIdc = repmat([2 2 2 1 1 1],NTypesN,1);            % number of synapse types for each connection type
ConParIdc(7,:) = [2 2 2 2 2 2]';                        % connections from InpNeurons

% Scaling factors for recurrent and input connection strengths
s1 = 0; 
s2 = 0; 
srec_EE = s(1); 
srec_IE = s(2); 
srec_EI = s(3); 
srec_II = s(4); 

% Cell array with 1) index of the parameter set to be used for each 
% synapse, 2) synapse type number for each synapse and 3) parameters sets
% (as seperate vector entries, if more than one), all of these for 
% each input->output pair
% Parameters: [wgt dtax p_fail]
% inv_con_PSP converts PSP values into peak conductances based on
% experimental simulations done using test_update_inv_con_PSP.m. Needs to
% be redone when neuron parameters change.
p_fail=0.3;     % probability of synaptic failure
ConParCont(1,1)   = {{[1 1] , [1 3], [inv_con_PSP(0.8405, 1, 1)*srec_EE  1.5465 p_fail]}};        % L23-E -> L23-E
ConParCont(1,8)   = {{[1 1] , [1 3], [inv_con_PSP(0.8378, 1, 8)*srec_EE  2.7533 p_fail]}};        % L5-E  -> L23-E
ConParCont(8,1)   = {{[1 1] , [1 3], [inv_con_PSP(0.9533, 8, 1)*srec_EE  1.9085 p_fail]}};        % L23-E -> L5-E 
ConParCont(8,8)   = {{[1 1] , [1 3], [inv_con_PSP(0.8818, 8, 8)*srec_EE  1.5667 p_fail]}};        % L5-E  -> L5-E


ConParCont(1,2)   = {{ 1    ,  2 ,   [inv_con_PSP(2.2615, 1, 2)*srec_IE  1.2491 p_fail]}};        % L23-I-L     -> L23-E
ConParCont(1,3)   = {{ 1    ,  2 ,   [inv_con_PSP(2.2615, 1, 3)*srec_IE  1.2491 p_fail]}};        % L23-I-L-d   -> L23-E
ConParCont(1,4)   = {{ 1    ,  2 ,   [inv_con_PSP(0.1800, 1, 4)*srec_IE  1.2491 p_fail]}};        % L23-I-CL    -> L23-E
ConParCont(1,5)   = {{ 1    ,  2 ,   [inv_con_PSP(0.1800, 1, 5)*srec_IE  1.2491 p_fail]}};        % L23-I-CL-AC -> L23-E
ConParCont(1,6)   = {{ 1    ,  2 ,   [inv_con_PSP(2.2615, 1, 6)*srec_IE  1.2491 p_fail]}};        % L23-I-CS    -> L23-E
ConParCont(1,7)   = {{ 1    ,  2 ,   [inv_con_PSP(1.8218, 1, 7)*srec_IE  1.2491 p_fail]}};        % L23-I-F     -> L23-E

ConParCont(1,9)   = {{ 1    ,  2 ,   [inv_con_PSP(0.2497, 1, 9)*srec_IE  1.4411 p_fail]}};        % L5-I-L      -> L23-E
ConParCont(1,10)  = {{ 1    ,  2 ,   [inv_con_PSP(0.2497, 1,10)*srec_IE  1.4411 p_fail]}};        % L5-I-L-d    -> L23-E
ConParCont(1,11)  = {{ 1    ,  2 ,   [inv_con_PSP(0.0556, 1,11)*srec_IE  1.4411 p_fail]}};        % L5-I-CL     -> L23-E
ConParCont(1,12)  = {{ 1    ,  2 ,   [inv_con_PSP(0.0556, 1,12)*srec_IE  1.4411 p_fail]}};        % L5-I-CL-AC  -> L23-E
ConParCont(1,13)  = {{ 1    ,  2 ,   [inv_con_PSP(0.2497, 1,13)*srec_IE  1.4411 p_fail]}};        % L5-I-CS     -> L23-E
ConParCont(1,14)  = {{ 1    ,  2 ,   [inv_con_PSP(0.2285, 1,14)*srec_IE  1.4411 p_fail]}};        % L5-I-F      -> L23-E

ConParCont(8,2)   = {{ 1    ,  2 ,   [inv_con_PSP(1.0503, 8, 2)*srec_IE  1.5415 p_fail]}};        % L23-I-L     -> L5-E
ConParCont(8,3)   = {{ 1    ,  2 ,   [inv_con_PSP(1.0503, 8, 3)*srec_IE  1.5415 p_fail]}};        % L23-I-L-d   -> L5-E
ConParCont(8,4)   = {{ 1    ,  2 ,   [inv_con_PSP(0.0836, 8, 4)*srec_IE  1.5415 p_fail]}};        % L23-I-CL    -> L5-E
ConParCont(8,5)   = {{ 1    ,  2 ,   [inv_con_PSP(0.0836, 8, 5)*srec_IE  1.5415 p_fail]}};        % L23-I-CL-AC -> L5-E
ConParCont(8,6)   = {{ 1    ,  2 ,   [inv_con_PSP(1.0503, 8, 6)*srec_IE  1.5415 p_fail]}};        % L23-I-CS    -> L5-E
ConParCont(8,7)   = {{ 1    ,  2 ,   [inv_con_PSP(0.8461, 8, 7)*srec_IE  1.5415 p_fail]}};        % L23-I-F     -> L5-E

ConParCont(8,9)   = {{ 1    ,  2 ,   [inv_con_PSP(1.7644, 8, 9)*srec_IE  0.8200 p_fail]}};        % L5-I-L      -> L5-E
ConParCont(8,10)  = {{ 1    ,  2 ,   [inv_con_PSP(1.7644, 8,10)*srec_IE  0.8200 p_fail]}};        % L5-I-L-d    -> L5-E
ConParCont(8,11)  = {{ 1    ,  2 ,   [inv_con_PSP(0.3932, 8,11)*srec_IE  0.8200 p_fail]}};        % L5-I-CL     -> L5-E
ConParCont(8,12)  = {{ 1    ,  2 ,   [inv_con_PSP(0.3932, 8,12)*srec_IE  0.8200 p_fail]}};        % L5-I-CL-AC  -> L5-E
ConParCont(8,13)  = {{ 1    ,  2 ,   [inv_con_PSP(1.7644, 8,13)*srec_IE  0.8200 p_fail]}};        % L5-I-CS     -> L5-E
ConParCont(8,14)  = {{ 1    ,  2 ,   [inv_con_PSP(1.6149, 8,14)*srec_IE  0.8200 p_fail]}};        % L5-I-F      -> L5-E


ConParCont(2,1)   = {{[1 1] , [1 3], [inv_con_PSP(1.3403, 2, 1)*srec_EI  0.9581 p_fail]}};        % L23-E -> L23-I-L
ConParCont(3,1)   = {{[1 1] , [1 3], [inv_con_PSP(1.3403, 3, 1)*srec_EI  0.9581 p_fail]}};        % L23-E -> L23-I-L-d
ConParCont(4,1)   = {{[1 1] , [1 3], [inv_con_PSP(0.4710, 4, 1)*srec_EI  0.9581 p_fail]}};        % L23-E -> L23-I-CL
ConParCont(5,1)   = {{[1 1] , [1 3], [inv_con_PSP(0.4710, 5, 1)*srec_EI  0.9581 p_fail]}};        % L23-E -> L23-I-CL-AC
ConParCont(6,1)   = {{[1 1] , [1 3], [inv_con_PSP(1.3403, 6, 1)*srec_EI  0.9581 p_fail]}};        % L23-E -> L23-I-CS
ConParCont(7,1)   = {{[1 1] , [1 3], [inv_con_PSP(0.2500, 7, 1)*srec_EI  0.9581 p_fail]}};        % L23-E -> L23-I-F

ConParCont(2,8)   = {{[1 1] , [1 3], [inv_con_PSP(0.7738, 2, 8)*srec_EI  1.0544 p_fail]}};        % L5-E  -> L23-I-L
ConParCont(3,8)   = {{[1 1] , [1 3], [inv_con_PSP(0.7738, 3, 8)*srec_EI  1.0544 p_fail]}};        % L5-E  -> L23-I-L-d
ConParCont(4,8)   = {{[1 1] , [1 3], [inv_con_PSP(0.2719, 4, 8)*srec_EI  1.0544 p_fail]}};        % L5-E  -> L23-I-CL
ConParCont(5,8)   = {{[1 1] , [1 3], [inv_con_PSP(0.2719, 5, 8)*srec_EI  1.0544 p_fail]}};        % L5-E  -> L23-I-CL-AC
ConParCont(6,8)   = {{[1 1] , [1 3], [inv_con_PSP(0.7738, 6, 8)*srec_EI  1.0544 p_fail]}};        % L5-E  -> L23-I-CS
ConParCont(7,8)   = {{[1 1] , [1 3], [inv_con_PSP(0.1443, 7, 8)*srec_EI  1.0544 p_fail]}};        % L5-E  -> L23-I-F

ConParCont(9,1)   = {{[1 1] , [1 3], [inv_con_PSP(1.5201, 9, 1)*srec_EI  1.1825 p_fail]}};        % L23-E -> L5-I-L
ConParCont(10,1)  = {{[1 1] , [1 3], [inv_con_PSP(1.5201,10, 1)*srec_EI  1.1825 p_fail]}};        % L23-E -> L5-I-L-d
ConParCont(11,1)  = {{[1 1] , [1 3], [inv_con_PSP(0.5342,11, 1)*srec_EI  1.1825 p_fail]}};        % L23-E -> L5-I-CL
ConParCont(12,1)  = {{[1 1] , [1 3], [inv_con_PSP(0.5342,12, 1)*srec_EI  1.1825 p_fail]}};        % L23-E -> L5-I-CL-AC
ConParCont(13,1)  = {{[1 1] , [1 3], [inv_con_PSP(1.5201,13, 1)*srec_EI  1.1825 p_fail]}};        % L23-E -> L5-I-CS
ConParCont(14,1)  = {{[1 1] , [1 3], [inv_con_PSP(0.2835,14, 1)*srec_EI  1.1825 p_fail]}};        % L23-E -> L5-I-F

ConParCont(9,8)   = {{[1 1] , [1 3], [inv_con_PSP(1.7431, 9, 8)*srec_EI  0.6000 p_fail]}};        % L5-E  -> L5-I-L
ConParCont(10,8)  = {{[1 1] , [1 3], [inv_con_PSP(1.7431,10, 8)*srec_EI  0.6000 p_fail]}};        % L5-E  -> L5-I-L-d
ConParCont(11,8)  = {{[1 1] , [1 3], [inv_con_PSP(0.8800,11, 8)*srec_EI  0.6000 p_fail]}};        % L5-E  -> L5-I-CL
ConParCont(12,8)  = {{[1 1] , [1 3], [inv_con_PSP(0.8800,12, 8)*srec_EI  0.6000 p_fail]}};        % L5-E  -> L5-I-CL-AC
ConParCont(13,8)  = {{[1 1] , [1 3], [inv_con_PSP(1.7431,13, 8)*srec_EI  0.6000 p_fail]}};        % L5-E  -> L5-I-CS
ConParCont(14,8)  = {{[1 1] , [1 3], [inv_con_PSP(0.2800,14, 8)*srec_EI  0.6000 p_fail]}};        % L5-E  -> L5-I-F

ConParCont(2,2)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 2, 2)*srec_II  1.1000 p_fail]}};        % L23-I-L     -> L23-I-L
ConParCont(2,3)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 2, 3)*srec_II  1.1000 p_fail]}};        % L23-I-L-d   -> L23-I-L
ConParCont(2,4)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 2, 4)*srec_II  1.1000 p_fail]}};        % L23-I-CL    -> L23-I-L
ConParCont(2,5)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 2, 5)*srec_II  1.1000 p_fail]}};        % L23-I-CL-AC -> L23-I-L
ConParCont(2,6)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 2, 6)*srec_II  1.1000 p_fail]}};        % L23-I-CS    -> L23-I-L
ConParCont(2,7)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 2, 7)*srec_II  1.1000 p_fail]}};        % L23-I-F     -> L23-I-L
ConParCont(3,2)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 3, 2)*srec_II  1.1000 p_fail]}};        % L23-I-L     -> L23-I-L-d
ConParCont(3,3)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 3, 3)*srec_II  1.1000 p_fail]}};        % L23-I-L-d   -> L23-I-L-d
ConParCont(3,4)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 3, 4)*srec_II  1.1000 p_fail]}};        % L23-I-CL    -> L23-I-L-d
ConParCont(3,5)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 3, 5)*srec_II  1.1000 p_fail]}};        % L23-I-CL-AC -> L23-I-L-d
ConParCont(3,6)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 3, 6)*srec_II  1.1000 p_fail]}};        % L23-I-CS    -> L23-I-L-d
ConParCont(3,7)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 3, 7)*srec_II  1.1000 p_fail]}};        % L23-I-F     -> L23-I-L-d
ConParCont(4,2)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 4, 2)*srec_II  1.1000 p_fail]}};        % L23-I-L     -> L23-I-CL
ConParCont(4,3)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 4, 3)*srec_II  1.1000 p_fail]}};        % L23-I-L-d   -> L23-I-CL
ConParCont(4,4)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 4, 4)*srec_II  1.1000 p_fail]}};        % L23-I-CL    -> L23-I-CL
ConParCont(4,5)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 4, 5)*srec_II  1.1000 p_fail]}};        % L23-I-CL-AC -> L23-I-CL
ConParCont(4,6)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 4, 6)*srec_II  1.1000 p_fail]}};        % L23-I-CS    -> L23-I-CL
ConParCont(4,7)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 4, 7)*srec_II  1.1000 p_fail]}};        % L23-I-F     -> L23-I-CL
ConParCont(5,2)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 5, 2)*srec_II  1.1000 p_fail]}};        % L23-I-L     -> L23-I-CL-AC
ConParCont(5,3)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 5, 3)*srec_II  1.1000 p_fail]}};        % L23-I-L-d   -> L23-I-CL-AC
ConParCont(5,4)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 5, 4)*srec_II  1.1000 p_fail]}};        % L23-I-CL    -> L23-I-CL-AC
ConParCont(5,5)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 5, 5)*srec_II  1.1000 p_fail]}};        % L23-I-CL-AC -> L23-I-CL-AC
ConParCont(5,6)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 5, 6)*srec_II  1.1000 p_fail]}};        % L23-I-CS    -> L23-I-CL-AC
ConParCont(5,7)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 5, 7)*srec_II  1.1000 p_fail]}};        % L23-I-F     -> L23-I-CL-AC
ConParCont(6,2)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 6, 2)*srec_II  1.1000 p_fail]}};        % L23-I-L     -> L23-I-CS
ConParCont(6,3)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 6, 3)*srec_II  1.1000 p_fail]}};        % L23-I-L-d   -> L23-I-CS
ConParCont(6,4)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 6, 4)*srec_II  1.1000 p_fail]}};        % L23-I-CL    -> L23-I-CS
ConParCont(6,5)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 6, 5)*srec_II  1.1000 p_fail]}};        % L23-I-CL-AC -> L23-I-CS
ConParCont(6,6)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 6, 6)*srec_II  1.1000 p_fail]}};        % L23-I-CS    -> L23-I-CS
ConParCont(6,7)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 6, 7)*srec_II  1.1000 p_fail]}};        % L23-I-F     -> L23-I-CS
ConParCont(7,2)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 7, 2)*srec_II  1.1000 p_fail]}};        % L23-I-L     -> L23-I-F
ConParCont(7,3)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 7, 3)*srec_II  1.1000 p_fail]}};        % L23-I-L-d   -> L23-I-F
ConParCont(7,4)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 7, 4)*srec_II  1.1000 p_fail]}};        % L23-I-CL    -> L23-I-F
ConParCont(7,5)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 7, 5)*srec_II  1.1000 p_fail]}};        % L23-I-CL-AC -> L23-I-F
ConParCont(7,6)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 7, 6)*srec_II  1.1000 p_fail]}};        % L23-I-CS    -> L23-I-F
ConParCont(7,7)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 7, 7)*srec_II  1.1000 p_fail]}};        % L23-I-F     -> L23-I-F

ConParCont(2,9)   = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-L      -> L23-I-L
ConParCont(2,10)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-L-d    -> L23-I-L
ConParCont(2,11)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CL     -> L23-I-L
ConParCont(2,12)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CL-AC  -> L23-I-L
ConParCont(2,13)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CS     -> L23-I-L
ConParCont(2,14)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-F      -> L23-I-L
ConParCont(3,9)   = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-L      -> L23-I-L-d
ConParCont(3,10)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-L-d    -> L23-I-L-d
ConParCont(3,11)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CL     -> L23-I-L-d
ConParCont(3,12)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CL-AC  -> L23-I-L-d
ConParCont(3,13)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CS     -> L23-I-L-d
ConParCont(3,14)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-F      -> L23-I-L-d
ConParCont(4,9)   = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-L      -> L23-I-CL
ConParCont(4,10)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-L-d    -> L23-I-CL
ConParCont(4,11)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CL     -> L23-I-CL
ConParCont(4,12)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CL-AC  -> L23-I-CL
ConParCont(4,13)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CS     -> L23-I-CL
ConParCont(4,14)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-F      -> L23-I-CL
ConParCont(5,9)   = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-L      -> L23-I-CL-AC
ConParCont(5,10)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-L-d    -> L23-I-CL-AC
ConParCont(5,11)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CL     -> L23-I-CL-AC
ConParCont(5,12)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CL-AC  -> L23-I-CL-AC
ConParCont(5,13)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CS     -> L23-I-CL-AC
ConParCont(5,14)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-F      -> L23-I-CL-AC
ConParCont(6,9)   = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-L      -> L23-I-CS
ConParCont(6,10)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-L-d    -> L23-I-CS
ConParCont(6,11)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CL     -> L23-I-CS
ConParCont(6,12)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CL-AC  -> L23-I-CS
ConParCont(6,13)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CS     -> L23-I-CS
ConParCont(6,14)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-F      -> L23-I-CS
ConParCont(7,9)   = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-L      -> L23-I-F
ConParCont(7,10)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-L-d    -> L23-I-F
ConParCont(7,11)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CL     -> L23-I-F
ConParCont(7,12)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CL-AC  -> L23-I-F
ConParCont(7,13)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-CS     -> L23-I-F
ConParCont(7,14)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L5-I-F      -> L23-I-F

ConParCont(9,2)   = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-L     -> L5-I-L
ConParCont(9,3)   = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-L-d   -> L5-I-L
ConParCont(9,4)   = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CL    -> L5-I-L
ConParCont(9,5)   = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CL-AC -> L5-I-L
ConParCont(9,6)   = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CS    -> L5-I-L
ConParCont(9,7)   = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-F     -> L5-I-L
ConParCont(10,2)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-L     -> L5-I-L-d
ConParCont(10,3)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-L-d   -> L5-I-L-d
ConParCont(10,4)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CL    -> L5-I-L-d
ConParCont(10,5)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CL-AC -> L5-I-L-d
ConParCont(10,6)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CS    -> L5-I-L-d
ConParCont(10,7)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-F     -> L5-I-L-d
ConParCont(11,2)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-L     -> L5-I-CL
ConParCont(11,3)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-L-d   -> L5-I-CL
ConParCont(11,4)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CL    -> L5-I-CL
ConParCont(11,5)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CL-AC -> L5-I-CL
ConParCont(11,6)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CS    -> L5-I-CL
ConParCont(11,7)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-F     -> L5-I-CL
ConParCont(12,2)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-L     -> L5-I-CL-AC
ConParCont(12,3)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-L-d   -> L5-I-CL-AC
ConParCont(12,4)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CL    -> L5-I-CL-AC
ConParCont(12,5)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CL-AC -> L5-I-CL-AC
ConParCont(12,6)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CS    -> L5-I-CL-AC
ConParCont(12,7)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-F     -> L5-I-CL-AC
ConParCont(13,2)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-L     -> L5-I-CS
ConParCont(13,3)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-L-d   -> L5-I-CS
ConParCont(13,4)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CL    -> L5-I-CS
ConParCont(13,5)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CL-AC -> L5-I-CS
ConParCont(13,6)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CS    -> L5-I-CS
ConParCont(13,7)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-F     -> L5-I-CS
ConParCont(14,2)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-L     -> L5-I-F
ConParCont(14,3)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-L-d   -> L5-I-F
ConParCont(14,4)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CL    -> L5-I-F
ConParCont(14,5)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CL-AC -> L5-I-F
ConParCont(14,6)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-CS    -> L5-I-F
ConParCont(14,7)  = {{ 1    ,  2 ,   [0                                  0 p_fail     ]}};        % L23-I-F     -> L5-I-F

ConParCont(9,9)   = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 9, 9)*srec_II  1.1000 p_fail]}};        % L5-I-L      -> L5-I-L
ConParCont(9,10)  = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 9,10)*srec_II  1.1000 p_fail]}};        % L5-I-L-d    -> L5-I-L
ConParCont(9,11)  = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 9,11)*srec_II  1.1000 p_fail]}};        % L5-I-CL     -> L5-I-L
ConParCont(9,12)  = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 9,12)*srec_II  1.1000 p_fail]}};        % L5-I-CL-AC  -> L5-I-L
ConParCont(9,13)  = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 9,13)*srec_II  1.1000 p_fail]}};        % L5-I-CS     -> L5-I-L
ConParCont(9,14)  = {{ 1    ,  2 ,   [inv_con_PSP(1.3500, 9,14)*srec_II  1.1000 p_fail]}};        % L5-I-F      -> L5-I-L
ConParCont(10,9)  = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,10, 9)*srec_II  1.1000 p_fail]}};        % L5-I-L      -> L5-I-L-d
ConParCont(10,10) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,10,10)*srec_II  1.1000 p_fail]}};        % L5-I-L-d    -> L5-I-L-d
ConParCont(10,11) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,10,11)*srec_II  1.1000 p_fail]}};        % L5-I-CL     -> L5-I-L-d
ConParCont(10,12) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,10,12)*srec_II  1.1000 p_fail]}};        % L5-I-CL-AC  -> L5-I-L-d
ConParCont(10,13) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,10,13)*srec_II  1.1000 p_fail]}};        % L5-I-CS     -> L5-I-L-d
ConParCont(10,14) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,10,14)*srec_II  1.1000 p_fail]}};        % L5-I-F      -> L5-I-L-d
ConParCont(11,9)  = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,11, 9)*srec_II  1.1000 p_fail]}};        % L5-I-L      -> L5-I-CL
ConParCont(11,10) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,11,10)*srec_II  1.1000 p_fail]}};        % L5-I-L-d    -> L5-I-CL
ConParCont(11,11) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,11,11)*srec_II  1.1000 p_fail]}};        % L5-I-CL     -> L5-I-CL
ConParCont(11,12) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,11,12)*srec_II  1.1000 p_fail]}};        % L5-I-CL-AC  -> L5-I-CL
ConParCont(11,13) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,11,13)*srec_II  1.1000 p_fail]}};        % L5-I-CS     -> L5-I-CL
ConParCont(11,14) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,11,14)*srec_II  1.1000 p_fail]}};        % L5-I-F      -> L5-I-CL
ConParCont(12,9)  = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,12, 9)*srec_II  1.1000 p_fail]}};        % L5-I-L      -> L5-I-CL-AC
ConParCont(12,10) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,12,10)*srec_II  1.1000 p_fail]}};        % L5-I-L-d    -> L5-I-CL-AC
ConParCont(12,11) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,12,11)*srec_II  1.1000 p_fail]}};        % L5-I-CL     -> L5-I-CL-AC
ConParCont(12,12) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,12,12)*srec_II  1.1000 p_fail]}};        % L5-I-CL-AC  -> L5-I-CL-AC
ConParCont(12,13) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,12,13)*srec_II  1.1000 p_fail]}};        % L5-I-CS     -> L5-I-CL-AC
ConParCont(12,14) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,12,14)*srec_II  1.1000 p_fail]}};        % L5-I-F      -> L5-I-CL-AC
ConParCont(13,9)  = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,13, 9)*srec_II  1.1000 p_fail]}};        % L5-I-L      -> L5-I-CS
ConParCont(13,10) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,13,10)*srec_II  1.1000 p_fail]}};        % L5-I-L-d    -> L5-I-CS
ConParCont(13,11) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,13,11)*srec_II  1.1000 p_fail]}};        % L5-I-CL     -> L5-I-CS
ConParCont(13,12) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,13,12)*srec_II  1.1000 p_fail]}};        % L5-I-CL-AC  -> L5-I-CS
ConParCont(13,13) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,13,13)*srec_II  1.1000 p_fail]}};        % L5-I-CS     -> L5-I-CS
ConParCont(13,14) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,13,14)*srec_II  1.1000 p_fail]}};        % L5-I-F      -> L5-I-CS
ConParCont(14,9)  = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,14, 9)*srec_II  1.1000 p_fail]}};        % L5-I-L      -> L5-I-F
ConParCont(14,10) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,14,10)*srec_II  1.1000 p_fail]}};        % L5-I-L-d    -> L5-I-F
ConParCont(14,11) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,14,11)*srec_II  1.1000 p_fail]}};        % L5-I-CL     -> L5-I-F
ConParCont(14,12) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,14,12)*srec_II  1.1000 p_fail]}};        % L5-I-CL-AC  -> L5-I-F
ConParCont(14,13) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,14,13)*srec_II  1.1000 p_fail]}};        % L5-I-CS     -> L5-I-F
ConParCont(14,14) = {{ 1    ,  2 ,   [inv_con_PSP(1.3500,14,14)*srec_II  1.1000 p_fail]}};        % L5-I-F      -> L5-I-F


ConParCont(1,15)  = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L23-E
ConParCont(2,15)  = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L23-I-L
ConParCont(3,15)  = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L23-I-L-d
ConParCont(4,15)  = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L23-I-CL
ConParCont(5,15)  = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L23-I-CL-AC
ConParCont(6,15)  = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L23-I-CS
ConParCont(7,15)  = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L23-I-F
ConParCont(8,15)  = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L5-E
ConParCont(9,15)  = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L5-I-L
ConParCont(10,15) = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L5-I-L-d
ConParCont(11,15) = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L5-I-CL
ConParCont(12,15) = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L5-I-CL-AC
ConParCont(13,15) = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L5-I-CS
ConParCont(14,15) = {{[1 1] , [1 3], [1*s1      3 p_fail]}};        % Inp1   -> L5-I-F

ConParCont(1,16)  = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L23-E
ConParCont(2,16)  = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L23-I-L
ConParCont(3,16)  = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L23-I-L-d
ConParCont(4,16)  = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L23-I-CL
ConParCont(5,16)  = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L23-I-CL-AC
ConParCont(6,16)  = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L23-I-CS
ConParCont(7,16)  = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L23-I-F
ConParCont(8,16)  = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L5-E
ConParCont(9,16)  = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L5-I-L
ConParCont(10,16) = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L5-I-L-d
ConParCont(11,16) = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L5-I-CL
ConParCont(12,16) = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L5-I-CL-AC
ConParCont(13,16) = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L5-I-CS
ConParCont(14,16) = {{ 1    ,  2 ,   [1*s2      3 p_fail]}};        % Inp2   -> L5-I-F


% Short-term synaptic plasticity parameters
mean_E1 = [0.28 194 507];   std_E1 = [0.02 18   37];
mean_E2 = [0.25 671 17 ];   std_E2 = [0.02 17    5];
mean_E3 = [0.29 329 326];   std_E3 = [0.03 53   66];
mean_I1 = [0.16  45 376];   std_I1 = [0.10 21  253];
mean_I2 = [0.25 706  21];   std_I2 = [0.13 405   9];
mean_I3 = [0.32 144  62];   std_I3 = [0.14  80  31];

% Use E2 or I2 only (whenever there is more than one possible) before using update_inv_con_PSP
ConParSTSP([1 8], [1 8])  = {{[0.45 0.38 0.17], {mean_E1, std_E1}, {mean_E2, std_E2}, {mean_E3, std_E3}}};     % E->E

ConParSTSP([2 9], [1 8])  = {{ 1.0,             {mean_E1, std_E1}                                      }};     % E->I-L
ConParSTSP([3 10],[1 8])  = {{ 1.0,             {mean_E2, std_E2}                                      }};     % E->I-L-d
ConParSTSP([4 11],[1 8])  = {{ 1.0,             {mean_E2, std_E2}                                      }};     % E->I-CL
ConParSTSP([5 12],[1 8])  = {{ 1.0,             {mean_E1, std_E1}                                      }};     % E->I-CL-AC
ConParSTSP([6 13],[1 8])  = {{ 1.0,             {mean_E1, std_E1}                                      }};     % E->I-CS
ConParSTSP([7 14],[1 8])  = {{ 1.0,             {mean_E2, std_E2}                                      }};     % E->I-F

ConParSTSP([1 8], [2 9])  = {{[0.25 0.50 0.25], {mean_I1, std_I1}, {mean_I2, std_I2}, {mean_I3, std_I3}}};     % I-L    ->E
ConParSTSP([1 8], [3 10]) = {{ 1.0,             {mean_I2, std_I2}                                      }};     % I-L-d  ->E
ConParSTSP([1 8], [4 11]) = {{ 1.0,             {mean_I2, std_I2}                                      }};     % I-CL   ->E
ConParSTSP([1 8], [5 12]) = {{ 1.0,             {mean_I2, std_I2}                                      }};     % I-CL-AC->E
ConParSTSP([1 8], [6 13]) = {{ 1.0,             {mean_I2, std_I2}                                      }};     % I-CS   ->E
ConParSTSP([1 8], [7 14]) = {{ 1.0,             {mean_I2, std_I2}                                      }};     % I-F    ->E

ConParSTSP(2:7, 2:7)      = {{[0.29 0.58 0.13], {mean_I1, std_I1}, {mean_I2, std_I2}, {mean_I3, std_I3}}};     % I->I
ConParSTSP(9:14, 9:14)    = {{[0.29 0.58 0.13], {mean_I1, std_I1}, {mean_I2, std_I2}, {mean_I3, std_I3}}};     % I->I


ConParSTSP(1:14,15)       = {{ 1.0,             {[1 0 0], [0 0 0]}                                     }};     % Inp1
ConParSTSP(1:14,16)       = {{ 1.0,             {[1 0 0], [0 0 0]}                                     }};     % Inp2


ConPar = {ConParIdc,ConParCont,ConParSTSP};                                       % (mean) connection parameters (weights etc.)


% Connection probabilities (for random networks)
pCon = zeros(14,16);                                                              
pCon([1 8], [1 8])  = p(1)*[0.1393    0.0449;    0.2333    0.0806];     % E->E

pCon([2 9], [1 8])  = p(2)*[0.3247    0.1875;    0.0870    0.3331];     % E->I-L
pCon([3 10],[1 8])  = p(2)*[0.3247    0.1875;    0.0870    0.3331];     % E->I-L-d
pCon([4 11],[1 8])  = p(2)*[0.1594    0.0920;    0.0800    0.0800];     % E->I-CL
pCon([5 12],[1 8])  = p(2)*[0.1594    0.0920;    0.0800    0.0800];     % E->I-CL-AC
pCon([6 13],[1 8])  = p(2)*[0.3247    0.1875;    0.0870    0.3331];     % E->I-CS
pCon([7 14],[1 8])  = p(2)*[0.2900    0.1674;    0.1500    0.3619];     % E->I-F

pCon([1 8], [2 9])  = p(3)*[0.4586    0.0991;    0.2130    0.7006];     % I-L    ->E
pCon([1 8], [3 10]) = p(3)*[0.4586    0.0991;    0.2130    0.7006];     % I-L-d  ->E
pCon([1 8], [4 11]) = p(3)*[0.4164    0.0321;    0.1934    0.2271];     % I-CL   ->E
pCon([1 8], [5 12]) = p(3)*[0.4164    0.0321;    0.1934    0.2271];     % I-CL-AC->E
pCon([1 8], [6 13]) = p(3)*[0.4586    0.0991;    0.2130    0.7006];     % I-CS   ->E
pCon([1 8], [7 14]) = p(3)*[0.6765    0.1287;    0.3142    0.9096];     % I-F    ->E

pCon(2:7, 2:7)      = p(4)*0.2500;                                      % I   ->I
pCon(9:14, 9:14)    = p(4)*0.6000;                                      % I   ->I

% IF: Get all existing connectivities to 0.1
% pCon(pCon>0)=0.1;
% pSelfCon = 0.1*ones(size(pSelfCon));


cluster_flag = zeros(14,16);                                % determines whether a common neighbour rule (1=recur, 2=cross) is applied or not (=0)
%cluster_flag(1,1) = 1;                                    % comment out those three lines before using  update_inv_con_PSP
%cluster_flag(8,8) = 1;
%cluster_flag(8,1) = 2;


% Remaining synapse parameter statistics (SD, min, max)           
S_sig = zeros(14,16,2);                                     % standard deviations of connection parameters (NTypesN x NTypesN x #syn. par.)
S_max = zeros(14,16,2);                                     % maxima of connection parameters (NTypesN x NTypesN x #syn. par.)
S_min = zeros(14,16,2);                                     % minima of connection parameters (NTypesN x NTypesN x #syn. par.)

% standard deviations of synaptic weights
S_sig([1 8], [1 8],1)  = [0.4695    0.1375;    0.3530    0.9653];     % E->E

S_sig([2 9], [1 8],1)  = [1.0855    0.6267;    0.8588    1.1194];     % E->I-L
S_sig([3 10],[1 8],1)  = [1.0855    0.6267;    0.8588    1.1194];     % E->I-L-d
S_sig([4 11],[1 8],1)  = [0.1999    0.1154;    0.1581    0.7033];     % E->I-CL
S_sig([5 12],[1 8],1)  = [0.1999    0.1154;    0.1581    0.7033];     % E->I-CL-AC
S_sig([6 13],[1 8],1)  = [1.0855    0.6267;    0.8588    1.1194];     % E->I-CS
S_sig([7 14],[1 8],1)  = [0.2000    0.1155;    0.1582    0.3000];     % E->I-F

S_sig([1 8], [2 9], 1) = [1.9462    0.0362;    0.9038    0.2557];     % I-L    ->E
S_sig([1 8], [3 10],1) = [1.9462    0.0362;    0.9038    0.2557];     % I-L-d  ->E
S_sig([1 8], [4 11],1) = [0.6634    0.0093;    0.3081    0.0655];     % I-CL   ->E
S_sig([1 8], [5 12],1) = [0.6634    0.0093;    0.3081    0.0655];     % I-CL-AC->E
S_sig([1 8], [6 13],1) = [1.9462    0.0362;    0.9038    0.2557];     % I-CS   ->E
S_sig([1 8], [7 14],1) = [3.6531    0.1828;    1.6966    1.2919];     % I-F    ->E

S_sig(2:7, 2:7,1)      =  0.3500;                                     % I->I
S_sig(9:14, 9:14,1)    =  0.3500;                                     % I->I


% standard deviations of synaptic delays
S_sig([1 8], [1 8],2)  = [0.3095    0.1825;    0.1651    0.4350];     % E->E

S_sig( 2:7,  [1 8],2)  = repmat([0.2489    0.0839],length(2:7),1);    % E->I
S_sig( 9:14, [1 8],2)  = repmat([0.1327    0.2000],length(9:14),1);   % E->I

S_sig([1 8],  2:7, 2)  = repmat([0.1786;    0.0940],1,length(2:7));   % I->E
S_sig([1 8],  9:14,2)  = repmat([0.0394;    0.0940],1,length(9:14));     % I->E

S_sig(2:7, 2:7,2)      =  0.4000;                                     % I->I
S_sig(9:14, 9:14,2)    =  0.4000;                                     % I->I

S_max(:,:,1) = 100;     % weight maxima are 100 times the mean (tail for log-normal distribution)
S_max(:,:,2) = 2;       % dtax maxima are twice the mean 


% ------------------------- Set Stripe parameters  ------------------------
if Nstripes>1
    ConParStripes{1} = [1 1; 8 13; 8 14];
else
    ConParStripes{1} = [];
end

ConParStripes{2}{1} = [0.909 1.4587];             % space constant lambda for exponential decay (wgt, p_con) and linear gain a (dtax)
ConParStripes{2}{2} = [0.909 1.0375];
ConParStripes{2}{3} = [0.909 1.0375];

ConParStripes{3}{1} = [-4 -2 2 4];                
ConParStripes{3}{2} = [-1 1];
ConParStripes{3}{3} = [-1 1]; 


% ------------------------- Set Noise parameters  ------------------------
% Parameters: [use tc_rec tc_fac mean_wgt sigma_wgt], one set for each synapse type
a = STypPar(1,1);
b = STypPar(1,3);

mean_back = 0;
std_back = 0;

NoisePar(1,:) = [mean_back*1000*a*0.012/(a+b)  std_back*1000*a^2*0.003/(a^2+b^2)];     % AMPA noise
NoisePar(2,:) = [mean_back*1000*0.057          std_back*1000*0.0066];                  % GABA noise
NoisePar(3,:) = [mean_back*1000*b*0.012/(a+b)  std_back*1000*b^2*0.003/(a^2+b^2)];      % NMDA noise

Y = 0:0.001:1;
X = -sqrt(2)*erfcinv(2*Y);                      % cummulative normal distribution
NoiseDistr = [X(2) X(2:end-1) X(end-1)];        % cut out infinity


% ------------------------------ Set Events  -----------------------------
% Fixed input from Cell(1,:)
if (max(abs(CellPar(1,:)))>0)
    i1 = 1;
    for i=1:length(NTypes)
        EvtMtx(i1:i1+NTypes(i)-1,1) = CellPar(1,i)*ones(NTypes(i),1);
        i1 = i1+NTypes(i);
    end;
    SimPar.EvtMtx = repmat(EvtMtx,Nstripes,1);
    SimPar.EvtTimes = [CtrPar(3); CtrPar(2)];
else
    SimPar.EvtMtx = [];
    SimPar.EvtTimes = [];
end;


% ----------------------------- Set Viewlist  -----------------------------
%ViewList=1:NTypes(1);                                                 % neurons to be recorded in detail
%ViewList = cumsum(NTypes);  % Record last neuron of each type
ViewList = 1;                % Record only the first neuron (default)
%ViewList = 1:N1*Nstripes;   % Record all neurons


% ------------------ Set spike trains for input neurons -------------------
%InpSTtrains(1,:) = -1;  % Empty spike trains
%InpSTtrains(2,:) = -1;

% Poisson distributed input spike trains
lambda = 50.0;                              % frequency 20 Hz
InpLambda = lambda*ones(sum(InpNTypes),1);
N_Inp = round(2*CtrPar(2)/lambda);
InpSTtrains = -1*ones(sum(InpNTypes), N_Inp);
SP_Pois = cumsum(-lambda*log(1-rand(sum(InpNTypes),N_Inp)),2);
for i=1:sum(InpNTypes)
    SP_Pois_act = SP_Pois(i,SP_Pois(i,:)<CtrPar(2));
    InpSTtrains(i,1:length(SP_Pois_act)) = SP_Pois_act;
end;


% ----------------------- Save random state -------------------------------
SimPar.RState = RandStream.getGlobalStream.State;


% ---------------------- Define cell assemblies ---------------------------
%CA(:,1) = [1;2;5;8];    % Indizes of CA members
%CA(:,2) = [1;2;3;4];    % Weight factor for outgoing synapses from each member
CA(:,1) = 1;    % dummy CA
CA(:,2) = 1;


% ----------------------- Pass parameters to SimPar -----------------------
% (lazyness... eh, work efficiency only, less code to change)

SimPar.CtrPar = CtrPar;
SimPar.Nstripes = Nstripes;
SimPar.NTypes = NTypes;
SimPar.InpNTypes = InpNTypes;

SimPar.CellPar = CellPar;
if exist('par_stats', 'var')
    SimPar.N_sig_mod = N_sig_mod;
    SimPar.N_min_mod = N_min_mod;
    SimPar.N_max_mod = N_max_mod;
end
SimPar.k_trans = k_trans;
SimPar.ParCov = ParCov;
SimPar.N_sig = N_sig;
SimPar.N_max = N_max;
SimPar.N_min = N_min;
SimPar.NPList = NPList;

SimPar.STypPar = STypPar;
SimPar.syn_scale = [srec_EE srec_IE srec_EI srec_II s1 s2];
SimPar.ConPar = ConPar;
SimPar.pCon = pCon;
SimPar.cluster_flag = cluster_flag;
SimPar.S_sig = S_sig;
SimPar.S_max = S_max;
SimPar.S_min = S_min;
SimPar.ConParStripes = ConParStripes;

SimPar.ViewList = ViewList;

SimPar.InpSTtrains = InpSTtrains;
SimPar.InpLambda = InpLambda;

SimPar.NoisePar = NoisePar;
SimPar.NoiseDistr = NoiseDistr;

SimPar.V0Par = V0Par;

SimPar.CA = CA;

SimPar.NeuronGroupsSaveArray = NeuronGroupsSaveArray;

if exist('UniqueNum', 'var')
    SimPar.UniqueNum = UniqueNum;
else 
    SimPar.UniqueNum = 0;
end;


% (c) 2016 J. Hass, L. Hertaeg and D. Durstewitz,
% Central Institute of Mental Health, Mannheim University of Heidelberg 
% and BCCN Heidelberg-Mannheim
