
#!/usr/bin/env python
# coding: utf-8

import sys
import math
import time
import numpy as np
from numpy import double, array
import pickle
import random as rndm
rndm.seed(1)
from scipy.spatial import distance

import steps.model as smod
import steps.geom as geom
import steps.utilities.meshio as smeshio
import steps.utilities.meshctrl as meshctrl
import steps.rng as srng
import steps.mpi
import steps.mpi.solver as solver
from steps.solver import UNKNOWN_INDEX


# Command line argument are, in order:
# (1) job_id: ID of job. Used to redirect sys.stdout per job and set the random number seed
# (2) model_path: Path to model components such as docking coordinates and the mesh
# (3) RATE: Firing rate of action potentials
# (4) INT: The simulation endtime
# (5) job_index: Multiplied by job_id to ensure a unique random number seed
# (6) data_path: Path to output data to

job_id = int(sys.argv[1])
model_path = sys.argv[2]
job_index = int(sys.argv[5])
data_path = sys.argv[6]

rng_seed = job_id*job_index


myrank =  steps.mpi.rank

# Only output on rank 0
if myrank == 0:
  f = open(data_path+str(rng_seed)+'_'+str(job_id)+'_stdout.txt', 'w+')
  sys.stdout=f
else:
  sys.stdout = None


mdist=10e-09

#Boolean for either static or mobile Ca channels
cachan_static = True
print('CDK5 kcat = 5')
if cachan_static == True:
    print('CA CHANNELS ARE STATIC')
else:
    print('CA CHANNELS ARE MOBILE')

print('TETHERS ARE ACTIVE')
print('TESTING DOCK SITES 3')
#cachan_static = False

#number of Ca channels
cachan_n = 15

#Choose dock triangle set
dock_tris = [42155, 76431, 84872, 61803, 21332, 71084, 79446, 89381]

#vesicle membrane protein binding
ves_protein_bind_on = True

tomo_init = 1500

can_init = 1500
print('TOTAL CALCINEURIN: ', can_init)

kcat_can_tomo = 1
print('kcat_can_tomo: ', kcat_can_tomo)

kcat_can_init = 0.5
print('kcat_can: '+str( kcat_can_init) )

#firing rate
RATE = int(sys.argv[3])


####################Functions##################
def getRandPosInTri(tri, mesh):
    verts = mesh.getTri(tri)
    v0 = mesh.getVertex(verts[0])
    v1 = mesh.getVertex(verts[1])
    v2 = mesh.getVertex(verts[2])

    u = sqrt(rndm.random())
    v=u*rndm.random()

    return (1-u)*array(v0) + (u-v)*array(v1) + v*array(v2)

tri_fixed_pos={}
def checkDistance(pos, dist):
    if pos[0]**2+pos[1]**2 > 3.6e-14: return False
    for tri in tri_fixed_pos.keys():
        positions =tri_fixed_pos[tri]
        for p in positions:
            distance = sqrt((pos[0]-p[0])**2+(pos[1]-p[1])**2+(pos[2]-p[2])**2)
            if distance<dist: return False
    return True


def getPosInTri(tri, mesh, idx):
    if tri in tri_fixed_pos.keys():
        if len(tri_fixed_pos[tri]) > idx: return tri_fixed_pos[tri][idx]
        else:
            new_pos = getRandPosInTri(tri, mesh)

            tri_fixed_pos[tri].append(new_pos)
            return tri_fixed_pos[tri][idx]
    else:
        tri_fixed_pos[tri]=[]
        assert(idx==0)
        tri_fixed_pos[tri].append(getRandPosInTri(tri, mesh))
        return tri_fixed_pos[tri][idx]

def getRandPosInTet(tet, mesh):
    verts = mesh.getTet(tet)
    v0 = mesh.getVertex(verts[0])
    v1 = mesh.getVertex(verts[1])
    v2 = mesh.getVertex(verts[2])
    v3 = mesh.getVertex(verts[3])

    s=rndm.random()
    t=rndm.random()
    u=rndm.random()

    if (s + t > 1.0):
        s = 1.0 - s
        t = 1.0 - t

    if (t + u > 1.0):
        tmp = u
        u = 1.0 - s - t
        t = 1.0 - tmp
    elif (s + t + u > 1.0):
        tmp = u
        u = s + t + u - 1.0
        s = 1 - t - tmp

    a = 1 - s - t - u

    return a*array(v0) + s*array(v1) + t*array(v2) + u*array(v3)

def SameSide(p1,p2, a,b):
    bsuba = np.subtract(b,a)
    p1suba = np.subtract(p1,a)
    p2suba = np.subtract(p2,a)
    cp1 = np.cross(bsuba, p1suba)
    cp2 = np.cross(bsuba, p2suba)
    if np.dot(cp1, cp2) >= 0:
        return True
    else:
        return False

def PointInTriangle(p, a,b,c):
    if SameSide(p,a, b,c) and SameSide(p,b, a,c) and SameSide(p,c, a,b):
        return True
    else:
        return False

#finding angles etc for defining endocytic zone
def unit_vector(vector):
    """ Returns the unit vector of the vector.  """
    return vector / np.linalg.norm(vector)

def angle_between(v1, v2):
    """ Returns the angle in radians between vectors 'v1' and 'v2'::

            >>> angle_between((1, 0, 0), (0, 1, 0))
            1.5707963267948966
            >>> angle_between((1, 0, 0), (1, 0, 0))
            0.0
            >>> angle_between((1, 0, 0), (-1, 0, 0))
            3.141592653589793
    """
    v1_u = unit_vector(v1)
    v2_u = unit_vector(v2)
    return np.arccos(np.clip(np.dot(v1_u, v2_u), -1.0, 1.0))
#################################### CONSTANTS ####################################

# Number of iterations; plotting dt; sim endtime:
NITER = 1

# The data collection time increment (s)
DT = 0.001

# The simulation endtime (s)
INT = int(sys.argv[4])

# Calcium pulse constants
ca_peak = 0#ca_peak_v

ca_pulse = 0.05

ca_width = 3

# Avogadro constant
Na = 6.02214129e23

start_time = 2000

##########  Calcium Square Pulse Function ##########

def square_pulse_ica(t,max_molar=float(ca_peak)*1e-6, duration=float(ca_width), start=start_time):
    if t < start or t > start + duration: return 0
    else: return max_molar

############# 2nd ORDER RATE CONSTANT CONVERSION FROM VOL TO SURFACE ################
cyto_vol = 2.18e-19 #cubic metres
cyto_vol_L = cyto_vol*1000 #litres

memb_area = 2.06e-12 #square metres

k_scale = memb_area/cyto_vol_L # m^2/L

########## Ca concentrations (M) #########

Ca_oconc = 2e-3
Ca_iconc = 45e-9

######## Temperature dependence (not used) ###########
#Qt = math.pow(Q10, ((TEMPERATURE-23)/10))

########## CaP channels density & permiability per channel ##########

# CaP_P is permeability per channel (m3/s)
# CaP_ro is channel/surface area (/m2)
# P in Ca Dynamics model is 0.95e-4 cm/s --> 0.95e-6 m/s

CaP_P = 2.5e-20
print('Ca Chan Perm: '+str (CaP_P))
CaP_ro = 3.8e13

#membrane SNARE proteins diffusiion scaling factor
#######SCALE SNARE MEM DIFF CONSTANTS#######
snare_diff_scale = 1

#Boolean for either presence or absence of Calmodulin
cam_present = True

########## Initial Membrane Potential #########
init_pot = -60e-3

v_step = [-50e-3, -40e-3, -30e-3, -20e-3, -10e-3, 0, 10e-3, 20e-3, 30e-3, 40e-3]

########## BULK RESISTIVITY ##########

Ra = 235.7*1.0e-2

########## MEMBRANE CAPACITANCE ##########

memb_capac = 1.5e-2

EF_DT = 1.0e-3         # The EField dt

##########CaP channel parameters ####################

#Units (mV)
vhalfm = -29.458
cvm = 8.429

def minf_cap(V):
    #Units (mV)
    vhalfm = -29.458
    cvm = 8.429
    vshift = 0.0

    return (1.0/(1.0 + math.exp(-(V-vhalfm-vshift)/cvm)))

def tau_cap(V):
    vshift = 0.0
    if (V-vshift) >= -40:
        return (0.2702 + 1.1622 * math.exp(-(V+26.798-vshift)*(V+26.798-vshift)/164.19))
    else:
        return (0.6923 * math.exp((V-vshift)/1089.372))

def alpha_cap(V):
    return (minf_cap(V)/tau_cap(V))

def beta_cap(V):
    return ((1.0-minf_cap(V))/tau_cap(V))


##Load docking site coordinates###
dock_x = open(model_path+ '/dock_coords_x_10.txt','r')
dock_x_list = dock_x.readlines()
dock_y = open(model_path+'/dock_coords_y_10.txt','r')
dock_y_list = dock_y.readlines()

dock_x_list = [double(i) for i in dock_x_list]
dock_y_list = [double(i) for i in dock_y_list]

dock_coords = []
dock_n = len(dock_x_list)
for i in range(0, dock_n):
    coord = [dock_x_list[i]*1e-09, dock_y_list[i]*1e-09]
    dock_coords.append(coord)

###Load active zone coordinates
azone_x = open(model_path+'/az_coords_x.txt','r')
azone_x_list = azone_x.readlines()
azone_y = open(model_path+'/az_coords_y.txt','r')
azone_y_list = azone_y.readlines()
azone_z = open(model_path+'/az_coords_z.txt','r')
azone_z_list = azone_z.readlines()

azone_x_list = [double(i) for i in azone_x_list]
azone_y_list = [double(i) for i in azone_y_list]
azone_z_list = [double(i) for i in azone_z_list]

azone_coords_n = len(azone_x_list)

azone_coords = []
#clust_ves_coords.append([])

for i in range(0, azone_coords_n):
    #coord = [azone_x_list[i]*25*1e-09, azone_y_list[i]*25*1e-09, azone_z_list[i]*25*1e-09]
    coord = [azone_x_list[i]*25*1e-09, azone_y_list[i]*25*1e-09]
    azone_coords.append(coord)

###Model Definition###
syt_model = smod.Model()

vsys = smod.Volsys('vsys', syt_model)
syn_sys = smod.Volsys('syn_sys', syt_model)
memsys = smod.Surfsys('memsys', syt_model)
ERsys = smod.Surfsys('ERsys', syt_model)
cytERsys = smod.Volsys('cytERsys', syt_model)

#General cytosolic DCST
DCST_CYT = 2.0e-12 #um^2/s
DCST_CYT_low = 0.028e-12
# Diffusion constant of Calcium
DCST_CA = 0.223e-9
#General membrane DCST
DCST_MEM = 0.25e-12
DCST_RAFT = 0
DCST_MEM_2 = 0.05e-12
DCST_MEM_VES = 0.05e-12
#Ca model
k1_pmca = 1.5e8
k2_pmca = 15
k3_pmca = 12
kl_pmca = 4.3
kon_serca_ca = 17147e6
koff_serca_ca = 8426.3
kflux_serca = 250
kleak = 1.8e-06
kon_pv_ca = 107e06
koff_pv_ca = 0.95
kon_pv_mg = 472e06
koff_pv_mg = 25.0
kon_cbs_ca = 5.5e06
koff_cbs_ca = 2.6
kon_cbf_ca = 43.5e06
koff_cbf_ca = 35.8
kinflux = 0.0

##### Species #####
Ca = smod.Spec('Ca', syt_model)
Ca.setValence(2)
PMCA_P0 = smod.Spec('PMCA_P0', syt_model)
PMCA_P1 = smod.Spec('PMCA_P1', syt_model)
SERCA = smod.Spec('SERCA',syt_model)
CaSERCA = smod.Spec('CaSERCA',syt_model)
Ca2SERCA = smod.Spec('Ca2SERCA',syt_model)
PV = smod.Spec('PV',syt_model)
PV_Ca = smod.Spec('PV_Ca',syt_model)
PV_2Ca = smod.Spec('PV_2Ca',syt_model)
MgPV = smod.Spec('MgPV',syt_model)
Mg2PV = smod.Spec('Mg2PV',syt_model)
Mg = smod.Spec('Mg',syt_model)
CBs = smod.Spec('CBs',syt_model)
CaCBs = smod.Spec('CaCBs',syt_model)
Ca2CBs = smod.Spec('Ca2CBs',syt_model)
CBf = smod.Spec('CBf',syt_model)
CaCBf = smod.Spec('CaCBf',syt_model)
Ca2CBf = smod.Spec('Ca2CBf',syt_model)

#Diffusion rules
diff_Ca = smod.Diff('diff_Ca', vsys, Ca,  DCST_CA)
diff_PV = smod.Diff('diff_PV', vsys, PV,  DCST_CYT)
diff_PV_Ca = smod.Diff('diff_PV_Ca', vsys, PV_Ca,  DCST_CYT)
diff_PV_2Ca = smod.Diff('diff_PV_2Ca', vsys, PV_2Ca,  DCST_CYT)
diff_MgPV = smod.Diff('diff_MgPV', vsys, MgPV,  DCST_CYT)
diff_Mg2PV = smod.Diff('diff_Mg2PV', vsys, Mg2PV,  DCST_CYT)
diff_Mg = smod.Diff('diff_Mg', vsys, Mg,  DCST_CYT)
diff_CBs = smod.Diff('diff_CBs', vsys, CBs,  DCST_CYT)
diff_CaCBs = smod.Diff('diff_CaCBs', vsys, CaCBs,  DCST_CYT)
diff_Ca2CBs = smod.Diff('diff_Ca2CBs', vsys, Ca2CBs,  DCST_CYT)
diff_CBf = smod.Diff('diff_CBf', vsys, CBf,  DCST_CYT)
diff_CaCBf = smod.Diff('diff_CaCBf', vsys, CaCBf,  DCST_CYT)
diff_Ca2CBf = smod.Diff('diff_Ca2CBf', vsys, Ca2CBf,  DCST_CYT)

diff_PMCA_P0 = smod.Diff('diff_PMCA_P0', memsys, PMCA_P0,  DCST_MEM)
diff_PMCA_P1 = smod.Diff('diff_PMCA_P1', memsys, PMCA_P1,  DCST_MEM)

diff_SERCA = smod.Diff('diff_SERCA', ERsys, SERCA,  DCST_MEM)
diff_CaSERCA = smod.Diff('diff_CaSERCA', ERsys, CaSERCA,  DCST_MEM)
diff_Ca2SERCA = smod.Diff('diff_Ca2SERCA', ERsys, Ca2SERCA,  DCST_MEM)


#Reactions
#Ca uncage
Ca_uncage = smod.Reac('Ca_uncage', vsys, rhs=[Ca], kcst = 0)

# Ca + PMCA <->  CaPMCA -> PMCA
Reac1 = smod.SReac('pump2_f', memsys, ilhs = [Ca], slhs =  [PMCA_P0], srhs = [PMCA_P1], kcst = k1_pmca)
Reac2 = smod.SReac('Reac2', memsys, slhs = [PMCA_P1], irhs = [Ca], srhs = [PMCA_P0], kcst = k2_pmca)
Reac3 = smod.SReac('Reac3', memsys, slhs = [PMCA_P1], srhs = [PMCA_P0], kcst = k3_pmca)
PMCA_leak = smod.SReac('PMCA_leak', memsys, slhs=[PMCA_P0], srhs=[PMCA_P0], irhs=[Ca], kcst=kl_pmca)


# Ca + SERCA <->  Ca1SERCA +Ca <->  Ca2SERCA  ->  SERCA
Reac9 = smod.SReac('Reac9', ERsys, olhs = [Ca], slhs = [SERCA], srhs = [CaSERCA])
Reac9.kcst = 17147e6
Reac10 = smod.SReac('Reac10', ERsys, slhs = [CaSERCA], orhs = [Ca], srhs = [SERCA])
Reac10.kcst = 8426.3
Reac11 = smod.SReac('Reac11', ERsys, olhs = [Ca], slhs = [CaSERCA], srhs = [Ca2SERCA])
Reac11.kcst = 17147e6
Reac12 = smod.SReac('Reac12', ERsys, slhs = [Ca2SERCA], orhs = [Ca], srhs = [CaSERCA])
Reac12.kcst = 8426.3
Reac13 = smod.SReac('Reac13', ERsys, slhs = [Ca2SERCA], srhs = [SERCA], irhs = [Ca,Ca])
Reac13.kcst = 250

# ////////////////////////////////////////////////////////////////////////////////////////////////////////


kreac_f_PV_Ca = smod.Reac('kreac_f_PV_Ca', vsys, lhs=[PV, Ca], rhs=[PV_Ca], kcst=107e6)
kreac_b_PV_Ca = smod.Reac('kreac_b_PV_Ca', vsys, lhs=[PV_Ca], rhs=[PV, Ca], kcst=0.95)

kreac_f_PV_2Ca = smod.Reac('kreac_f_PV_2Ca', vsys, lhs=[PV_Ca, Ca], rhs=[PV_2Ca], kcst=107e6)
kreac_b_PV_2Ca = smod.Reac('kreac_b_PV_2Ca', vsys, lhs=[PV_2Ca], rhs=[PV_Ca, Ca], kcst=0.95)

# ////////////////////////////////////////////////////////////////////////////////////////////////////////
CBhi = smod.Spec('CBhi', syt_model)         # CALBINDIN 1 (D-28K) (we consider the 2 (hi-aff) : 2 (lo-aff) scenario)
CBhi_Ca = smod.Spec('CBhi_Ca', syt_model)   # Binding Kinetics of Calbindin-D28k Determined by Flash Photolysis of Caged Ca2 (Nagerl 2000)
CBhi_2Ca = smod.Spec('CBhi_2Ca', syt_model)
CBlo = smod.Spec('CBlo', syt_model)
CBlo_Ca = smod.Spec('CBlo_Ca', syt_model)
CBlo_2Ca = smod.Spec('CBlo_2Ca', syt_model)


kreac_f_CBhi_Ca = smod.Reac('kreac_f_CBhi_Ca', vsys, lhs = [CBhi, Ca], rhs = [CBhi_Ca], kcst = 1.1e7)
kreac_b_CBhi_Ca = smod.Reac('kreac_b_CBhi_Ca', vsys, lhs = [CBhi_Ca], rhs = [CBhi, Ca], kcst = 2.607)

kreac_f_CBhi_2Ca = smod.Reac('kreac_f_CBhi_2Ca', vsys, lhs = [CBhi_Ca, Ca], rhs = [CBhi_2Ca], kcst = 1.1e7)
kreac_b_CBhi_2Ca = smod.Reac('kreac_b_CBhi_2Ca', vsys, lhs = [CBhi_2Ca], rhs = [CBhi_Ca, Ca], kcst = 2.607)

kreac_f_CBlo_Ca = smod.Reac('kreac_f_CBlo_Ca', vsys, lhs = [CBlo, Ca], rhs = [CBlo_Ca], kcst = 8.7e7)
kreac_b_CBlo_Ca = smod.Reac('kreac_b_CBlo_Ca', vsys, lhs = [CBlo_Ca], rhs = [CBlo, Ca], kcst = 35.76)

kreac_f_CBlo_2Ca = smod.Reac('kreac_f_CBlo_2Ca', vsys, lhs = [CBlo_Ca, Ca], rhs = [CBlo_2Ca], kcst = 8.7e7)
kreac_b_CBlo_2Ca = smod.Reac('kreac_b_CBlo_2Ca', vsys, lhs = [CBlo_2Ca], rhs = [CBlo_Ca, Ca], kcst = 35.76)

# ////////////////////////////////////////////////////////////////////////////////////////////////////////
#Calretinin Buffer Model
CRTT     = smod.Spec('CRTT', syt_model)     # pairs of cooperative binding sites MULTIPLY BY TWO to get 4
CRTR_Ca  = smod.Spec('CRTR_Ca', syt_model)
CRRR_2Ca = smod.Spec('CRRR_2Ca', syt_model)
CRind     = smod.Spec('CRind', syt_model)
CRind_Ca  = smod.Spec('CRind_Ca', syt_model) # independent binding site

kon_T  = 1.8e6
koff_T = 53
kon_R  = 3.1e8
koff_R = 20
kon_ind  = 7.3e6
koff_ind = 252

# pair 1 (you MUST multiply its concentration by 2) because WE'VE GOT TWO PAIRS OF COOPERATIVE CA2+ BINDING SITES
kreac_f_CRTT1_Ca = smod.Reac('kreac_f_CRTT1_Ca', vsys, lhs = [CRTT, Ca], rhs = [CRTR_Ca], kcst=2*kon_T) # two domains are free, so we multiply the association rate by 2
kreac_b_CRTT1_Ca = smod.Reac('kreac_b_CRTT1_Ca', vsys, lhs = [CRTR_Ca], rhs = [CRTT, Ca], kcst=koff_T)
kreac_f_CRRR1_2Ca = smod.Reac('kreac_f_CRRR1_2Ca', vsys, lhs = [CRTR_Ca, Ca], rhs = [CRRR_2Ca], kcst=kon_T)
kreac_b_CRRR1_2Ca = smod.Reac('kreac_b_CRRR1_2Ca', vsys, lhs = [CRRR_2Ca], rhs = [CRTR_Ca, Ca], kcst = 2*koff_R) # two domains are occupied, so we multiply the dissociation rate by 2

# independent Ca2+ binding site:
kreac_f_CRind_Ca = smod.Reac('kreac_f_CRind_Ca', vsys, lhs = [CRind, Ca], rhs = [CRind_Ca], kcst=kon_ind)
kreac_b_CRind_Ca = smod.Reac('kreac_b_CRind_Ca', vsys, lhs = [CRind_Ca], rhs = [CRind, Ca], kcst=koff_ind)

#######VESICLES#########
ves_diameter = 40e-9
ves_radius = ves_diameter/2
ves_diff_k = 0.11e-12 #Gaffield 2006 Neuron
ves_diff_k_slow = 0.0011e-12 #Rizzoli

#free vesicles
ves = smod.Vesicle('ves', syt_model, ves_diameter, ves_diff_k)#0.003e-12
#non-diffusable vesicles that will dock at the beginning of the simulation
ves_d = smod.Vesicle('ves_d', syt_model, ves_diameter, 0)#0.003e-12
#recycled vesicles
ves_r = smod.Vesicle('ves_r', syt_model, ves_diameter, ves_diff_k)#0.003e-12
#cluster (immobile) vesicles
ves_c = smod.Vesicle('ves_c', syt_model, ves_diameter, 0)#0.003e-12
ves_ssys = smod.VesSurfsys('ves_ssys', syt_model)

#vesicles to block true vesicles from leaving bouton
block_ves_diameter = 1e-9
block_ves_diff_k = 0
#free vesicles
ves_block = smod.Vesicle('ves_block', syt_model, block_ves_diameter, block_ves_diff_k)

#######Synaptotagmin Model#######
#Parameters
kon_SNARE_syt_CXN_ca1 = 1000e06
koff_SNARE_syt_CXN_ca1 = 1270
koff_SNARE_syt_CXN_ca1_s = 1000 #Millet 2002

kon_SNARE_syt_CXN_ca2 = 1000e06
koff_SNARE_syt_CXN_ca2 = 227670
koff_SNARE_syt_CXN_ca2_s = 2000

kon_SNARE_syt_CXN_ca3 = 1000e06
koff_SNARE_syt_CXN_ca3 = 12370
koff_SNARE_syt_CXN_ca3_s = 5000

kon_SNARE_syt_CXN_bca1 = 1000e06
koff_SNARE_syt_CXN_bca1 = 50000

kon_SNARE_syt_CXN_bca2 = 1000e06
koff_SNARE_syt_CXN_bca2 = 25780

kon_SNARE_syt_CXN_mem  = 2500
koff_SNARE_syt_CXN_mem = 700

#Species
syt = smod.Spec('syt',syt_model)
glu = smod.Spec('glu', syt_model)
SNARE_syt = smod.Spec('SNARE_syt', syt_model)
SNARE_syt_CXN = smod.Spec('SNARE_syt_CXN',syt_model)
SNARE_syt_CXN_Ca = smod.Spec('SNARE_syt_CXN_Ca',syt_model)
SNARE_syt_CXN_Ca2 = smod.Spec('SNARE_syt_CXN_Ca2',syt_model)
SNARE_syt_CXN_Ca3 = smod.Spec('SNARE_syt_CXN_Ca3',syt_model)
SNARE_syt_CXN_bCa = smod.Spec('SNARE_syt_CXN_bCa',syt_model)
SNARE_syt_CXN_bCa2 = smod.Spec('SNARE_syt_CXN_bCa2',syt_model)

SNARE_syt_CXN_Ca_bCa = smod.Spec('SNARE_syt_CXN_Ca_bCa',syt_model)
SNARE_syt_CXN_Ca_bCa2 = smod.Spec('SNARE_syt_CXN_Ca_bCa2',syt_model)
SNARE_syt_CXN_Ca2_bCa = smod.Spec('SNARE_syt_CXN_Ca2_bCa',syt_model)
SNARE_syt_CXN_Ca2_bCa2 = smod.Spec('SNARE_syt_CXN_Ca2_bCa2',syt_model)
SNARE_syt_CXN_Ca3_bCa = smod.Spec('SNARE_syt_CXN_Ca3_bCa',syt_model)
SNARE_syt_CXN_Ca3_bCa2 = smod.Spec('SNARE_syt_CXN_Ca3_bCa2',syt_model)

#Diffusion rules
diff_syt = smod.Diff('diff_syt', memsys, syt,  0.25e-12)
diff_SNARE_syt = smod.Diff('diff_SNARE_syt', memsys, SNARE_syt,  0)
diff_SNARE_syt_CXN = smod.Diff('diff_SNARE_syt_CXN', memsys, SNARE_syt_CXN,  0)
diff_SNARE_syt_CXN_Ca = smod.Diff('diff_SNARE_syt_CXN_Ca', memsys, SNARE_syt_CXN_Ca,  0)
diff_SNARE_syt_CXN_Ca2 = smod.Diff('diff_SNARE_syt_CXN_Ca2', memsys, SNARE_syt_CXN_Ca2,  0)
diff_SNARE_syt_CXN_Ca3 = smod.Diff('diff_SNARE_syt_CXN_Ca3', memsys, SNARE_syt_CXN_Ca3,  0)
diff_SNARE_syt_CXN_bCa = smod.Diff('diff_SNARE_syt_CXN_bCa', memsys, SNARE_syt_CXN_bCa,  0)
diff_SNARE_syt_CXN_bCa2 = smod.Diff('diff_SNARE_syt_CXN_bCa2', memsys, SNARE_syt_CXN_bCa2,  0)
diff_SNARE_syt_CXN_Ca_bCa = smod.Diff('diff_SNARE_syt_CXN_Ca_bCa', memsys, SNARE_syt_CXN_Ca_bCa,  0)
diff_SNARE_syt_CXN_Ca_bCa2 = smod.Diff('diff_SNARE_syt_CXN_Ca_bCa2', memsys, SNARE_syt_CXN_Ca_bCa2,  0)
diff_SNARE_syt_CXN_Ca2_bCa = smod.Diff('diff_SNARE_syt_CXN_Ca2_bCa', memsys, SNARE_syt_CXN_Ca2_bCa,  0)
diff_SNARE_syt_CXN_Ca2_bCa2 = smod.Diff('diff_SNARE_syt_CXN_Ca2_bCa2', memsys, SNARE_syt_CXN_Ca2_bCa2,  0)
diff_SNARE_syt_CXN_Ca3_bCa = smod.Diff('diff_SNARE_syt_CXN_Ca3_bCa', memsys, SNARE_syt_CXN_Ca3_bCa,  0)
diff_SNARE_syt_CXN_Ca3_bCa2 = smod.Diff('diff_SNARE_syt_CXN_Ca3_bCa2', memsys, SNARE_syt_CXN_Ca3_bCa2,  0)
diff_SNARE_syt_CXN_Ca3_bCa2m = smod.Diff('diff_SNARE_syt_CXN_Ca3_bCa2m', memsys, SNARE_syt_CXN_Ca3_bCa2,  0)

#vesicle Diffusion of Synaptotagmin
vdiff_syt = smod.VesSDiff('vdiff_syt', ves_ssys, syt, DCST_MEM_VES)
#Diffusion of glutamate in the cleft
diff_glu = smod.Diff('diff_glu', syn_sys, glu, DCST_CYT)
sdiff_glu = smod.Diff('sdiff_glu', memsys, glu, 0)
#Reactions
SNARE_syt_CXN_ca_on_1 = smod.VesSReac('SNARE_syt_CXN_ca_on_1', ves_ssys, vlhs = [SNARE_syt_CXN], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca], kcst = kon_SNARE_syt_CXN_ca1)
SNARE_syt_CXN_ca_off_1 = smod.VesSReac('SNARE_syt_CXN_ca_off_1', ves_ssys, vlhs = [SNARE_syt_CXN_Ca], vrhs = [SNARE_syt_CXN], orhs = [Ca], kcst = koff_SNARE_syt_CXN_ca1)

SNARE_syt_CXN_ca_on_2 = smod.VesSReac('SNARE_syt_CXN_ca_on_2', ves_ssys, vlhs = [SNARE_syt_CXN_Ca], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca2], kcst = kon_SNARE_syt_CXN_ca2)
SNARE_syt_CXN_ca_off_2 = smod.VesSReac('SNARE_syt_CXN_ca_off_2', ves_ssys, vlhs = [SNARE_syt_CXN_Ca2], vrhs = [SNARE_syt_CXN_Ca], orhs = [Ca], kcst = koff_SNARE_syt_CXN_ca2)

SNARE_syt_CXN_ca_on_3 = smod.VesSReac('SNARE_syt_CXN_ca_on_3', ves_ssys, vlhs = [SNARE_syt_CXN_Ca2], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca3 ], kcst = kon_SNARE_syt_CXN_ca3)
SNARE_syt_CXN_ca_off_3 = smod.VesSReac('SNARE_syt_CXN_ca_off_3', ves_ssys, vlhs = [SNARE_syt_CXN_Ca3], vrhs = [SNARE_syt_CXN_Ca2], orhs = [Ca], kcst = koff_SNARE_syt_CXN_ca3)

SNARE_syt_CXN_ca_on_4 = smod.VesSReac('SNARE_syt_CXN_ca_on_4', ves_ssys, vlhs = [SNARE_syt_CXN], olhs = [Ca], vrhs = [SNARE_syt_CXN_bCa], kcst = kon_SNARE_syt_CXN_bca1)
SNARE_syt_CXN_ca_off_4 = smod.VesSReac('SNARE_syt_CXN_ca_off_4', ves_ssys, vlhs = [SNARE_syt_CXN_bCa], vrhs = [SNARE_syt_CXN], orhs = [Ca], kcst = koff_SNARE_syt_CXN_bca1)

SNARE_syt_CXN_ca_on_5 = smod.VesSReac('SNARE_syt_CXN_ca_on_5', ves_ssys, vlhs = [SNARE_syt_CXN_bCa], olhs = [Ca], vrhs = [SNARE_syt_CXN_bCa2], kcst = kon_SNARE_syt_CXN_bca2)
SNARE_syt_CXN_ca_off_5 = smod.VesSReac('SNARE_syt_CXN_ca_off_5', ves_ssys, vlhs = [SNARE_syt_CXN_bCa2], vrhs = [SNARE_syt_CXN_bCa], orhs = [Ca], kcst = koff_SNARE_syt_CXN_bca2)

SNARE_syt_CXN_ca_on_6 = smod.VesSReac('SNARE_syt_CXN_ca_on_6', ves_ssys, vlhs = [SNARE_syt_CXN_bCa], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca_bCa], kcst = kon_SNARE_syt_CXN_ca1)
SNARE_syt_CXN_ca_off_6 = smod.VesSReac('SNARE_syt_CXN_ca_off_6', ves_ssys, vlhs = [SNARE_syt_CXN_Ca_bCa], vrhs = [SNARE_syt_CXN_bCa], orhs = [Ca], kcst = koff_SNARE_syt_CXN_ca1)

SNARE_syt_CXN_ca_on_7 = smod.VesSReac('SNARE_syt_CXN_ca_on_7', ves_ssys, vlhs = [SNARE_syt_CXN_Ca_bCa], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca2_bCa], kcst = kon_SNARE_syt_CXN_ca2)
SNARE_syt_CXN_ca_off_7 = smod.VesSReac('SNARE_syt_CXN_ca_off_7', ves_ssys, vlhs = [SNARE_syt_CXN_Ca2_bCa], vrhs = [SNARE_syt_CXN_Ca_bCa], orhs = [Ca], kcst = koff_SNARE_syt_CXN_ca2)

SNARE_syt_CXN_ca_on_8 = smod.VesSReac('SNARE_syt_CXN_ca_on_8', ves_ssys, vlhs = [SNARE_syt_CXN_Ca2_bCa], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca3_bCa], kcst = kon_SNARE_syt_CXN_ca3)
SNARE_syt_CXN_ca_off_8 = smod.VesSReac('SNARE_syt_CXN_ca_off_8', ves_ssys, vlhs = [SNARE_syt_CXN_Ca3_bCa], vrhs = [SNARE_syt_CXN_Ca2_bCa], orhs = [Ca], kcst = koff_SNARE_syt_CXN_ca3)

SNARE_syt_CXN_ca_on_9 = smod.VesSReac('SNARE_syt_CXN_ca_on_9', ves_ssys, vlhs = [SNARE_syt_CXN_Ca], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca_bCa], kcst = kon_SNARE_syt_CXN_bca1)
SNARE_syt_CXN_ca_off_9 = smod.VesSReac('SNARE_syt_CXN_ca_off_9', ves_ssys, vlhs = [SNARE_syt_CXN_Ca_bCa], vrhs = [SNARE_syt_CXN_Ca], orhs = [Ca], kcst = koff_SNARE_syt_CXN_bca1)

SNARE_syt_CXN_ca_on_10 = smod.VesSReac('SNARE_syt_CXN_ca_on_10', ves_ssys, vlhs = [SNARE_syt_CXN_Ca_bCa], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca_bCa2], kcst = kon_SNARE_syt_CXN_bca2)
SNARE_syt_CXN_ca_off_10 = smod.VesSReac('SNARE_syt_CXN_ca_off_10', ves_ssys, vlhs = [SNARE_syt_CXN_Ca_bCa2], vrhs = [SNARE_syt_CXN_Ca_bCa], orhs = [Ca], kcst = koff_SNARE_syt_CXN_bca2)

SNARE_syt_CXN_ca_on_11 = smod.VesSReac('SNARE_syt_CXN_ca_on_11', ves_ssys, vlhs = [SNARE_syt_CXN_Ca2], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca2_bCa], kcst = kon_SNARE_syt_CXN_bca1)
SNARE_syt_CXN_ca_off_11 = smod.VesSReac('SNARE_syt_CXN_ca_off_11', ves_ssys, vlhs = [SNARE_syt_CXN_Ca2_bCa], vrhs = [SNARE_syt_CXN_Ca2], orhs = [Ca], kcst = koff_SNARE_syt_CXN_bca1)

SNARE_syt_CXN_ca_on_12 = smod.VesSReac('SNARE_syt_CXN_ca_on_12', ves_ssys, vlhs = [SNARE_syt_CXN_Ca2_bCa], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca2_bCa2], kcst = kon_SNARE_syt_CXN_bca2)
SNARE_syt_CXN_ca_off_12 = smod.VesSReac('SNARE_syt_CXN_ca_off_12', ves_ssys, vlhs = [SNARE_syt_CXN_Ca2_bCa2], vrhs = [SNARE_syt_CXN_Ca2_bCa], orhs = [Ca], kcst = koff_SNARE_syt_CXN_bca2)

SNARE_syt_CXN_ca_on_13 = smod.VesSReac('SNARE_syt_CXN_ca_on_13', ves_ssys, vlhs = [SNARE_syt_CXN_Ca3], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca3_bCa], kcst = kon_SNARE_syt_CXN_bca1)
SNARE_syt_CXN_ca_off_13 = smod.VesSReac('SNARE_syt_CXN_ca_off_13', ves_ssys, vlhs = [SNARE_syt_CXN_Ca3_bCa], vrhs = [SNARE_syt_CXN_Ca3], orhs = [Ca], kcst = koff_SNARE_syt_CXN_bca1)

SNARE_syt_CXN_ca_on_14 = smod.VesSReac('SNARE_syt_CXN_ca_on_14', ves_ssys, vlhs = [SNARE_syt_CXN_Ca3_bCa], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca3_bCa2], kcst = kon_SNARE_syt_CXN_bca2)
SNARE_syt_CXN_ca_off_14 = smod.VesSReac('SNARE_syt_CXN_ca_off_14', ves_ssys, vlhs = [SNARE_syt_CXN_Ca3_bCa2], vrhs = [SNARE_syt_CXN_Ca3_bCa], orhs = [Ca], kcst = koff_SNARE_syt_CXN_bca2)

SNARE_syt_CXN_ca_on_15 = smod.VesSReac('SNARE_syt_CXN_ca_on_15', ves_ssys, vlhs = [SNARE_syt_CXN_bCa2], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca_bCa2], kcst = kon_SNARE_syt_CXN_ca1)
SNARE_syt_CXN_ca_off_15 = smod.VesSReac('SNARE_syt_CXN_ca_off_15', ves_ssys, vlhs = [SNARE_syt_CXN_Ca_bCa2], vrhs = [SNARE_syt_CXN_bCa2], orhs = [Ca], kcst = koff_SNARE_syt_CXN_ca1_s)

SNARE_syt_CXN_ca_on_16 = smod.VesSReac('SNARE_syt_CXN_ca_on_16', ves_ssys, vlhs = [SNARE_syt_CXN_Ca_bCa2], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca2_bCa2], kcst = kon_SNARE_syt_CXN_ca2)
SNARE_syt_CXN_ca_off_16 = smod.VesSReac('SNARE_syt_CXN_ca_off_16', ves_ssys, vlhs = [SNARE_syt_CXN_Ca2_bCa2], vrhs = [SNARE_syt_CXN_Ca_bCa2], orhs = [Ca], kcst = koff_SNARE_syt_CXN_ca2_s)

SNARE_syt_CXN_ca_on_17 = smod.VesSReac('SNARE_syt_CXN_ca_on_17', ves_ssys, vlhs = [SNARE_syt_CXN_Ca2_bCa2], olhs = [Ca], vrhs = [SNARE_syt_CXN_Ca3_bCa2], kcst = kon_SNARE_syt_CXN_ca3)
SNARE_syt_CXN_ca_off_17 = smod.VesSReac('SNARE_syt_CXN_ca_off_17', ves_ssys, vlhs = [SNARE_syt_CXN_Ca3_bCa2], vrhs = [SNARE_syt_CXN_Ca2_bCa2], orhs = [Ca], kcst = koff_SNARE_syt_CXN_ca3_s)



###### CaP channel ##############

CaPchan = smod.Chan('CaPchan', syt_model)

CaP_m0 = smod.ChanState('CaP_m0', syt_model, CaPchan)
CaP_m1 = smod.ChanState('CaP_m1', syt_model, CaPchan)
CaP_m2 = smod.ChanState('CaP_m2', syt_model, CaPchan)
CaP_m3 = smod.ChanState('CaP_m3', syt_model, CaPchan)

if cachan_static == True:
    diff_CaP_m0 = smod.Diff('diff_CaP_m0', memsys, CaP_m0,  0)#0.01e-12)
    diff_CaP_m1 = smod.Diff('diff_CaP_m1', memsys, CaP_m1,  0)#0.01e-12)
    diff_CaP_m2 = smod.Diff('diff_CaP_m2', memsys, CaP_m2,  0)#0.01e-12)
    diff_CaP_m3 = smod.Diff('diff_CaP_m3', memsys, CaP_m3,  0)#0.01e-12)
else:
    diff_CaP_m0 = smod.Diff('diff_CaP_m0', memsys, CaP_m0,  0.01e-12)
    diff_CaP_m1 = smod.Diff('diff_CaP_m1', memsys, CaP_m1,  0.01e-12)
    diff_CaP_m2 = smod.Diff('diff_CaP_m2', memsys, CaP_m2,  0.01e-12)
    diff_CaP_m3 = smod.Diff('diff_CaP_m3', memsys, CaP_m3,  0.01e-12)

## Intitial conditions (proportion of each channel state)

CaP_m0_p = 0.92402
CaP_m1_p = 0.073988
CaP_m2_p = 0.0019748
CaP_m3_p = 1.7569e-05


CaPm0m1 = smod.VDepSReac('CaPm0m1', memsys, slhs = [CaP_m0], srhs = [CaP_m1], k= lambda V: 1.0e3 *3.* alpha_cap(V*1.0e3))
CaPm1m2 = smod.VDepSReac('CaPm1m2', memsys, slhs = [CaP_m1], srhs = [CaP_m2], k= lambda V: 1.0e3 *2.* alpha_cap(V*1.0e3))
CaPm2m3 = smod.VDepSReac('CaPm2m3', memsys, slhs = [CaP_m2], srhs = [CaP_m3], k= lambda V: 1.0e3 *1.* alpha_cap(V*1.0e3))

CaPm3m2 = smod.VDepSReac('CaPm3m2', memsys, slhs = [CaP_m3], srhs = [CaP_m2], k= lambda V: 1.0e3 *3.* beta_cap(V*1.0e3))
CaPm2m1 = smod.VDepSReac('CaPm2m1', memsys, slhs = [CaP_m2], srhs = [CaP_m1], k= lambda V: 1.0e3 *2.* beta_cap(V*1.0e3))
CaPm1m0 = smod.VDepSReac('CaPm1m0', memsys, slhs = [CaP_m1], srhs = [CaP_m0], k= lambda V: 1.0e3 *1.* beta_cap(V*1.0e3))

#Calculate GHK current
OC_CaP = smod.GHKcurr('OC_CaP', memsys, CaP_m3, Ca, virtual_oconc = Ca_oconc, computeflux = True)

#Set single channel permeability
OC_CaP.setP(CaP_P)


#######Vesicle Docking########
#######RIM binds and activates Munc13 homodimer
#species
RIM = smod.Spec('RIM',syt_model)
M13 = smod.Spec('M13',syt_model)
RIM_M13 = smod.Spec('RIM_M13',syt_model)

#parameters
kon_rim_m13 = 10e06
koff_rim_m13 = 0

# Things will get transported by endocytosis- so even though they will be clamped to 0 in some places need to define them in patches
diff_RIM = smod.Diff('diff_RIM', memsys, RIM, 0.0)
diff_M13 = smod.Diff('diff_M13', vsys, M13, 2.2796e-12)
sdiff_M13 = smod.Diff('sdiff_M13', memsys, M13, 0)
diff_RIM_M13 = smod.Diff('diff_RIM_M13', memsys, RIM_M13, 0.0)

#reactions
rim_m13_on = smod.SReac('rim_m13_on', memsys, slhs = [RIM], ilhs = [M13], srhs = [RIM_M13], kcst = kon_rim_m13)
rim_m13_off = smod.SReac('rim_m13_off', memsys, slhs = [RIM_M13], irhs = [M13], srhs = [RIM], kcst = koff_rim_m13)


#######Vesicle docks by Rab3 via RIM_M13 interaction
#species
Rab3 = smod.Spec('Rab3',syt_model)
RIM_M13_Rab3 = smod.Spec('RIM_M13_Rab3',syt_model)

#diff
diff_Rab3 = smod.Diff('diff_Rab3', vsys, Rab3, 1.7793e-12)
sdiff_Rab3 = smod.Diff('sdiff_Rab3', memsys, Rab3, 0.25e-12)
vdiff_Rab3 = smod.VesSDiff('vdiff_rab3', ves_ssys, Rab3, 0.5e-12) #estimated from similar proteins
diff_RIM_M13_Rab3 = smod.Diff('diff_RIM_M13_Rab3', memsys, RIM_M13_Rab3, 0.0)

#parameters
kon_rim_rab3 = 1000e9
koff_rim_rab3 = 0#15

#reactions
dock_on = smod.VesSReac('dock_on', ves_ssys, vlhs = [Rab3], slhs = [RIM_M13], vrhs = [RIM_M13_Rab3], kcst=kon_rim_rab3, immobility=1, max_dist=mdist)
dock_off = smod.VesSReac('dock_off', ves_ssys, vlhs = [RIM_M13_Rab3], vrhs = [Rab3], srhs = [RIM_M13], kcst=koff_rim_rab3, immobility=-1)

####### SNARE COMPLEX FORMATION [PRIMING I]#######
#######Active Munc13 binds Syntaxin-1_Munc18 complex
#species
SYX = smod.Spec('SYX',syt_model)
M18 = smod.Spec('M18',syt_model)
SYX_M18 = smod.Spec('SYX_M18',syt_model)
RIM_M13_Rab3_SYX_M18 = smod.Spec('RIM_M13_Rab3_SYX_M18',syt_model)
RIM_M13_Rab3_SYX_M18_SNP25 = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25',syt_model)

SNP25 = smod.Spec('SNP25',syt_model)
SYB = smod.Spec('SYB',syt_model)

SNARE = smod.Spec('SNARE',syt_model)

#diffs
diff_SYX = smod.Diff('diff_SYX', memsys, SYX, 0.2429e-12)
diff_SYX_M18 = smod.Diff('diff_SYX_M18', memsys, SYX_M18, 0.2429e-12)
diff_M18 = smod.Diff('diff_M18', vsys, M18, 2.2357e-12)
sdiff_M18 = smod.Diff('sdiff_M18', memsys, M18, 0)
diff_RIM_M13_Rab3_SYX_M18= smod.Diff('diff_RIM_M13_Rab3_SYX_M18', memsys, RIM_M13_Rab3_SYX_M18, 0)
diff_RIM_M13_Rab3_SYX_M18_SNP25  = smod.Diff('diff_RIM_M13_Rab3_SYX_M18_SNP25', memsys, RIM_M13_Rab3_SYX_M18_SNP25, 0)
diff_SNP25 = smod.Diff('diff_SNP25', memsys, SNP25, 0.651e-12)
diff_SYB = smod.Diff('diff_SYB', memsys, SYB, 0.25e-12)
diff_SNARE = smod.Diff('diff_SNARE', memsys, SNARE, 0.0)
#vesicle Diffusion of Synaptobrevin
vdiff_syb = smod.VesSDiff('vdiff_syb', ves_ssys, SYB, 0.3245e-12)


#parameters
kon_syx_m18 = 5e06 #Burkhardt 2008
koff_syx_m18 = 0.0011 #Burkhardt 2008
kon_m13_syx = 20e06 #Mezer 2004
koff_m13_syx = 2.6 #Mezer 2004
kon_syx_snp25 = 10e06
koff_syx_snp25 = 1.26
kon_syb_syx = 2.35e06
koff_syb_syx = 0.0047

##Blocking of Syntaxin-1 by MUNC18a
syx_m18_on = smod.SReac('syx_m18_on', memsys, slhs = [SYX], ilhs = [M18], srhs = [SYX_M18], kcst = kon_syx_m18)
syx_m18_off = smod.SReac('syx_m18_off', memsys, slhs = [SYX_M18], irhs = [M18], srhs = [SYX], kcst = koff_syx_m18)
##Munc13 (active form in complex with RIM)  binds and opens syntaxin-1 (which will allow it to bind SNAP25) (Lai 2017)
m13_syx_on = smod.VesSReac('m13_syx_on', ves_ssys, vlhs = [RIM_M13_Rab3], slhs = [SYX_M18], vrhs = [RIM_M13_Rab3_SYX_M18], kcst=kon_m13_syx)
m13_syx_off = smod.VesSReac('m13_syx_off', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18], srhs = [SYX_M18], vrhs = [RIM_M13_Rab3], kcst=koff_m13_syx)


#######SNAP25 is incorporated into the SNARE complex (binds to SYX) (Lai 2017)
syx_snp25_on = smod.VesSReac('syx_snp25_on', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18], slhs = [SNP25], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25], kcst=kon_syx_snp25)
syx_snp25_off = smod.VesSReac('syx_snp25_off', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25], srhs = [SNP25], vrhs = [RIM_M13_Rab3_SYX_M18], kcst=koff_syx_snp25)

#######Synaptobrevin is incorporated into the SNARE complex (binds to SYX_SNP25 complex)
syx_syb_on = smod.VesSReac('syx_syb_on', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25, SYB], vrhs = [SNARE], kcst=kon_syb_syx)
syx_syb_off = smod.VesSReac('syx_syb_off', ves_ssys, vlhs = [SNARE], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25, SYB], kcst=koff_syb_syx)


#######PRIMING II (binding of Synaptotagmin (syt) and complexin (CXN) to SNARE complex)
CXN = smod.Spec('CXN',syt_model)

#Diffusion rules
diff_CXN = smod.Diff('diff_CXN', vsys, CXN, 1.7793e-12)
sdiff_CXN = smod.Diff('sdiff_CXN', memsys, CXN, 0)
#vdiff_syt = smod.VesSDiff('vdiff_syt', ves_ssys, syt, DCST_MEM_VES) (already defined!!)

#Parameters
kon_syt_snare = 57e06 #32e06 #Mezer 2004
koff_syt_snare = 0.24 #9.4 #Mezer 2004

kon_cxn_snare = 30e06 #7e06 #Mezer 2004
koff_cxn_snare = 0.33 #Mezer 2004

##Binding of syt to SNARE
prime_reac_1 = smod.VesSReac('prime_reac_1', ves_ssys, vlhs = [SNARE, syt], vrhs = [SNARE_syt], kcst=kon_syt_snare)
prime_reac_2 = smod.VesSReac('prime_reac_2', ves_ssys, vlhs = [SNARE_syt], vrhs = [SNARE, syt], kcst=koff_syt_snare)

##Binding of CXN to SNARE
prime_reac_3 = smod.VesSReac('prime_reac_3', ves_ssys, vlhs = [SNARE_syt], olhs = [CXN], vrhs = [SNARE_syt_CXN], kcst=kon_cxn_snare)
prime_reac_4 = smod.VesSReac('prime_reac_4', ves_ssys, vlhs = [SNARE_syt_CXN], vrhs = [SNARE_syt], orhs = [CXN], kcst=koff_cxn_snare)

##VESICLE FUSION EVENT (EXOCYTOSIS)
raft_sys = smod.Raftsys('raft_sys', syt_model)
Raft = smod.Raft('Raft', syt_model, 30e-9, DCST_RAFT)

fusion2 = smod.Exocytosis('fusion2', ves_ssys, deps = [SNARE_syt_CXN_Ca3_bCa2, SNARE_syt_CXN_Ca3_bCa2], kcst=3e03)
fusion3 = smod.Exocytosis('fusion3', ves_ssys, deps = [SNARE_syt_CXN_Ca3_bCa2, SNARE_syt_CXN_Ca3_bCa2, SNARE_syt_CXN_Ca3_bCa2], kcst=3e04)
fusion4 = smod.Exocytosis('fusion4', ves_ssys, deps = [SNARE_syt_CXN_Ca3_bCa2, SNARE_syt_CXN_Ca3_bCa2, SNARE_syt_CXN_Ca3_bCa2, SNARE_syt_CXN_Ca3_bCa2], kcst=3e05)

#POST-FUSION SORTING IN RAFTS
aSNAP = smod.Spec('aSNAP', syt_model)
diff_aSNAP = smod.Diff('diff_aSNAP', vsys,aSNAP, 2.1959e-12)
SNARE_aSNAP = smod.Spec('SNARE_aSNAP', syt_model)
NSF = smod.Spec('NSF', syt_model)
diff_NSF = smod.Diff('diff_NSF', vsys, NSF, 2.2788e-12)
cisSNARE = smod.Spec('cisSNARE', syt_model)
SNARE_aSNAP_NSF = smod.Spec('SNARE_aSNAP_NSF', syt_model)
SNARE_aSNAP_NSF_1 = smod.Spec('SNARE_aSNAP_NSF_1', syt_model)
SNARE_aSNAP_NSF_2 = smod.Spec('SNARE_aSNAP_NSF_2', syt_model)
SNARE_aSNAP_NSF_3 = smod.Spec('SNARE_aSNAP_NSF_3', syt_model)
SNARE_aSNAP_NSF_4 = smod.Spec('SNARE_aSNAP_NSF_4', syt_model)
SNARE_aSNAP_NSF_5 = smod.Spec('SNARE_aSNAP_NSF_5', syt_model)
SNARE_aSNAP_NSF_6 = smod.Spec('SNARE_aSNAP_NSF_6', syt_model)
SNARE_aSNAP_NSF_7 = smod.Spec('SNARE_aSNAP_NSF_7', syt_model)
SNARE_aSNAP_NSF_8 = smod.Spec('SNARE_aSNAP_NSF_8', syt_model)
SNARE_aSNAP_NSF_9 = smod.Spec('SNARE_aSNAP_NSF_9', syt_model)
#Factor that signals SNARE has been dismantled (biological nature not established)
SNARE_DISS = smod.Spec('SNARE_DISS', syt_model)

kon_snare_snap = 1.7e05
koff_snare_snap = 0.26
kon_nsf_snap = 20e06
kcat_nsf = 0.116 #Vivona 2013
#
# #initial loss of CXN, Ca,  syt, etc from all SNARE variants
k_rim_diss = 1000e12
snare_diss_reac00 = smod.SReac('snare_diss_reac00', memsys, slhs = [SNARE], srhs = [RIM_M13, cisSNARE], irhs = [Rab3], kcst = k_rim_diss)
snare_diss_reac01 = smod.SReac('snare_diss_reac01', memsys, slhs = [SNARE_syt_CXN_Ca3_bCa2], irhs = [CXN, Ca, Ca, Ca, Ca, Ca], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)
snare_diss_reac02 = smod.SReac('snare_diss_reac02', memsys, slhs = [SNARE_syt_CXN_Ca3_bCa], irhs = [CXN, Ca, Ca, Ca, Ca], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)
snare_diss_reac03 = smod.SReac('snare_diss_reac03', memsys, slhs = [SNARE_syt_CXN_Ca2_bCa2], irhs = [CXN, Ca, Ca, Ca, Ca], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)
snare_diss_reac04 = smod.SReac('snare_diss_reac04', memsys, slhs = [SNARE_syt_CXN_Ca2_bCa], irhs = [CXN, Ca, Ca, Ca], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)
snare_diss_reac05 = smod.SReac('snare_diss_reac05', memsys, slhs = [SNARE_syt_CXN_Ca_bCa2], irhs = [CXN, Ca, Ca, Ca], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)
snare_diss_reac06 = smod.SReac('snare_diss_reac06', memsys, slhs = [SNARE_syt_CXN_Ca_bCa], irhs = [CXN, Ca, Ca], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)
snare_diss_reac07 = smod.SReac('snare_diss_reac07', memsys, slhs = [SNARE_syt_CXN_bCa2], irhs = [CXN, Ca, Ca], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)
snare_diss_reac08 = smod.SReac('snare_diss_reac08', memsys, slhs = [SNARE_syt_CXN_bCa], irhs = [CXN, Ca], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)
snare_diss_reac09 = smod.SReac('snare_diss_reac09', memsys, slhs = [SNARE_syt_CXN_Ca3], irhs = [CXN, Ca, Ca, Ca], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)
snare_diss_reac010 = smod.SReac('snare_diss_reac010', memsys, slhs = [SNARE_syt_CXN_Ca2], irhs = [CXN, Ca, Ca], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)
snare_diss_reac011 = smod.SReac('snare_diss_reac011', memsys, slhs = [SNARE_syt_CXN_Ca], irhs = [CXN, Ca], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)
snare_diss_reac012 = smod.SReac('snare_diss_reac012', memsys, slhs = [SNARE_syt_CXN], irhs = [CXN], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)
snare_diss_reac013 = smod.SReac('snare_diss_reac013', memsys, slhs = [SNARE_syt], srhs = [RIM_M13, cisSNARE, syt, Rab3], kcst = k_rim_diss)

# # #disassembly of SNARE by NSF/aSNAP
snare_diss_reac1 = smod.SReac('snare_diss_reac1', memsys, slhs = [cisSNARE], ilhs = [aSNAP], srhs = [SNARE_aSNAP], kcst = kon_snare_snap)
snare_diss_reac2 = smod.SReac('snare_diss_reac2', memsys, slhs = [SNARE_aSNAP], irhs = [aSNAP], srhs = [SNARE], kcst = koff_snare_snap)
snare_diss_reac3 = smod.SReac('snare_diss_reac3', memsys, slhs = [SNARE_aSNAP], ilhs = [NSF], srhs = [SNARE_aSNAP_NSF], kcst = kon_nsf_snap)
snare_diss_reac4 = smod.SReac('snare_diss_reac4', memsys, slhs = [SNARE_aSNAP_NSF], srhs = [SYB, SNARE_DISS, SYX, SNP25], irhs = [NSF, aSNAP, M18], kcst = kcat_nsf)

###############DYNAMIN MODEL###################
#Calmodulin calcium model
#Faas 2011 CaM Ca Binding model

CaM_N_0_C_0 = smod.Spec('CaM_N_0_C_0', syt_model)
CaM_N_1_C_0 = smod.Spec('CaM_N_1_C_0', syt_model)
CaM_N_0_C_1 = smod.Spec('CaM_N_0_C_1', syt_model)
CaM_N_1_C_1 = smod.Spec('CaM_N_1_C_1', syt_model)
CaM_N_0_C_2 = smod.Spec('CaM_N_0_C_2', syt_model)
CaM_N_2_C_0 = smod.Spec('CaM_N_2_C_0', syt_model)
CaM_N_2_C_1 = smod.Spec('CaM_N_2_C_1', syt_model)
CaM_N_1_C_2 = smod.Spec('CaM_N_1_C_2', syt_model)
CaM_N_2_C_2 = smod.Spec('CaM_N_2_C_2', syt_model)


diff_CaM_N_0_C_0 = smod.Diff('diff_CaM_N_0_C_0', vsys, CaM_N_0_C_0,  1.6496e-12)
diff_CaM_N_1_C_0 = smod.Diff('diff_CaM_N_1_C_0', vsys, CaM_N_1_C_0,  1.6496e-12)
diff_CaM_N_0_C_1 = smod.Diff('diff_CaM_N_0_C_1', vsys, CaM_N_0_C_1,  1.6496e-12)
diff_CaM_N_1_C_1 = smod.Diff('diff_CaM_N_1_C_1', vsys, CaM_N_1_C_1,  1.6496e-12)
diff_CaM_N_0_C_2 = smod.Diff('diff_CaM_N_0_C_2', vsys, CaM_N_0_C_2,  1.6496e-12)
diff_CaM_N_2_C_0 = smod.Diff('diff_CaM_N_2_C_0', vsys, CaM_N_2_C_0,  1.6496e-12)
diff_CaM_N_2_C_1 = smod.Diff('diff_CaM_N_2_C_1', vsys, CaM_N_2_C_1,  1.6496e-12)
diff_CaM_N_1_C_2 = smod.Diff('diff_CaM_N_1_C_2', vsys, CaM_N_1_C_2,  1.6496e-12)
diff_CaM_N_2_C_2 = smod.Diff('diff_CaM_N_2_C_2', vsys, CaM_N_2_C_2,  1.6496e-12)

kon_cam_NT = 7.7e08
kon_cam_NR = 3.2e10
kon_cam_CT = 8.4e07
kon_cam_CR = 2.5e07

koff_cam_NT = 1.6e05
koff_cam_CT = 2.2e04
koff_cam_NR = 2.6e03
koff_cam_CR = 6.5

#ON reacs
cam_reac_1 = smod.Reac('cam_reac_1', vsys, lhs = [CaM_N_0_C_0, Ca], rhs = [CaM_N_1_C_0], kcst = 2*kon_cam_NT)
cam_reac_2 = smod.Reac('cam_reac_2', vsys, lhs = [CaM_N_0_C_0, Ca], rhs = [CaM_N_0_C_1], kcst = 2*kon_cam_CT)
cam_reac_3 = smod.Reac('cam_reac_3', vsys, lhs = [CaM_N_1_C_0, Ca], rhs = [CaM_N_1_C_1], kcst = 2*kon_cam_CT)
cam_reac_4 = smod.Reac('cam_reac_4', vsys, lhs = [CaM_N_1_C_0, Ca], rhs = [CaM_N_2_C_0], kcst = kon_cam_NR)
cam_reac_5 = smod.Reac('cam_reac_5', vsys, lhs = [CaM_N_0_C_1, Ca], rhs = [CaM_N_0_C_2], kcst = kon_cam_CR)
cam_reac_6 = smod.Reac('cam_reac_6', vsys, lhs = [CaM_N_0_C_1, Ca], rhs = [CaM_N_1_C_1], kcst = 2*kon_cam_NT)
cam_reac_7 = smod.Reac('cam_reac_7', vsys, lhs = [CaM_N_1_C_1, Ca], rhs = [CaM_N_2_C_1], kcst = kon_cam_NR)
cam_reac_8 = smod.Reac('cam_reac_8', vsys, lhs = [CaM_N_1_C_1, Ca], rhs = [CaM_N_1_C_2], kcst = kon_cam_CR)
cam_reac_9 = smod.Reac('cam_reac_9', vsys, lhs = [CaM_N_2_C_0, Ca], rhs = [CaM_N_2_C_1], kcst = 2*kon_cam_CT)
cam_reac_10 = smod.Reac('cam_reac_10', vsys, lhs = [CaM_N_0_C_2, Ca], rhs = [CaM_N_1_C_2], kcst = 2*kon_cam_NT)
cam_reac_11 = smod.Reac('cam_reac_11', vsys, lhs = [CaM_N_1_C_1, Ca], rhs = [CaM_N_2_C_1], kcst = kon_cam_NR)
cam_reac_12 = smod.Reac('cam_reac_12', vsys, lhs = [CaM_N_1_C_1, Ca], rhs = [CaM_N_1_C_2], kcst = kon_cam_CR)
cam_reac_13 = smod.Reac('cam_reac_13', vsys, lhs = [CaM_N_2_C_1, Ca], rhs = [CaM_N_2_C_2], kcst = kon_cam_CR)
cam_reac_14 = smod.Reac('cam_reac_14', vsys, lhs = [CaM_N_1_C_2, Ca], rhs = [CaM_N_2_C_2], kcst = kon_cam_NR)

#OFF reacs
cam_reac_15 = smod.Reac('cam_reac_15', vsys, lhs = [CaM_N_1_C_0], rhs = [CaM_N_0_C_0, Ca], kcst = koff_cam_NT)
cam_reac_16 = smod.Reac('cam_reac_16', vsys, lhs = [CaM_N_0_C_1], rhs = [CaM_N_0_C_0, Ca], kcst = koff_cam_CT)
cam_reac_17 = smod.Reac('cam_reac_17', vsys, lhs = [CaM_N_1_C_1], rhs = [CaM_N_0_C_1, Ca], kcst = koff_cam_NT)
cam_reac_18 = smod.Reac('cam_reac_18', vsys, lhs = [CaM_N_1_C_1], rhs = [CaM_N_1_C_0, Ca], kcst = koff_cam_CT)
cam_reac_19 = smod.Reac('cam_reac_19', vsys, lhs = [CaM_N_2_C_0], rhs = [CaM_N_1_C_0, Ca], kcst = 2*koff_cam_NR)
cam_reac_20 = smod.Reac('cam_reac_20', vsys, lhs = [CaM_N_0_C_2], rhs = [CaM_N_0_C_1, Ca], kcst = 2*koff_cam_CR)
cam_reac_21 = smod.Reac('cam_reac_21', vsys, lhs = [CaM_N_2_C_1], rhs = [CaM_N_1_C_1, Ca], kcst = 2*koff_cam_NR)
cam_reac_22 = smod.Reac('cam_reac_22', vsys, lhs = [CaM_N_2_C_1], rhs = [CaM_N_2_C_0, Ca], kcst = koff_cam_CT)
cam_reac_23 = smod.Reac('cam_reac_23', vsys, lhs = [CaM_N_1_C_2], rhs = [CaM_N_0_C_2, Ca], kcst = koff_cam_NT)
cam_reac_24 = smod.Reac('cam_reac_24', vsys, lhs = [CaM_N_1_C_2], rhs = [CaM_N_1_C_1, Ca], kcst = 2*koff_cam_CR)
cam_reac_25 = smod.Reac('cam_reac_25', vsys, lhs = [CaM_N_2_C_2], rhs = [CaM_N_1_C_2, Ca], kcst = 2*koff_cam_NR)
cam_reac_26 = smod.Reac('cam_reac_26', vsys, lhs = [CaM_N_2_C_2], rhs = [CaM_N_2_C_1, Ca], kcst = 2*koff_cam_CR)

####Calcineurin binding to CaM (Quintana 2005)
CaN = smod.Spec('CaN', syt_model)
CaN_CaM = smod.Spec('CaN_CaM', syt_model)
diff_CaN = smod.Diff('diff_CaN', vsys, CaN,  DCST_CYT)
diff_CaN_CaM = smod.Diff('diff_CaN_CaM', vsys, CaN_CaM,  DCST_CYT)

kon_can_cam = 46e06
koff_can_cam = 1.2

kon_can_dyn = 10e06
koff_can_dyn = 1
kcat_can = 0.5

#Calcineurin binding to CaM
can_reac_1 = smod.Reac('can_reac_1', vsys, lhs = [CaN, CaM_N_2_C_2], rhs = [CaN_CaM], kcst = kon_can_cam)
can_reac_2 = smod.Reac('can_reac_2', vsys, lhs = [CaN_CaM], rhs = [CaN, CaM_N_2_C_2], kcst = koff_can_cam)

#Calcineurin dephosphorylation of dynamin
DYNpp = smod.Spec('DYNpp', syt_model)
DYNp = smod.Spec('DYNp', syt_model)
DYN = smod.Spec('DYN', syt_model)
CaN_DYNpp = smod.Spec('CaN_DYNpp', syt_model)
CaN_DYNp = smod.Spec('CaN_DYNp', syt_model)
diff_DYNpp = smod.Diff('diff_DYNpp', vsys, DYNpp,  3.2464e-12)
sdiff_DYNpp = smod.Diff('sdiff_DYNpp', memsys, DYNpp,  3.2464e-12)
diff_DYNp = smod.Diff('diff_DYNp', vsys, DYNp,  3.2464e-12)
sdiff_DYNp = smod.Diff('sdiff_DYNp', memsys, DYNp,  3.2464e-12)
diff_DYN = smod.Diff('diff_DYN', vsys, DYN,  3.2464e-12)
sdiff_DYN = smod.Diff('sdiff_DYN', memsys, DYN,  3.2464e-12)
sdiff_CaN_DYNpp = smod.Diff('sdiff_CaN_DYNpp', memsys, CaN_DYNpp,  3.2464e-12)
sdiff_CaN_DYNp = smod.Diff('sdiff_CaN_DYNp', memsys, CaN_DYNp,  3.2464e-12)


#cytosolic dephosphorylation
can_reac_3a = smod.Reac('can_reac_3a', vsys, lhs = [CaN_CaM, DYNpp], rhs = [CaN_DYNpp], kcst = kon_can_dyn)
can_reac_3b = smod.Reac('can_reac_3b', vsys, lhs = [CaN_DYNpp], rhs = [CaN_CaM, DYNpp], kcst = koff_can_dyn)
can_reac_3c = smod.Reac('can_reac_3c', vsys, lhs = [CaN_DYNpp], rhs = [CaN_CaM, DYNp], kcst = kcat_can)

can_reac_4a = smod.Reac('can_reac_4a', vsys, lhs = [CaN_CaM, DYNp], rhs = [CaN_DYNp], kcst = kon_can_dyn)
can_reac_4b = smod.Reac('can_reac_4b', vsys, lhs = [CaN_DYNp], rhs = [CaN_CaM, DYNp], kcst = koff_can_dyn)
can_reac_4c = smod.Reac('can_reac_4c', vsys, lhs = [CaN_DYNp], rhs = [CaN_CaM, DYN], kcst = kcat_can)

#raft dephosphorylation
can_reac_3a_raft = smod.RaftSReac('can_reac_3a_raft', raft_sys, ilhs = [CaN_CaM], rlhs = [DYNpp], rrhs = [CaN_DYNpp], kcst = kon_can_dyn)
can_reac_3b_raft = smod.RaftSReac('can_reac_3b_raft', raft_sys, rlhs = [CaN_DYNpp], irhs = [CaN_CaM], rrhs = [DYNpp], kcst = koff_can_dyn)
can_reac_3c_raft = smod.RaftSReac('can_reac_3c_raft', raft_sys, rlhs = [CaN_DYNpp], irhs = [CaN_CaM], rrhs = [DYNp], kcst = kcat_can)

can_reac_4a_raft = smod.RaftSReac('can_reac_4a_raft', raft_sys, ilhs = [CaN_CaM], rlhs = [DYNp], rrhs = [CaN_DYNp], kcst = kon_can_dyn)
can_reac_4b_raft = smod.RaftSReac('can_reac_4b_raft', raft_sys, rlhs = [CaN_DYNp], irhs = [CaN_CaM], rrhs = [DYNp], kcst = koff_can_dyn)
can_reac_4c_raft = smod.RaftSReac('can_reac_4c_raft', raft_sys, rlhs = [CaN_DYNp], irhs = [CaN_CaM], rrhs = [DYN], kcst = kcat_can)

#binding of dynamin to syndapin 1 (SYN1)
SYN1 = smod.Spec('SYN1', syt_model)
DYN_SYN1 = smod.Spec('DYN_SYN1', syt_model)

diff_SYN1 = smod.Diff('diff_SYN1', vsys, SYN1,  2.1959e-12)
diff_DYN_SYN1 = smod.Diff('diff_DYN_SYN1', vsys, DYN_SYN1,  3.2464e-12)
sdiff_DYN_SYN1 = smod.Diff('sdiff_DYN_SYN1', memsys, DYN_SYN1,  3.2464e-12)

kon_dyn_syn = 1000e06
koff_dyn_syn = 1

kon_dyn_mem = 0#1000e06
koff_dyn_mem = 0

dyn_syn_on = smod.Reac('dyn_syn_on', vsys, lhs = [DYN, SYN1], rhs = [DYN_SYN1], kcst = kon_dyn_syn)
dyn_syn_off = smod.Reac('dyn_syn_off', vsys, lhs = [DYN_SYN1], rhs = [DYN, SYN1], kcst = koff_dyn_syn)

dyn_mem_on = smod.SReac('dyn_mem_on', memsys, ilhs = [DYN_SYN1], srhs = [DYN_SYN1], kcst = kon_dyn_mem)
dyn_mem_off = smod.SReac('dyn_mem_off', memsys, slhs = [DYN_SYN1], irhs = [DYN_SYN1], kcst = koff_dyn_mem)

dyn_mem_on_2 = smod.SReac('dyn_mem_on_2', memsys, ilhs = [DYNp], srhs = [DYNp], kcst = kon_dyn_mem)
dyn_mem_off_2 = smod.SReac('dyn_mem_off_2', memsys, slhs = [DYNp], irhs = [DYNp], kcst = koff_dyn_mem)

dyn_mem_on_3 = smod.SReac('dyn_mem_on_3', memsys, ilhs = [DYN], srhs = [DYN], kcst = kon_dyn_mem)
dyn_mem_off_3 = smod.SReac('dyn_mem_off_3', memsys, slhs = [DYN], irhs = [DYN], kcst = koff_dyn_mem)

#Raft binding of dynamin to syndapin 1 (SYN1)
dyn_syn_on_raft = smod.RaftSReac('dyn_syn_on_raft', raft_sys, ilhs = [SYN1], rlhs = [DYN], rrhs = [DYN_SYN1], kcst = kon_dyn_syn)
dyn_syn_off_raft = smod.RaftSReac('dyn_syn_off_raft', raft_sys, rlhs = [DYN_SYN1], irhs = [SYN1], rrhs = [DYN], kcst = koff_dyn_syn)
#CDK5 phosphorylation of dynamin
CDK5 = smod.Spec('CDK5', syt_model)
DYN_CDK5 = smod.Spec('DYN_CDK5', syt_model)
DYNp_CDK5 = smod.Spec('DYNp_CDK5', syt_model)
DYN_SYN1_CDK5 = smod.Spec('DYN_SYN1_CDK5', syt_model)

diff_CDK5 = smod.Diff('diff_CDK5', vsys, CDK5,  DCST_CYT)
diff_DYN_CDK5 = smod.Diff('diff_DYN_CDK5', vsys, DYN_CDK5,  DCST_CYT)
diff_DYNp_CDK5 = smod.Diff('diff_DYNp_CDK5', vsys, DYNp_CDK5,  DCST_CYT)
diff_DYN_SYN1_CDK5 = smod.Diff('diff_DYN_SYN1_CDK5', vsys, DYN_SYN1_CDK5,  DCST_CYT)
sdiff_DYN_SYN1_CDK5 = smod.Diff('sdiff_DYN_SYN1_CDK5', memsys, DYN_SYN1_CDK5,  DCST_MEM)


kon_cdk_dyn = 3e06
koff_cdk_dyn = 10
kcat_cdk = 0.5

#phosphorylation of dynamin in the cytosol
dyn_phos_1 = smod.Reac('dyn_phos_1', vsys, lhs = [DYN, CDK5], rhs = [DYN_CDK5], kcst = kon_cdk_dyn)
dyn_phos_2 = smod.Reac('dyn_phos_2', vsys, lhs = [DYN_CDK5], rhs = [DYN, CDK5], kcst = koff_cdk_dyn)
dyn_phos_3 = smod.Reac('dyn_phos_3', vsys, lhs = [DYN_CDK5], rhs = [DYNp, CDK5], kcst = kcat_cdk)

dyn_phos_1a = smod.Reac('dyn_phos_1a', vsys, lhs = [DYN_SYN1, CDK5], rhs = [DYN_SYN1_CDK5], kcst = kon_cdk_dyn)
dyn_phos_2a = smod.Reac('dyn_phos_2a', vsys, lhs = [DYN_SYN1_CDK5], rhs = [DYN_SYN1, CDK5], kcst = koff_cdk_dyn)
dyn_phos_3a = smod.Reac('dyn_phos_3a', vsys, lhs = [DYN_SYN1_CDK5], rhs = [DYNp, SYN1, CDK5], kcst = kcat_cdk)

#phosphorylation of dynamin in the membrane
dyn_phos_1a = smod.SReac('dyn_phos_1a', memsys, slhs = [DYN_SYN1], ilhs = [CDK5], srhs = [DYN_SYN1_CDK5], kcst = kon_cdk_dyn)
dyn_phos_2a = smod.SReac('dyn_phos_2a', memsys, slhs = [DYN_SYN1_CDK5], srhs = [DYN_SYN1], irhs = [CDK5], kcst = koff_cdk_dyn)
dyn_phos_3a = smod.SReac('dyn_phos_3a', memsys, slhs = [DYN_SYN1_CDK5], irhs = [DYNp, SYN1, CDK5], kcst = kcat_cdk)

dyn_phos_4a = smod.SReac('dyn_phos_4a', memsys, slhs = [DYN, CDK5], srhs = [DYN_CDK5], kcst = kon_cdk_dyn)
dyn_phos_5a = smod.SReac('dyn_phos_5a', memsys, slhs = [DYN_CDK5], srhs = [DYN, CDK5], kcst = koff_cdk_dyn)
dyn_phos_6a = smod.SReac('dyn_phos_6a', memsys, slhs = [DYN_CDK5], srhs = [DYNp, CDK5], kcst = kcat_cdk)

#Reclustering of vesicle proteins in rafts
#Syb adaptor AP180
AP180 = smod.Spec('AP180',syt_model)
kon_syb_ap180 = 2000e06
ap180_on = smod.RaftSReac('ap180_on', raft_sys, slhs = [SYB], rlhs = [AP180], rrhs = [SYB], kcst=kon_syb_ap180)
#Syt adaptor AP2
AP2 = smod.Spec('AP2',syt_model)
kon_syt_ap2 = 2000e06
ap2_on = smod.RaftSReac('ap2_on', raft_sys, slhs = [syt], rlhs = [AP2], rrhs = [syt], kcst=kon_syt_ap2)
#Rab3 adaptor RabAd
RabAd = smod.Spec('RabAd',syt_model)
kon_rab3_rabad = 2000e06
rabad_on = smod.RaftSReac('rabad_on', raft_sys, slhs = [Rab3], rlhs = [RabAd], rrhs = [Rab3], kcst=kon_rab3_rabad)
#attachment of dynamin to raft
DYN_AD = smod.Spec('DYN_AD',syt_model)
kon_dyn_raft = 2000e06
koff_dyn_raft = 0
dyn_raft_on = smod.RaftSReac('dyn_raft_on', raft_sys, slhs = [DYN_SYN1], rlhs = [DYN_AD], rrhs = [DYN_SYN1], kcst=kon_dyn_raft)
dyn_raft_on_2 = smod.RaftSReac('dyn_raft_on_2', raft_sys, slhs = [DYNp], rlhs = [DYN_AD], rrhs = [DYNp], kcst=kon_dyn_raft)
dyn_raft_on_3 = smod.RaftSReac('dyn_raft_on_3', raft_sys, slhs = [DYN], rlhs = [DYN_AD], rrhs = [DYN], kcst=kon_dyn_raft)
dyn_raft_off = smod.RaftSReac('dyn_raft_off', raft_sys, rlhs = [DYN_SYN1], srhs = [DYN_SYN1], rrhs = [DYN_AD], kcst=koff_dyn_raft)
dyn_raft_off_2 = smod.RaftSReac('dyn_raft_off_2', raft_sys, rlhs = [DYNp], srhs = [DYNp], rrhs = [DYN_AD], kcst=koff_dyn_raft)
dyn_raft_off_3 = smod.RaftSReac('dyn_raft_off_3', raft_sys, rlhs = [DYN], srhs = [DYN], rrhs = [DYN_AD], kcst=koff_dyn_raft)
##formation of new vesicle by endocytosis
raftendo = smod.RaftEndocytosis('raftendo', raft_sys, irhs = ves, deps = 68*[SYB]+26*[syt]+10*[Rab3]+[DYN_SYN1], kcst=1000)

#removal of DYN_SYN1 from newly endocytosed vesicle
dynamin_ves__off = smod.VesSReac('dynamin_ves__off', ves_ssys, vlhs = [DYN_SYN1], orhs = [DYNp, SYN1], kcst = 1e9)

sdiff_AP2 = smod.Diff('sdiff_AP2', memsys, AP2, 0)
sdiff_RabAd = smod.Diff('sdiff_RabAd', memsys, RabAd, 0)
sdiff_DYN_AD = smod.Diff('sdiff_DYN_AD', memsys, DYN_AD, 0)
sdiff_AP180 = smod.Diff('sdiff_AP180', memsys, AP180, 0)

################Adenylyl Cyclase Activation Model##################
AC18 = smod.Spec('AC18', syt_model) #0.5e-6
AC18_CaM = smod.Spec('AC18_CaM', syt_model)
cAMP = smod.Spec('cAMP', syt_model)

diff_AC18 = smod.Diff('diff_AC18', vsys, AC18,  DCST_CYT)
diff_AC18_CaM = smod.Diff('diff_AC18_CaM', vsys, AC18_CaM,  DCST_CYT)
diff_cAMP = smod.Diff('diff_cAMP', vsys, cAMP,  DCST_CYT)

#Parameters
kon_ac_cam = 500e06
koff_ac_cam = 0.1
kcat_ac18 = 18
kdeg_camp = 1
#Reactions
ac_cam_on = smod.Reac('ac_cam_on', vsys, lhs = [CaM_N_2_C_2, AC18], rhs = [AC18_CaM], kcst = kon_ac_cam)
ac_cam_off = smod.Reac('ac_cam_off', vsys, lhs = [AC18_CaM], rhs = [CaM_N_2_C_2, AC18], kcst = koff_ac_cam)
cAMP_synth = smod.Reac('cAMP_synth', vsys, lhs = [AC18_CaM], rhs = [AC18_CaM, cAMP], kcst = kcat_ac18)
cAMP_deg = smod.Reac('cAMP_deg', vsys, lhs = [cAMP], rhs = [], kcst = kdeg_camp)

##############################PKA Activation Model##############################
##########PKA Activation Model########
#Species
PKA = smod.Spec('PKA', syt_model)
R2C2 = smod.Spec('R2C2', syt_model)#0.2e-06
R2C2_cAMP = smod.Spec('R2C2_cAMP', syt_model)
R2C2_2cAMPbb = smod.Spec('R2C2_2cAMPbb', syt_model)
R2C2_2cAMPab = smod.Spec('R2C2_2cAMPab', syt_model)
R2C2_3cAMP = smod.Spec('R2C2_3cAMP', syt_model)
R2C2_4cAMP = smod.Spec('R2C2_4cAMP', syt_model)
R2C_2cAMPab = smod.Spec('R2C_2cAMPab', syt_model)
R2C_3cAMP = smod.Spec('R2C_3cAMP', syt_model)
R2C_4cAMP = smod.Spec('R2C_4cAMP', syt_model)
R2_4cAMP = smod.Spec('R2_4cAMP', syt_model)

diff_PKA = smod.Diff('diff_PKA', vsys, PKA,  DCST_CYT)
diff_R2C2 = smod.Diff('diff_R2C2', vsys, R2C2,  DCST_CYT)
diff_R2C2_cAMP = smod.Diff('diff_R2C2_cAMP', vsys, R2C2_cAMP,  DCST_CYT)
diff_R2C2_2cAMPbb = smod.Diff('diff_R2C2_2cAMPbb', vsys, R2C2_2cAMPbb,  DCST_CYT)
diff_R2C2_2cAMPab = smod.Diff('diff_R2C2_2cAMPab', vsys, R2C2_2cAMPab,  DCST_CYT)
diff_R2C2_3cAMP = smod.Diff('diff_R2C2_3cAMP', vsys, R2C2_3cAMP,  DCST_CYT)
diff_R2C2_4cAMP = smod.Diff('diff_R2C2_4cAMP', vsys, R2C2_4cAMP,  DCST_CYT)
diff_R2C_2cAMPab = smod.Diff('diff_R2C_2cAMPab', vsys, R2C_2cAMPab,  DCST_CYT)
diff_R2C_3cAMP = smod.Diff('diff_R2C_3cAMP', vsys, R2C_3cAMP,  DCST_CYT)
diff_R2C_4cAMP = smod.Diff('diff_R2C_4cAMP', vsys, R2C_4cAMP,  DCST_CYT)
diff_R2_4cAMP  = smod.Diff('diff_R2_4cAMP', vsys, R2_4cAMP ,  DCST_CYT)

#Parameters
kon_r2c2_camp1 = 2e06
koff_r2c2_camp1 = 0.75

kon_r2c2_camp2bb = 1e06
koff_r2c2_camp2bb = 1.5

kon_r2c2_camp2ab = 10e06
koff_r2c2_camp2ab = 7.5

kon_r2c2_camp3bb = 20e06
koff_r2c2_camp3bb = 7.5

kon_r2c2_camp3ab = 1e06
koff_r2c2_camp3ab = 0.75

kon_r2c2_camp4 = 10e06
koff_r2c2_camp4 = 15

kact_pka_23 = 0.005
kdeact_pka_23 = 5e06

kact_pka_4 = 6
kdeact_pka_4 = 5e06

kon_r2c_camp3 = 1e06
koff_r2c_camp3 = 0.75

kon_r2c_camp4 = 10e06
koff_r2c_camp4 = 7.5

kact_pka_r2c = 3
kdeact_pka_r2c = 10e06

#Reactions
pka_reac_1 = smod.Reac('pka_reac_1', vsys, lhs = [R2C2, cAMP], rhs = [R2C2_cAMP], kcst = kon_r2c2_camp1)
pka_reac_2 = smod.Reac('pka_reac_2', vsys, lhs = [R2C2_cAMP], rhs = [R2C2, cAMP], kcst = koff_r2c2_camp1)
pka_reac_3 = smod.Reac('pka_reac_3', vsys, lhs = [R2C2_cAMP, cAMP], rhs = [R2C2_2cAMPbb], kcst = kon_r2c2_camp2bb)
pka_reac_4 = smod.Reac('pka_reac_4', vsys, lhs = [R2C2_2cAMPbb], rhs = [R2C2_cAMP, cAMP], kcst = koff_r2c2_camp2bb)
pka_reac_5 = smod.Reac('pka_reac_5', vsys, lhs = [R2C2_cAMP, cAMP], rhs = [R2C2_2cAMPab], kcst = kon_r2c2_camp2ab)
pka_reac_6  = smod.Reac('pka_reac_6', vsys, lhs = [R2C2_2cAMPab], rhs = [R2C2_cAMP, cAMP], kcst = koff_r2c2_camp2ab)
pka_reac_7 = smod.Reac('pka_reac_7', vsys, lhs = [R2C2_2cAMPbb, cAMP], rhs = [R2C2_3cAMP], kcst = kon_r2c2_camp3bb)
pka_reac_8 = smod.Reac('pka_reac_8', vsys, lhs = [R2C2_3cAMP], rhs = [R2C2_2cAMPbb, cAMP], kcst = koff_r2c2_camp3bb)
pka_reac_9=smod.Reac('pka_reac_9',vsys,lhs=[R2C2_2cAMPab,cAMP],rhs=[R2C2_3cAMP],kcst=kon_r2c2_camp3ab)
pka_reac_10=smod.Reac('pka_reac_10',vsys,lhs=[R2C2_3cAMP],rhs=[R2C2_2cAMPab],kcst=koff_r2c2_camp3ab)
pka_reac_11=smod.Reac('pka_reac_11',vsys,lhs=[R2C2_3cAMP,cAMP],rhs=[R2C2_4cAMP],kcst=kon_r2c2_camp4)
pka_reac_12=smod.Reac('pka_reac_12',vsys,lhs=[R2C2_4cAMP],rhs=[R2C2_3cAMP,cAMP],kcst=koff_r2c2_camp4)
pka_reac_13=smod.Reac('pka_reac_13',vsys,lhs=[R2C2_2cAMPab],rhs=[R2C_2cAMPab,PKA],kcst=kact_pka_23)
pka_reac_14=smod.Reac('pka_reac_14',vsys,lhs=[R2C_2cAMPab,PKA],rhs=[R2C2_2cAMPab],kcst=kdeact_pka_23)
pka_reac_15=smod.Reac('pka_reac_15',vsys,lhs=[R2C2_3cAMP],rhs=[R2C_3cAMP,PKA],kcst=kact_pka_23)
pka_reac_16=smod.Reac('pka_reac_16',vsys,lhs=[R2C_3cAMP,PKA],rhs=[R2C2_3cAMP],kcst=kdeact_pka_23)
pka_reac_17=smod.Reac('pka_reac_17',vsys,lhs=[R2C2_4cAMP],rhs=[R2C_4cAMP,PKA],kcst=kact_pka_4)
pka_reac_18=smod.Reac('pka_reac_18',vsys,lhs=[R2C_4cAMP,PKA],rhs=[R2C2_4cAMP],kcst=kdeact_pka_4)
pka_reac_19=smod.Reac('pka_reac_19',vsys,lhs=[R2C_2cAMPab,cAMP],rhs=[R2C_3cAMP],kcst=kon_r2c_camp3)
pka_reac_20=smod.Reac('pka_reac_20',vsys,lhs=[R2C_3cAMP],rhs=[R2C_2cAMPab,cAMP],kcst=koff_r2c_camp3)
pka_reac_21=smod.Reac('pka_reac_21',vsys,lhs=[R2C_3cAMP,cAMP],rhs=[R2C_4cAMP],kcst=kon_r2c_camp4)
pka_reac_22=smod.Reac('pka_reac_22',vsys,lhs=[R2C_4cAMP],rhs=[R2C_3cAMP,cAMP],kcst=koff_r2c_camp4)
pka_reac_23=smod.Reac('pka_reac_23',vsys,lhs=[R2C_4cAMP],rhs=[R2_4cAMP,PKA],kcst=kact_pka_r2c)
pka_reac_24=smod.Reac('pka_reac_24',vsys,lhs=[R2_4cAMP,PKA],rhs=[R2C_4cAMP],kcst=kdeact_pka_r2c)


#######SYNAPSIN VESICLE CLUSTERING MODEL#######
#####Rebuilt for MPI version########

#synapsin phos. by PKA
#synapsin dephos. by PP2A

#TOMO phos. by CDK5
#TOMO dephos. by CaN_CaM

#####FREE CYTOSOLIC SYNAPSIN#####
synapsin = smod.Spec('synapsin', syt_model)
synapsin_p = smod.Spec('synapsin_p', syt_model)
diff_synapsin = smod.Diff('diff_synapsin', vsys, synapsin,  2.65e-12)
diff_synapsin_mem = smod.VesSDiff('diff_synapsin_mem', ves_ssys, synapsin,  DCST_MEM_VES)
sdiff_synapsin = smod.Diff('sdiff_synapsin', memsys, synapsin,  0)

diff_synapsin_p = smod.Diff('diff_synapsin_p', vsys, synapsin_p,  2.65e-12)
diff_synapsin_p_mem = smod.VesSDiff('diff_synapsin_p_mem', ves_ssys, synapsin_p,  DCST_MEM_VES)
sdiff_synapsin_p = smod.Diff('sdiff_synapsin_p', memsys, synapsin_p,  0)
#PKA phos.
PKA_synapsin = smod.Spec('PKA_synapsin', syt_model)
diff_synapsin_pka = smod.Diff('diff_synapsin_pka', vsys, PKA_synapsin,  2.65e-12)
diff_synapsin_pka_mem = smod.VesSDiff('diff_synapsin_pka_mem', ves_ssys, PKA_synapsin,  DCST_MEM_VES)
sdiff_synapsin_pka = smod.Diff('sdiff_synapsin_pka', memsys, PKA_synapsin,  0)
kon_PKA_syn = 10e06
koff_PKA_syn = 1
kcat_PKA_syn = 10

#PP2A dephos.
PP2A = smod.Spec('PP2A', syt_model)
PP2A_synapsin_p = smod.Spec('PP2A_synapsin_p', syt_model)
diff_synapsin_pp2a = smod.Diff('diff_synapsin_pp2a', vsys, PP2A_synapsin_p,  2.65e-12)
diff_synapsin_pp2a_ves = smod.VesSDiff('diff_synapsin_pp2a_ves', ves_ssys, PP2A_synapsin_p,  DCST_MEM_VES)
sdiff_synapsin_pp2a = smod.Diff('sdiff_synapsin_pp2a', memsys, PP2A_synapsin_p,  0)
kon_pp2a_synapsin = 3e06 #Gallimore, Cell Reports, 2018 (from PP2A CaMKII dephos parameters)
koff_pp2a_synapsin = 10
kcat_pp2a_syn = 0.5
#dephos of cytosolic synapsin_p
pp2a_syn_on_c = smod.Reac('pp2a_syn_on_c', vsys, lhs = [PP2A, synapsin_p], rhs = [PP2A_synapsin_p], kcst = kon_pp2a_synapsin)
pp2a_syn_off_c = smod.Reac('pp2a_syn_off_c', vsys, lhs = [PP2A_synapsin_p], rhs = [PP2A, synapsin_p], kcst = koff_pp2a_synapsin)
pp2a_syn_phos_c = smod.Reac('pp2a_syn_phos_c', vsys, lhs = [PP2A_synapsin_p], rhs = [PP2A, synapsin], kcst = kcat_pp2a_syn)


######SYNAPSIN BOUND TO ACTIN#####
actin = smod.Spec('actin', syt_model)
synapsin_actin = smod.Spec('synapsin_actin', syt_model) #(only the vesicle-bound form)
synapsin_actin_p = smod.Spec('synapsin_actin_p', syt_model)#(only the vesicle-bound form)
Rab3_TOMO_synapsin_actin = smod.Spec('Rab3_TOMO_synapsin_actin', syt_model)
Rab3_TOMO_p_synapsin_actin = smod.Spec('Rab3_TOMO_p_synapsin_actin', syt_model)
Rab3_TOMO_synapsin_actin_p = smod.Spec('Rab3_TOMO_synapsin_actin_p', syt_model)
Rab3_TOMO_p_synapsin_actin_p = smod.Spec('Rab3_TOMO_p_synapsin_actin_p', syt_model)
diff_actin = smod.Diff('diff_actin', vsys, actin,  0)
diff_synapsin_actin = smod.Diff('diff_synapsin_actin', vsys, synapsin_actin,  0)
diff_synapsin_p_actin = smod.Diff('diff_synapsin_p_actin', vsys, synapsin_actin_p,  0)
diff_Rab3_TOMO_synapsin_actin = smod.Diff('diff_Rab3_TOMO_synapsin_actin', vsys, Rab3_TOMO_synapsin_actin,  0)
sdiff_Rab3_TOMO_synapsin_actin = smod.Diff('sdiff_Rab3_TOMO_synapsin_actin', memsys, Rab3_TOMO_synapsin_actin,  0)
diff_Rab3_TOMO_p_synapsin_actin = smod.Diff('diff_Rab3_TOMO_p_synapsin_actin', vsys, Rab3_TOMO_p_synapsin_actin,  0)
sdiff_Rab3_TOMO_p_synapsin_actin = smod.Diff('sdiff_Rab3_TOMO_p_synapsin_actin', memsys, Rab3_TOMO_p_synapsin_actin,  0)

#synapsin bound to vesicle binds to actin
kon_synapsin_actin = 1000e06
koff_synapsin_actin = 1
koff_synapsin_actin_p = 1000
synapsin_actin_on = smod.VesSReac('synapsin_actin_on', ves_ssys, vlhs = [synapsin], olhs = [actin], vrhs = [synapsin_actin], kcst = kon_synapsin_actin, immobility=1)
synapsin_actin_off = smod.VesSReac('synapsin_actin_off', ves_ssys, vlhs = [synapsin_actin], orhs = [actin], vrhs = [synapsin], kcst = koff_synapsin_actin, immobility=-1)
synapsin_p_actin_on = smod.VesSReac('synapsin_p_actin_on', ves_ssys, vlhs = [synapsin_p], olhs = [actin], vrhs = [synapsin_actin_p], kcst = kon_synapsin_actin, immobility=1)
synapsin_p_actin_off = smod.VesSReac('synapsin_p_actin_off', ves_ssys, vlhs = [synapsin_actin_p], orhs = [actin], vrhs = [synapsin_p], kcst = koff_synapsin_actin_p, immobility=-1)

#####NON-DIMERISED SYNAPSINS#####
ves_syn1_site = smod.Spec('ves_syn1_site', syt_model)
#synapsin (already defined)
#synapsin_p ("")
Rab3_TOMO_synapsin = smod.Spec('Rab3_TOMO_synapsin', syt_model)
Rab3_TOMO_synapsin_p = smod.Spec('Rab3_TOMO_synapsin_p', syt_model)
Rab3_TOMO_p_synapsin = smod.Spec('Rab3_TOMO_p_synapsin', syt_model)
Rab3_TOMO_p_synapsin_p = smod.Spec('Rab3_TOMO_p_synapsin_p', syt_model)
sdiff_Rab3_TOMO_synapsin = smod.Diff('sdiff_Rab3_TOMO_synapsin', memsys, Rab3_TOMO_synapsin,  0)
diff_Rab3_TOMO_synapsin_ves = smod.VesSDiff('diff_Rab3_TOMO_synapsin_ves', ves_ssys, Rab3_TOMO_synapsin,  DCST_MEM_VES)
diff_Rab3_TOMO_synapsin_p_ves = smod.VesSDiff('diff_Rab3_TOMO_synapsin_p_ves', ves_ssys, Rab3_TOMO_synapsin_p,  DCST_MEM_VES)
sdiff_Rab3_TOMO_synapsin_p_ves = smod.Diff('sdiff_Rab3_TOMO_synapsin_p_ves', memsys, Rab3_TOMO_synapsin_p,  0)
sdiff_Rab3_TOMO_p_synapsin = smod.Diff('sdiff_Rab3_TOMO_p_synapsin', memsys, Rab3_TOMO_p_synapsin, 0)
diff_Rab3_TOMO_p_synapsin_ves  = smod.VesSDiff('diff_Rab3_TOMO_p_synapsin_ves', ves_ssys, Rab3_TOMO_p_synapsin,  DCST_MEM_VES)
sdiff_Rab3_TOMO_p_synapsin_p = smod.Diff('sdiff_Rab3_TOMO_p_synapsin_p', memsys, Rab3_TOMO_p_synapsin_p, 0)
diff_Rab3_TOMO_p_synapsin_p_ves = smod.VesSDiff('diff_Rab3_TOMO_p_synapsin_p_ves', ves_ssys, Rab3_TOMO_p_synapsin_p,  DCST_MEM_VES)

#Tomosyn (TOMO) states (not including synapsin-bound states earlier)
TOMO_p = smod.Spec('TOMO_p', syt_model)
Rab3_TOMO_p = smod.Spec('Rab3_TOMO_p', syt_model)
TOMO = smod.Spec('TOMO', syt_model)
Rab3_TOMO = smod.Spec('Rab3_TOMO', syt_model)

diff_TOMO_p = smod.Diff('diff_TOMO_p', vsys, TOMO_p, 1.7793e-12)
diff_TOMO = smod.Diff('diff_TOMO', vsys, TOMO, 1.7793e-12)
diff_Rab3_TOMO_p_ves = smod.VesSDiff('diff_Rab3_TOMO_p_ves', ves_ssys, Rab3_TOMO_p,  DCST_MEM_VES)
sdiff_Rab3_TOMO_p = smod.Diff('sdiff_Rab3_TOMO_p', memsys, Rab3_TOMO_p, 0)
diff_Rab3_TOMO_ves = smod.VesSDiff('diff_Rab3_TOMO_ves', ves_ssys, Rab3_TOMO,  DCST_MEM_VES)
sdiff_Rab3_TOMO = smod.Diff('sdiff_Rab3_TOMO', memsys, Rab3_TOMO, 0)

#CDK5 phos.
TOMO_CDK5 = smod.Spec('TOMO_CDK5', syt_model)
Rab3_TOMO_CDK5 = smod.Spec('Rab3_TOMO_CDK5', syt_model)
diff_TOMO_CDK5 = smod.Diff('diff_TOMO_CDK5', vsys, TOMO_CDK5, 1.7793e-12)
diff_Rab3_TOMO_CDK5_ves = smod.VesSDiff('diff_Rab3_TOMO_CDK5_ves', ves_ssys, Rab3_TOMO_CDK5,  0.5e-12)
sdiff_Rab3_TOMO_CDK5 = smod.Diff('sdiff_Rab3_TOMO_CDK5', memsys, Rab3_TOMO_CDK5,  0)

kon_cdk_tomo = 3e06
koff_cdk_tomo = 10
kcat_cdk_tomo = 0.5

TOMO_CDK5_on = smod.Reac('TOMO_CDK5_on', vsys, lhs = [TOMO, CDK5], rhs = [TOMO_CDK5], kcst = kon_cdk_tomo)
TOMO_CDK5_off = smod.Reac('TOMO_CDK5_off', vsys, lhs = [TOMO_CDK5], rhs = [TOMO, CDK5], kcst = koff_cdk_tomo)
TOMO_CDK5_cat = smod.Reac('TOMO_CDK5_cat', vsys, lhs = [TOMO_CDK5], rhs = [TOMO_p, CDK5], kcst = kcat_cdk_tomo)

Rab3_TOMO_CDK5_on = smod.VesSReac('Rab3_TOMO_CDK5_on', ves_ssys, vlhs = [Rab3_TOMO], olhs = [CDK5], vrhs = [Rab3_TOMO_CDK5], kcst = kon_cdk_tomo)
Rab3_TOMO_CDK5_off = smod.VesSReac('Rab3_TOMO_CDK5_off', ves_ssys, vlhs = [Rab3_TOMO_CDK5], orhs = [CDK5], vrhs = [Rab3_TOMO], kcst = koff_cdk_tomo)
Rab3_TOMO_CDK5_cat = smod.VesSReac('Rab3_TOMO_CDK5_cat', ves_ssys, vlhs = [Rab3_TOMO_CDK5], orhs = [CDK5], vrhs = [Rab3_TOMO_p], kcst = kcat_cdk_tomo)
#CaN dephos.
TOMO_p_CaN_CaM = smod.Spec('TOMO_p_CaN_CaM', syt_model)
Rab3_TOMO_p_CaN_CaM = smod.Spec('Rab3_TOMO_p_CaN_CaM', syt_model)
diff_TOMO_p_CaN = smod.Diff('diff_TOMO_p_CaN', vsys, TOMO_p_CaN_CaM, 1.7793e-12)
diff_Rab3_TOMO_p_CaN_CaM_ves = smod.VesSDiff('diff_Rab3_TOMO_p_CaN_CaM_ves', ves_ssys, Rab3_TOMO_p_CaN_CaM,  DCST_MEM_VES)
sdiff_Rab3_TOMO_p_CaN_CaM = smod.Diff('sdiff_Rab3_TOMO_p_CaN_CaM', memsys, Rab3_TOMO_p_CaN_CaM,  0)

kon_can_tomo = 1e06
koff_can_tomo = 10
# kcat_can_tomo = 1#0.5 (given at init)

TOMO_p_CaN_on = smod.Reac('TOMO_p_CaN_on', vsys, lhs = [TOMO_p, CaN_CaM], rhs = [TOMO_p_CaN_CaM], kcst = kon_can_tomo)
TOMO_p_CaN_off = smod.Reac('TOMO_p_CaN_off', vsys, lhs = [TOMO_p_CaN_CaM], rhs = [TOMO_p, CaN_CaM], kcst = koff_can_tomo)
TOMO_p_CaN_cat = smod.Reac('TOMO_p_CaN_cat', vsys, lhs = [TOMO_p_CaN_CaM], rhs = [TOMO, CaN_CaM], kcst = kcat_can_tomo)

Rab3_TOMO_p_CaN_on = smod.VesSReac('Rab3_TOMO_p_CaN_on', ves_ssys, vlhs = [Rab3_TOMO_p], olhs = [CaN_CaM], vrhs = [Rab3_TOMO_p_CaN_CaM], kcst = kon_can_tomo)
Rab3_TOMO_p_CaN_off = smod.VesSReac('Rab3_TOMO_p_CaN_off', ves_ssys, vlhs = [Rab3_TOMO_p_CaN_CaM], orhs = [CaN_CaM], vrhs = [Rab3_TOMO_p], kcst = koff_can_tomo)
Rab3_TOMO_p_CaN_cat = smod.VesSReac('Rab3_TOMO_p_CaN_cat', ves_ssys, vlhs = [Rab3_TOMO_p_CaN_CaM], orhs = [CaN_CaM], vrhs = [Rab3_TOMO], kcst = kcat_can_tomo)

#TOMOSYN binds Rab3
kon_rab3_TOMO_p = 1e06
koff_rab3_TOMO_p = 10
rab3_TOMO_p_on = smod.VesSReac('rab3_TOMO_p_on', ves_ssys, olhs = [TOMO_p], vlhs = [Rab3], vrhs = [Rab3_TOMO_p], kcst = kon_rab3_TOMO_p)
rab3_TOMO_p_off = smod.VesSReac('rab3_TOMO_p_off', ves_ssys, vlhs = [Rab3_TOMO_p], vrhs = [Rab3], orhs = [TOMO_p], kcst = koff_rab3_TOMO_p)
#rab3_TOMO_on = smod.VesSReac('rab3_TOMO_on', ves_ssys, olhs = [TOMO], vlhs = [Rab3], vrhs = [Rab3_TOMO], kcst = kon_rab3_TOMO_p)
rab3_TOMO_off = smod.VesSReac('rab3_TOMO_off', ves_ssys, vlhs = [Rab3_TOMO], vrhs = [Rab3], orhs = [TOMO], kcst = koff_rab3_TOMO_p)

#Rab3-TOMO binds synapsin (non-dimerised)
kon_TOMO_p_synapsin = 100e06
koff_TOMO_p_synapsin = 1
kon_TOMO_synapsin = 100e06
koff_TOMO_synapsin = 100 #increase 100x with TOMO_p dephosphorylation

TOMO_p_synapsin_on = smod.VesSReac('TOMO_p_synapsin_on', ves_ssys, vlhs = [Rab3_TOMO_p, synapsin], vrhs = [Rab3_TOMO_p_synapsin], kcst = kon_TOMO_p_synapsin)
TOMO_p_synapsin_off = smod.VesSReac('TOMO_p_synapsin_off', ves_ssys, vlhs = [Rab3_TOMO_p_synapsin], vrhs = [Rab3_TOMO_p, synapsin], kcst = koff_TOMO_p_synapsin)

TOMO_p_synapsin_p_on = smod.VesSReac('TOMO_p_synapsin_p_on', ves_ssys, vlhs = [Rab3_TOMO_p, synapsin_p], vrhs = [Rab3_TOMO_p_synapsin_p], kcst = kon_TOMO_p_synapsin)
TOMO_p_synapsin_p_off = smod.VesSReac('TOMO_p_synapsin_p_off', ves_ssys, vlhs = [Rab3_TOMO_p_synapsin_p], vrhs = [Rab3_TOMO_p, synapsin_p], kcst = koff_TOMO_p_synapsin)

TOMO_synapsin_on = smod.VesSReac('TOMO_synapsin_on', ves_ssys, vlhs = [Rab3_TOMO, synapsin], vrhs = [Rab3_TOMO_synapsin], kcst = kon_TOMO_synapsin)
TOMO_synapsin_off = smod.VesSReac('TOMO_synapsin_off', ves_ssys, vlhs = [Rab3_TOMO_synapsin], vrhs = [Rab3_TOMO, synapsin], kcst = koff_TOMO_synapsin)

TOMO_synapsin_p_on = smod.VesSReac('TOMO_synapsin_p_on', ves_ssys, vlhs = [Rab3_TOMO, synapsin_p], vrhs = [Rab3_TOMO_synapsin_p], kcst = kon_TOMO_synapsin)
TOMO_synapsin_p_off = smod.VesSReac('TOMO_synapsin_p_off', ves_ssys, vlhs = [Rab3_TOMO_synapsin_p], vrhs = [Rab3_TOMO, synapsin_p], kcst = koff_TOMO_synapsin)

#Enzyme-complex states of non-dimerised ves-bound synapsins
#PKA phos.
#PKA_synapsin (already defined)
Rab3_TOMO_synapsin_PKA = smod.Spec('Rab3_TOMO_synapsin_PKA', syt_model)
Rab3_TOMO_p_synapsin_PKA = smod.Spec('Rab3_TOMO_p_synapsin_PKA', syt_model)
diff_Rab3_TOMO_synapsin_PKA_ves = smod.VesSDiff('diff_Rab3_TOMO_synapsin_PKA_ves', ves_ssys, Rab3_TOMO_synapsin_PKA,  DCST_MEM_VES)
sdiff_Rab3_TOMO_synapsin_PKA = smod.Diff('sdiff_Rab3_TOMO_synapsin_PKA', memsys, Rab3_TOMO_synapsin_PKA,  0)
diff_Rab3_TOMO_p_synapsin_PKA_ves = smod.VesSDiff('diff_Rab3_TOMO_p_synapsin_PKA_ves', ves_ssys, Rab3_TOMO_p_synapsin_PKA,  DCST_MEM_VES)
sdiff_Rab3_TOMO_p_synapsin_PKA = smod.Diff('sdiff_Rab3_TOMO_p_synapsin_PKA', memsys, Rab3_TOMO_p_synapsin_PKA,  0)

#free vesicle synapsin phosphorylation
PKA_syn_on = smod.VesSReac('PKA_syn_on', ves_ssys, olhs = [PKA], vlhs = [synapsin], vrhs = [PKA_synapsin], kcst = kon_PKA_syn)
PKA_syn_off = smod.VesSReac('PKA_syn_off', ves_ssys, vlhs = [PKA_synapsin], orhs = [PKA], vrhs = [synapsin], kcst = koff_PKA_syn)
PKA_syn_phos = smod.VesSReac('PKA_syn_phos', ves_ssys, vlhs = [PKA_synapsin], orhs = [PKA], vrhs = [synapsin_p], kcst = kcat_PKA_syn)

PKA_Rab3_TOMO_synves_on = smod.VesSReac('PKA_Rab3_TOMO_synves_on', ves_ssys, olhs = [PKA], vlhs = [Rab3_TOMO_synapsin], vrhs = [Rab3_TOMO_synapsin_PKA], kcst = kon_PKA_syn)
PKA_Rab3_TOMO_synves_off = smod.VesSReac('PKA_Rab3_TOMO_synves_off', ves_ssys, vlhs = [Rab3_TOMO_synapsin_PKA], orhs = [PKA], vrhs = [Rab3_TOMO_synapsin], kcst = koff_PKA_syn)
PKA_Rab3_TOMO_synves_phos = smod.VesSReac('PKA_Rab3_TOMO_synves_phos', ves_ssys, vlhs = [Rab3_TOMO_synapsin_PKA], orhs = [PKA], vrhs = [Rab3_TOMO_synapsin_p], kcst = kcat_PKA_syn)

PKA_Rab3_TOMO_p_synves_on = smod.VesSReac('PKA_Rab3_TOMO_p_synves_on', ves_ssys, olhs = [PKA], vlhs = [Rab3_TOMO_p_synapsin], vrhs = [Rab3_TOMO_p_synapsin_PKA], kcst = kon_PKA_syn)
PKA_Rab3_TOMO_p_synves_off = smod.VesSReac('PKA_Rab3_TOMO_p_synves_off', ves_ssys, vlhs = [Rab3_TOMO_p_synapsin_PKA], orhs = [PKA], vrhs = [Rab3_TOMO_p_synapsin], kcst = koff_PKA_syn)
PKA_Rab3_TOMO_p_synves_phos = smod.VesSReac('PKA_Rab3_TOMO_p_synves_phos', ves_ssys, vlhs = [Rab3_TOMO_p_synapsin_PKA], orhs = [PKA], vrhs = [Rab3_TOMO_p_synapsin_p], kcst = kcat_PKA_syn)

#PP2A dephos.
#PP2A_synapsin_p (already defined)
Rab3_TOMO_p_synapsin_p_PP2A = smod.Spec('Rab3_TOMO_p_synapsin_p_PP2A', syt_model)
Rab3_TOMO_synapsin_p_PP2A = smod.Spec('Rab3_TOMO_synapsin_p_PP2A', syt_model)
#dephos of vesicle synapsin_p
pp2a_syn_on_v = smod.VesSReac('pp2a_syn_on_v', ves_ssys, olhs = [PP2A], vlhs = [synapsin_p], vrhs = [PP2A_synapsin_p], kcst = kon_pp2a_synapsin)
pp2a_syn_off_v = smod.VesSReac('pp2a_syn_off_v', ves_ssys, vlhs = [PP2A_synapsin_p], orhs = [PP2A], vrhs = [synapsin_p], kcst = koff_pp2a_synapsin)
pp2a_syn_phos_v = smod.VesSReac('pp2a_syn_phos_v', ves_ssys, vlhs = [PP2A_synapsin_p], orhs = [PP2A], vrhs = [synapsin], kcst = kcat_pp2a_syn)

pp2a_Rab3_TOMO_p_syn_on_v = smod.VesSReac('pp2a_Rab3_TOMO_p_syn_on_v', ves_ssys, olhs = [PP2A], vlhs = [Rab3_TOMO_p_synapsin_p], vrhs = [Rab3_TOMO_p_synapsin_p_PP2A], kcst = kon_pp2a_synapsin)
pp2a_Rab3_TOMO_p_syn_off_v = smod.VesSReac('pp2a_Rab3_TOMO_p_syn_off_v', ves_ssys, vlhs = [Rab3_TOMO_p_synapsin_p_PP2A], orhs = [PP2A], vrhs = [Rab3_TOMO_p_synapsin_p], kcst = koff_pp2a_synapsin)
pp2a_Rab3_TOMO_p_syn_phos_v = smod.VesSReac('pp2a_Rab3_TOMO_p_syn_phos_v', ves_ssys, vlhs = [Rab3_TOMO_p_synapsin_p_PP2A], orhs = [PP2A], vrhs = [Rab3_TOMO_p_synapsin], kcst = kcat_pp2a_syn)

pp2a_Rab3_TOMO_syn_on_v = smod.VesSReac('pp2a_Rab3_TOMO_syn_on_v', ves_ssys, olhs = [PP2A], vlhs = [Rab3_TOMO_synapsin_p], vrhs = [Rab3_TOMO_synapsin_p_PP2A], kcst = kon_pp2a_synapsin)
pp2a_Rab3_TOMO_syn_off_v = smod.VesSReac('pp2a_Rab3_TOMO_syn_off_v', ves_ssys, vlhs = [Rab3_TOMO_synapsin_p_PP2A], orhs = [PP2A], vrhs = [Rab3_TOMO_synapsin_p], kcst = koff_pp2a_synapsin)
pp2a_Rab3_TOMO_syn_phos_v = smod.VesSReac('pp2a_Rab3_TOMO_syn_phos_v', ves_ssys, vlhs = [Rab3_TOMO_synapsin_p_PP2A], orhs = [PP2A], vrhs = [Rab3_TOMO_synapsin], kcst = kcat_pp2a_syn)

#CDK5 phos.
Rab3_TOMO_synapsin_CDK5 = smod.Spec('Rab3_TOMO_synapsin_CDK5', syt_model)
Rab3_TOMO_synapsin_p_CDK5 = smod.Spec('Rab3_TOMO_synapsin_p_CDK5', syt_model)
diff_Rab3_TOMO_synapsin_CDK5_ves = smod.VesSDiff('diff_Rab3_TOMO_synapsin_CDK5_ves', ves_ssys, Rab3_TOMO_synapsin_CDK5,  DCST_MEM_VES)
sdiff_Rab3_TOMO_synapsin_CDK5 = smod.Diff('sdiff_Rab3_TOMO_synapsin_CDK5', memsys, Rab3_TOMO_synapsin_CDK5,  0)

diff_Rab3_TOMO_synapsin_p_CDK5_ves = smod.VesSDiff('diff_Rab3_TOMO_synapsin_p_CDK5_ves', ves_ssys, Rab3_TOMO_synapsin_p_CDK5,  DCST_MEM_VES)
sdiff_Rab3_TOMO_synapsin_p_CDK5 = smod.Diff('sdiff_Rab3_TOMO_synapsin_p_CDK5', memsys, Rab3_TOMO_synapsin_p_CDK5,  0)

Rab3_TOMO_synapsin_CDK5_on = smod.VesSReac('Rab3_TOMO_synapsin_CDK5_on', ves_ssys, vlhs = [Rab3_TOMO_synapsin], olhs = [CDK5], vrhs = [Rab3_TOMO_synapsin_CDK5], kcst = kon_cdk_tomo)
Rab3_TOMO_synapsin_CDK5_off = smod.VesSReac('Rab3_TOMO_synapsin_CDK5_off', ves_ssys, vlhs = [Rab3_TOMO_synapsin_CDK5], orhs = [CDK5], vrhs = [Rab3_TOMO_synapsin], kcst = koff_cdk_tomo)
Rab3_TOMO_synapsin_CDK5_cat = smod.VesSReac('Rab3_TOMO_synapsin_CDK5_cat', ves_ssys, vlhs = [Rab3_TOMO_synapsin_CDK5], orhs = [CDK5], vrhs = [Rab3_TOMO_p_synapsin], kcst = kcat_cdk_tomo)

Rab3_TOMO_synapsin_p_CDK5_on = smod.VesSReac('Rab3_TOMO_synapsin_p_CDK5_on', ves_ssys, vlhs = [Rab3_TOMO_synapsin_p], olhs = [CDK5], vrhs = [Rab3_TOMO_synapsin_p_CDK5], kcst = kon_cdk_tomo)
Rab3_TOMO_synapsin_p_CDK5_off = smod.VesSReac('Rab3_TOMO_synapsin_p_CDK5_off', ves_ssys, vlhs = [Rab3_TOMO_synapsin_p_CDK5], orhs = [CDK5], vrhs = [Rab3_TOMO_synapsin_p], kcst = koff_cdk_tomo)
Rab3_TOMO_synapsin_p_CDK5_cat = smod.VesSReac('Rab3_TOMO_synapsin_p_CDK5_cat', ves_ssys, vlhs = [Rab3_TOMO_synapsin_p_CDK5], orhs = [CDK5], vrhs = [Rab3_TOMO_p_synapsin_p], kcst = kcat_cdk_tomo)

#CaN dephos.
Rab3_TOMO_p_synapsin_CaN_CaM = smod.Spec('Rab3_TOMO_p_synapsin_CaN_CaM', syt_model)
Rab3_TOMO_p_synapsin_p_CaN_CaM = smod.Spec('Rab3_TOMO_p_synapsin_p_CaN_CaM', syt_model)
diff_Rab3_TOMO_p_synapsin_CaN_CaM_ves = smod.VesSDiff('diff_Rab3_TOMO_p_synapsin_CaN_CaM_ves', ves_ssys, Rab3_TOMO_p_synapsin_CaN_CaM,  DCST_MEM_VES)
sdiff_Rab3_TOMO_p_synapsin_CaN_CaM = smod.Diff('sdiff_Rab3_TOMO_p_synapsin_CaN_CaM', memsys, Rab3_TOMO_p_synapsin_CaN_CaM,  0)
diff_Rab3_TOMO_p_synapsin_p_CaN_CaM_ves = smod.VesSDiff('diff_Rab3_TOMO_p_synapsin_p_CaN_CaM_ves', ves_ssys, Rab3_TOMO_p_synapsin_p_CaN_CaM,  DCST_MEM_VES)
sdiff_Rab3_TOMO_p_synapsin_p_CaN_CaM = smod.Diff('sdiff_Rab3_TOMO_p_synapsin_p_CaN_CaM', memsys, Rab3_TOMO_p_synapsin_p_CaN_CaM,  0)

Rab3_TOMO_p_synapsin_CaN_on = smod.VesSReac('Rab3_TOMO_p_synapsin_CaN_on', ves_ssys, vlhs = [Rab3_TOMO_p_synapsin], olhs = [CaN_CaM], vrhs = [Rab3_TOMO_p_synapsin_CaN_CaM], kcst = kon_can_tomo)
Rab3_TOMO_p_synapsin_CaN_off = smod.VesSReac('Rab3_TOMO_p_synapsin_CaN_off', ves_ssys, vlhs = [Rab3_TOMO_p_synapsin_CaN_CaM ], orhs = [CaN_CaM], vrhs = [Rab3_TOMO_p_synapsin], kcst = koff_can_tomo)
Rab3_TOMO_p_synapsin_CaN_cat = smod.VesSReac('Rab3_TOMO_p_synapsin_CaN_cat', ves_ssys, vlhs = [Rab3_TOMO_p_synapsin_CaN_CaM], orhs = [CaN_CaM], vrhs = [Rab3_TOMO_synapsin], kcst = kcat_can_tomo)

Rab3_TOMO_p_synapsin_p_CaN_on = smod.VesSReac('Rab3_TOMO_p_synapsin_p_CaN_on', ves_ssys, vlhs = [Rab3_TOMO_p_synapsin_p], olhs = [CaN_CaM], vrhs = [Rab3_TOMO_p_synapsin_p_CaN_CaM], kcst = kon_can_tomo)
Rab3_TOMO_p_synapsin_p_CaN_off = smod.VesSReac('Rab3_TOMO_p_synapsin_p_CaN_off', ves_ssys, vlhs = [Rab3_TOMO_p_synapsin_p_CaN_CaM], orhs = [CaN_CaM], vrhs = [Rab3_TOMO_p_synapsin_p], kcst = koff_can_tomo)
Rab3_TOMO_p_synapsin_p_CaN_cat = smod.VesSReac('Rab3_TOMO_p_synapsin_p_CaN_cat', ves_ssys, vlhs = [Rab3_TOMO_p_synapsin_p_CaN_CaM], orhs = [CaN_CaM], vrhs = [Rab3_TOMO_synapsin_p], kcst = kcat_can_tomo)

#Basic states of dimerised synapsins (each "half" of the link between vesicles)
#None of the dimers need explicitly defining.
synapsin_dimer = smod.LinkSpec('synapsin_dimer', syt_model, 0.1e-12)
synapsin_dimer_p = smod.LinkSpec('synapsin_dimer_p', syt_model, 0.1e-12)
Rab3_TOMO_synapsin_dimer = smod.LinkSpec('Rab3_TOMO_synapsin_dimer', syt_model,   0.1e-12) #(2 formation routes)
Rab3_TOMO_synapsin_dimer_p = smod.LinkSpec('Rab3_TOMO_synapsin_dimer_p', syt_model,   0.1e-12) #(2 formation routes)
Rab3_TOMO_p_synapsin_dimer = smod.LinkSpec('Rab3_TOMO_p_synapsin_dimer', syt_model,   0.1e-12) #(2 formation routes)
Rab3_TOMO_p_synapsin_dimer_p = smod.LinkSpec('Rab3_TOMO_p_synapsin_dimer_p', syt_model,   0.1e-12) #(2 formation routes)

#####Rab3-TOMO binds to the synapsin dimers#####
TOMO_p_synapsin_dimer_on = smod.VesSReac('TOMO_p_synapsin_dimer_on', ves_ssys, vlhs = [Rab3_TOMO_p], llhs = [synapsin_dimer], lrhs = [Rab3_TOMO_p_synapsin_dimer], kcst = kon_TOMO_p_synapsin)
TOMO_p_synapsin_dimer_off = smod.VesSReac('TOMO_p_synapsin_dimer_off', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer ], vrhs = [Rab3_TOMO_p], lrhs = [synapsin_dimer], kcst = koff_TOMO_p_synapsin)

TOMO_p_synapsin_dimer_p_on = smod.VesSReac('TOMO_p_synapsin_dimer_p_on', ves_ssys, vlhs = [Rab3_TOMO_p], llhs = [synapsin_dimer_p], lrhs = [Rab3_TOMO_p_synapsin_dimer_p], kcst = kon_TOMO_p_synapsin)
TOMO_p_synapsin_dimer_p_off = smod.VesSReac('TOMO_p_synapsin_dimer_p_off', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer_p], vrhs = [Rab3_TOMO_p], lrhs = [synapsin_dimer_p], kcst = koff_TOMO_p_synapsin)

TOMO_synapsin_dimer_on = smod.VesSReac('TOMO_synapsin_dimer_on', ves_ssys, vlhs = [Rab3_TOMO], llhs = [synapsin_dimer], lrhs = [Rab3_TOMO_synapsin_dimer], kcst = kon_TOMO_synapsin)
TOMO_synapsin_dimer_off = smod.VesSReac('TOMO_synapsin_dimer_off', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer], vrhs = [Rab3_TOMO], lrhs = [synapsin_dimer], kcst = koff_TOMO_synapsin)

TOMO_synapsin_dimer_p_on = smod.VesSReac('TOMO_synapsin_dimer_p_on', ves_ssys, vlhs = [Rab3_TOMO], llhs = [synapsin_dimer_p], lrhs = [Rab3_TOMO_synapsin_dimer_p], kcst = kon_TOMO_synapsin)
TOMO_synapsin_dimer_p_off = smod.VesSReac('TOMO_synapsin_dimer_p_off', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer_p], vrhs = [Rab3_TOMO], lrhs = [synapsin_dimer_p], kcst = koff_TOMO_synapsin)

kon_synapsin_dimer = 10e06
koff_synapsin_dimer = 10
koff_synapsin_dimer_p = 10000

#initial dimerisation depends on synapsin_actin dimer [5 different forms] (ensures actin is the focus of the clustering)
synapsin_dimerise_1a = smod.VesBind('synapsin_dimerise_1a', vsys, ves1=ves, r1=synapsin, ves2=ves, r2=synapsin, l1=synapsin_dimer, l2 = synapsin_dimer, length_max = 10e-8, length_min = 0, vdeps1 = [synapsin_actin], kcst=kon_synapsin_dimer, immobility=0)
synapsin_dimerise_1b = smod.VesBind('synapsin_dimerise_1b', vsys, ves1=ves, r1=synapsin, ves2=ves, r2=synapsin, l1=synapsin_dimer, l2 = synapsin_dimer, length_max = 10e-8, length_min = 0, vdeps1 = [Rab3_TOMO_synapsin_actin], kcst=kon_synapsin_dimer, immobility=0)
synapsin_dimerise_1c = smod.VesBind('synapsin_dimerise_1c', vsys, ves1=ves, r1=synapsin, ves2=ves, r2=synapsin, l1=synapsin_dimer, l2 = synapsin_dimer, length_max = 10e-8, length_min = 0, vdeps1 = [Rab3_TOMO_p_synapsin_actin], kcst=kon_synapsin_dimer, immobility=0)
synapsin_dimerise_1d = smod.VesBind('synapsin_dimerise_1d', vsys, ves1=ves, r1=synapsin, ves2=ves, r2=synapsin, l1=synapsin_dimer, l2 = synapsin_dimer, length_max = 10e-8, length_min = 0, vdeps1 = [Rab3_TOMO_synapsin_actin_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_dimerise_1e = smod.VesBind('synapsin_dimerise_1e', vsys, ves1=ves, r1=synapsin, ves2=ves, r2=synapsin, l1=synapsin_dimer, l2 = synapsin_dimer, length_max = 10e-8, length_min = 0, vdeps1 = [Rab3_TOMO_p_synapsin_actin_p], kcst=kon_synapsin_dimer, immobility=0)


#Dimer states (need both VesBind and VesUnbind reactions for each, and ldeps for VesBinds -- 6, one for each state)
#synapsin - synapsin -> synapsin_dimer - synapsin_dimer
synapsin_synapsin_dimerise_a = smod.VesBind('synapsin_synapsin_dimerise_a', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=synapsin, l1=synapsin_dimer, l2 = synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_synapsin_dimerise_b = smod.VesBind('synapsin_synapsin_dimerise_b', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=synapsin, l1=synapsin_dimer, l2 = synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_synapsin_dimerise_c = smod.VesBind('synapsin_synapsin_dimerise_c', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=synapsin, l1=synapsin_dimer, l2 = synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_synapsin_dimerise_d = smod.VesBind('synapsin_synapsin_dimerise_d', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=synapsin, l1=synapsin_dimer, l2 = synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_synapsin_dimerise_e = smod.VesBind('synapsin_synapsin_dimerise_e', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=synapsin, l1=synapsin_dimer, l2 = synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_synapsin_dimerise_f = smod.VesBind('synapsin_synapsin_dimerise_f', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=synapsin, l1=synapsin_dimer, l2 = synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

synapsin_synapsin_unbind = smod.VesUnbind('synapsin_synapsin_unbind', vsys, synapsin_dimer, synapsin_dimer, ves, synapsin, ves, synapsin, koff_synapsin_dimer, immobility=0)

#synapsin - synapsin_p -> synapsin_dimer - synapsin_dimer_p
synapsin_synapsin_p_dimerise_a = smod.VesBind('synapsin_synapsin_p_dimerise_a', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=synapsin_p, l1=synapsin_dimer, l2 = synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_synapsin_p_dimerise_b = smod.VesBind('synapsin_synapsin_p_dimerise_b', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=synapsin_p, l1=synapsin_dimer, l2 = synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_synapsin_p_dimerise_c = smod.VesBind('synapsin_synapsin_p_dimerise_c', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=synapsin_p, l1=synapsin_dimer, l2 = synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_synapsin_p_dimerise_d = smod.VesBind('synapsin_synapsin_p_dimerise_d', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=synapsin_p, l1=synapsin_dimer, l2 = synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_synapsin_p_dimerise_e = smod.VesBind('synapsin_synapsin_p_dimerise_e', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=synapsin_p, l1=synapsin_dimer, l2 = synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_synapsin_p_dimerise_f = smod.VesBind('synapsin_synapsin_p_dimerise_f', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=synapsin_p, l1=synapsin_dimer, l2 = synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

synapsin_synapsin_p_unbind = smod.VesUnbind('synapsin_synapsin_p_unbind', vsys, synapsin_dimer, synapsin_dimer_p, ves, synapsin, ves, synapsin_p, koff_synapsin_dimer_p, immobility=0)

#synapsin - Rab3_TOMO_synapsin -> synapsin_dimer - Rab3_TOMO_synapsin_dimer
synapsin_Rab3_TOMO_synapsin_dimerise_a = smod.VesBind('synapsin_Rab3_TOMO_synapsin_dimerise_a', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_synapsin_dimerise_b = smod.VesBind('synapsin_Rab3_TOMO_synapsin_dimerise_b', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_synapsin_dimerise_c = smod.VesBind('synapsin_Rab3_TOMO_synapsin_dimerise_c', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_synapsin_dimerise_d = smod.VesBind('synapsin_Rab3_TOMO_synapsin_dimerise_d', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_synapsin_dimerise_e = smod.VesBind('synapsin_Rab3_TOMO_synapsin_dimerise_e', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_synapsin_dimerise_f = smod.VesBind('synapsin_Rab3_TOMO_synapsin_dimerise_f', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

synapsin_Rab3_TOMO_synapsin_unbind = smod.VesUnbind('synapsin_Rab3_TOMO_synapsin_unbind', vsys, synapsin_dimer, Rab3_TOMO_synapsin_dimer, ves, synapsin, ves, Rab3_TOMO_synapsin, koff_synapsin_dimer, immobility=0)

#synapsin - Rab3_TOMO_synapsin_p -> synapsin_dimer - Rab3_TOMO_synapsin_dimer_p
synapsin_Rab3_TOMO_synapsin_p_dimerise_a = smod.VesBind('synapsin_Rab3_TOMO_synapsin_p_dimerise_a', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_synapsin_p_dimerise_b = smod.VesBind('synapsin_Rab3_TOMO_synapsin_p_dimerise_b', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_synapsin_p_dimerise_c = smod.VesBind('synapsin_Rab3_TOMO_synapsin_p_dimerise_c', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_synapsin_p_dimerise_d = smod.VesBind('synapsin_Rab3_TOMO_synapsin_p_dimerise_d', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_synapsin_p_dimerise_e = smod.VesBind('synapsin_Rab3_TOMO_synapsin_p_dimerise_e', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_synapsin_p_dimerise_f = smod.VesBind('synapsin_Rab3_TOMO_synapsin_p_dimerise_f', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

synapsin_Rab3_TOMO_synapsin_p_unbind = smod.VesUnbind('synapsin_Rab3_TOMO_synapsin_p_unbind', vsys, synapsin_dimer, Rab3_TOMO_synapsin_dimer_p, ves, synapsin, ves, Rab3_TOMO_synapsin_p, koff_synapsin_dimer_p, immobility=0)

#synapsin - Rab3_TOMO_p_synapsin -> synapsin_dimer - Rab3_TOMO_p_synapsin_dimer
synapsin_Rab3_TOMO_p_synapsin_dimerise_a = smod.VesBind('synapsin_Rab3_TOMO_p_synapsin_dimerise_a', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_p_synapsin_dimerise_b = smod.VesBind('synapsin_Rab3_TOMO_p_synapsin_dimerise_b', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_p_synapsin_dimerise_c = smod.VesBind('synapsin_Rab3_TOMO_p_synapsin_dimerise_c', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_p_synapsin_dimerise_d = smod.VesBind('synapsin_Rab3_TOMO_p_synapsin_dimerise_d', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_p_synapsin_dimerise_e = smod.VesBind('synapsin_Rab3_TOMO_p_synapsin_dimerise_e', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_p_synapsin_dimerise_f = smod.VesBind('synapsin_Rab3_TOMO_p_synapsin_dimerise_f', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

synapsin_Rab3_TOMO_p_synapsin_unbind = smod.VesUnbind('synapsin_Rab3_TOMO_p_synapsin_unbind', vsys, synapsin_dimer, Rab3_TOMO_p_synapsin_dimer, ves, synapsin, ves, Rab3_TOMO_p_synapsin, koff_synapsin_dimer, immobility=0)

#synapsin - Rab3_TOMO_p_synapsin_p -> synapsin_dimer - Rab3_TOMO_p_synapsin_dimer_p
synapsin_Rab3_TOMO_p_synapsin_p_dimerise_a = smod.VesBind('synapsin_Rab3_TOMO_p_synapsin_p_dimerise_a', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_p_synapsin_p_dimerise_b = smod.VesBind('synapsin_Rab3_TOMO_p_synapsin_p_dimerise_b', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_p_synapsin_p_dimerise_c = smod.VesBind('synapsin_Rab3_TOMO_p_synapsin_p_dimerise_c', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_p_synapsin_p_dimerise_d = smod.VesBind('synapsin_Rab3_TOMO_p_synapsin_p_dimerise_d', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_p_synapsin_p_dimerise_e = smod.VesBind('synapsin_Rab3_TOMO_p_synapsin_p_dimerise_e', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_Rab3_TOMO_p_synapsin_p_dimerise_f = smod.VesBind('synapsin_Rab3_TOMO_p_synapsin_p_dimerise_f', vsys,  ves1=ves, r1=synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

synapsin_Rab3_TOMO_p_synapsin_p_unbind = smod.VesUnbind('synapsin_Rab3_TOMO_p_synapsin_p_unbind', vsys, synapsin_dimer, Rab3_TOMO_p_synapsin_dimer_p, ves, synapsin, ves, Rab3_TOMO_p_synapsin_p, koff_synapsin_dimer_p, immobility=0)

#synapsin_p - synapsin_p -> synapsin_dimer_p - synapsin_dimer_p
synapsin_p_synapsin_p_dimerise_a = smod.VesBind('synapsin_p_synapsin_p_dimerise_a', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=synapsin_p, l1=synapsin_dimer_p, l2 = synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_synapsin_p_dimerise_b = smod.VesBind('synapsin_p_synapsin_p_dimerise_b', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=synapsin_p, l1=synapsin_dimer_p, l2 = synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_synapsin_p_dimerise_c = smod.VesBind('synapsin_p_synapsin_p_dimerise_c', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=synapsin_p, l1=synapsin_dimer_p, l2 = synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_synapsin_p_dimerise_d = smod.VesBind('synapsin_p_synapsin_p_dimerise_d', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=synapsin_p, l1=synapsin_dimer_p, l2 = synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_synapsin_p_dimerise_e = smod.VesBind('synapsin_p_synapsin_p_dimerise_e', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=synapsin_p, l1=synapsin_dimer_p, l2 = synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_synapsin_p_dimerise_f = smod.VesBind('synapsin_p_synapsin_p_dimerise_f', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=synapsin_p, l1=synapsin_dimer_p, l2 = synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

synapsin_p_synapsin_p_dimerise_unbind = smod.VesUnbind('synapsin_p_synapsin_p_dimerise_unbind', vsys, synapsin_dimer_p, synapsin_dimer_p, ves, synapsin_p, ves, synapsin_p, koff_synapsin_dimer_p, immobility=0)

#synapsin_p - Rab3_TOMO_synapsin -> synapsin_dimer_p - Rab3_TOMO_synapsin_dimer
synapsin_p_Rab3_TOMO_synapsin_dimerise_a = smod.VesBind('synapsin_p_Rab3_TOMO_synapsin_dimerise_a', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_synapsin_dimerise_b = smod.VesBind('synapsin_p_Rab3_TOMO_synapsin_dimerise_b', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_synapsin_dimerise_c = smod.VesBind('synapsin_p_Rab3_TOMO_synapsin_dimerise_c', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_synapsin_dimerise_d = smod.VesBind('synapsin_p_Rab3_TOMO_synapsin_dimerise_d', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_synapsin_dimerise_e = smod.VesBind('synapsin_p_Rab3_TOMO_synapsin_dimerise_e', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_synapsin_dimerise_f = smod.VesBind('synapsin_p_Rab3_TOMO_synapsin_dimerise_f', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

synapsin_p_Rab3_TOMO_synapsin_dimerise_unbind = smod.VesUnbind('synapsin_p_Rab3_TOMO_synapsin_dimerise_unbind', vsys, synapsin_dimer_p, Rab3_TOMO_synapsin_dimer, ves, synapsin_p, ves, Rab3_TOMO_synapsin, koff_synapsin_dimer_p, immobility=0)

#synapsin_p - Rab3_TOMO_synapsin_p -> synapsin_dimer_p - Rab3_TOMO_synapsin_dimer_p
synapsin_p_Rab3_TOMO_synapsin_p_dimerise_a = smod.VesBind('synapsin_p_Rab3_TOMO_synapsin_p_dimerise_a', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_synapsin_p_dimerise_b = smod.VesBind('synapsin_p_Rab3_TOMO_synapsin_p_dimerise_b', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_synapsin_p_dimerise_c = smod.VesBind('synapsin_p_Rab3_TOMO_synapsin_p_dimerise_c', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_synapsin_p_dimerise_d = smod.VesBind('synapsin_p_Rab3_TOMO_synapsin_p_dimerise_d', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_synapsin_p_dimerise_e = smod.VesBind('synapsin_p_Rab3_TOMO_synapsin_p_dimerise_e', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_synapsin_p_dimerise_f = smod.VesBind('synapsin_p_Rab3_TOMO_synapsin_p_dimerise_f', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

synapsin_p_Rab3_TOMO_synapsin_p_unbind = smod.VesUnbind('synapsin_p_Rab3_TOMO_synapsin_p_unbind', vsys, synapsin_dimer_p, Rab3_TOMO_synapsin_dimer_p, ves, synapsin_p, ves, Rab3_TOMO_synapsin_p, koff_synapsin_dimer_p, immobility=0)

#synapsin_p - Rab3_TOMO_p_synapsin -> synapsin_dimer_p - Rab3_TOMO_p_synapsin_dimer
synapsin_p_Rab3_TOMO_p_synapsin_dimerise_a = smod.VesBind('synapsin_p_Rab3_TOMO_p_synapsin_dimerise_a', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_p_synapsin_dimerise_b = smod.VesBind('synapsin_p_Rab3_TOMO_p_synapsin_dimerise_b', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_p_synapsin_dimerise_c = smod.VesBind('synapsin_p_Rab3_TOMO_p_synapsin_dimerise_c', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_p_synapsin_dimerise_d = smod.VesBind('synapsin_p_Rab3_TOMO_p_synapsin_dimerise_d', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_p_synapsin_dimerise_e = smod.VesBind('synapsin_p_Rab3_TOMO_p_synapsin_dimerise_e', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_p_synapsin_dimerise_f = smod.VesBind('synapsin_p_Rab3_TOMO_p_synapsin_dimerise_f', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

synapsin_p_Rab3_TOMO_p_synapsin_unbind = smod.VesUnbind('synapsin_p_Rab3_TOMO_p_synapsin_unbind', vsys, synapsin_dimer_p, Rab3_TOMO_p_synapsin_dimer, ves, synapsin_p, ves, Rab3_TOMO_p_synapsin, koff_synapsin_dimer_p, immobility=0)

#synapsin_p - Rab3_TOMO_p_synapsin_p -> synapsin_dimer_p - Rab3_TOMO_p_synapsin_dimer_p
synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_a = smod.VesBind('synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_a', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_b = smod.VesBind('synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_b', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_c = smod.VesBind('synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_c', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_d = smod.VesBind('synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_d', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_e = smod.VesBind('synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_e', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_f = smod.VesBind('synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_f', vsys,  ves1=ves, r1=synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

synapsin_p_Rab3_TOMO_p_synapsin_p_unbind = smod.VesUnbind('synapsin_p_Rab3_TOMO_p_synapsin_p_unbind', vsys, synapsin_dimer_p, Rab3_TOMO_p_synapsin_dimer_p, ves, synapsin_p, ves, Rab3_TOMO_p_synapsin_p, koff_synapsin_dimer_p, immobility=0)

#Rab3_TOMO_synapsin - Rab3_TOMO_synapsin -> Rab3_TOMO_synapsin_dimer - Rab3_TOMO_synapsin_dimer
Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_dimerise_a = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_dimerise_a', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_dimerise_b = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_dimerise_b', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_dimerise_c = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_dimerise_c', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_dimerise_d = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_dimerise_d', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_dimerise_e = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_dimerise_e', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_dimerise_f = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_dimerise_f', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_unbind = smod.VesUnbind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_unbind', vsys, Rab3_TOMO_synapsin_dimer, Rab3_TOMO_synapsin_dimer, ves, Rab3_TOMO_synapsin, ves, Rab3_TOMO_synapsin, koff_synapsin_dimer, immobility=0)

#Rab3_TOMO_synapsin - Rab3_TOMO_synapsin_p -> Rab3_TOMO_synapsin_dimer - Rab3_TOMO_synapsin_dimer_p
Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_dimerise_a = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_dimerise_a', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_dimerise_b = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_dimerise_b', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_dimerise_c = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_dimerise_c', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_dimerise_d = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_dimerise_d', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_dimerise_e = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_dimerise_e', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_dimerise_f = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_dimerise_f', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_unbind = smod.VesUnbind('Rab3_TOMO_synapsin_Rab3_TOMO_synapsin_p_unbind', vsys, Rab3_TOMO_synapsin_dimer, Rab3_TOMO_synapsin_dimer_p, ves, Rab3_TOMO_synapsin, ves, Rab3_TOMO_synapsin_p, koff_synapsin_dimer_p, immobility=0)

#Rab3_TOMO_synapsin - Rab3_TOMO_p_synapsin -> Rab3_TOMO_synapsin_dimer - Rab3_TOMO_p_synapsin_dimer
Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_dimerise_a = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_dimerise_a', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_dimerise_b = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_dimerise_b', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_dimerise_c = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_dimerise_c', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_dimerise_d = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_dimerise_d', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_dimerise_e = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_dimerise_e', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_dimerise_f = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_dimerise_f', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

Rab3_TOMO_synapsin_Rab3_TOMO_p_unbind = smod.VesUnbind('Rab3_TOMO_synapsin_Rab3_TOMO_p_unbind', vsys, Rab3_TOMO_synapsin_dimer, Rab3_TOMO_p_synapsin_dimer, ves, Rab3_TOMO_synapsin, ves, Rab3_TOMO_p_synapsin, koff_synapsin_dimer, immobility=0)

#Rab3_TOMO_synapsin - Rab3_TOMO_p_synapsin_p -> Rab3_TOMO_synapsin_dimer - Rab3_TOMO_p_synapsin_dimer_p
Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_a = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_a', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_b = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_b', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_c = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_c', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_d = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_d', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_e = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_e', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_f = smod.VesBind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_f', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_unbind = smod.VesUnbind('Rab3_TOMO_synapsin_Rab3_TOMO_p_synapsin_p_unbind', vsys, Rab3_TOMO_synapsin_dimer, Rab3_TOMO_p_synapsin_dimer_p, ves, Rab3_TOMO_synapsin, ves, Rab3_TOMO_p_synapsin_p, koff_synapsin_dimer_p, immobility=0)

#Rab3_TOMO_synapsin_p - Rab3_TOMO_synapsin_p -> Rab3_TOMO_synapsin_dimer_p - Rab3_TOMO_synapsin_dimer_p
Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_dimerise_a = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_dimerise_a', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_dimerise_b = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_dimerise_b', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_dimerise_c = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_dimerise_c', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_dimerise_d = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_dimerise_d', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_dimerise_e = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_dimerise_e', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_dimerise_f = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_dimerise_f', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_synapsin_p, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_unbind = smod.VesUnbind('Rab3_TOMO_synapsin_p_Rab3_TOMO_synapsin_p_unbind', vsys, Rab3_TOMO_synapsin_dimer_p, Rab3_TOMO_synapsin_dimer_p, ves, Rab3_TOMO_synapsin_p, ves, Rab3_TOMO_synapsin_p, koff_synapsin_dimer_p, immobility=0)

#Rab3_TOMO_synapsin_p - Rab3_TOMO_p_synapsin -> Rab3_TOMO_synapsin_dimer_p - Rab3_TOMO_p_synapsin_dimer
Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_dimerise_a = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_dimerise_a', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_dimerise_b = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_dimerise_b', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_dimerise_c = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_dimerise_c', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_dimerise_d = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_dimerise_d', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_dimerise_e = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_dimerise_e', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_dimerise_f = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_dimerise_f', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_unbind = smod.VesUnbind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_unbind', vsys, Rab3_TOMO_synapsin_dimer_p, Rab3_TOMO_p_synapsin_dimer, ves, Rab3_TOMO_synapsin_p, ves, Rab3_TOMO_p_synapsin, koff_synapsin_dimer_p, immobility=0)

#Rab3_TOMO_synapsin_p - Rab3_TOMO_p_synapsin_p -> Rab3_TOMO_synapsin_dimer_p - Rab3_TOMO_p_synapsin_dimer_p
Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_a = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_a', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_b = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_b', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_c = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_c', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_d = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_d', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_e = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_e', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_f = smod.VesBind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_f', vsys,  ves1=ves, r1=Rab3_TOMO_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_unbind = smod.VesUnbind('Rab3_TOMO_synapsin_p_Rab3_TOMO_p_synapsin_p_unbind', vsys, Rab3_TOMO_synapsin_dimer_p, Rab3_TOMO_p_synapsin_dimer_p, ves, Rab3_TOMO_synapsin_p, ves, Rab3_TOMO_p_synapsin_p, koff_synapsin_dimer_p, immobility=0)

#Rab3_TOMO_p_synapsin - Rab3_TOMO_p_synapsin -> Rab3_TOMO_p_synapsin_dimer - Rab3_TOMO_p_synapsin_dimer
Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_dimerise_a = smod.VesBind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_dimerise_a', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_p_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_dimerise_b = smod.VesBind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_dimerise_b', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_p_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_dimerise_c = smod.VesBind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_dimerise_c', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_p_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_dimerise_d = smod.VesBind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_dimerise_d', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_p_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_dimerise_e = smod.VesBind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_dimerise_e', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_p_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_dimerise_f = smod.VesBind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_dimerise_f', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin, l1=Rab3_TOMO_p_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_unbind = smod.VesUnbind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_unbind', vsys, Rab3_TOMO_p_synapsin_dimer, Rab3_TOMO_p_synapsin_dimer, ves, Rab3_TOMO_p_synapsin, ves, Rab3_TOMO_p_synapsin, koff_synapsin_dimer, immobility=0)

#Rab3_TOMO_p_synapsin - Rab3_TOMO_p_synapsin_p -> Rab3_TOMO_p_synapsin_dimer - Rab3_TOMO_p_synapsin_dimer_p
Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_a = smod.VesBind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_a', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_p_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_b = smod.VesBind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_b', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_p_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_c = smod.VesBind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_c', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_p_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_d = smod.VesBind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_d', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_p_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_e = smod.VesBind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_e', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_p_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_f = smod.VesBind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_dimerise_f', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_p_synapsin_dimer, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_unbind = smod.VesUnbind('Rab3_TOMO_p_synapsin_Rab3_TOMO_p_synapsin_p_unbind', vsys, Rab3_TOMO_p_synapsin_dimer, Rab3_TOMO_p_synapsin_dimer_p, ves, Rab3_TOMO_p_synapsin, ves, Rab3_TOMO_p_synapsin_p, koff_synapsin_dimer_p, immobility=0)

#Rab3_TOMO_p_synapsin_p - Rab3_TOMO_p_synapsin_p -> Rab3_TOMO_p_synapsin_dimer_p - Rab3_TOMO_p_synapsin_dimer_p
Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_a = smod.VesBind('Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_a', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_p_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_b = smod.VesBind('Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_b', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_p_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_c = smod.VesBind('Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_c', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_p_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_d = smod.VesBind('Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_d', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_p_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_e = smod.VesBind('Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_e', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_p_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer], kcst=kon_synapsin_dimer, immobility=0)
Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_f = smod.VesBind('Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_dimerise_f', vsys,  ves1=ves, r1=Rab3_TOMO_p_synapsin_p,  ves2=ves, r2=Rab3_TOMO_p_synapsin_p, l1=Rab3_TOMO_p_synapsin_dimer_p, l2 = Rab3_TOMO_p_synapsin_dimer_p, length_max = 10e-8, length_min = 0, ldeps1 = [Rab3_TOMO_p_synapsin_dimer_p], kcst=kon_synapsin_dimer, immobility=0)

Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_unbind = smod.VesUnbind('Rab3_TOMO_p_synapsin_p_Rab3_TOMO_p_synapsin_p_unbind', vsys, Rab3_TOMO_p_synapsin_dimer_p, Rab3_TOMO_p_synapsin_dimer_p, ves, Rab3_TOMO_p_synapsin_p, ves, Rab3_TOMO_p_synapsin_p, koff_synapsin_dimer_p, immobility=0)

#Enzyme-complex states of ves-bound dimerised synapsins (assume the complexed enzyme blocks the dimerisation/unbinding reaction)
PKA_synapsin_dimer = smod.LinkSpec('PKA_synapsin_dimer', syt_model, 0.1e-12)
Rab3_TOMO_synapsin_dimer_PKA = smod.LinkSpec('Rab3_TOMO_synapsin_dimer_PKA', syt_model,   0.1e-12)
Rab3_TOMO_p_synapsin_dimer_PKA = smod.LinkSpec('Rab3_TOMO_p_synapsin_dimer_PKA', syt_model,   0.1e-12)
#PKA phos.
PKA_syndim_on = smod.VesSReac('PKA_syndim_on', ves_ssys, olhs = [PKA], llhs = [synapsin_dimer], lrhs = [PKA_synapsin_dimer], kcst = kon_PKA_syn)
PKA_syndim_off = smod.VesSReac('PKA_syndim_off', ves_ssys, llhs = [PKA_synapsin_dimer], orhs = [PKA], lrhs = [synapsin_dimer], kcst = koff_PKA_syn)
PKA_syndim_phos = smod.VesSReac('PKA_syndim_phos', ves_ssys, llhs = [PKA_synapsin_dimer], orhs = [PKA], lrhs = [synapsin_dimer_p], kcst = kcat_PKA_syn)

Rab3_TOMO_synapsin_dimer_PKA_on = smod.VesSReac('Rab3_TOMO_synapsin_dimer_PKA_on', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer], olhs = [PKA], lrhs = [Rab3_TOMO_synapsin_dimer_PKA], kcst = kon_PKA_syn)
Rab3_TOMO_synapsin_dimer_PKA_off = smod.VesSReac('Rab3_TOMO_synapsin_dimer_PKA_off', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer_PKA], orhs = [PKA], lrhs = [Rab3_TOMO_synapsin_dimer], kcst = koff_PKA_syn)
Rab3_TOMO_synapsin_dimer_PKA_cat = smod.VesSReac('Rab3_TOMO_synapsin_dimer_PKA_cat', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer_PKA], orhs = [PKA], lrhs = [Rab3_TOMO_synapsin_dimer_p], kcst = kcat_PKA_syn)

Rab3_TOMO_p_synapsin_dimer_PKA_on = smod.VesSReac('Rab3_TOMO_p_synapsin_dimer_PKA_on', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer], olhs = [PKA], lrhs = [Rab3_TOMO_p_synapsin_dimer_PKA], kcst = kon_PKA_syn)
Rab3_TOMO_p_synapsin_dimer_PKA_off = smod.VesSReac('Rab3_TOMO_p_synapsin_dimer_PKA_off', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer_PKA], orhs = [PKA], lrhs = [Rab3_TOMO_p_synapsin_dimer], kcst = koff_PKA_syn)
Rab3_TOMO_p_synapsin_dimer_PKA_cat = smod.VesSReac('Rab3_TOMO_p_synapsin_dimer_PKA_cat', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer_PKA], orhs = [PKA], lrhs = [Rab3_TOMO_p_synapsin_dimer_p], kcst = kcat_PKA_syn)

#PP2A dephos.
PP2A_synapsin_dimer_p = smod.LinkSpec('PP2A_synapsin_dimer_p', syt_model, 0.1e-12)
Rab3_TOMO_synapsin_dimer_p_PP2A = smod.LinkSpec('Rab3_TOMO_synapsin_dimer_p_PP2A', syt_model, 0.1e-12)
Rab3_TOMO_p_synapsin_dimer_p_PP2A = smod.LinkSpec('Rab3_TOMO_p_synapsin_dimer_p_PP2A', syt_model, 0.1e-12)

pp2a_syndim_on = smod.VesSReac('pp2a_syndim_on', ves_ssys, olhs = [PP2A], llhs = [synapsin_dimer_p], lrhs = [PP2A_synapsin_dimer_p], kcst = kon_pp2a_synapsin)
pp2a_syndim_off = smod.VesSReac('pp2a_syndim_off', ves_ssys, llhs = [PP2A_synapsin_dimer_p], orhs = [PP2A], lrhs = [synapsin_dimer_p], kcst = koff_pp2a_synapsin)
pp2a_syndim_dephos = smod.VesSReac('pp2a_syndim_dephos', ves_ssys, llhs = [PP2A_synapsin_dimer_p], orhs = [PP2A], lrhs = [synapsin_dimer], kcst = kcat_pp2a_syn)

Rab3_TOMO_synapsin_dimer_p_PP2A_on = smod.VesSReac('Rab3_TOMO_synapsin_dimer_p_PP2A_on', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer_p], olhs = [PP2A], lrhs = [Rab3_TOMO_synapsin_dimer_p_PP2A], kcst = kon_pp2a_synapsin)
Rab3_TOMO_synapsin_dimer_p_PP2A_off = smod.VesSReac('Rab3_TOMO_synapsin_dimer_p_PP2A_off', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer_p_PP2A], orhs = [PP2A], lrhs = [Rab3_TOMO_synapsin_dimer_p], kcst = koff_pp2a_synapsin)
Rab3_TOMO_synapsin_dimer_p_PP2A_cat = smod.VesSReac('Rab3_TOMO_synapsin_dimer_p_PP2A_cat', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer_p_PP2A], orhs = [PP2A], lrhs = [Rab3_TOMO_synapsin_dimer], kcst = kcat_pp2a_syn)

Rab3_TOMO_p_synapsin_dimer_p_PP2A_on = smod.VesSReac('Rab3_TOMO_p_synapsin_dimer_p_PP2A_on', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer_p], olhs = [PP2A], lrhs = [Rab3_TOMO_p_synapsin_dimer_p_PP2A], kcst = kon_pp2a_synapsin)
Rab3_TOMO_p_synapsin_dimer_p_PP2A_off = smod.VesSReac('Rab3_TOMO_p_synapsin_dimer_p_PP2A_off', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer_p_PP2A], orhs = [PP2A], lrhs = [Rab3_TOMO_p_synapsin_dimer_p], kcst = koff_pp2a_synapsin)
Rab3_TOMO_p_synapsin_dimer_p_PP2A_cat = smod.VesSReac('Rab3_TOMO_p_synapsin_dimer_p_PP2A_cat', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer_p_PP2A], orhs = [PP2A], lrhs = [Rab3_TOMO_p_synapsin_dimer], kcst = kcat_pp2a_syn)

#CDK5 phos.
Rab3_TOMO_synapsin_dimer_CDK5 = smod.LinkSpec('Rab3_TOMO_synapsin_dimer_CDK5', syt_model,   0.1e-12)
Rab3_TOMO_synapsin_dimer_p_CDK5 = smod.LinkSpec('Rab3_TOMO_synapsin_dimer_p_CDK5', syt_model,   0.1e-12)

Rab3_TOMO_synapsin_dimer_CDK5_on = smod.VesSReac('Rab3_TOMO_synapsin_dimer_CDK5_on', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer], olhs = [CDK5], lrhs = [Rab3_TOMO_synapsin_dimer_CDK5], kcst = kon_cdk_tomo)
Rab3_TOMO_synapsin_dimer_CDK5_off = smod.VesSReac('Rab3_TOMO_synapsin_dimer_CDK5_off', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer_CDK5], orhs = [CDK5], lrhs = [Rab3_TOMO_synapsin_dimer], kcst = koff_cdk_tomo)
Rab3_TOMO_synapsin_dimer_CDK5_cat = smod.VesSReac('Rab3_TOMO_synapsin_dimer_CDK5_cat', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer_CDK5], orhs = [CDK5], lrhs = [Rab3_TOMO_p_synapsin_dimer], kcst = kcat_cdk_tomo)

Rab3_TOMO_synapsin_dimer_p_CDK5_on = smod.VesSReac('Rab3_TOMO_synapsin_dimer_p_CDK5_on', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer_p], olhs = [CDK5], lrhs = [Rab3_TOMO_synapsin_dimer_p_CDK5], kcst = kon_cdk_tomo)
Rab3_TOMO_synapsin_dimer_p_CDK5_off = smod.VesSReac('Rab3_TOMO_synapsin_dimer_p_CDK5_off', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer_p_CDK5], orhs = [CDK5], lrhs = [Rab3_TOMO_synapsin_dimer_p], kcst = koff_cdk_tomo)
Rab3_TOMO_synapsin_dimer_p_CDK5_cat = smod.VesSReac('Rab3_TOMO_synapsin_dimer_p_CDK5_cat', ves_ssys, llhs = [Rab3_TOMO_synapsin_dimer_p_CDK5], orhs = [CDK5], lrhs = [Rab3_TOMO_p_synapsin_dimer_p], kcst = kcat_cdk_tomo)
#CaN dephos.
Rab3_TOMO_p_synapsin_dimer_CaN_CaM = smod.LinkSpec('Rab3_TOMO_p_synapsin_dimer_CaN_CaM', syt_model,   0.1e-12)
Rab3_TOMO_p_synapsin_dimer_p_CaN_CaM = smod.LinkSpec('Rab3_TOMO_p_synapsin_dimer_p_CaN_CaM', syt_model,   0.1e-12)

Rab3_TOMO_p_synapsin_dimer_CaN_on = smod.VesSReac('Rab3_TOMO_p_synapsin_dimer_CaN_on', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer], olhs = [CaN_CaM], lrhs = [Rab3_TOMO_p_synapsin_dimer_CaN_CaM], kcst = kon_can_tomo)
Rab3_TOMO_p_synapsin_dimer_CaN_off = smod.VesSReac('Rab3_TOMO_p_synapsin_dimer_CaN_off', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer_CaN_CaM], orhs = [CaN_CaM], lrhs = [Rab3_TOMO_p_synapsin_dimer], kcst = koff_can_tomo)
Rab3_TOMO_p_synapsin_dimer_CaN_cat = smod.VesSReac('Rab3_TOMO_p_synapsin_dimer_CaN_cat', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer_CaN_CaM], orhs = [CaN_CaM], lrhs = [Rab3_TOMO_synapsin_dimer], kcst = kcat_can_tomo)

Rab3_TOMO_p_synapsin_dimer_p_CaN_on = smod.VesSReac('Rab3_TOMO_p_synapsin_dimer_p_CaN_on', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer_p], olhs = [CaN_CaM], lrhs = [Rab3_TOMO_p_synapsin_dimer_p_CaN_CaM], kcst = kon_can_tomo)
Rab3_TOMO_p_synapsin_dimer_p_CaN_off = smod.VesSReac('Rab3_TOMO_p_synapsin_dimer_p_CaN_off', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer_p_CaN_CaM], orhs = [CaN_CaM], lrhs = [Rab3_TOMO_p_synapsin_dimer_p], kcst = koff_can_tomo)
Rab3_TOMO_p_synapsin_dimer_p_CaN_cat = smod.VesSReac('Rab3_TOMO_p_synapsin_dimer_p_CaN_cat', ves_ssys, llhs = [Rab3_TOMO_p_synapsin_dimer_p_CaN_CaM], orhs = [CaN_CaM], lrhs = [Rab3_TOMO_synapsin_dimer_p], kcst = kcat_can_tomo)

# #2. Inhibition of vesicle priming (competition with SYB, same rate)
kon_tomo_syx = 2.35e06
koff_tomo_syx = 0.0047
kdisp_tomo_syx = 10e06
kflip_tomo_f = 0.01
kflip_tomo_P_f = 10
kflip_tomo_r = 1
kon_tomo_pka = 10e06
koff_tomo_pka = 1
kcat_pka_tomo = 50

kon_tomo_PP2A = 2e06 #Gallimore, Cell Reports, 2018 (from PP2A CaMKII dephos parameters)
koff_tomo_PP2A = 8
kcat_PP2A_tomo = 2

RIM_M13_Rab3_SYX_M18_SNP25_TOMO = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25_TOMO', syt_model)
RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p', syt_model)
RIM_M13_Rab3_SYX_M18_SNP25_TOMOx = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25_TOMOx', syt_model)
RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p', syt_model)
RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P', syt_model)
RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_P = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_P', syt_model)
RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P', syt_model)
RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p_P = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p_P', syt_model)
RIM_M13_Rab3_SYX_M18_SNP25_TOMO_PKA = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25_TOMO_PKA', syt_model)
RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_PKA = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_PKA', syt_model)
RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P_PP2A = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P_PP2A', syt_model)
RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P_PP2A = smod.Spec('RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P_PP2A', syt_model)

sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO = smod.Diff('sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO', memsys, RIM_M13_Rab3_SYX_M18_SNP25_TOMO,  0)
sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p = smod.Diff('sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p', memsys, RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p,  0)
sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx = smod.Diff('sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx', memsys, RIM_M13_Rab3_SYX_M18_SNP25_TOMOx,  0)
sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p = smod.Diff('sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p', memsys, RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p,  0)
sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P = smod.Diff('sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P', memsys, RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P,  0)
sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_P = smod.Diff('sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_P', memsys, RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_P,  0)
sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P = smod.Diff('sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P', memsys, RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P,  0)
sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p_P = smod.Diff('sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p_P', memsys, RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p_P,  0)
sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_PKA = smod.Diff('sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_PKA', memsys, RIM_M13_Rab3_SYX_M18_SNP25_TOMO_PKA,  0)
sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_PKA = smod.Diff('sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_PKA', memsys, RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_PKA,  0)
sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P_PP2A = smod.Diff('sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P_PP2A', memsys, RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P_PP2A,  0)
sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P_PP2A = smod.Diff('sdiff_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P_PP2A', memsys, RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P_PP2A,  0)

#binding of TOMO to SNARE (independent of CDK5 phos state, indicated by small p)
#######TOMO is incorporated into the SNARE complex (binds to SYX_SNP25 complex)
syx_tomo_on = smod.VesSReac('syx_tomo_on', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25], olhs = [TOMO], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO], kcst=kon_tomo_syx)
syx_tomo_off = smod.VesSReac('syx_tomo_off', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO], orhs = [TOMO], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25], kcst=koff_tomo_syx)

syx_tomo_p_on = smod.VesSReac('syx_tomo_p_on', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25], olhs = [TOMO_p], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p], kcst=kon_tomo_syx)
syx_tomo_p_off = smod.VesSReac('syx_tomo_p_off', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p], orhs = [TOMO_p], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25], kcst=koff_tomo_syx)

#switching of TOMO to SYB-displaceable state (unphosphorylated, slow)
flip_tomo_f = smod.VesSReac('flip_tomo_f', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx], kcst=kflip_tomo_f)
flip_tomo_r = smod.VesSReac('flip_tomo_r', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO], kcst=kflip_tomo_r)

flip_tomo_p_f = smod.VesSReac('flip_tomo_p_f', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p], kcst=kflip_tomo_f)
flip_tomo_p_r = smod.VesSReac('flip_tomo_p_r', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p], kcst=kflip_tomo_r)

#switching of TOMO to SYB-displaceable state (phosphorylated by PKA indicated by big P, fast)
flip_tomo_P_f = smod.VesSReac('flip_tomo_P_f', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_P], kcst=kflip_tomo_P_f)
flip_tomo_P_r = smod.VesSReac('flip_tomo__Pr', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_P], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P], kcst=kflip_tomo_r)

flip_tomo_p_P_f = smod.VesSReac('flip_tomo_p_P_f', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p], kcst=kflip_tomo_P_f)
flip_tomo_p_P_r = smod.VesSReac('flip_tomo_p_P_r', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p_P], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P], kcst=kflip_tomo_r)

#displacement of TOMO by SYB
disp_tomo_f = smod.VesSReac('disp_tomo_f', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx, SYB], vrhs = [SNARE, TOMO], kcst=kdisp_tomo_syx)
disp_tomo_p_f = smod.VesSReac('disp_tomo_p_f', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p, SYB], vrhs = [SNARE, TOMO_p], kcst=kdisp_tomo_syx)

#displacement of TOMO_P by SYB
disp_tomo_P_f = smod.VesSReac('disp_tomo_P_f', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_P, SYB], vrhs = [SNARE, TOMO], kcst=kdisp_tomo_syx)
disp_tomo_p_P_f = smod.VesSReac('disp_tomo_p_P_f', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p_P, SYB], vrhs = [SNARE, TOMO_p], kcst=kdisp_tomo_syx)

#phosphorylation of TOMO by PKA
tomo_pka_on = smod.VesSReac('tomo_pka_on', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO], olhs = [PKA], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_PKA], kcst=kon_tomo_pka)
tomo_pka_off = smod.VesSReac('tomo_pka_off', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_PKA], orhs = [PKA], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO], kcst=koff_tomo_pka)
tomo_pka_cat = smod.VesSReac('tomo_pka_cat', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_PKA], orhs = [PKA], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P], kcst=kcat_pka_tomo)

tomo_p_pka_on = smod.VesSReac('tomo_p_pka_on', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p], olhs = [PKA], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_PKA], kcst=kon_tomo_pka)
tomo_p_pka_off = smod.VesSReac('tomo_p_pka_off', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_PKA], orhs = [PKA], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p], kcst=koff_tomo_pka)
tomo_p_pka_cat = smod.VesSReac('tomo_p_pka_cat', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_PKA], orhs = [PKA], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P], kcst=kcat_pka_tomo)

#dephosphorylation of TOMO by PP2A
tomo_PP2A_on = smod.VesSReac('tomo_PP2A_on', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P], olhs = [PP2A], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P_PP2A], kcst=kon_tomo_PP2A)
tomo_PP2A_off = smod.VesSReac('tomo_PP2A_off', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P_PP2A], orhs = [PP2A], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P], kcst=koff_tomo_PP2A)
tomo_PP2A_cat = smod.VesSReac('tomo_PP2A_cat', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P_PP2A], orhs = [PP2A], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO], kcst=kcat_PP2A_tomo)

tomo_p_PP2A_on = smod.VesSReac('tomo_p_PP2A_on', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P], olhs = [PP2A], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P_PP2A], kcst=kon_tomo_PP2A)
tomo_p_PP2A_off = smod.VesSReac('tomo_p_PP2A_off', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P_PP2A], orhs = [PP2A], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P], kcst=koff_tomo_PP2A)
tomo_p_PP2A_cat = smod.VesSReac('tomo_p_PP2A_cat', ves_ssys, vlhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P_PP2A], orhs = [PP2A], vrhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p], kcst=kcat_PP2A_tomo)

#####BINDING OF PROTEINS TO VESICLES#######
######Binding of synapsin to vesicles
kon_synapsin_ves = 10.7e06
koff_synapsin_ves = 0.123
koff_synapsin_p_ves = 123
synapsin_ves_on = smod.VesSReac('synapsin_ves_on', ves_ssys, vlhs = [ves_syn1_site], olhs = [synapsin], vrhs = [synapsin], kcst = kon_synapsin_ves)
synapsin_ves_off = smod.VesSReac('synapsin_ves_off', ves_ssys, vlhs = [synapsin], orhs = [synapsin], vrhs = [ves_syn1_site], kcst = koff_synapsin_ves)
synapsin_ves_p_off = smod.VesSReac('synapsin_ves_p_off', ves_ssys, vlhs = [synapsin_p], orhs = [synapsin_p], vrhs = [ves_syn1_site], kcst = koff_synapsin_p_ves)
#####Binding of Complexin to vesicles, Zdanowicz 2017######
#CXN = smod.Spec('CXN', syt_model)
ves_CXN_site = smod.Spec('ves_CXN_site', syt_model)
sdiff_CXN_site = smod.Diff('sdiff_CXN_site', memsys, ves_CXN_site, 0)

if ves_protein_bind_on:
    kon_cxn_ves = 1.2e06
else:
    kon_cxn_ves = 0
koff_cxn_ves = 1000

cxn_ves_on = smod.VesSReac('cxn_ves_on', ves_ssys, vlhs = [ves_CXN_site], olhs = [CXN], vrhs = [CXN], kcst = kon_cxn_ves)
cxn_ves_off = smod.VesSReac('cxn_ves_off', ves_ssys, vlhs = [CXN], orhs = [CXN], vrhs = [ves_CXN_site], kcst = koff_cxn_ves)

#####Binding of Rab3 to vesicles, Zdanowicz 2017######
ves_rab3_site = smod.Spec('ves_rab3_site', syt_model)
sdiff_rab3_site = smod.Diff('sdiff_rab3_site', memsys, ves_rab3_site, 0)

if ves_protein_bind_on:
    kon_rab3_ves = 10e06
else:
    kon_rab3_ves = 0

koff_rab3_ves = 1

rab3_ves_on = smod.VesSReac('rab3_ves_on', ves_ssys, vlhs = [ves_rab3_site], olhs = [Rab3], vrhs = [Rab3], kcst = kon_rab3_ves)
rab3_ves_off = smod.VesSReac('rab3_ves_off', ves_ssys, vlhs = [Rab3], orhs = [Rab3], vrhs = [ves_rab3_site], kcst = koff_rab3_ves)

#####Binding of M13/18 to vesicles, Quade 2019#####
ves_m13_site = smod.Spec('ves_m13_site', syt_model)
sdiff_M13_site = smod.Diff('sdiff_M13_site', memsys, ves_m13_site, 0)

if ves_protein_bind_on:
    kon_m13_ves = 1.6e06
else:
    kon_m13_ves = 0

koff_m13_ves = 1000

m13_ves_on = smod.VesSReac('m13_ves_on', ves_ssys, vlhs = [ves_m13_site], olhs = [M13], vrhs = [M13], kcst = kon_m13_ves)
m13_ves_off = smod.VesSReac('m13_ves_off', ves_ssys, vlhs = [M13], orhs = [M13], vrhs = [ves_m13_site], kcst = koff_m13_ves)

ves_m18_site = smod.Spec('ves_m18_site', syt_model)
sdiff_M18_site = smod.Diff('sdiff_M18_site', memsys, ves_m18_site, 0)

if ves_protein_bind_on:
    kon_m18_ves = 1.6e06
else:
    kon_m18_ves = 0

koff_m18_ves = 1000

m18_ves_on = smod.VesSReac('m18_ves_on', ves_ssys, vlhs = [ves_m18_site], olhs = [M18], vrhs = [M18], kcst = kon_m18_ves)
m18_ves_off = smod.VesSReac('m18_ves_off', ves_ssys, vlhs = [M18], orhs = [M18], vrhs = [ves_m18_site], kcst = koff_m18_ves)

#####Binding of Syndapin1 to vesicles, Quan 2012#####
ves_syndap_site = smod.Spec('ves_syndap_site', syt_model)
sdiff_syndap_site = smod.Diff('sdiff_syndap_site', memsys, ves_syndap_site, 0)
sdiff_SYN1 = smod.Diff('sdiff_SYN1', memsys, SYN1,  0)
if ves_protein_bind_on:
    kon_syn1_ves = 2.8e06
else:
    kon_syn1_ves = 0
koff_syn1_ves = 1000

syn1_ves_on = smod.VesSReac('syn1_ves_on', ves_ssys, vlhs = [ves_syndap_site], olhs = [SYN1], vrhs = [SYN1], kcst = kon_syn1_ves)
syn1_ves_off = smod.VesSReac('syn1_ves_off', ves_ssys, vlhs = [SYN1], orhs = [SYN1], vrhs = [ves_syndap_site], kcst = koff_syn1_ves)

#####Binding of aSNAP to vesicles, Steel 1997#####
ves_asnap_site = smod.Spec('ves_asnap_site', syt_model)
sdiff_asnap_site = smod.Diff('sdiff_asnap_site', memsys, ves_asnap_site, 0)
sdiff_asnap = smod.Diff('sdiff_asnap', memsys, aSNAP, 0)

if ves_protein_bind_on:
    kon_asnap_ves = 2.8e06
else:
    kon_asnap_ves = 0
koff_asnap_ves = 1000

asnap_ves_on = smod.VesSReac('asnap_ves_on', ves_ssys, vlhs = [ves_asnap_site], olhs = [aSNAP], vrhs = [aSNAP], kcst = kon_asnap_ves)
asnap_ves_off = smod.VesSReac('asnap_ves_off', ves_ssys, vlhs = [aSNAP], orhs = [aSNAP], vrhs = [ves_asnap_site], kcst = koff_asnap_ves)

#####Binding of NSF to vesicles######
ves_nsf_site = smod.Spec('ves_nsf_site', syt_model)
sdiff_nsf_site = smod.Diff('sdiff_nsf_site', memsys, ves_nsf_site, 0)
sdiff_NSF = smod.Diff('sdiff_NSF', memsys, NSF, 0)
if ves_protein_bind_on:
    kon_nsf_ves = 0.9e06
else:
    kon_nsf_ves = 0
koff_nsf_ves = 1000

nsf_ves_on = smod.VesSReac('nsf_ves_on', ves_ssys, vlhs = [ves_nsf_site], olhs = [NSF], vrhs = [NSF], kcst = kon_nsf_ves)
nsf_ves_off = smod.VesSReac('nsf_ves_off', ves_ssys, vlhs = [NSF], orhs = [NSF], vrhs = [ves_nsf_site], kcst = koff_nsf_ves)

#####Remove vesicle-protein sites and other proteins from plasma membrane after vesicle fusion

remove_ves_syn1_site = smod.SReac('remove_ves_syn1_site', memsys, slhs = [ves_syn1_site], kcst = 1e06)
remove_ves_CXN_site = smod.SReac('remove_ves_CXN_site', memsys, slhs = [ves_CXN_site], kcst = 1e06)
remove_ves_m13_site = smod.SReac('remove_ves_m13_site', memsys, slhs = [ves_m13_site], kcst = 1e06)
remove_ves_m18_site = smod.SReac('remove_ves_m18_site', memsys, slhs = [ves_m18_site], kcst = 1e06)
remove_ves_syndap_site = smod.SReac('remove_ves_syndap_site', memsys, slhs = [ves_syndap_site], kcst = 1e06)
remove_ves_asnap_site = smod.SReac('remove_ves_asnap_site', memsys, slhs = [ves_asnap_site], kcst = 1e06)
remove_ves_nsf_site = smod.SReac('remove_ves_nsf_site', memsys, slhs = [ves_nsf_site], kcst = 1e06)
remove_ves_rab3_site = smod.SReac('remove_ves_rab3_site', memsys, slhs = [ves_rab3_site], kcst = 1e06)
remove_nsf = smod.SReac('remove_nsf', memsys, slhs = [NSF], irhs = [NSF], kcst = 1e06)

remove_synapsin = smod.SReac('remove_synapsin', memsys, slhs = [synapsin], irhs = [synapsin], kcst = 1e06)
remove_synapsin_p = smod.SReac('remove_synapsin_p', memsys, slhs = [synapsin_p], irhs = [synapsin], kcst = 1e06)
remove_PKA_synapsin = smod.SReac('remove_PKA_synapsin', memsys, slhs = [PKA_synapsin], irhs = [PKA_synapsin], kcst = 1e06)
remove_PP2A_synapsin_p = smod.SReac('remove_PP2A_synapsin_p', memsys, slhs = [PP2A_synapsin_p], irhs = [PP2A_synapsin_p], kcst = 1e06)

remove_TOMO_p = smod.SReac('remove_TOMO_p', memsys, slhs = [TOMO_p], irhs = [TOMO_p], kcst = 1e06)
Rab3_TOMO_p_diss_reac = smod.SReac('Rab3_TOMO_p_diss_reac', memsys, slhs = [Rab3_TOMO_p], irhs = [TOMO_p, Rab3], kcst = 1e06)
Rab3_TOMO_diss_reac = smod.SReac('Rab3_TOMO_diss_reac', memsys, slhs = [Rab3_TOMO], irhs = [TOMO, Rab3], kcst = 1e06)
Rab3_TOMO_p_synapsin_diss_reac = smod.SReac('Rab3_TOMO_p_synapsin_diss_reac', memsys, slhs = [Rab3_TOMO_p_synapsin], irhs = [Rab3, TOMO_p, synapsin], kcst = 1e06)
Rab3_TOMO_synapsin_diss_reac = smod.SReac('Rab3_TOMO_synapsin_diss_reac', memsys, slhs = [Rab3_TOMO_synapsin], irhs = [Rab3, TOMO, synapsin], kcst = 1e06)
Rab3_TOMO_p_synapsin_p_diss_reac = smod.SReac('Rab3_TOMO_p_synapsin_p_diss_reac', memsys, slhs = [Rab3_TOMO_p_synapsin_p], irhs = [Rab3, TOMO_p, synapsin_p], kcst = 1e06)
Rab3_TOMO_synapsin_p_diss_reac = smod.SReac('Rab3_TOMO_synapsin_p_diss_reac', memsys, slhs = [Rab3_TOMO_synapsin_p], irhs = [Rab3, TOMO, synapsin_p], kcst = 1e06)

remove_actin = smod.SReac('remove_actin', memsys, slhs = [actin], irhs = [actin], kcst = 1e06)
remove_synapsin_actin = smod.SReac('remove_synapsin_actin', memsys, slhs = [synapsin_actin], irhs = [synapsin, actin], kcst = 1e06)
remove_Rab3_TOMO_synapsin_actin = smod.SReac('remove_Rab3_TOMO_synapsin_actin', memsys, slhs = [Rab3_TOMO_synapsin_actin], irhs = [Rab3, TOMO, synapsin, actin], kcst = 1e06)
remove_Rab3_TOMO_p_synapsin_actin = smod.SReac('remove_Rab3_TOMO_p_synapsin_actin', memsys, slhs = [Rab3_TOMO_p_synapsin_actin], irhs = [Rab3, TOMO_p, synapsin, actin], kcst = 1e06)
remove_Rab3_TOMO_synapsin_PKA = smod.SReac('remove_Rab3_TOMO_synapsin_PKA', memsys, slhs = [Rab3_TOMO_synapsin_PKA], irhs = [Rab3, TOMO, synapsin, PKA], kcst = 1e06)
remove_Rab3_TOMO_p_synapsin_PKA = smod.SReac('remove_Rab3_TOMO_p_synapsin_PKA', memsys, slhs = [Rab3_TOMO_p_synapsin_PKA], irhs = [Rab3, TOMO_p, synapsin, PKA], kcst = 1e06)
# remove_PKA_synapsin_actin = smod.SReac('remove_PKA_synapsin_actin', memsys, slhs = [PKA_synapsin_actin], irhs = [PKA, synapsin, actin], kcst = 1e06)
remove_synapsin_actin_p = smod.SReac('remove_synapsin_actin_p', memsys, slhs = [synapsin_actin_p], irhs = [synapsin, actin], kcst = 1e06)
remove_Rab3_TOMO_p_CaN_CaM = smod.SReac('remove_Rab3_TOMO_p_CaN_CaM', memsys, slhs = [Rab3_TOMO_p_CaN_CaM], irhs = [Rab3, TOMO_p, CaN_CaM], kcst = 1e06)
remove_Rab3_TOMO_p_synapsin_CaN_CaM = smod.SReac('remove_Rab3_TOMO_p_synapsin_CaN_CaM', memsys, slhs = [Rab3_TOMO_p_synapsin_CaN_CaM], irhs = [Rab3, TOMO_p, synapsin, CaN_CaM], kcst = 1e06)
remove_Rab3_TOMO_p_synapsin_p_CaN_CaM = smod.SReac('remove_Rab3_TOMO_p_synapsin_p_CaN_CaM', memsys, slhs = [Rab3_TOMO_p_synapsin_p_CaN_CaM], irhs = [Rab3, TOMO_p, synapsin_p, CaN_CaM], kcst = 1e06)
remove_Rab3_TOMO_CDK5 = smod.SReac('remove_Rab3_TOMO_CDK5', memsys, slhs = [Rab3_TOMO_CDK5], irhs = [Rab3, TOMO, CDK5], kcst = 1e06)
remove_Rab3_TOMO_synapsin_CDK5 = smod.SReac('remove_Rab3_TOMO_synapsin_CDK5', memsys, slhs = [Rab3_TOMO_synapsin_CDK5], irhs = [Rab3, TOMO, synapsin, CDK5], kcst = 1e06)
remove_Rab3_TOMO_synapsin_p_CDK5 = smod.SReac('remove_Rab3_TOMO_synapsin_p_CDK5', memsys, slhs = [Rab3_TOMO_synapsin_p_CDK5], irhs = [Rab3, TOMO, synapsin_p, CDK5], kcst = 1e06)
remove_Rab3_TOMO_p_synapsin_p_PP2A = smod.SReac('remove_Rab3_TOMO_p_synapsin_p_PP2A', memsys, slhs = [Rab3_TOMO_p_synapsin_p_PP2A], irhs = [Rab3, TOMO_p, synapsin_p, PP2A], kcst = 1e06)
remove_Rab3_TOMO_synapsin_p_PP2A = smod.SReac('remove_Rab3_TOMO_synapsin_p_PP2A', memsys, slhs = [Rab3_TOMO_synapsin_p_PP2A], irhs = [Rab3, TOMO, synapsin_p, PP2A], kcst = 1e06)
# remove_PP2A_synapsin_actin_p = smod.SReac('remove_PP2A_synapsin_actin_p', memsys, slhs = [PP2A_synapsin_actin_p], irhs = [PP2A, synapsin, actin], kcst = 1e06)

remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO = smod.SReac('remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO', memsys, slhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO], srhs = [RIM_M13, SYX, SNP25], irhs = [Rab3, M18, TOMO], kcst = 1e06)
remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p = smod.SReac('remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p', memsys, slhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p], srhs = [RIM_M13, SYX, SNP25], irhs = [Rab3, M18, TOMO_p], kcst = 1e06)
remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx = smod.SReac('remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx', memsys, slhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx], srhs = [RIM_M13, SYX, SNP25], irhs = [Rab3, M18, TOMO], kcst = 1e06)
remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p = smod.SReac('remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p', memsys, slhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p], srhs = [RIM_M13, SYX, SNP25], irhs = [Rab3, M18, TOMO_p], kcst = 1e06)
remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P = smod.SReac('remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P', memsys, slhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P], srhs = [RIM_M13, SYX, SNP25], irhs = [Rab3, M18, TOMO], kcst = 1e06)
remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_P = smod.SReac('remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_P', memsys, slhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_P], srhs = [RIM_M13, SYX, SNP25], irhs = [Rab3, M18, TOMO], kcst = 1e06)
remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P = smod.SReac('remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P', memsys, slhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P], srhs = [RIM_M13, SYX, SNP25], irhs = [Rab3, M18, TOMO_p], kcst = 1e06)
remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p_P = smod.SReac('remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p_P', memsys, slhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p_P], srhs = [RIM_M13, SYX, SNP25], irhs = [Rab3, M18, TOMO_p], kcst = 1e06)
remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_PKA = smod.SReac('remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_PKA', memsys, slhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_PKA], srhs = [RIM_M13, SYX, SNP25], irhs = [Rab3, M18, TOMO, PKA], kcst = 1e06)
remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_PKA = smod.SReac('remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_PKA', memsys, slhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_PKA], srhs = [RIM_M13, SYX, SNP25], irhs = [Rab3, M18, TOMO_p, PKA], kcst = 1e06)
remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P_PP2A = smod.SReac('remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P_PP2A', memsys, slhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P_PP2A], srhs = [RIM_M13, SYX, SNP25], irhs = [Rab3, M18, TOMO, PP2A], kcst = 1e06)
remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P_PP2A = smod.SReac('remove_RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P_PP2A', memsys, slhs = [RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P_PP2A], srhs = [RIM_M13, SYX, SNP25], irhs = [Rab3, M18, TOMO_p, PP2A], kcst = 1e06)

####RELEASE INDICATOR##############
REL_IND = smod.Spec('REL_IND',syt_model)
diff_REL_IND = smod.Diff('diff_REL_IND', syn_sys, REL_IND, 0)
####################################
#FIND ALL SPECIES WITHOUT MEMBRANE DIFFUSION RULE
all_cyt_specs = vsys.getAllSpecs()
all_cyt_specs_ID = []
for s in all_cyt_specs:
    spec_id = s.getID()
    all_cyt_specs_ID.append(spec_id)
print('TOTAL CYTOSOL SPECIES: '+str( len(all_cyt_specs_ID)))

all_surf_specs = memsys.getAllSpecs()
all_surf_specs_ID = []
for s in all_surf_specs:
    spec_id = s.getID()
    all_surf_specs_ID.append(spec_id)
print('TOTAL MEMBRANE SPECIES: '+str(len(all_surf_specs_ID)))

all_ves_specs = ves_ssys.getAllSpecs()
all_ves_specs_ID = []
for s in all_ves_specs:
    spec_id = s.getID()
    all_ves_specs_ID.append(spec_id)
print('TOTAL VES SPECIES: '+str(len(all_ves_specs_ID)) )
#find intersect of vesicle species and surface species
all_cyt_specs_set = set(all_cyt_specs_ID)
all_ves_specs_set = set(all_ves_specs_ID)
all_ves_cyt_specs_set = all_cyt_specs_set.union(all_ves_specs_set)
all_surf_specs_set = set(all_surf_specs_ID)
ves_surf_spec_diff = all_ves_specs_set.difference(all_surf_specs_set)
print('There are '+str(len(ves_surf_spec_diff))+ ' ves species not in the surface system')
print('MISSING SURF SPECIES: ')
ves_surf_spec_diff_list = list(ves_surf_spec_diff)
for s in ves_surf_spec_diff_list:
    print(s)
print('END OF SPECIES')
all_raft_specs = raft_sys.getAllSpecs()
all_raft_specs_ID = []
for s in all_raft_specs:
    spec_id = s.getID()
    all_raft_specs_ID.append(spec_id)
print('TOTAL RAFT SPECIES: '+str( len(all_raft_specs_ID)) )
all_raft_specs_set = set(all_raft_specs_ID)
raft_surf_spec_diff = all_raft_specs_set.difference(all_surf_specs_set)
print('There are ' + str(len(raft_surf_spec_diff))+ ' raft species not in the surface system')
print('MISSING RAFT SPECIES: ')
raft_surf_spec_diff_list = list(raft_surf_spec_diff)
for s in raft_surf_spec_diff_list:
    print(s)
print('END OF SPECIES')

####Geometry Definition####
#import mesh
#coarse mesh
scale = 1e-9
scale_100 = 1e-7

mesh = smeshio.loadMesh(model_path+'pyr_axon_2021')[0]

ER_tets = mesh.getROIData("er")
cyt_tets_full= mesh.getROIData("spine")
outer_tets = mesh.getROIData("outer")
vac1_tets = mesh.getROIData("vec1")
vac2_tets = mesh.getROIData("vec2")
vac3_tets = mesh.getROIData("vec3")
vac4_tets = mesh.getROIData("vec4")
vac5_tets = mesh.getROIData("vec5")
mit_tets = mesh.getROIData("mit")
print("MESH LOADED")

axon_tets = []

#find min and max y-values of outer_tets bounding box
outer_tets_y = []
for t in outer_tets:
    center = mesh.getTetBarycenter(t)
    y_coord = center[1]
    outer_tets_y.append(y_coord)
cyt_y_min = min(outer_tets_y)
cyt_y_max = max(outer_tets_y)

print("cyt_y_min"+str(cyt_y_min))
print("cyt_y_max"+str(cyt_y_max))

print('FOUND OUTER TETS')

cyt_tets = []
for c in cyt_tets_full:
    center = mesh.getTetBarycenter(c)
    y_coord = center[1]
    if (y_coord < cyt_y_max) and (y_coord > cyt_y_min):
        cyt_tets.append(c)

for t in cyt_tets_full:
    if t not in cyt_tets:
        axon_tets.append(t)

total_tets = mesh.ntets

print('ALL tets in mesh: '+str (total_tets))
print('Total cytosol tets: '+str ( len(cyt_tets_full)))
print('Total axonal tets: '+str ( len(axon_tets)))
print('Total bouton tets: '+str ( len(cyt_tets)))
print('Sum bouton + axomal : '+str ( len(axon_tets)+len(cyt_tets)))
print('Sum bouton, axon, ER, outer: '+str ( len(cyt_tets_full)+len(ER_tets)+len(outer_tets)))

ERmemb_tris = meshctrl.findOverlapTris(mesh, cyt_tets, ER_tets)
memb_tris = meshctrl.findOverlapTris(mesh, cyt_tets, outer_tets)
print('Number of triangles on membrane = '+str(len(memb_tris)))

#open actin_tets list
print('OPENING ACTIN TETS...')

with open(model_path+'actin_tets_215nm.txt', 'rb') as f:
    actin_tets = pickle.load(f)
print('ACTIN TETS LOADED: '+str( len(actin_tets)) )

######FIND AZONE TRIANGLES######
azone_tris = []
#find membrane triangles within ellipse enclosing coordinates
#ellipse foci
focus1 = [2036.7e-9, 2982.6e-9]
focus2 = [2185.8e-9, 3280.8e-9]
#find memb triangles closest to foci
#focus1
distances = {}
for m in memb_tris:
    center_2d = [mesh.getTriBarycenter(m)[0], mesh.getTriBarycenter(m)[1]]
    dist = distance.euclidean(center_2d, focus1)
    distances[m] = dist
focus1_tri = min(distances, key=distances.get)
print("Ellipse focus 1 tri: "+str( focus1_tri) )
#focus2
distances = {}
for m in memb_tris:
    center_2d = [mesh.getTriBarycenter(m)[0], mesh.getTriBarycenter(m)[1]]
    dist = distance.euclidean(center_2d, focus2)
    distances[m] = dist
focus2_tri = min(distances, key=distances.get)
print("Ellipse focus 2 tri: "+str( focus2_tri) )

radius_maj = 380e-9#349.8e-9
focus1_tri_center = mesh.getTriBarycenter(focus1_tri)
focus2_tri_center = mesh.getTriBarycenter(focus2_tri)
for t in memb_tris:
    center = mesh.getTriBarycenter(t)
    dist1 = distance.euclidean(center, focus1_tri_center)
    dist2 = distance.euclidean(center, focus2_tri_center)
    dist_tot = dist1+dist2
    if dist_tot < radius_maj:
        azone_tris.append(t)

print('Number of triangles in active zone = '+str(len(azone_tris)))
#find active zone area
azone_area = 0;
for t in azone_tris:
    azone_area = azone_area+mesh.getTriArea(t)
print("active zone area = "+ str(azone_area))

######FIND ENDOCYTIC ZONE TRIANGLES######
endo_tris = []
radius_maj_endo = 550e-9 #550e-9#349.8e-9
for t in memb_tris:
    center = mesh.getTriBarycenter(t)
    dist1 = distance.euclidean(center, focus1_tri_center)
    dist2 = distance.euclidean(center, focus2_tri_center)
    dist_tot = dist1+dist2
    if dist_tot < radius_maj_endo:
        if not(t in azone_tris):
            endo_tris.append(t)

print('Number of triangles in endo zone: '+str (len(endo_tris)) )
#find membrane area
endo_area = 0;
for t in endo_tris:
    endo_area = endo_area+mesh.getTriArea(t)
print("endo zone area = "+ str(endo_area))

#find membrane area
mem_area = 0;
for t in memb_tris:
    mem_area = mem_area+mesh.getTriArea(t)
print("membrane area = "+ str(mem_area))

#find ER vol
ER_vol = 0;
for t in ER_tets:
    ER_vol = ER_vol+mesh.getTetVol(t)

print("ER vol ="+ str(ER_vol))

#create compartments
#cytosol includes all tets not in the recycling vesicle AZ cluster
axon = geom.TmComp('axon', mesh, axon_tets)
cytosol = geom.TmComp('cytosol', mesh, cyt_tets)
cytER = geom.TmComp('cytER', mesh, ER_tets)
cleft = geom.TmComp('cleft', mesh, outer_tets)

vac1 = geom.TmComp('vac1', mesh, vac1_tets)
vac2 = geom.TmComp('vac2', mesh, vac2_tets)
vac3 = geom.TmComp('vac3', mesh, vac3_tets)
vac4 = geom.TmComp('vac4', mesh, vac4_tets)
vac5 = geom.TmComp('vac5', mesh, vac5_tets)
mito = geom.TmComp('mito', mesh, mit_tets)
#find cytosol vol
cyt_vol = 0;
for t in cyt_tets:
    cyt_vol = cyt_vol+mesh.getTetVol(t)

print("cytosol vol ="+ str(cyt_vol))

#find axon vol
axon_vol = 0;
for t in axon_tets:
    axon_vol = axon_vol+mesh.getTetVol(t)

print("axon vol ="+ str(axon_vol))

#TmPatch
#get surface triangles of compartments
cyt_tris = meshctrl.findSurfTrisInComp(mesh, cytosol)
ERmemb = geom.TmPatch('ERmemb', mesh, ERmemb_tris, icomp = cytER, ocomp = cytosol)
memb = geom.TmPatch('memb', mesh, memb_tris, icomp = cytosol, ocomp = cleft)

#find vertices in cyt_tets
cyt_verts = set()
for t in cyt_tets:
    cyt_verts.update(mesh.getTet(t))

cyt_verts_list = list(cyt_verts)
cyt_verts_n = len(cyt_verts_list)

print("Number of cyt_verts = "+str( cyt_verts_n) )

#Create excitable membrane
membrane = geom.Memb('membrane', mesh, [memb])


#find ER membrane area
ERmem_area = 0;
for t in ERmemb_tris:
    ERmem_area = ERmem_area+mesh.getTriArea(t)

#find tetrahedra in restricted diffusion "tethering" zone around Azone.
tether_zone = set()
for a in azone_tris:
    tri_center = mesh.getTriBarycenter(a)
    for c in cyt_tets:
        tet_center = mesh.getTetBarycenter(c)
        dist = distance.euclidean(tri_center, tet_center)
        if dist < 100e-09:
            tether_zone.add(c)
tether_zone = list(tether_zone)
print("Tetrahedra in tether zone: "+str( len(tether_zone)) )


###########FIND ACTIVE ZONE NEIGHBOURS TO BLOCK DIFFUSION OF CA CHANNELS######
#Find neighbours of all active zone triangles
azone_neighbs = set([])
for t in azone_tris:
    neighbs = mesh.getTriTriNeighb(t, memb)
    azone_neighbs.update(neighbs)

#remove triangles in the active zone
azone_neighbs = list(set(azone_neighbs)-set(azone_tris))
print('AZONE NEIGHBS: ' +str(len(azone_neighbs)))

# # #find edges between active zone and neighbouring triangles (to block diffusion of VGCCs)
azone_edge_tris = []
for t in azone_neighbs:
    neighbs = mesh.getTriTriNeighb(t, memb)
    for i in range(3):
        if neighbs[i] in azone_tris:
            azone_edge_tris.append(neighbs[i])
print('EDGE TRIS: '+str( len(azone_edge_tris)))

#create dictionary: azone triangle:azone_neighbs
azone_endo_dict = {}
for tri in azone_edge_tris:
    neighbs = mesh.getTriTriNeighb(tri, memb)
    for i in range(3):
        if neighbs[i] in azone_neighbs:
            azone_endo_dict[tri] = neighbs[i]
print('AZONE BLOCKED TRI: '+str( len(azone_endo_dict)))

#create dictionary of dock tris and surrounding triangles (to capture escaped RIM)
dock_tri_to_surround_tris = {}
all_surr_tris = []
azone_and_endo_tris = azone_tris+endo_tris
for tri in dock_tris:
    dock_tri_to_surround_tris[tri] = []
    dists = []
    dock_cent = mesh.getTriBarycenter(tri)
    for t in azone_and_endo_tris:
        tris_cent = mesh.getTriBarycenter(t)
        dock_dist = (distance.euclidean(dock_cent, tris_cent))*1e09
        if dock_dist < 45:
            dists.append(t)
            all_surr_tris.append(t)
    print('Dock tri '+str( tri) + ' has '+str( len(dists))+ ' surrounding tris.')
    print('Dock tri '+str( tri) + ' has '+str( dists)+ ' surrounding tris.')
#check for duplicates
all_surr_tris_set = set(all_surr_tris)
contain_duplicates = len(all_surr_tris) != len(all_surr_tris_set)
print('TRI SURROUNDING HAD DUPLICATES: '+str( contain_duplicates))
print(str(len(all_surr_tris))+ ' vs '+ str( len(all_surr_tris_set)))

all_surr_tris_unique = list(all_surr_tris_set)
#find duplicates
seen = {}
dupes = []
for t in all_surr_tris:
    if t not in seen:
        seen[t] = 1
    else:
        if seen[t] == 1:
            dupes.append(t)
        seen[t] += 1
print('DUPLICATES: '+str( dupes) )

#add each surrounding triangle to closest dock triangle
for t in all_surr_tris_unique:
    dists = []
    t_cent = mesh.getTriBarycenter(t)
    for d in dock_tris:
        d_cent = mesh.getTriBarycenter(d)
        dock_dist = (distance.euclidean(d_cent, t_cent))*1e09
        dists.append(dock_dist)
    min_dist_ind = np.argmin(dists)
    #add triangle to closests dock tri
    min_dist_tri = dock_tris[min_dist_ind]
    dock_tri_to_surround_tris[min_dist_tri].append(t)

surr_tri_count = 0
for d in dock_tris:
    surr_tri = dock_tri_to_surround_tris[d]
    if d in surr_tri:
        dock_tri_to_surround_tris[d].remove(d)
for d in dock_tris:
    print('Dock tris'+str( d) + ' surrond tris: ' +str( dock_tri_to_surround_tris[d]))
    surr_tri_count += len(dock_tri_to_surround_tris[d])
print('TOTAL SURR TRIS: '+str( surr_tri_count))


#create dictionary of dock tris and "tether path" ends (tri:[[x0,y0],[x1,y1]])
dock_tri_to_tether_path = {}
#find norms for all dock_tris
print('FINDING TETHER PATH POINTS')
for tri in dock_tris:
    #get tri barycenter
    coord = mesh.getTriBarycenter(tri)
    print('DOCK TRI COORD: '+str( coord))
    #find dock triangle norm
    tri_norm = mesh.getTriNorm(tri)
    #convert to nm (multiply by 1e-09)
    tri_norm = np.multiply(tri_norm, 1e-09)
    mag = np.sqrt(np.sum(tri_norm**2))
    print('Dock tri norm mag:'+str( mag )) #should be 1e-09
    #add norm to tri barycenter
    tri_norm_bary = np.add(tri_norm, coord)
    #test if norm pointing into or out of cytosol
    norm_tet = mesh.findTetByPoint(tri_norm_bary)
    print('Norm points in: ', norm_tet)
    if not(norm_tet in cyt_tets):
        print('NORM NOT IN CYTOSOL.... INVERTING')
        tri_norm = np.multiply(tri_norm, -1)
        tri_norm_bary = np.add(tri_norm, coord)
        norm_tet = mesh.findTetByPoint(tri_norm_bary)
        print('Inverted norm points in: '+str( norm_tet))
    if (norm_tet in cyt_tets):
        print('NORM POINTS IN CYTOSOL')
    #create tether path point 100nm from membrane on outside
    dock_tri_to_tether_path[tri] = [[],[]]
    path_ext_dist = -100
    path_ext_norm = np.multiply(tri_norm, path_ext_dist)
    path_ext_point = np.add(path_ext_norm, coord)
    dock_tri_to_tether_path[tri][0] = path_ext_point
    print(tri, ' OUTSIDE PATH LEN: '+str(distance.euclidean(coord, dock_tri_to_tether_path[tri][0])))
    #create tether path point 60nm from membrane on inside
    path_int_dist = 100
    path_int_norm = np.multiply(tri_norm, path_int_dist)
    path_int_point = np.add(path_int_norm, coord)
    dock_tri_to_tether_path[tri][1] = path_int_point
    print(tri, ' INSIDE PATH LEN: '+str( distance.euclidean(coord, dock_tri_to_tether_path[tri][1])))
print('FOUND ALL '+str( len(dock_tri_to_tether_path)) + ' TETHER PATH POINTS')


#find min distance between dock_tris barycenters
min_dist_dock = []
for d in dock_tris:
    coord1 = mesh.getTriBarycenter(d)
    x1 = coord1[0]
    y1 = coord1[1]
    distances = []
    for e in dock_tris:
        if e!=d:
            coord2 = mesh.getTriBarycenter(e)
            x2 = coord2[0]
            y2 = coord2[1]
            # dist = sqrt((x2-x1)*(x2-x1) + (y2-y1)*(y2-y1))
            dist = distance.euclidean(coord1, coord2)
            distances.append(dist)
    min_distance = min(distances)
    min_dist_dock.append(min_distance)
print('DOCK MIN DIST:', (min(min_dist_dock))*1e09, ' nm')

#find active zone area
azone_area = 0;
for t in azone_tris:
    azone_area = azone_area+mesh.getTriArea(t)
print("active zone area = "+ str(azone_area))

#find endo zone area
endo_area = 0;
for t in endo_tris:
    endo_area = endo_area+mesh.getTriArea(t)
print("endo zone area = "+ str(endo_area))

#find membrane area
mem_area = 0;
for t in cyt_tris:
    mem_area = mem_area+mesh.getTriArea(t)
print("membrane area = "+ str(mem_area))

#find ER vol
ER_vol = 0;
for t in ER_tets:
    ER_vol = ER_vol+mesh.getTetVol(t)

print("ER vol ="+ str(ER_vol))

print("MEMBRANE CREATED")

#find vertices in cyt_tets
cyt_verts = set()
for t in cyt_tets:
    cyt_verts.update(mesh.getTet(t))

cyt_verts_list = list(cyt_verts)
cyt_verts_n = len(cyt_verts_list)

print("Number of cyt_verts = "+str( cyt_verts_n))

#find ER membrane area
ERmem_area = 0;
for t in ERmemb_tris:
    ERmem_area = ERmem_area+mesh.getTriArea(t)


cytosol.addVolsys('vsys')
memb.addSurfsys('memsys')
cytER.addVolsys('cytERsys')
ERmemb.addSurfsys('ERsys')
cleft.addVolsys('syn_sys')
# #add raft system to rafts
Raft.addRaftsys('raft_sys')
#add vesicle system to vesicles
ves.addVesSurfsys('ves_ssys')
ves_d.addVesSurfsys('ves_ssys')
ves_r.addVesSurfsys('ves_ssys')

#################################### SIMULATION DATA FUNCTIONS ETC ####################################

def getAllSyt(tet):
    syt_total = sim.getTetCount(tet, 'syt') + sim.getTetCount(tet, 'syt_Ca') + sim.getTetCount(tet, 'syt_Ca2')\
        + sim.getTetCount(tet, 'syt_Ca3') + sim.getTetCount(tet, 'syt_bCa') + sim.getTetCount(tet, 'syt_bCa2')\
        + sim.getTetCount(tet, 'syt_Ca_bCa') + sim.getTetCount(tet, 'syt_Ca_bCa2') + sim.getTetCount(tet, 'syt_Ca2_bCa')\
        + sim.getTetCount(tet, 'syt_Ca2_bCa2') + sim.getTetCount(tet, 'syt_Ca3_bCa') + sim.getTetCount(tet, 'syt_Ca3_bCa2')
    return syt_total

#function to count all cis-SNAREs in rafs
def getRaftSpecTypeCount(raftdict):
    count = 0
    for v in raftdict:
        count += raftdict[v]
    return count

def getRaftSnares():
    dict01 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt_CXN_Ca3_bCa')
    dict02 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt_CXN_Ca2_bCa2')
    dict03 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt_CXN_Ca2_bCa')
    dict04 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt_CXN_Ca_bCa2')
    dict05 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt_CXN_Ca_bCa')
    dict06 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt_CXN_bCa2')
    dict07 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt_CXN_bCa')
    dict08 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt_CXN_Ca3')
    dict09 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt_CXN_Ca2')
    dict010 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt_CXN_Ca')
    dict011 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt_CXN')
    dict012 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt')
    dict0 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_syt_CXN_Ca3_bCa2')
    dict1a = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE')
    dict1b = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'cisSNARE')
    dict2 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_aSNAP')
    dict3 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_aSNAP_NSF')
    dict4 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_aSNAP_NSF_1')
    dict5 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_aSNAP_NSF_2')
    dict6 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_aSNAP_NSF_3')
    dict7 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_aSNAP_NSF_4')
    dict8 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_aSNAP_NSF_5')
    dict9 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_aSNAP_NSF_6')
    dict10 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_aSNAP_NSF_7')
    dict11 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_aSNAP_NSF_8')
    dict12 = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_aSNAP_NSF_9')

    snare_count = getRaftSpecTypeCount(dict01)+getRaftSpecTypeCount(dict02)+getRaftSpecTypeCount(dict03)\
    +getRaftSpecTypeCount(dict04)+getRaftSpecTypeCount(dict05)+getRaftSpecTypeCount(dict06)\
    +getRaftSpecTypeCount(dict07)+getRaftSpecTypeCount(dict08)+getRaftSpecTypeCount(dict09)\
    +getRaftSpecTypeCount(dict010)+getRaftSpecTypeCount(dict011)+getRaftSpecTypeCount(dict012)\
    +getRaftSpecTypeCount(dict1a)+getRaftSpecTypeCount(dict1b)+getRaftSpecTypeCount(dict2)\
    +getRaftSpecTypeCount(dict3)+getRaftSpecTypeCount(dict4)+getRaftSpecTypeCount(dict5)\
    +getRaftSpecTypeCount(dict6)+getRaftSpecTypeCount(dict7)+getRaftSpecTypeCount(dict8)\
    +getRaftSpecTypeCount(dict9)+getRaftSpecTypeCount(dict10)+getRaftSpecTypeCount(dict11)\
    +getRaftSpecTypeCount(dict12)+getRaftSpecTypeCount(dict0)

    return snare_count

def getTOMOcount():
    TOMO_count = sim.getCompSpecCount('cytosol', 'TOMO')+\
    sim.getCompSpecCount('cytosol', 'TOMO_CDK5')+\
    sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO')+\
    sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin')+\
    sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_actin')+\
    sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_PKA')+\
    sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_p')+\
    sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_CaN_CaM')+\
    sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_CDK5')+\
    sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_CDK5')+\
    sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_p_PP2A')+\
    sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_p_PP2A')+\
    sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_PKA')+\
    sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer')+\
    sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_p')+\
    sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_CDK5')+\
    sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_p_CDK5')

    return TOMO_count

def getVesicleLinkSpecs(index):
    dimer_count = sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'synapsin_dimer')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'synapsin_dimer_p')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_p')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_p')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'PKA_synapsin_dimer')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_PKA')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_PKA')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'PP2A_synapsin_dimer_p')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_p_PP2A')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_p_PP2A')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_CDK5')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_p_CDK5')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_CaN_CaM')[index]\
    +sim.getCompVesicleSurfaceLinkSpecCountDict('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_p_CaN_CaM')[index]

    return dimer_count

def getPhosphoSynapsins():
    syn_phos_count = sim.getCompSpecCount('cytosol', 'synapsin_p')+sim.getCompSpecCount('cytosol', 'synapsin_actin_p')\
    +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'synapsin_p')\
    +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_actin_p')\
    +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_actin_p')\
    +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_p')\
    +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_p')\
    +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_p_PP2A')\
    +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_p_PP2A')\
    +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_p_CDK5')\
    +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_p_CaN_CaM')\
    +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'synapsin_dimer_p')\
    +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_p')\
    +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_p')\
    +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'PP2A_synapsin_dimer_p')\
    +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_p_PP2A')\
    +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_p_PP2A')\
    +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_p_CDK5')\
    +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_p_CaN_CaM')

    return syn_phos_count


#create RNG
rng = srng.create('mt19937', 512)
rng.initialize(rng_seed)

#Create sim solver object (with E-field)
print ("Creating MPI solver...")
use_efield = True
if use_efield: sim = solver.TetVesicle(syt_model, mesh, rng, solver.EF_DV_PETSC)
else: sim = solver.TetVesicle(syt_model, mesh, rng, solver.EF_NONE)
sim.setOutputSync(True, 0);
print ("MPI solver created.")


#create arrays to record data
#create array of time points
tpnts=np.arange(0.0,INT,DT)
#find number of time points
ntpnts = tpnts.shape[0]

print('Number of timepoints = '+ str(ntpnts))

FUSED_VES = []

def generateDockVesicleRandTri():
    global dock_count
    global dock_tris
    index = sim.addCompVesicle('cytosol', 'ves')
    selected = False
    min_dists = []
    while selected == False:
        rand_tri = rndm.choice(azone_tris)
        if not(rand_tri in dock_tris):
            coord = mesh.getTriBarycenter(rand_tri)
            #find dock triangle norm
            tri_norm = mesh.getTriNorm(rand_tri)
            #convert to nm (multiply by 1e-09)
            tri_norm = np.multiply(tri_norm, 1e-09)
            mag = np.sqrt(np.sum(tri_norm**2))
            print('Tri norm mag:'+str(mag))
            #add norm to tri barycenter
            tri_norm_bary = np.add(tri_norm, coord)
            #test if norm pointing into or out of cytosol
            norm_tet = mesh.findTetByPoint(tri_norm_bary)
            print('Norm points in: '+str(norm_tet))
            if not(norm_tet in cyt_tets):
                print('NORM NOT IN CYTOSOL.... INVERTING')
                tri_norm = np.multiply(tri_norm, -1)
                tri_norm_bary = np.add(tri_norm, coord)
                norm_tet = mesh.findTetByPoint(tri_norm_bary)
                print('Inverted norm points in: '+str( norm_tet))
            if (norm_tet in cyt_tets):
                print('NORM POINTS IN CYTOSOL')
            #multiply norm to get distance from membrane for vesicle
            ves_dist = (ves_diameter/2)+1e-09
            ves_dist_nm = ((ves_diameter/2)+1e-09)*1e09
            tri_norm_ext = np.multiply(tri_norm, ves_dist_nm)
            mag = np.sqrt(np.sum(tri_norm_ext**2))
            print('Vesicle position from membrane: '+str(ves_dist))
            print('Extended tri norm mag: '+str( mag))
            ves_loc = np.add(coord, tri_norm_ext)
            orig_pos = sim.getSingleVesiclePos('ves', index)
            sim.setCompSingleVesiclePos('cytosol', 'ves', index, ves_loc)
            print('Current dock tris: '+str( dock_tris))
            #calculate minimum distance between dock sites
            dists = []
            for t in dock_tris:
                pos = mesh.getTriBarycenter(t)
                tri_pos = mesh.getTriBarycenter(rand_tri)
                dist = (distance.euclidean(list(pos), list(tri_pos)))*1e09
                dists.append(dist)
            print('Dists: '+str( dists) )
            if dists:
                min_dist = min(dists)
                print('Min dist: '+str( min_dist))
                min_dists.append(min_dist)
            if min_dists:
                print('Max dist found: '+str( max(min_dists)))

            #test vesciel actually moved
            new_pos = sim.getSingleVesiclePos( 'ves', index)
            new_dock_dist = distance.euclidean(list(new_pos), list(ves_loc))
            if new_dock_dist < 1e-09:
                if dists:
                    if min_dist > 75:
                        dock_positions.append(ves_loc)
                        print("Added dock vesicle\n")
                        dock_count += 1
                        dock_tris.append(rand_tri)
                        selected = True
                else:
                    dock_positions.append(ves_loc)
                    print("Added dock vesicle\n")
                    dock_count += 1
                    dock_tris.append(rand_tri)
                    selected = True
    return index

def generateDockVesicle(tri):
    global dock_count
    global dock_tris
    index = sim.addCompVesicle('cytosol', 'ves')
    coord = mesh.getTriBarycenter(tri)
    #find dock triangle norm
    tri_norm = mesh.getTriNorm(tri)
    #convert to nm (multiply by 1e-09)
    tri_norm = np.multiply(tri_norm, 1e-09)
    mag = np.sqrt(np.sum(tri_norm**2))
    #add norm to tri barycenter
    tri_norm_bary = np.add(tri_norm, coord)
    #test if norm pointing into or out of cytosol
    norm_tet = mesh.findTetByPoint(tri_norm_bary)
    if not(norm_tet in cyt_tets):
        print('NORM NOT IN CYTOSOL.... INVERTING')
        tri_norm = np.multiply(tri_norm, -1)
        tri_norm_bary = np.add(tri_norm, coord)
        norm_tet = mesh.findTetByPoint(tri_norm_bary)
    if (norm_tet in cyt_tets):
        print('NORM POINTS IN CYTOSOL')
        #multiply norm to get distance from membrane for vesicle
        ves_dist = (ves_diameter/2)+1e-09
        ves_dist_nm = ((ves_diameter/2)+1e-09)*1e09
        tri_norm_ext = np.multiply(tri_norm, ves_dist_nm)
        mag = np.sqrt(np.sum(tri_norm_ext**2))
        print('Extended tri norm mag: '+str( mag) )
        ves_loc = np.add(coord, tri_norm_ext)
        orig_pos = sim.getSingleVesiclePos( 'ves', index)
        sim.setCompSingleVesiclePos('cytosol', 'ves', index, ves_loc)
        #test vesicle actually moved
        new_pos = sim.getSingleVesiclePos( 'ves', index)
        new_dock_dist = distance.euclidean(list(new_pos), list(ves_loc))
        if new_dock_dist < 1e-09:
            dock_positions.append(ves_loc)
            print("Added dock vesicle\n")
            dock_count += 1
        else:
            print("VESICLE DOCK ERROR!\n")
    else:
        print('VESICLE NORM WRONG DIRECTION!')
    return index


# Record voltage from the central tetrahedron
cent_tet = mesh.findTetByPoint([0.0,0.0,0.0])
###record number of times a fusion event occurs
fusion_count = 0
for j in range(NITER):
    #sim.reset()

    if use_efield:
        sim.setEfieldDT(EF_DT)
        sim.setMembPotential('membrane', init_pot)
        sim.setMembVolRes('membrane', Ra)
        sim.setMembCapac('membrane', memb_capac)
        sim.setMembPotential('membrane', init_pot)
        for v in cyt_verts_list:
            sim.setVertVClamped(v, True)
        print("E-FIELD IS ACTIVE")
    else:
        print("E-FIELD IS INACTIVE")

    # initial conditions


    #################ADD DOCK VESICLES#####################
    rec_ves_ind = []

    #number of initial dock vesicles
    dock_ves_init_n = len(dock_tris)
    #list of initial dock vesicle indices
    dock_ves_init = []
    dock_positions = []
    dock_count = 0
    #build dictionary connecting ves index with dock site triangle
    vesicle_to_docktri = {}
    #build dictionary connecting dock triangles to position
    dock_tri_to_pos = {}
    for d in dock_tris:
        pos = mesh.getTriBarycenter(d)
        dock_tri_to_pos[d] = pos
    #generating dock vesicles and add surface species
    for d in dock_tris:
        ind = generateDockVesicle(d)
        #add index and triangle to dictionary
        vesicle_to_docktri[ind] = d
        dock_ves_init.append(ind)
        print('Added ves: '+str( ind) )
        sim.setSingleVesicleSurfaceSpecCount( 'ves', ind, 'Rab3', 10)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', ind, 'SYB', 69)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', ind, 'syt', 27)
        sim.setSingleVesicleInnerSpecCount( 'ves', ind, 'REL_IND', 1 )

    print('Dock vesicles added: '+str( dock_count))
    print('Dock triangles: '+str( dock_tris))

    #find tetrahedron for each dock vesicle
    dock_ves_init_tets = {}
    for d in dock_ves_init:
        center = sim.getSingleVesiclePos( 'ves', d)
        tet = mesh.findTetByPoint(center)
        dock_ves_init_tets[d] = tet

    #set initial diffusion rate for dock vesicles to zero
    for t in dock_ves_init_tets.values():
        sim.setTetVesicleDcst(t, 'ves', 0)
        print('SET DOCK TET DIFF TO ZERO: '+str( t) )
    #########################################################################
    #non-recycling vesicles
####################################################################
    #add non-recycling vesicles to bouton
    #non-recycling vesicles
    ves_n = 250
    for v in range(ves_n):
        sim.addCompVesicle('cytosol', 'ves')

    print("Vesicle Count: "+ str(sim.getCompVesicleCount('cytosol', 'ves')) )

    #find indices of non-recycling vesicles
    nonrec_ves_ind = list(sim.getCompVesicleSurfaceSpecCountDict('cytosol', 'ves', 'Rab3').keys())
    #remove dock ind from nonrec_ves_ind
    nonrec_ves_ind = [ind for ind in nonrec_ves_ind if ind not in dock_ves_init]
    #save reserve vesicles at init
    nonrec_ves_ind_init = list(sim.getCompVesicleSurfaceSpecCountDict('cytosol', 'ves', 'Rab3').keys())
    #remove dock ind from nonrec_ves_ind
    nonrec_ves_ind_init = [ind for ind in nonrec_ves_ind_init if ind not in dock_ves_init]
    print('Non rec ves ind: ', nonrec_ves_ind_init)
    for v in nonrec_ves_ind:

        #add synapsin to non-recycling vesicles
        #Add vesicle membrane species
        sim.setSingleVesicleSurfaceSpecCount( 'ves', v, 'ves_syn1_site', 10)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', v, 'ves_CXN_site', 1000)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', v, 'ves_m13_site', 1000)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', v, 'ves_m18_site', 1000)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', v, 'ves_syndap_site', 1000)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', v, 'ves_asnap_site', 1000)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', v, 'ves_nsf_site', 1000)

        #adding Rab3 directly for release prob tests
        sim.setSingleVesicleSurfaceSpecCount( 'ves', v, 'Rab3', 10)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', v, 'SYB', 69)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', v, 'syt', 27)
        sim.setSingleVesicleInnerSpecCount( 'ves', v, 'REL_IND', 1)
    print("Added synapsin to non-recycling vesicles")

    #add recycling vesicles without synapsin
    rec_ves_n = 50
    rec_ves_count = 0
    #rec_ves_ind = [] ####NOW CREATE BEFORE DOCK VESICLES
    for r in range(rec_ves_n):
        index = sim.addCompVesicle('cytosol', 'ves')
        rec_ves_ind.append(index)
        #Add vesicle membrane species
        sim.setSingleVesicleSurfaceSpecCount( 'ves', index, 'ves_CXN_site', 1000)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', index, 'ves_m13_site', 1000)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', index, 'ves_m18_site', 1000)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', index, 'ves_syndap_site', 1000)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', index, 'ves_asnap_site', 1000)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', index, 'ves_nsf_site', 1000)
        #adding Rab3 directly for release prob tests
        sim.setSingleVesicleSurfaceSpecCount( 'ves', index, 'Rab3', 10)#Takamori 2006
        sim.setSingleVesicleSurfaceSpecCount( 'ves', index, 'SYB', 69)
        sim.setSingleVesicleSurfaceSpecCount( 'ves', index, 'syt', 27)
        #Add glutamate to each vesicle
        sim.setSingleVesicleInnerSpecCount( 'ves', index, 'REL_IND', 1)
    print('Added recycling vesicles: '+str( len(rec_ves_ind)) )
    #save recycling vesicles at init
    rec_ves_ind_init = rec_ves_ind+dock_ves_init
    print('Rec ves ind: ', rec_ves_ind_init)

    RAFT_N = 40
    raft_count = 0
    while raft_count < RAFT_N :
        rand_tri = rndm.choice(endo_tris)
        raft_unique_idx = sim.addTriRaft(rand_tri, 'Raft')
        if (raft_unique_idx == UNKNOWN_INDEX): continue
        raft_count+=1
    raft_count_init = len(sim.getPatchRaftSpecCountDict('memb', 'Raft', 'Rab3').keys())
    print (sim.getPatchRaftSpecCount('memb', 'Raft', 'Rab3'))

    for n in range(raft_count):
        sim.setSingleRaftSpecCount('Raft', n, 'Rab3', 10) #Takamori 2006
        sim.setSingleRaftSpecCount('Raft', n, 'SYB', 69)
        sim.setSingleRaftSpecCount('Raft', n, 'syt', 27)
        sim.setSingleRaftSpecCount('Raft', n, 'DYNp', 1)
        sim.setSingleRaftSpecCount('Raft', n, 'ves_CXN_site', 1000)
        sim.setSingleRaftSpecCount('Raft', n, 'ves_m13_site', 1000)
        sim.setSingleRaftSpecCount('Raft', n, 'ves_m18_site', 1000)
        sim.setSingleRaftSpecCount('Raft', n, 'ves_syndap_site', 1000)
        sim.setSingleRaftSpecCount('Raft', n, 'ves_asnap_site', 1000)
        sim.setSingleRaftSpecCount('Raft', n, 'ves_nsf_site', 1000)
    syb_raft_indices = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'AP180').keys()
    count = 0
    for s in syb_raft_indices:
        count = count+sim.getPatchRaftSpecCountDict('memb', 'Raft', 'AP180')[s]
    raft_dict = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'AP180')
    rafts_n = len(raft_dict)


    #populate docking sites with RIM randomly
    RIM_n = 38
    #mean RIMs per docking site
    RIM_n_site = 4
    for d in dock_tris:
        sim.setTriSpecCount(d, 'RIM_M13', RIM_n_site)

    if cachan_static == True:

        ca_tris = dock_tris
        for c in ca_tris:
            sim.setTriSpecCount(c, 'CaP_m0', 4)
    else:
        #add 15 Ca channels randomly over active zone
        ca_n = cachan_n
        ca_tris = []
        for n in range(ca_n):
            tri_selected = False
            while tri_selected == False:
                ca_tri = rndm.choice(azone_tris)
                if not(ca_tri in ca_tris):
                    ca_tris.append(ca_tri)
                    tri_selected = True
        for c in ca_tris:
            sim.setTriSpecCount(c, 'CaP_m0', 1)

    print('Added '+str( sim.getPatchSpecCount('memb', 'RIM_M13'))+ ' RIM to membrane')
    print('Added '+str( sim.getPatchSpecCount('memb', 'CaP_m0'))+ ' CaP_m0 to membrane')

    #scale membrane proteins by 1.6 (area of mesh membrane/cylinder membrane)
    mem_factor = 1.6

    sim.setCompSpecConc('cytosol', 'Ca', 0.045e-6)
    # total CB in rat HC: 1.98e-6
    sim.setCompSpecConc('cytosol', 'CBhi', 0.99e-6) # i.e. 1/2 of total CB molarity (1.98*10e-6)
    sim.setCompSpecConc('cytosol', 'CBlo', 0.99e-6) # i.e. 1/2 of total CB molarity (1.98*10e-6)

    # total CR in rat HC: 2.47e-6
    # WE MODEL TWO IDENTICAL PAIRS OF COOPERATIVE BINDING SITES AS ONE SPECIES.
    # SO WE SET ITS CONCENTRATION TO 4/5 OF THE TOTAL CONCENTRATION OF CR. THE REMAINING 1/5 IS THE INDEPENDENT SITE/
    sim.setCompSpecConc('cytosol', 'CRTT', 0.1976e-6)   # 4/5 of total CR concentration (2 pairs of cooperative sites)
    sim.setCompSpecConc('cytosol', 'CRind', 0.494e-6)  # 1/5 of total CR

    # total PV in rat HC: 4.55e-6
    sim.setCompSpecConc('cytosol', 'PV', 4.55e-6)

    sim.setCompSpecConc('cytosol', 'PV_Ca', 8.4e-6)
    sim.setCompSpecConc('cytosol', 'MgPV', 30.45e-6)

    #Add AC18 and PKAinact
    sim.setCompSpecConc('cytosol', 'AC18', 0.2e-6)#0.5e-6
    sim.setCompSpecConc('cytosol', 'R2C2', 1e-6)#0.5e-6

    SERCA_ro = 1000*1e12 # Bartol 2015. Computational reconstitution...
    PMCA_ro = 180*1e12 # Bartol 2015. Computational reconstitution...
    SERCA_count = ERmem_area * SERCA_ro
    PMCA_count = mem_area * PMCA_ro
    sim.setPatchSpecCount('ERmemb', 'SERCA', SERCA_count)
    sim.setPatchSpecCount('memb', 'PMCA_P0', PMCA_count)
    print('Added SERCA: '+str( sim.getPatchSpecCount('ERmemb', 'SERCA')))
    print('Added PMCA: '+str( sim.getPatchSpecCount('memb', 'PMCA_P0')))

    sim.setCompSpecConc('cytER', 'Ca', 150e-6)
    sim.setCompSpecClamped('cytER', 'Ca', True) # clamped means the conc won't change as simulation runs.

    sim.setCompSpecConc('cytosol', 'M18', 28.4e-6)
    m18_total = sim.getCompSpecCount('cytosol', 'M18')
    print('Added M18 to cytosol: '+ str( m18_total))
    sim.setCompSpecConc('cytosol', 'M13', 10.36e-6)
    m13_total = sim.getCompSpecCount('cytosol', 'M13')
    print('Added M13 to cytosol: '+ str( m13_total))
    sim.setCompSpecConc('cytosol', 'CXN', 16.59e-6)
    cxn_total = sim.getCompSpecCount('cytosol', 'CXN')
    print('Added complexin to cytosol: '+ str( cxn_total))
    sim.setCompSpecConc('cytosol', 'Rab3', 0)#125.85e-6)
    rab3_total = sim.getCompSpecCount('cytosol', 'Rab3')
    print('Added Rab3 to cytosol: '+ str( rab3_total))
    sim.setCompSpecConc('cytosol', 'aSNAP', 7.68e-6)
    asnap_total = sim.getCompSpecCount('cytosol', 'aSNAP')
    print('Added aSNAP to cytosol: '+ str( asnap_total))
    sim.setCompSpecConc('cytosol', 'NSF', 27.14e-6)
    nsf_total = sim.getCompSpecCount('cytosol', 'NSF')
    print('Added NSF to cytosol: '+ str( nsf_total))
    sim.setCompSpecConc('cytosol', 'synapsin', 156e-6)
    synapsin_total = sim.getCompSpecCount('cytosol', 'synapsin')
    print('Added synapsins to cytosol: '+ str( synapsin_total))

    #add PP2A to cytocol
    sim.setCompSpecConc('cytosol', 'PP2A', 0.5e-6)

    sim.setPatchSpecCount('memb', 'SYX', 20096*mem_factor)
    sim.setPatchSpecCount('memb', 'SNP25', 26686*mem_factor)#10

    if cam_present:
        sim.setCompSpecConc('cytosol', 'CaM_N_0_C_0', 60e-6) #Wilhelm 2014
        print("CALMODULIN PRESENT")
    else:
        sim.setCompSpecConc('cytosol', 'CaM_N_0_C_0', 0)
        print("CALMODULIN NOT PRESENT")
    sim.setCompSpecConc('cytosol', 'CaN', 5e-6)
    sim.setCompSpecCount('cytosol', 'DYNp', 0)#20) #plus 40 in initial rafts
    sim.setCompSpecConc('cytosol', 'CDK5', 1e-6)
    sim.setCompSpecConc('cytosol', 'Rab3', 125.85e-6)

    sim.setCompSpecConc('cytosol', 'SYN1', 21.37e-6)
    syn1_total = sim.getCompSpecCount('cytosol', 'SYN1')
    print('Added Syndapin1 to cytosol: '+str( syn1_total) )

    #add TOMOSYN
    sim.setCompSpecCount('cytosol', 'TOMO_p', tomo_init)

    for t in actin_tets:
        sim.setTetSpecCount(t, 'actin', 1)

    pka_act_time = 50000
    camkii_deact_time = 50000

    print("Running iteration"+str( j+1) )
    print( "Time points: ")

    #create list of vesicles already shifted to docking sites
    docked_ves = []

    #create list of pulse times
    endtime = INT*1000
    firing_rate = RATE
    ap_interval = 1000/firing_rate
    start_firing = 5000
    pulse_times = np.arange(start_firing, endtime, ap_interval)
    pulse_times_off = pulse_times+1
    check_release_times_pre = pulse_times-1
    check_release_times = pulse_times+5

    fusion_events_tot = 0
    #check release indicator in cleft pre and post run
    rel_ind_count_pre = 0
    rel_ind_count_post = 0
    #create empty list to contain recycled vescicles
    recycled_ves_free = []
    recycled_ves_all = []
    #dictionary of Rafts to Vesicle indices (connecting endocytosis)
    raft_to_vesicle = {}
    #record total raft endocytosis events
    total_raft_endo = 0
    total_time = 0

    #####SCALE DIFFUSION OF SYNTAXIN AND SNAP25#####################
    for tri in azone_tris:
        sim.setTriDiffD(tri, 'diff_SYX', 0.2429e-12*snare_diff_scale)
        sim.setTriDiffD(tri, 'diff_SYX_M18', 0.2429e-12*snare_diff_scale)
        sim.setTriDiffD(tri, 'diff_SNP25', 0.651e-12*snare_diff_scale)

    action_potentials_fired = 0
    #Save release probs
    ap_n = len(pulse_times)
    rel_probs = np.zeros(ap_n) #10 APs
    #count fusions from "recycling" vs "reserve" pool
    REC_FUSIONS = []
    RES_FUSIONS = []
    add_tethers_time = 700
    #get all initial vesicle indices
    init_vesicle_ind = sim.getCompVesicleSurfaceSpecCountDict('cytosol', 'ves', 'Rab3').keys()
    #build list for vesicles (f)used from initial pools
    init_vesicles_used = []
    init_rec_vesicles_used = []
    init_nonrec_vesicles_used = []
    new_vesicles_used = []
    fused_ind_all = []
    #record total recycled vesicles
    recycled_vesicles_tot = 0
    #list of new rafts
    new_rafts = []
    filled_rafts = []
    for i in range(0,ntpnts):

        if i == add_tethers_time:
            #create tether paths for each dock triangle
            print("CREATING TETHER PATHS...")
            tether_paths = []
            for d in dock_tris:
                path_name = 'actin_'+str(d)
                sim.createPath(path_name)
                tether_paths.append(path_name)
                point_1 = dock_tri_to_tether_path[d][1]
                point_2 = dock_tri_to_tether_path[d][0]
                print(path_name+ ' PATH LENGTH: '+str((distance.euclidean(point_1, point_2))*1e09)+ ' nm')
                sim.addPathPoint(path_name, 1, point_1)
                sim.addPathPoint(path_name, 2, point_2)
                #connect path points
                sim.addPathBranch(path_name, 1, {2:1.0})
            print("TETHER PATHS CREATED")

            #add vesicle diffusion rates to paths
            print('ADDING VESICLES TO TETHER PATHS...')
            tether_pull_rate = 200.0 #um/min
            tether_pull_rate_s = tether_pull_rate/60 #um/s
            tether_pull_rate_m_s = tether_pull_rate_s*1e-06 #m/s
            print('TETHER RATE: ' +str(tether_pull_rate_m_s) + ' m/s')
            for p in tether_paths:
                #Rab3 must be on vesicle for path binding
                sim.addPathVesicle(p, 'ves', tether_pull_rate_m_s, {'Rab3':8}) # Speeds of ~1 to 10 um/minute are reasonable
                print('Added vesicle to path: ' +str( p) )
            print('VESICLES ADDED TO TETHER PATHS')
        #######SAVE SIMPLE NUMBER COUNT DATA##########
        if myrank == 0:
            afile_DATA = open(data_path+str(rng_seed)+'_results.txt', 'a+')
            afile_RP = open(data_path+str(rng_seed)+'_rel_probs.txt', 'a+')
            afile_RAB3 = open(data_path+str(rng_seed)+'_rab3s.txt', 'a+')
            afile_RAB3_rec = open(data_path+str(rng_seed)+'_rab3s_rec.txt', 'a+')
            afile_RAB3_res = open(data_path+str(rng_seed)+'_rab3s_res.txt', 'a+')

        res = np.zeros(32) #29 species
        #time point
        res[0] = i
        print(str(i))
        if i==pka_act_time:
            sim.setCompSpecCount('cytosol', 'PKA', 60)
        if i==camkii_deact_time:
            sim.setCompReacK('cytosol', 'camkii_activate', 0)
        #get all vesicles and rafts before the next time step to test for new vesicles by endoctosis
        all_ves_indices_prerun = sim.getCompVesicleSurfaceSpecCountDict('cytosol', 'ves', 'Rab3').keys()
        raft_indices_prerun = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'Rab3').keys()
        raft_positions_prerun = {}
        for r in raft_indices_prerun:
            raft_pos = sim.getSingleRaftPos( 'Raft', r)
            raft_positions_prerun[r] = raft_pos
        #get number of exocytosis events pre-run
        exo_extent_pre = sim.getExocytosisExtent('fusion4')+sim.getExocytosisExtent('fusion3')+sim.getExocytosisExtent('fusion2')
        #count release indicators in cleft pre run
        rel_ind_count_pre = sim.getCompSpecCount('cleft', 'REL_IND')
        print('Exo_extent_pre = '+ str(rel_ind_count_pre))
        ##########RUN TIME STEP###########
        start = time.time()
        sim.run(tpnts[i])
        end = time.time()
        ##################################
        #count release indicators in cleft post run
        rel_ind_count_post = sim.getCompSpecCount('cleft', 'REL_IND')
        print("rel_ind_count_post: "+str(rel_ind_count_post))
        exo_extent_post = sim.getExocytosisExtent('fusion4')+sim.getExocytosisExtent('fusion3')+sim.getExocytosisExtent('fusion2')
        print('Exo_extent_post = '+ str(rel_ind_count_post))
        print("ACTION POTENTIALS FIRED: "+str( action_potentials_fired) )
        #fusion_events = exo_extent_post-exo_extent_pre
        fusion_events = rel_ind_count_post-rel_ind_count_pre
        print("fusion_events: "+str(fusion_events))
        fusion_events_tot += fusion_events
        print('fusion_events_tot = ' + str(fusion_events_tot))

        time_took = (end - start)
        print('Took: '+str( time_took) )
        if not(i==0):
            total_time += time_took
            mean_time = float(total_time)/float(i)
            print('Mean time: '+str( mean_time) )
            time_remain_sec = (ntpnts-i)*float(mean_time)
            time_remain_hour = float(time_remain_sec)/float(3600)
            time_remain_days = float(time_remain_hour)/float(24)
            print('Time remaining (hours): '+str( time_remain_hour))
            print('Time remaining (days): '+str( time_remain_days), flush=True)

        #COUNT CA CHANNELS IN ACTIVE AZONE
        ca_chan_azone_n = 0
        for tri in azone_tris:
            ca_chan_azone_n += sim.getTriSpecCount(tri, 'CaP_m0')
            ca_chan_azone_n += sim.getTriSpecCount(tri, 'CaP_m1')
            ca_chan_azone_n += sim.getTriSpecCount(tri, 'CaP_m2')
            ca_chan_azone_n += sim.getTriSpecCount(tri, 'CaP_m3')

        #move any docked vesicles to their docking site
        #get list of all vesicles (some may have appeared by endocytosis)
        all_ves_indices = sim.getCompVesicleSurfaceSpecCountDict('cytosol', 'ves', 'Rab3').keys()
        #get list of newly formed vesicles
        new_ves = []
        for v in all_ves_indices:
            if not(v in all_ves_indices_prerun):
                new_ves.append(v)
                #add REL_IND to vesicle
                sim.setSingleVesicleInnerSpecCount( 'ves', v, 'REL_IND', 1)
        #check that all docked_ves still exist (i.e. haven't fused)
        for v in all_ves_indices_prerun:
            if (v not in all_ves_indices):
                FUSED_VES.append(v)

        #calculate the mean [Ca] next to membrane triangles containing CaP channels
        ca_mem_conc = 0
        for p in ca_tris:
            tets = mesh.getTriTetNeighb(p)
            conc = 0
            for t in tets:
                if t in cyt_tets:
                    conc = conc + sim.getTetSpecConc(t, 'Ca')
            ca_mem_conc = ca_mem_conc + conc

        if use_efield:
            if i in pulse_times:
                sim.setMembPotential('membrane', 40e-3)
                print('Action potential ON')
            if i in pulse_times_off:
                sim.setMembPotential('membrane', init_pot)
                print('Action potential OFF')
            if i in check_release_times_pre:
                rel_ind_count_pre_ap = sim.getCompSpecCount('cleft', 'REL_IND')
                all_ves_pre_ap = sim.getCompVesicleSurfaceSpecCountDict('cytosol', 'ves', 'Rab3').keys()
            if i in check_release_times:
                all_ves_post_ap = sim.getCompVesicleSurfaceSpecCountDict('cytosol', 'ves', 'Rab3').keys()
                rel_ind_count_post_ap = sim.getCompSpecCount('cleft', 'REL_IND')
                fusions = rel_ind_count_post_ap-rel_ind_count_pre_ap
                if not (fusions == 0):
                    rel_probs[action_potentials_fired] = 1
                    #get all current vesicles to check for fusion
                    fused_ind = []
                    for v in all_ves_pre_ap:
                        if (v not in all_ves_post_ap):
                            fused_ind.append(v)
                            fused_ind_all.append(v)
                            #add fused vesicle to used vesicle lists
                    print('FUSED VESICLES: ', fused_ind)
                    for v in fused_ind:
                        print('Fused indices: ', fused_ind)
                        if v in init_vesicle_ind:
                            init_vesicles_used.append(v)
                            print(v, ' is an init vesicle type')
                        if v in rec_ves_ind_init:
                            init_rec_vesicles_used.append(v)
                            print(v, ' is a recycling vesicle type')
                        if v in nonrec_ves_ind_init:
                            init_nonrec_vesicles_used.append(v)
                            print(v, ' is a reserve vesicle type')
                        if v not in init_vesicle_ind:
                            new_vesicles_used.append(v)
                            print(v, ' is a new vesicle type')
                        print (v, ' is not in the list ', nonrec_ves_ind_init)
                        print ('init_nonrec_vesicles_used', init_nonrec_vesicles_used)
                            #add indices to ves_docktris
                    #create a new Rafts for each fused vesicle
                    count = 0
                    attempts = 0
                    print('CREATING NEW RAFT')
                    while count < fusions:
                        rand_tri = rndm.choice(endo_tris)
                        print('RAND TRI SELECTED: '+str(rand_tri))
                        index = sim.addTriRaft(rand_tri, 'Raft')
                        print ('INDEX: ' + str(index) + 'UNK INDEX: ' + str(UNKNOWN_INDEX) )
                        if (index != UNKNOWN_INDEX):
                            print('ADDING SPECS TO NEW RAFT')
                            sim.setSingleRaftSpecCount('Raft', index, 'AP180', 69)
                            sim.setSingleRaftSpecCount('Raft', index, 'AP2', 27)
                            sim.setSingleRaftSpecCount('Raft', index, 'Rab3', 10)
                            sim.setSingleRaftSpecCount('Raft', index, 'DYN_AD', 1)
                            sim.setSingleRaftSpecCount('Raft', index, 'ves_CXN_site', 1000)
                            sim.setSingleRaftSpecCount('Raft', index, 'ves_m13_site', 1000)
                            sim.setSingleRaftSpecCount('Raft', index, 'ves_m18_site', 1000)
                            sim.setSingleRaftSpecCount('Raft', index, 'ves_syndap_site', 1000)
                            sim.setSingleRaftSpecCount('Raft', index, 'ves_asnap_site', 1000)
                            sim.setSingleRaftSpecCount('Raft', index, 'ves_nsf_site', 1000)
                            count+=1
                            new_rafts.append(index)
                        else:
                            attempts+=1
                            print('RAND TRI SELECTION ATTEMPTS: '+str(attempts))
                            continue
                else:
                    rel_probs[action_potentials_fired] = 0
                action_potentials_fired += 1

        ca_conc = 0
        for t in cyt_tets:
            conc = sim.getTetSpecConc(t, 'Ca')
            ca_conc = ca_conc + conc

        docked_ves = []
        for v in all_ves_indices:
            #check immobility
            if (sim.getSingleVesicleImmobility('ves', v) != 0): #ONLY DOCKED VESICLES ARE IMMOBILE IN rec_ves_ind
                surf_actin_n = sim.getSingleVesicleSurfaceSpecCount('ves', v, 'synapsin_actin')\
                +sim.getSingleVesicleSurfaceSpecCount('ves', v, 'Rab3_TOMO_p_synapsin_actin')\
                +sim.getSingleVesicleSurfaceSpecCount('ves', v, 'Rab3_TOMO_synapsin_actin')
                if surf_actin_n == 0: #Check not a non-recycling vesicle bound to actin
                    synapsin_actin_count = sim.getSingleVesicleSurfaceSpecCount('ves', v, 'synapsin_actin')
                    Rab3_TOMO_p_synapsin_actin_count = sim.getSingleVesicleSurfaceSpecCount('ves', v, 'Rab3_TOMO_p_synapsin_actin')
                    Rab3_TOMO_synapsin_actin_count = sim.getSingleVesicleSurfaceSpecCount('ves', v, 'Rab3_TOMO_synapsin_actin')

                    distances_to_dock_pos = []
                    distances_to_tri_bary = []
                    ves_pos = sim.getSingleVesiclePos( 'ves', v)
                    for d in range(len(dock_tris)):
                        dist = distance.euclidean(dock_positions[d], list(ves_pos))
                        distances_to_dock_pos.append(dist)
                    min_dist_ind = distances_to_dock_pos.index(min(distances_to_dock_pos)) #find closest dock position

                    for t in dock_tris:
                        tri_bary = dock_tri_to_pos[t]
                        dist = distance.euclidean(tri_bary, list(ves_pos))
                        distances_to_tri_bary.append(dist)
                    min_dist_ind_2 = distances_to_tri_bary.index(min(distances_to_tri_bary))

                    dock_pos = dock_positions[min_dist_ind_2]
                    orig_dock_dist = distance.euclidean(list(ves_pos), list(dock_pos))
                    orig_dock_dist_nm = orig_dock_dist*1e9

                    #only move vesicle if closer enough to dock site (i.e. not actin bound)
                    if orig_dock_dist < 100e-9:
                        print("Attempted new position: "+ str(dock_pos))
                        sim.setCompSingleVesiclePos('cytosol', 'ves', v, dock_pos)
                        new_pos = sim.getSingleVesiclePos( 'ves', v)
                        new_dock_dist = distance.euclidean(list(new_pos), list(dock_pos))
                        #test vesicle has actually been moved to dock site (i.e. not blocked by another vesicle)
                        if new_dock_dist < 1e-09:
                            docked_ves.append(v)
                        print('Ves current pos: '+ str(ves_pos))
                        print('Ves new pos: '+ str(new_pos))
        print('Dock ves ind: '+ str(docked_ves))

        #remove dock ves from nonrec_ves_ind (if docked)
        for d in docked_ves:
            if d in nonrec_ves_ind:
                nonrec_ves_ind.remove(d)
            if d in rec_ves_ind:
                rec_ves_ind.remove(d)
        for d in FUSED_VES:
            if d in nonrec_ves_ind:
                nonrec_ves_ind.remove(d)
            if d in nonrec_ves_ind_init:
                if d not in RES_FUSIONS:
                    RES_FUSIONS.append(d)
            if d in rec_ves_ind:
                rec_ves_ind.remove(d)
            if d in rec_ves_ind_init:
                if d not in REC_FUSIONS:
                    REC_FUSIONS.append(d)

        #create dictionary of docked vesicles and their dock sites (ves_ind -> dock_tri)
        ves_docktris = {}
        for v in docked_ves:
            dists = {}
            vespos = sim.getSingleVesiclePos( 'ves', v)
            for d in dock_tris:
                center = mesh.getTriBarycenter(d)
                dists[d] = distance.euclidean(center, vespos)
            nearest_tri = min(dists, key=dists.get)
            ves_docktris[v] = nearest_tri

        #find Raft each new vesicle came from.
        raft_indices = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'Rab3').keys()
        rafts_endocytosed = []
        for r in raft_indices_prerun:
            if r not in raft_indices:
                rafts_endocytosed.append(r)
        total_raft_endo += len(rafts_endocytosed)
        print('Rafts endocytosed: '+str( total_raft_endo) )
        for v in new_ves:
            ves_pos = sim.getSingleVesiclePos( 'ves', v)
            distances = {}
            for r in rafts_endocytosed:
                raft_pos = raft_positions_prerun[r]
                dist_to_ves = distance.euclidean(raft_pos, ves_pos)
                distances[r] = dist_to_ves
            #get closest raft
            closest_raft = min(distances, key=distances.get)
            raft_to_vesicle[closest_raft] = v
            rafts_endocytosed.remove(closest_raft)
            print('Raft '+str(closest_raft) + ' formed vesicle '+str( v) )


        #######RELEASE PROBABILITY TEST####################################################
        #add SNARE_SYT_CXN complexes to each docked vesicle
        for d in docked_ves:
            if i == 50:
                print('Priming vesicle: '+ str(d))
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt_CXN', 4)
                #remove RIM and related structures from vesicle dock sites
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'RIM_M13_Rab3', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'RIM_M13_Rab3_SYX_M18', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'RIM_M13_Rab3_SYX_M18_SNP25', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt_CXN_Ca', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt_CXN_Ca2', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt_CXN_Ca3', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt_CXN_bCa', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt_CXN_bCa2', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt_CXN_Ca_bCa', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt_CXN_Ca_bCa2', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt_CXN_Ca2_bCa', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt_CXN_Ca2_bCa2', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt_CXN_Ca3_bCa', 0)
                sim.setSingleVesicleSurfaceSpecCount( 'ves', d, 'SNARE_syt_CXN_Ca3_bCa2', 0)
        if i == 50:
            #set normal diffusion rate for dock vesicles
            for t in dock_ves_init_tets.values():
                sim.setTetVesicleDcst(t, 'ves', ves_diff_k)
                print('SET DOCK TET DIFF TO ZERO: '+str( t) )

        print('Dock ves ind: '+ str(docked_ves))
        ###################################################################################

        #shift all SNARE complexes to pole of vesicle (closest to membrane)
        for v in docked_ves:
            ves_center = sim.getSingleVesiclePos( 'ves', v)
            dock_tri = ves_docktris[v]
            tri_center = mesh.getTriBarycenter(dock_tri)
            #Cartesian position
            coord_cart = np.subtract(tri_center, ves_center)
            coord_x = coord_cart[0]
            coord_y = coord_cart[1]
            coord_z = coord_cart[2]
            theta = np.arctan2(coord_y, coord_x)
            phi = np.arccos(coord_z/ves_radius)

            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE_syt', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE_syt_CXN', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE_syt_CXN_Ca', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE_syt_CXN_Ca2', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE_syt_CXN_Ca3', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE_syt_CXN_bCa', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE_syt_CXN_bCa2', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE_syt_CXN_Ca_bCa', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE_syt_CXN_Ca_bCa2', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE_syt_CXN_Ca2_bCa', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE_syt_CXN_Ca3_bCa', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'SNARE_syt_CXN_Ca3_bCa2', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'RIM_M13_Rab3', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'RIM_M13_Rab3_SYX_M18', [theta, phi])
            sim.setSingleVesicleSurfaceSpecPosSpherical('ves', v, 'RIM_M13_Rab3_SYX_M18_SNP25', [theta, phi])

        #count docked vesicles
        rec_ves_free = []
        for v in all_ves_indices:
            if ((v not in docked_ves)and(v not in nonrec_ves_ind)):
                rec_ves_free.append(v)

        #separate non-recycling vesicles into free and clustered
        nonrec_ves_free = []
        nonrec_ves_clust = []
        actin_bound_ves = []
        total_link_specs = 0
        for v in nonrec_ves_ind:
            dimer_count = getVesicleLinkSpecs(v)
            total_link_specs+=dimer_count
            if dimer_count < 1:
                nonrec_ves_free.append(v)
            else:
                nonrec_ves_clust.append(v)
            ves_actin_count = sim.getSingleVesicleSurfaceSpecCount('ves', v, 'synapsin_actin') + sim.getSingleVesicleSurfaceSpecCount('ves', v, 'Rab3_TOMO_p_synapsin_actin')
            if (ves_actin_count != 0):
                actin_bound_ves.append(v)
        #find recycled vesicles
        for v in new_ves:
            if (v not in docked_ves)and(v not in FUSED_VES):
                recycled_ves_free.append(v)
        #check no recycled vesicles have docked
        rec_docked = []
        for v in recycled_ves_free:
            if (v in docked_ves)or(v in FUSED_VES):
                rec_docked.append(v)
        for v in rec_docked:
            recycled_ves_free.remove(v)
        #remove recycled vesicles in rec_ves_free
        ves_free_to_remove = []
        for v in rec_ves_free:
            if v in recycled_ves_free:
                ves_free_to_remove.append(v)
        for v in ves_free_to_remove:
            rec_ves_free.remove(v)

        #concatenate rec_ves_free and nonrec_ves_free to get all free vesicles
        all_ves_free = rec_ves_free+nonrec_ves_free

        print('Number of docked vesicles = '+str( len(docked_ves)))
        print('Number of recycled vesicles = '+str( len(recycled_ves_free)))
        #sanity check
        print('Total vesicles: '+str( len(all_ves_indices)))
        ves_sum = docked_ves+all_ves_free+recycled_ves_free
        print('Summed vesicles: '+str( len(ves_sum)+len(nonrec_ves_clust)))

        #Get positions of all recycling vesicles
        for v in docked_ves:
            ves_pos = array(sim.getSingleVesiclePos( 'ves', v))
            #find which triangle vesicle is docked at
            distances_to_dock_tris = {}
            for t in dock_tris:
                dock_tri_center = mesh.getTriBarycenter(t)
                dist_to_dock = distance.euclidean(dock_tri_center, list(new_pos))
                distances_to_dock_tris[t] = dist_to_dock
            #get closest triangle
            closest_dock_tri = min(distances_to_dock_tris, key=distances_to_dock_tris.get)
            #get norm of triangle
            tri_norm = mesh.getTriNorm(closest_dock_tri)
            #convert to nm (multiply by 1e-09)
            tri_norm_nm = np.multiply(tri_norm, 1e-09)
            mag = np.sqrt(np.sum(tri_norm_nm**2))
            #get tri barycenter
            tri_center = mesh.getTriBarycenter(closest_dock_tri)
            #add norm to tri barycenter
            tri_norm_bary = np.add(tri_norm_nm, tri_center)
            #test if norm pointing into or out of cytosol
            norm_tet = mesh.findTetByPoint(tri_norm_bary)
            if norm_tet in cyt_tets:
                print('NORM POINTING IN CYTOSOL.... INVERTING')
                tri_norm = np.multiply(tri_norm, -1)
            #scale position of vesicle
            ves_pos = ves_pos/scale_100
            #concatenate position and norm as 6-tuple
            norm_array = array(tri_norm)
            pos_norm = np.concatenate((ves_pos, norm_array))

        #Get positions of all recycled free vesicles

        print('rec_ves_free: '+str( len(rec_ves_free)))
        print('nonrec_ves_free: '+str( len(nonrec_ves_free)))
        print('all_ves_free: '+str( len(all_ves_free)))

        # RAFT_POS_NORM.append({})
        raft_indices = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'Rab3').keys()
        for r in raft_indices:
            raft_pos = array(sim.getSingleRaftPos( 'Raft', r))
            #find the raft norm pointing inwards
            #find which triangle raft is docked at
            distances_to_raft_tris = {}
            for t in endo_tris:
                    dock_tri_center = mesh.getTriBarycenter(t)
                    dist_to_dock = distance.euclidean(dock_tri_center, list(raft_pos))
                    distances_to_raft_tris[t] = dist_to_dock
            #get closest triangle
            closest_dock_tri = min(distances_to_raft_tris, key=distances_to_raft_tris.get)
            #get norm of triangle
            tri_norm = mesh.getTriNorm(closest_dock_tri)
            #convert to nm (multiply by 1e-09)
            tri_norm_nm = np.multiply(tri_norm, 1e-09)
            mag = np.sqrt(np.sum(tri_norm_nm**2))
            #get tri barycenter
            tri_center = mesh.getTriBarycenter(closest_dock_tri)
            #add norm to tri barycenter
            tri_norm_bary = np.add(tri_norm_nm, tri_center)
            #test if norm pointing into or out of cytosol
            norm_tet = mesh.findTetByPoint(tri_norm_bary)
            if norm_tet not in cyt_tets:
                tri_norm = np.multiply(tri_norm, -1)
            #scale position of raft
            raft_pos = raft_pos/scale_100
            #concatenate position and norm as 6-tuple
            norm_array = array(tri_norm)
            pos_norm = np.concatenate((raft_pos, norm_array))
            #RAFT_POS_NORM[i][r] = pos_norm
        rafts_n = len(raft_indices)
        print("N rafts: " +str( rafts_n))

        syb_dict = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SYB')
        print('Number of SYB in rafts: '+ str(getRaftSpecTypeCount(syb_dict)))

        raft_indices = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'Rab3').keys()
        raft_syb = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SYB')
        raft_syt = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'syt')

        print("N rafts: " +str( sim.getPatchRaftCount('memb', 'Raft')) )


        print('SYB in membrane = '+ str(sim.getPatchSpecCount('memb', 'SYB')))

        #####calculate SYX and SNAP25 in AZONE####
        syx_azone = 0
        snp25_azone = 0
        for tri in azone_tris:
            syx_azone += (sim.getTriSpecCount(tri, 'SYX')+sim.getTriSpecCount(tri, 'SYX_M18'))
            snp25_azone += sim.getTriSpecCount(tri, 'SNP25')
        print('SYX in AZone: '+str(syx_azone))
        print('SNAP25 in AZone: '+str(snp25_azone))

        endo_events = raft_count_init-rafts_n
        print('Total endo events: ' +str(endo_events))

        print( 'Ca4CaM count = '+ str(sim.getCompSpecCount('cytosol', 'CaM_N_2_C_2')))
        print( 'CaN_CaM count = '+ str(sim.getCompSpecCount('cytosol', 'CaN_CaM')))
        print( 'DYNpp count = '+ str(sim.getCompSpecCount('cytosol', 'DYNpp')))
        print( 'DYN count = '+ str(sim.getCompSpecCount('cytosol', 'DYN')))
        print( 'DYN MEM count = '+ str(sim.getPatchSpecCount('memb', 'DYN_SYN1')+sim.getPatchSpecCount('memb', 'DYN_SYN1_CDK5')))

        synapsin_n = sim.getCompSpecCount('cytosol', 'synapsin')+sim.getCompSpecCount('cytosol', 'synapsin_actin')
        print('Bound synapsin: '+ str( synapsin_total-synapsin_n) )
        synapsin_cluster_perc = (1-(float(synapsin_n)/float(synapsin_total)))*100
        #SYNAPSIN_CLUST_PERC[j, i] = synapsin_cluster_perc
        print('Percentage SYNAPSIN in cluster: '+str( synapsin_cluster_perc) )
        print('TOTAL CXN in CYTOSOL: ' +str( sim.getCompSpecCount('cytosol', 'CXN')))
        print('TOTAL CXN in AXON: '+str( sim.getCompSpecCount('axon', 'CXN')))

        print("FUSION COUNT: "+str( fusion_count))
        print('TOTAL Ca in CYTOSOL: '+str( 1e06*sim.getCompSpecConc('cytosol', 'Ca')))
        print('TOTAL CaM_N_2_C_2 in CYTOSOL: '+str( sim.getCompSpecCount('cytosol', 'CaM_N_2_C_2')))
        print('TOTAL AC18 in CYTOSOL: '+str( sim.getCompSpecCount('cytosol', 'AC18')))
        print('TOTAL AC18_CaM in CYTOSOL: '+str( sim.getCompSpecCount('cytosol', 'AC18_CaM')))
        print('TOTAL cAMP in CYTOSOL: '+str( sim.getCompSpecCount('cytosol', 'cAMP')))
        print('TOTAL R2C2 in CYTOSOL: '+str( sim.getCompSpecCount('cytosol', 'R2C2')))
        print('TOTAL R2C2_cAMP in CYTOSOL: '+str( sim.getCompSpecCount('cytosol', 'R2C2_cAMP')))
        print('TOTAL PKA in CYTOSOL: '+str( sim.getCompSpecCount('cytosol', 'PKA')))

        res[1] = sim.getCompSpecCount('cytosol', 'CaM_N_2_C_2')
        res[2] = sim.getCompSpecCount('cytosol', 'AC18_CaM')
        res[3] = sim.getCompSpecCount('cytosol', 'cAMP')
        res[4] = sim.getCompSpecCount('cytosol', 'PKA')
        res[5] = sim.getCompSpecCount('cytosol', 'Ca')

        print('Number of vesicles bound to actin = '+str( len(actin_bound_ves)) )
        syn_dimers_n = sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'synapsin_dimer')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'synapsin_dimer_p')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_p')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_p')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'PKA_synapsin_dimer')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_PKA')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_PKA')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'PP2A_synapsin_dimer_p')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_p_PP2A')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_p_PP2A')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_CDK5')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_synapsin_dimer_p_CDK5')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_CaN_CaM')\
        +sim.getCompVesicleSurfaceLinkSpecCount('cytosol', 'ves', 'Rab3_TOMO_p_synapsin_dimer_p_CaN_CaM')
        print('Number of synapsin dimers = '+ str( syn_dimers_n) )

        res[6] = len(nonrec_ves_clust)
        res[7] = len(nonrec_ves_free)
        res[8] = len(docked_ves)

        print('NON-CLUSTERED VESICLES: '+str( len(nonrec_ves_free)))
        print('CLUSTERED VESICLES: ' +str(len(nonrec_ves_clust)))
        print('TOTAL PHOSPHO SYNAPSIN in CYTOSOL: '+str( sim.getCompSpecCount('cytosol', 'synapsin_p')))

        ind_count = sim.getCompSpecCount('cleft', 'REL_IND')
        res[9] = ind_count

        #count active Calcineurin all forms in cytosol
        can_count = sim.getCompSpecCount('cytosol', 'CaN_CaM')+sim.getCompSpecCount('cytosol', 'CaN_DYNpp')+sim.getCompSpecCount('cytosol', 'CaN_DYNp')
        res[10] = can_count

        #count all dephosphorylated TOMO
        TOMO_dephos = getTOMOcount()
        res[11] = TOMO_dephos
        #count all free cytosolic TOMO
        TOMO_free = sim.getCompSpecCount('cytosol', 'TOMO')+sim.getCompSpecCount('cytosol', 'TOMO_p')
        res[12] = TOMO_free

        #Count TOMO bound to Syx in non-displaceable state
        TOMO_SYX_nondisp = sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'RIM_M13_Rab3_SYX_M18_SNP25_TOMO')\
        +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p')\
        +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'RIM_M13_Rab3_SYX_M18_SNP25_TOMO_P')\
        +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_P')\
        +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'RIM_M13_Rab3_SYX_M18_SNP25_TOMO_PKA')\
        +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'RIM_M13_Rab3_SYX_M18_SNP25_TOMO_p_PKA')
        res[13] = TOMO_SYX_nondisp
        #Count TOMO bound to Syx in displaceable state
        TOMO_SYX_disp = sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'RIM_M13_Rab3_SYX_M18_SNP25_TOMOx')\
        +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p')\
        +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_P')\
        +sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'RIM_M13_Rab3_SYX_M18_SNP25_TOMOx_p_P')
        res[14] = TOMO_SYX_disp
        #count recycled vesicles
        recycled_vesicles_tot = sim.getRaftEndocytosisExtent('raftendo')
        res[15] = recycled_vesicles_tot

        DYNpp_raft = sum(sim.getPatchRaftSpecCountDict('memb', 'Raft', 'DYNpp').values()) + sum(sim.getPatchRaftSpecCountDict('memb', 'Raft', 'CaN_DYNpp').values())
        res[16] = DYNpp_raft
        DYNp_raft = sum(sim.getPatchRaftSpecCountDict('memb', 'Raft', 'DYNp').values()) + sum(sim.getPatchRaftSpecCountDict('memb', 'Raft', 'CaN_DYNp').values())
        res[17] = DYNp_raft

        DYN_raft = sum(sim.getPatchRaftSpecCountDict('memb', 'Raft', 'DYN').values())
        res[18] = DYN_raft

        DYN_SYN1_raft = sum(sim.getPatchRaftSpecCountDict('memb', 'Raft', 'DYN_SYN1').values())+sum(sim.getPatchRaftSpecCountDict('memb', 'Raft', 'DYN_SYN1_CDK5').values())
        res[19] = DYN_SYN1_raft

        #count disassembled SNAREs in Rafts
        sd_dict = sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SNARE_DISS')
        res[20] = getRaftSpecTypeCount(sd_dict)
        print('Number of disassembled cis-SNAREs: '+ str(getRaftSpecTypeCount(sd_dict)))
        #count SYB in membrane
        res[21] = sim.getPatchSpecCount('memb', 'SYB')
        #count SYB in Rafts
        res[22] = sum(sim.getPatchRaftSpecCountDict('memb', 'Raft', 'SYB').values())
        #count used vesicles
        used_ves_n = len(init_vesicles_used)
        res[23] = used_ves_n
        rec_vesicles_used_n = len(init_rec_vesicles_used)
        res[24] = rec_vesicles_used_n
        nonrec_vesicles_used_n = len(init_nonrec_vesicles_used)
        res[25] = nonrec_vesicles_used_n
        new_vesicles_used_n = len(new_vesicles_used)
        res[26] = new_vesicles_used_n
        all_used_ves = rec_vesicles_used_n+nonrec_vesicles_used_n+new_vesicles_used_n
        res[27] = all_used_ves
        fused_ind_all.sort()
        print('Fused vesicles: ', fused_ind_all)
        #count Rafts
        all_rafts_n = sim.getPatchRaftCount('memb', 'Raft')
        res[28] = all_rafts_n
        #count free Rab3 on vesicles
        res[29] = sim.getCompVesicleSurfaceSpecCount('cytosol', 'ves', 'Rab3')
        #count all link specs
        res[30] = total_link_specs
        #count phosphorylated synapsin
        res[31] = getPhosphoSynapsins()
        print('Total phosho synapsins: ', getPhosphoSynapsins())
        print('Total rafts: '+str(all_rafts_n) )
        total_dynamin_cytosol = sim.getCompSpecCount('cytosol', 'DYN')+sim.getCompSpecCount('cytosol', 'DYNp')+sim.getCompSpecCount('cytosol', 'DYN_SYN1')
        print('Total dynamin in cytosol: '+str( total_dynamin_cytosol) )
        total_dynamin_membrane = sim.getPatchSpecCount('memb', 'DYN')+sim.getPatchSpecCount('memb', 'DYNp')+sim.getPatchSpecCount('memb', 'DYN_SYN1')
        print('Total dynamin in membrane: '+str( total_dynamin_membrane) )
        total_dynamin_rafts = DYNpp_raft+DYNp_raft+DYN_raft+DYN_SYN1_raft
        print('Total dynamin in rafts: '+str( total_dynamin_rafts) )
        print('DYNpp in rafts: '+str( DYNpp_raft) )
        print('DYNp in rafts: '+str( DYNp_raft) )
        print('DYN in rafts: '+str( DYN_raft) )
        print('DYN_SYN1 in rafts: '+ str( DYN_SYN1_raft))

        #check dynamin saturation and add to raft as necessary (only one per time step in chronological order)
        print('Total new rafts: ', len(new_rafts))
        print('Total filled rafts: ', len(filled_rafts))
        if total_dynamin_rafts < 60:
            for raft in new_rafts:
                ves_protein_count = sim.getSingleRaftSpecCount('Raft', raft, 'SYB')+sim.getSingleRaftSpecCount('Raft', raft, 'syt')
                if ves_protein_count>93:
                    sim.setSingleRaftSpecCount('Raft', raft, 'DYNp', 1)
                    sim.setSingleRaftSpecCount('Raft', raft, 'DYN_AD', 0)
                    new_rafts.remove(raft)
                    filled_rafts.append(raft)
                    break
                else:
                    continue
                break

        #save data this time point
        if myrank == 0:
            np.savetxt(afile_DATA, res, newline = " ")
            afile_DATA.write("\n")
            afile_DATA.close()

        #count RIM_M13 in dock_tris
        for d in dock_tris:
            count = sim.getTriSpecCount(d, 'RIM_M13')
            nbs = dock_tri_to_surround_tris[d]
            for n in nbs:
                count_nbs = sim.getTriSpecCount(n, 'RIM_M13')+sim.getTriSpecCount(n, 'RIM')

        #maintain RIM_M13 in dock tris
        dock_tris_occupied = []
        for d in ves_docktris:
            tri = ves_docktris[d]
            dock_tris_occupied.append(tri)
        for d in dock_tris:
            if d not in dock_tris_occupied:
                sim.setTriSpecCount(d, 'RIM_M13', 4)
        for t in azone_and_endo_tris:
            if t not in dock_tris:
                sim.setTriSpecCount(t, 'RIM_M13', 0)

        #separate vesicles into RECYCLING and RESERVE POOLS
        rec_pool = []
        res_pool =[]
        #create lists of Rab3 counts
        VES_RAB3_ALL = []
        VES_RAB3_REC = []
        VES_RAB3_RES = []
        RAB3_dict = sim.getCompVesicleSurfaceSpecCountDict('cytosol', 'ves', 'Rab3')
        rab3_tot_count = 0
        for v in RAB3_dict:
            rab3_count = RAB3_dict[v]
            VES_RAB3_ALL.append(rab3_count)
            rab3_tot_count += rab3_count
            if rab3_count>0:
                rec_pool.append(v)
            else:
                res_pool.append(v)
        for v in rec_ves_ind:
            VES_RAB3_REC.append(RAB3_dict[v])
        for v in nonrec_ves_ind:
            VES_RAB3_RES.append(RAB3_dict[v])

        if myrank == 0:
            np.savetxt(afile_RAB3, VES_RAB3_ALL, newline = " ", fmt='%s')
            afile_RAB3.write("\n")
            afile_RAB3.close()
            np.savetxt(afile_RAB3_rec, VES_RAB3_REC, newline = " ", fmt='%s')
            afile_RAB3_rec.write("\n")
            afile_RAB3_rec.close()
            np.savetxt(afile_RAB3_res, VES_RAB3_RES, newline = " ", fmt='%s')
            afile_RAB3_res.write("\n")
            afile_RAB3_res.close()

    #save release probs and tris
    print ('SAVING RELEASE PROBS 1')
    if myrank == 0:
        print ('SAVING RELEASE PROBS 2')
        np.savetxt(afile_RP, rel_probs, newline = " ", fmt='%s')
        afile_RP.write("\n")
        afile_RP.close()
        print ('SAVING RELEASE PROBS COMPLETE')
