'''
This is the netParams.py file for the expanded receptive field (RF) model
Code by L Medlock (Nov 15 2024)
'''

from netpyne import specs, sim
from neuron import h
import numpy as np
import random
from RF_indexing import RF_indx
import json
import pandas as pd

try: 
    from __main__ import cfg
except:
    from cfg_RF import cfg

netParams = specs.NetParams()

###############################################################################
# TOUCH STIM 
###############################################################################

# TouchStim ---- Spatially Ordered Touch Stim (9x9 PANs): 
netParams.PAN5 = [78,79,80,81,82,83,84,85,86]  
netParams.PAN6 = [93,94,95,96,97,98,99,100,101] 
netParams.PAN7 = [108,109,110,111,112,113,114,115,116]  
netParams.PAN8 = [123,124,125,126,127,128,129,130,131]  
netParams.PAN9 = [138,139,140,141,142,143,144,145,146]
netParams.PAN10 = [153,154,155,156,157,158,159,160,161] 
netParams.PAN11 = [168,169,170,171,172,173,174,175,176] 
netParams.PAN12 = [183,184,185,186,187,188,189,190,191]
netParams.PAN13 = [198,199,200,201,202,203,204,205,206]
netParams.TouchStim = netParams.PAN5 + netParams.PAN6  + netParams.PAN7  + netParams.PAN8  + netParams.PAN9   + netParams.PAN10  + netParams.PAN11  + netParams.PAN12 + netParams.PAN13

# TouchStimRand ---- Spatially Disordered Touch Stim (rand PANs in different #s (25, 49, 81)):
#netParams.TouchStimRand = random.sample(range(0,285), 81)   # Spatially disordered PAN activation (not SCS)

###############################################################################
# CELL PARAMETERS
###############################################################################

##############   AdEx Models + Balachandar et al. (2018) + Ratte et al. (2015)   #####################
##Inhibitory Spinal Neurons (Tonic Spiking)
netParams.importCellParams(
        label='SpinalNeuron_AdEx_Tonic',
        conds={'cellType': 'I_tonic', 'cellModel': 'I_tonic'},
        fileName=('SpinalNeuron_AdEx.hoc'),
        cellName='SDH_Tonic_AdEx',
        importSynMechs=True)

# # # Excitatory Spinal Neurons (Single Spiking)
netParams.importCellParams(
        label='SpinalNeuron_AdEx_Single',
        conds={'cellType': 'E_single', 'cellModel': 'E_single'},
        fileName=('SpinalNeuron_AdEx.hoc'),
        cellName='SDH_Single_AdEx',
        importSynMechs=True)

# # # Excitatory Spinal Neurons (Delayed Spiking)
netParams.importCellParams(
        label='SpinalNeuron_AdEx_Delay',
        conds={'cellType': 'E_delay', 'cellModel': 'E_delay'},
        fileName=('SpinalNeuron_AdEx.hoc'),
        cellName='SDH_Delay_AdEx',
        importSynMechs=True)

# # Excitatory Spinal Neurons (Hybrid Spiking)
netParams.importCellParams(
        label='SpinalNeuron_AdEx_Hybrid',
        conds={'cellType': 'E_hybrid', 'cellModel': 'E_hybrid'},
        fileName=('SpinalNeuron_AdEx.hoc'),
        cellName='SDH_Hybrid_AdEx',
        importSynMechs=True)

# Primary Afferent (I&F Only)
netParams.importCellParams(
        label='PAN_AdEx',
        conds={'cellType': 'PAN', 'cellModel': 'PAN'},
        fileName=('SpinalNeuron_AdEx.hoc'),
        cellName='PAN_AdEx',
        importSynMechs=True)

###############################################################################
# POPULATION PARAMETERS
###############################################################################

# PANs --- Cell Model of PANs 
netParams.popParams['PAN'] = {'cellType':'PAN', 
                                  'gridSpacing': 1750,
                                  'xRange' : [0,31500], 
                                  'yRange' : [0,24500], 
                                  'zRange' : [0,0], 
                                  'cellModel': 'PAN'}

# Tonic Pop (Inhibitory)
netParams.popParams['I_layer'] = {'cellType': 'I_tonic',
                                    'gridSpacing': 1750,
                                    'cellModel': 'I_tonic',
                                    'xRange' : [12250,19250], 
                                    'yRange' : [10500,14000], 
                                    'zRange' : [4000,4000] }

# Excitatory Pop (Delay, Single, Hybrid)
netParams.popParams['E_cell1'] = {'cellType':'E_delay', 
                                  'numCells': 1, 
                                  'cellModel': 'E_delay',
                                  'xRange' : [10500,10500], 
                                  'yRange' : [12250,12250], 
                                  'zRange' : [4000,4000] }

# Excitatory Pop (Delay, Single, Hybrid)
netParams.popParams['E_cell2'] = {'cellType':'E_delay', 
                                  'numCells': 1, 
                                  'cellModel': 'E_delay',
                                  'xRange' : [21000,21000], 
                                  'yRange' : [12250,12250], 
                                  'zRange' : [4000,4000] }

# Excitatory Pop (Delay, Single, Hybrid)
netParams.popParams['Proj'] = {'cellType':'E_delay', 
                                  'numCells': 1, 
                                  'cellModel': 'E_delay',
                                  'xRange' : [16500,16500], 
                                  'yRange' : [20000,20000], 
                                  'zRange' : [4000,4000] }

###############################################################################
# SCS STIM 
###############################################################################

## ------- VecStim of PAN Spike Trains from SCS -------- #
netParams.PAN_num = 285    # Number of PANs to stimulate with SCS (set to 0 if no SCS)
netParams.PAN_total = 285  # total number of PANs in PAN Layer

# Choose 1kHz or 50Hz SCS (update the excel file name):
netParams.df = pd.read_excel('50Hz-spike-times-expanded.xlsx')  # importing SCS spike times for PANS

netParams.PANStim = random.sample(range(0,netParams.PAN_total), netParams.PAN_num)   # Randomly select PANs to stimulate with SCS
netParams.spkt_PAN = [0] * netParams.PAN_total
netParams.df1 = netParams.df.transpose()
netParams.df1 = netParams.df1.fillna(0)
netParams.df1 = netParams.df1 * 1000
netParams.list1 = netParams.df1.values.tolist()
del netParams.df, netParams.df1  # delete the data frames once they are turned to list so that it can save
netParams.spkt_SCS1 = [0] * len(netParams.list1)
for y in range(len(netParams.list1)):    # removing 0s from spike trains
    list1_ = netParams.list1[y]
    list2_ = [i for i in list1_ if i != 0]
    netParams.spkt_SCS1[y] = list2_
netParams.spkt_SCS = random.sample(netParams.spkt_SCS1, len(netParams.spkt_SCS1))   # shuffle SCS
for i in range(netParams.PAN_num):      # insert spike trains for different PANs
        t = netParams.PANStim[i]
        spkt_PAN_ = netParams.spkt_SCS[i]
        netParams.spkt_PAN[t]=[spkt_PAN_]
for h in range(netParams.PAN_total):    # empty spike train when neuron is not selected
    if netParams.spkt_PAN[h] == 0:
        netParams.spkt_PAN[h] = []

netParams.popParams['SCS'] = {'cellModel': 'VecStim', 
                              'spkTimes': netParams.spkt_PAN,
                              'xRange' : [15750,15750], 
                              'yRange' : [1000,1000], 
                              'numCells': 285}  

###############################################################################
# SYNAPTIC PARAMETERS
###############################################################################

netParams.synMechParams['AMPA'] = {'mod': 'Exp2Syn', 'tau1': 0.1, 'tau2': 5.0, 'e': 0}  # excitatory synaptic mechanism
netParams.synMechParams['GABA'] = {'mod': 'Exp2Syn', 'tau1': 0.1, 'tau2': 20.0, 'e': -55}  # inhibitory synaptic mechanism (set to -55 mV for CD in SCS simulations)

###############################################################################
# RECEPTIVE FIELD STRUCTURE
###############################################################################
# ----------------- Local Output 1 ----------------- #:
# PAN to E and I indexing based on the center of the RF and size of PAN layer (height)
[PANtoE1, PANtoICenter1, PANtoITop1, PANtoIBottom1, PANtoILeft1, PANtoIRight1] = RF_indx(97, 15)   # RF_indx(center of RF, # of PANs -- height)
# PANs --> Output Neuron 
netParams.RFE1 = PANtoE1  # Index of PANs in E1 Receptive Field 
netParams.E1 = np.zeros(len(netParams.RFE1), dtype='int8')                                 # Index of E1 Neuron
netParams.PrePostRF1 = []
netParams.PrePostRF1 = list(zip(netParams.RFE1, netParams.E1))                            # Zip PANs indexes with E1 index (for connectivity)              
# PANs --> ICenter
netParams.RF1_ICenter = PANtoICenter1                                                    # 
netParams.ICenter1 = np.ones(len(netParams.RF1_ICenter), dtype='int8') * 4                 # Cell index for center neuron (ICenter) of inhibitory layer
netParams.PP_RF1_ICenter = []
netParams.PP_RF1_ICenter = list(zip(netParams.RF1_ICenter, netParams.ICenter1))           # Zip PANs indexes with ICenter index (for connectivity)    
# PANs --> ITop
netParams.RF1_ITop = PANtoITop1                                          
netParams.ITop1 = np.ones(len(netParams.RF1_ITop), dtype='int8') * 3                       # Cell index for top neuron (ITop) of inhibitory layer
netParams.PP_RF1_ITop = []
netParams.PP_RF1_ITop = list(zip(netParams.RF1_ITop, netParams.ITop1))                    # Zip PANs indexes with ITop index (for connectivity)   
# PANs --> IBottom
netParams.RF1_IBottom = PANtoIBottom1                                      
netParams.IBottom1 = np.ones(len(netParams.RF1_IBottom), dtype='int8') * 5                 # Cell index for bottom neuron (IBottom) of inhibitory layer
netParams.PP_RF1_IBottom = []
netParams.PP_RF1_IBottom = list(zip(netParams.RF1_IBottom, netParams.IBottom1))           # Zip PANs indexes with IBottom index (for connectivity)   
# PANs --> ILeft
netParams.RF1_ILeft = PANtoILeft1                                    
netParams.ILeft1 = np.ones(len(netParams.RF1_ILeft), dtype='int8')                         # Cell index for left neuron (ILeft) of inhibitory layer
netParams.PP_RF1_ILeft = []
netParams.PP_RF1_ILeft = list(zip(netParams.RF1_ILeft, netParams.ILeft1))                 # Zip PANs indexes with ILeft index (for connectivity)   
# PANs --> IRight
netParams.RF1_IRight = PANtoIRight1                                  
netParams.IRight1 = np.ones(len(netParams.RF1_IRight), dtype='int8') * 7                   # Cell index for right neuron (IRight) of inhibitory layer
netParams.PP_RF1_IRight = []
netParams.PP_RF1_IRight = list(zip(netParams.RF1_IRight, netParams.IRight1))              # Zip PANs indexes with IRight index (for connectivity)   

# ----------------- Local Output 2 ----------------- #:
# PAN to E and I indexing based on the center of the RF and size of PAN layer (height)
[PANtoE2, PANtoICenter2, PANtoITop2, PANtoIBottom2, PANtoILeft2, PANtoIRight2] = RF_indx(187, 15)   # RF_indx(center of RF, PAN layer height)
# PANs --> Output Neuron 
netParams.RFE2 = PANtoE2  # Index of PANs in E1 Receptive Field 
netParams.E2 = np.zeros(len(netParams.RFE2), dtype='int8')                                 # Index of E2 Neuron
netParams.PrePostRF2 = []
netParams.PrePostRF2 = list(zip(netParams.RFE2, netParams.E2))                            # Zip PANs indexes with E2 index (for connectivity)              
# PANs --> ICenter
netParams.RF2_ICenter = PANtoICenter2                                                    # 
netParams.ICenter2 = np.ones(len(netParams.RF2_ICenter), dtype='int8') * 10                # Cell index for center neuron (ICenter) of inhibitory layer
netParams.PP_RF2_ICenter = []
netParams.PP_RF2_ICenter = list(zip(netParams.RF2_ICenter, netParams.ICenter2))           # Zip PANs indexes with ICenter index (for connectivity)    
# PANs --> ITop
netParams.RF2_ITop = PANtoITop2                                          
netParams.ITop2 = np.ones(len(netParams.RF2_ITop), dtype='int8') * 9                      # Cell index for top neuron (ITop) of inhibitory layer
netParams.PP_RF2_ITop = []
netParams.PP_RF2_ITop = list(zip(netParams.RF2_ITop, netParams.ITop2))                    # Zip PANs indexes with ITop index (for connectivity)   
# PANs --> IBottom
netParams.RF2_IBottom = PANtoIBottom2                                      
netParams.IBottom2 = np.ones(len(netParams.RF2_IBottom), dtype='int8') * 11                 # Cell index for bottom neuron (IBottom) of inhibitory layer
netParams.PP_RF2_IBottom = []
netParams.PP_RF2_IBottom = list(zip(netParams.RF2_IBottom, netParams.IBottom2))           # Zip PANs indexes with IBottom index (for connectivity)   
# PANs --> ILeft
netParams.RF2_ILeft = PANtoILeft2                                    
netParams.ILeft2 = np.ones(len(netParams.RF2_ILeft), dtype='int8') * 7                        # Cell index for left neuron (ILeft) of inhibitory layer
netParams.PP_RF2_ILeft = []
netParams.PP_RF2_ILeft = list(zip(netParams.RF2_ILeft, netParams.ILeft2))                 # Zip PANs indexes with ILeft index (for connectivity)   
# PANs --> IRight
netParams.RF2_IRight = PANtoIRight2                                  
netParams.IRight2 = np.ones(len(netParams.RF2_IRight), dtype='int8') * 13                   # Cell index for right neuron (IRight) of inhibitory layer
netParams.PP_RF2_IRight = []
netParams.PP_RF2_IRight = list(zip(netParams.RF2_IRight, netParams.IRight2))              # Zip PANs indexes with IRight index (for connectivity)   

# ----------------- PAN to extra I Cells (for Global Output Neuron surround) ----------------- #:
netParams.RF2_I10 = [0,1,2,3,4,15,16,17,18,19,30,31,32,33,34,45,46,47,48,49,60,61,62,63,64]                                 
netParams.I10 = np.ones(len(netParams.RF2_I10), dtype='int8') * 0                   # Cell index for I10 in I Layer
netParams.PP_RF_I10 = []
netParams.PP_RF_I10 = list(zip(netParams.RF2_I10, netParams.I10))        

netParams.RF2_I11 = [105,106,107,108,109,120,121,122,123,124,135,136,137,138,139,150,151,152,153,154,165,166,167,168,169]                                 
netParams.I11 = np.ones(len(netParams.RF2_I11), dtype='int8') * 6                   # Cell index for I11 in I Layer
netParams.PP_RF_I11 = []
netParams.PP_RF_I11 = list(zip(netParams.RF2_I11, netParams.I11))     

netParams.RF2_I12 = [210,211,212,213,214,225,226,227,228,229,240,241,242,243,244,255,256,257,258,259,270,271,272,273,274]                                 
netParams.I12 = np.ones(len(netParams.RF2_I12), dtype='int8') * 12                   # Cell index for I12 in I Layer
netParams.PP_RF_I12 = []
netParams.PP_RF_I12 = list(zip(netParams.RF2_I12, netParams.I12))   

netParams.RF2_I13 = [10,11,12,13,14,25,26,27,28,29,40,41,42,43,44,55,56,57,58,59,70,71,72,73,74]                               
netParams.I13 = np.ones(len(netParams.RF2_I13), dtype='int8') * 2                   # Cell index for I13 in I Layer
netParams.PP_RF_I13 = []
netParams.PP_RF_I13 = list(zip(netParams.RF2_I13, netParams.I13))   

netParams.RF2_I14 = [115,116,117,118,119,130,131,132,133,134,145,146,147,148,149,160,161,162,163,164,175,176,177,178,179]                                 
netParams.I14 = np.ones(len(netParams.RF2_I14), dtype='int8') * 8                   # Cell index for I14 in I Layer
netParams.PP_RF_I14 = []
netParams.PP_RF_I14 = list(zip(netParams.RF2_I14, netParams.I14))   

netParams.RF2_I15 = [220,221,222,223,224,235,236,237,238,239,250,251,252,253,254,265,266,267,268,269,280,281,282,283,284]                                
netParams.I15 = np.ones(len(netParams.RF2_I15), dtype='int8') * 14                   # Cell index for I15 in I Layer
netParams.PP_RF_I15 = []
netParams.PP_RF_I15 = list(zip(netParams.RF2_I15, netParams.I15))   

## SCS to PANs
netParams.SCS_IDs = np.arange(0,285)
netParams.PAN_IDs = np.arange(0,285)
netParams.SCS_to_PAN = list(zip(netParams.PAN_IDs, netParams.SCS_IDs)) 

###############################################################################
# CONNECTIVITY PARAMETERS
###############################################################################

# ## ----------------- Connectivity for LO1 ----------------- ##
netParams.connParams['PAN->E1'] = {    #  PAN --> LO1
        'preConds': {'pop':'PAN'},
        'postConds': {'pop':'E_cell1'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.001*(0.9*exp(-((pre_x-10500)**2 + (pre_y-12250)**2)/8100000)+0.1*exp(-((pre_x-10500)**2 + (pre_y-12250)**2)/65000000)))',   # Max 0.0010 for E cells
        #'weight': 'max(0,0.0005*exp(-((pre_x-10500)**2 + (pre_y-12250)**2)/12500000))',    # Max 0.0005 for tonic
        'prob': 0.2,                      # synaptic probability
        'delay': 10,                      # transmission delay (ms)
        'connList': netParams.PrePostRF1  # connection between specific neurons
        }
netParams.connParams['PAN->I_Center1'] = {    #  PAN --> I_Center
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-10500)**2 + (pre_y-12250)**2)/12500000))',   # Gaussian with center at [2500,2500]          
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms) 
        'connList': netParams.PP_RF1_ICenter # connection between specific neurons
        }
netParams.connParams['PAN->I_Top1'] = {    #  PAN --> I_Top
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-10500)**2 + (pre_y-7000)**2)/12500000))',   # Gaussian with center at [2000,1000]              
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF1_ITop # connection between specific neurons
        }
netParams.connParams['PAN->I_Bottom1'] = {    #  PAN --> I_Bottom
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-10500)**2 + (pre_y-17500)**2)/12500000))', # Gaussian with center at [2500,4000]                
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF1_IBottom # connection between specific neurons
        }
netParams.connParams['PAN->I_Left1'] = {    #  PAN --> I_Left
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-5250)**2 + (pre_y-12250)**2)/12500000))',     # Gaussian with center at [1000,2500]                
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF1_ILeft # connection between specific neurons
        }
netParams.connParams['PAN->I_Right1'] = {    #  PAN --> I_Right
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-15750)**2 + (pre_y-12250)**2)/12500000))',    # Gaussian with center at [4000,2500]         
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF1_IRight # connection between specific neurons
        }
netParams.connParams['I_layer->E_cell1'] = {    #  I_layer --> E_cell  
        'preConds': {'pop': 'I_layer'},
        'postConds': {'pop': 'E_cell1'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'GABA',              # target synaptic mechanism
        'weight':'max(0,0.002*exp(-((pre_x-14000)**2 + (pre_y-12250)**2)/12500000))',       # Max = 0.002 when delay, single, gap   
        #'weight':'max(0,0.001*exp(-((pre_x-14000)**2 + (pre_y-12250)**2)/12500000))',      # Max = 0.001 when tonic                                                   
        'prob': 0.2,                    # synaptic probability
        'delay': 2,                     # transmission delay (ms)
        'connList': [[1,0],[3,0],[4,0],[5,0],[7,0]]       # connections between I_Layer neurons and E1
        }

## ----------------- Connectivity for LO2 ----------------- ##
netParams.connParams['PAN->E2'] = {    #  PAN --> LO2
        'preConds': {'pop':'PAN'},
        'postConds': {'pop':'E_cell2'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.001*(0.9*exp(-((pre_x-21000)**2 + (pre_y-12250)**2)/8100000)+0.1*exp(-((pre_x-21000)**2 + (pre_y-12250)**2)/65000000)))',  # Max 0.0010 for E cells
        #'weight': 'max(0,0.0005*exp(-((pre_x-21000)**2 + (pre_y-12250)**2)/12500000))',     # Max 0.0005 for tonic
        'prob': 0.2,                      # synaptic probability
        'delay': 10,                      # transmission delay (ms)
        'connList': netParams.PrePostRF2 # connection between specific neurons
        }
netParams.connParams['PAN->I_Center2'] = {    #  PAN --> I_Center
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-21000)**2 + (pre_y-12250)**2)/12500000))',  # Gaussian with center at      
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF2_ICenter # connection between specific neurons
        }
netParams.connParams['PAN->I_Top2'] = {    #  PAN --> I_Top
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-21000)**2 + (pre_y-7000)**2)/12500000))',    # Gaussian with center at [2000,1000]              
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF2_ITop # connection between specific neurons
        }
netParams.connParams['PAN->I_Bottom2'] = {    #  PAN --> I_Bottom
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-21000)**2 + (pre_y-17500)**2)/12500000))',   # Gaussian with center at [2500,4000]                
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF2_IBottom # connection between specific neurons
        }
netParams.connParams['PAN->I_Left2'] = {    #  PAN --> I_Left
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-15750)**2 + (pre_y-12250)**2)/12500000))',     # Gaussian with center at [1000,2500]                
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF2_ILeft # connection between specific neurons
        }
netParams.connParams['PAN->I_Right2'] = {    #  PAN --> I_Right
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-26250)**2 + (pre_y-12250)**2)/12500000))',     # Gaussian with center at [4000,2500]         
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF2_IRight # connection between specific neurons
        }
netParams.connParams['I_layer->E_cell2'] = {    #  I_layer --> E_cell  
        'preConds': {'pop': 'I_layer'},
        'postConds': {'pop': 'E_cell2'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'GABA',              # target synaptic mechanism
        'weight':'max(0,0.002*exp(-((pre_x-17500)**2 + (pre_y-12250)**2)/12500000))',       # Max = 0.002 when delay, single, gap   
        #'weight':'max(0,0.001*exp(-((pre_x-17500)**2 + (pre_y-12250)**2)/12500000))',      # Max = 0.001 when tonic                                                         
        'prob': 0.2,                    # synaptic probability
        'delay': 2,                     # transmission delay (ms)
        'connList': [[7,0],[9,0],[10,0],[11,0],[13,0]]       # connections between I_Layer neurons and E1
        }

### ----------------- PAN to extra I Cells and I Layer to Proj ----------------- ##
netParams.connParams['PAN->I_10'] = {    #  PAN --> I_Center
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-3500)**2 + (pre_y-3500)**2)/12500000))',   # Gaussian with center at           
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms) 
        'connList': netParams.PP_RF_I10 # connection between specific neurons
        }
netParams.connParams['PAN->I_11'] = {    #  PAN --> I_Top
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-15750)**2 + (pre_y-3500)**2)/12500000))',   # Gaussian with center at              
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF_I11 # connection between specific neurons
        }
netParams.connParams['PAN->I_12'] = {    #  PAN --> I_Bottom
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-28000)**2 + (pre_y-3500)**2)/12500000))', # Gaussian with center at               
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF_I12 # connection between specific neurons
        }
netParams.connParams['PAN->I_13'] = {    #  PAN --> I_Left
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-3500)**2 + (pre_y-21000)**2)/12500000))',     # Gaussian with center at [1000,2500]                
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF_I13 # connection between specific neurons
        }
netParams.connParams['PAN->I_14'] = {    #  PAN --> I_Right
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-15750)**2 + (pre_y-21000)**2)/12500000))',    # Gaussian with center at          
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF_I14 # connection between specific neurons
        }
netParams.connParams['PAN->I_15'] = {    #  PAN --> I_Right
        'preConds': {'pop': 'PAN'},
        'postConds': {'pop': 'I_layer'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 'max(0,0.0005*exp(-((pre_x-28000)**2 + (pre_y-21000)**2)/12500000))',    # Gaussian with center at          
        'prob': 0.2,                    # synaptic probability
        'delay': 10,                     # transmission delay (ms)
        'connList': netParams.PP_RF_I15 # connection between specific neurons
        }

##-------- Local Output and Inh. Layer to Global Output -------------#
netParams.connParams['E1->P'] = {       #  Output Cell 1 to Projection Neuron
        'preConds': {'pop':'E_cell1'},
        'postConds': {'pop':'Proj'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 0.005,                  
        'prob': 1,                      # synaptic probability
        'delay': 2,                     # transmission delay (ms)
        }
netParams.connParams['E2->P'] = {       #  Output Cell 2 to Projection Neuron
        'preConds': {'pop':'E_cell2'},
        'postConds': {'pop':'Proj'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'AMPA',              # target synaptic mechanism
        'weight': 0.005,                
        'prob': 1,                      # synaptic probability
        'delay': 2,                      # transmission delay (ms)
        }

netParams.connParams['I_layer->Proj'] = {    #  I_layer --> E_cell  
        'preConds': {'pop': 'I_layer'},
        'postConds': {'pop': 'Proj'},
        'sec': 'soma',                  # target postsyn section
        'synMech': 'GABA',              # target synaptic mechanism
        'weight':0.0025,                # Max = 0.002 when delay, single, gap                                                   
        'prob': 1,                      # synaptic probability
        'delay': 2                      # transmission delay (ms)     
        }

## ----------------- Connectivity SCS to PAN Layer ----------------- ##
netParams.connParams['SCS->PAN'] = {    #  
        'preConds': {'pop':'SCS'},
        'postConds': {'pop':'PAN'},
        'sec': 'soma',                    # target postsyn section
        'synMech': 'AMPA',                # target synaptic mechanism
        'weight': 0.3, 
        'delay': 0,                       # transmission delay (ms)
        'connList': netParams.SCS_to_PAN  # connection between SCS to PAN (1:1)
        }

###############################################################################
# STIMULATION PARAMETERS
###############################################################################

## ------------ Touch Stimuli --------------##
netParams.stimSourceParams['Mech1'] = { 'type': 'NetStim',  # NETSTIM
                                        'rate' : 50,
                                        'start': 0,
                                        'noise': 0.6 }   # 0 = sync, 0.6 = async

netParams.stimTargetParams['Input->PAN'] = {'source': 'Mech1',  # Mech --> PAN
                                                  'sec':'soma',
                                                  'loc': 0.5,
                                                  'weight': 0.2,  
                                                  'delay' : 1,
                                                  'conds': {'pop':'PAN','cellList': netParams.TouchStim }}   # TouchStim = 9x9 PANs (spatially ordered), TouchStimRand = spatially disordered

                                                   





