% sweepFitBoltzmannHnap.m
% Jessica Parker, October 12, 2024
%
% This Matlab script fits the measured steady-state inactivation levels with a first order Boltzmann function, estimating the voltage of
% half-inactivation and steepness.

close all;
clear all;

vrsn = 'A';
msrRealVrsn = 'A';

run1 = 3;
run2 = 0;
run3 = 0;
run4 = 4;

boltzType = 'h'; %Set to m or h depending on whether it is an activation or inactivation curve.
FitMinHnap = 0; % Set to 1 to fit the min hNaP, set to 0 to use the minimum measured hNaP for this parameter.
PlotResidualError = 1; % Set to 1 to plot residual error
DisplayParameters = 1;
DisplayMinHnap = 1;
percStep = 0.005; % Sets the resolution you use to find the optimal parameter values

skpPnts = 2; % Plotting every other point so that the final curve does not look to cluttered. Set to 1 to plot every point.
xmin = -80;
xmax = -10;
PlotFit = 1;
CloseFigures = 0;
fntnm = 'Lucida Sans';
fntsz = 14;
fntwght = 'bold';
textYpostn = 0.8;
textXpostn = 0.45;

%% Starting fit parameters
VhNaP = -53;
khNaP = 3.9; %Negative for activation, positive for inactivation
hNaPmin = 0.4;

dir2 = [num2str(run1) '_' num2str(run2) '_' num2str(run3)];
dir3 = [dir2 '_' num2str(run4)];

realhNaP = load(['data/' boltzType 'napMsrd' dir3 msrRealVrsn '.txt']);

Vclamp = realhNaP(2,:);
hNaP = realhNaP(1,:);

if FitMinHnap == 0 && contains(boltzType,'h')
  hNaPmin = min(hNaP);
end

Vm = Vclamp;
hNaPfit = (1-hNaPmin)./(1.0+exp((Vm-VhNaP)/khNaP)) + hNaPmin; % Calculate steady-state hNaP curve using starting parameters guesses
edif = (hNaPfit - hNaP).*(hNaPfit - hNaP); % Used to calculate residual error
reserr = sum(edif); % residual error

VhNaPI = VhNaP-0.2*abs(VhNaP); % Start of VhNaP values to sweep
khNaPI = khNaP-0.2*abs(khNaP); % Start of khNaP values to sweep
hNaPminI = hNaPmin-0.2*abs(hNaPmin); % Start of min hNaP values to sweep if FitMinHnap is set to 1.
nruns = round(0.4/percStep)+1; % Number of instances in each sweep

parstep = percStep*abs(VhNaP); % Step size of VhNaP sweep
for aa = 1:nruns
  VhNaP0 = VhNaPI + (aa-1)*parstep; % Sweeping VhNaP
  hNaPfit0 = (1-hNaPmin)./(1.0+exp((Vm-VhNaP0)/khNaP)) + hNaPmin; % Calculating hNaP curve at this VhNaP value
  edif0 = (hNaPfit0 - hNaP).*(hNaPfit0 - hNaP);
  reserrs(aa) = sum(edif0); % Residual error
  aa = aa+1;
end
[bf, bfIndx] = min(reserrs); % Finding best fit from sweep
VhNaP = VhNaPI + (bfIndx-1)*parstep;
reserr(1) = bf;

if PlotResidualError
  allpars = VhNaPI:parstep:VhNaPI+(nruns-1)*parstep;
  f = figure()
  plot(allpars,reserrs);
  hold on;
  plot(VhNaP,bf,'rx');
  xlabel(['V_{' boltzType 'NaP}']);
end

parstep = percStep*abs(khNaP);
for aa = 1:nruns
  khNaP0 = khNaPI + (aa-1)*parstep;
  hNaPfit0 = (1-hNaPmin)./(1.0+exp((Vm-VhNaP)/khNaP0)) + hNaPmin;
  edif0 = (hNaPfit0 - hNaP).*(hNaPfit0 - hNaP);
  reserrs(aa) = sum(edif0);
  aa = aa+1;
end
[bf, bfIndx] = min(reserrs);
khNaP = khNaPI + (bfIndx-1)*parstep;
reserr(2) = bf;

if PlotResidualError
  allpars = khNaPI:parstep:khNaPI+(nruns-1)*parstep;
  f = figure()
  plot(allpars,reserrs);
  hold on;
  plot(khNaP,bf,'rx');
  xlabel(['k_{' boltzType 'NaP}']);
end

if FitMinHnap
  parstep = percStep*abs(hNaPmin);
  for aa = 1:nruns
    hNaPmin0 = hNaPminI + (aa-1)*parstep;
    hNaPfit0 = (1-hNaPmin0)./(1.0+exp((Vm-VhNaP)/khNaP)) + hNaPmin0;
    edif0 = (hNaPfit0 - hNaP).*(hNaPfit0 - hNaP);
    reserrs(aa) = sum(edif0);
    aa = aa+1;
  end
  [bf, bfIndx] = min(reserrs);
  hNaPmin = hNaPminI + (bfIndx-1)*parstep;
  reserr(3) = bf;

  if PlotResidualError
    allpars = hNaPminI:parstep:hNaPminI+(nruns-1)*parstep;
    f = figure()
    plot(allpars,reserrs);
    hold on;
    plot(hNaPmin,bf,'rx');
    xlabel(['min ' boltzType '_{NaP}']);
  end
end

Vm = xmin:0.01:xmax;
hNaPfit = (1-hNaPmin)./(1.0+exp((Vm-VhNaP)/khNaP)) + hNaPmin;
disp(hNaPmin);

Vclmp = Vclamp(1:skpPnts:end);
hNaPplt = hNaP(1:skpPnts:end);

f = figure();
f.PaperPositionMode = 'manual';
f.PaperUnits = 'inches';
f.Units = 'inches';
f.OuterPosition = [1 1 6.0 5.0];
f.InnerPosition = [0.25 0.25 5.5 4.5];
f.PaperPosition = [0.25 0.25 5.0 4.0];
f.RendererMode = 'manual';

axes('position',[0.18 0.17 0.78 0.79]);
hold on;
if PlotFit
  plot(Vm,hNaPfit,'color',[0.7 0.7 0.7],'linewidth',2);
end
plot(Vclmp,hNaPplt,'r.','markersize',18);
if DisplayParameters
  text(xmin+textXpostn*(xmax-xmin),textYpostn+0.13,['V_{1/2' boltzType 'NaP} = ' num2str(round(VhNaP,1)) ' mV'],'fontsize',fntsz-1,'fontname',fntnm,'fontweight',fntwght);
  text(xmin+textXpostn*(xmax-xmin),textYpostn,['k_{' boltzType 'NaP} = ' num2str(round(khNaP,2)) ' mV'],'fontsize',fntsz-1,'fontname',fntnm,'fontweight',fntwght);
end
if DisplayMinHnap
  text(xmin+textXpostn*(xmax-xmin),textYpostn-0.13,['min ' boltzType '_{NaP} = ' num2str(round(hNaPmin,3))],'fontsize',fntsz-1,'fontname',fntnm,'fontweight',fntwght);
end
ylabel([boltzType '_{NaP}']);
xlabel('Membrane Potential (mV)');
xlim([xmin xmax]);
ylim([0 1]);
box off;
ax = gca;
ax.FontName = fntnm;
ax.FontSize = fntsz;
ax.FontWeight = fntwght;
ax.LineWidth = 3.0;

print(f,['plots/' boltzType 'NaPfitLeastSquares' dir3 msrRealVrsn '_' vrsn '.eps'],'-depsc','-r0');

if CloseFigures
  close(f);
end

dlmwrite(['data/' boltzType 'NaPfitParameters' dir3 msrRealVrsn '_' vrsn '.txt'],[VhNaP,khNaP,hNaPmin]);
