% sweepFitTimeConstant.m
% Jessica Parker, October 12, 2024
%
% This Matlab script measures the time constant of inactivation by fitting the inactivation level reached after the deinactivating step
% versus the duration of the deinactivating step (indexed by run5).

close all;
clear all;

vrsn = 'A';
msrRealVrsn = 'A';

run1 = 3;
run2 = 0;
run3 = 0;
run4 = 6;
dataDir = 'data/';
figDir = 'plots/';

boltzType = 'h'; %Set to m or h depending on whether it is an activation or inactivation
percStep = 0.002;  % Sets the resolution you use to find the optimal parameter values
percRange = 0.4; % Width of the range of each parameter sweep as a proportion of the current parameter value
nruns = round(percRange/percStep)+1; % Number of instances in each sweep

tauStrt = -1.9; % Starting value of tau_hNaP parameter fit
strtYintl = 0.58; % Starting value of recovery at t=0 parameter
maxYstrt = 1.01; % Starting value of maximum hNaP recovery parameter if FitMaxY is set to 1

%% Plot Properties
xmin = 0;
xmax = 15;
ymin = 0.4;
ymax = 1;
VariableYrange = 1;
PlotFit = 1;
CloseFigures = 0;
fntnm = 'Lucida Sans';
fntsz = 16;
fntwght = 'bold';
textYpostn = 0.7;
textXpostn = 0.4;
fitColor = [22, 150, 149]/255;

dir2 = [num2str(run1) '_' num2str(run2) '_' num2str(run3)];


dir3 = [dir2 '_' num2str(run4)];

rcvry0 = load([dataDir boltzType 'napRecovery' dir3 msrRealVrsn '.txt']); % Loading recovery data as analyzed by measureTauHnap.m
holdDurs = rcvry0(2,:); % Durations of deinactivating step
rcvry = rcvry0(1,:); % Level of recovery (hNaP) reached by the end of the deinactivating step

% Setting initial estimate for Fit Parameters
tau = tauStrt; %Positive for decreasing exponential, negative for increasing exponential
maxY = maxYstrt;
strtY = strtYintl;

tm = holdDurs;

rcvryFit = (strtY-maxY)*exp(tm/tau)+maxY; % Equation of fit
edif = (rcvryFit - rcvry).*(rcvryFit - rcvry); % Used to calculate residual error
reserr = sum(edif); % Residual error

chngPar = 1.0; % This variable is used to measure how much the fit parameters change at each loop run
bb = 1; % Ticker parameter to avoid infinite while loop
sfty = 500; % Safety parameter to avoid infinite while loop
while chngPar > 0.001 && bb < sfty % Once the parameters are no longer changing significantly the loop breaks

    tauI = tau-0.5*percRange*abs(tau); % Beginning of the range of tau_hNaP values to sweep
    parstep = percStep*abs(tau); % Step size of tau_hNaP sweep
    lastPar = tau; % Best fit tau from the last run of the loop

    for aa = 1:nruns
        taux0 = tauI + (aa-1)*parstep; % Sweeping tau_hNaP
        rcvryFit0 = (strtY-maxY)*exp(tm/taux0)+maxY; % Fit equation
        edif0 = (rcvryFit0 - rcvry).*(rcvryFit0 - rcvry);
        reserrs(aa) = sum(edif0); % Residual error
        aa = aa+1;
    end
    [bf, bfIndx] = min(reserrs); % Finding best fit from this sweep
    tau = tauI + (bfIndx-1)*parstep; % Tau_hNaP of best fit at least for this run of the loop
    reserr(1) = bf; % Residual error of best fit
    chngPar1 = abs(tau - lastPar)/lastPar; % The proportion change of parameter from the last loop run

    strtYi = strtY-0.5*percRange*abs(strtY); % Beginning of range of recovery at t=0 values to sweep
    parstep = percStep*abs(strtY); % Step size of sweep for recovery at t=0
    lastPar = strtY; % strtY of best fit from the last run of the loop

    for aa = 1:nruns
        strtY0 = strtYi + (aa-1)*parstep; % Sweeping recovery at t=0
        rcvryFit0 = (strtY0-maxY)*exp(tm/tau)+maxY; % Fit equation
        edif0 = (rcvryFit0 - rcvry).*(rcvryFit0 - rcvry);
        reserrs(aa) = sum(edif0); % Residual error
        aa = aa+1;
    end
    [bf, bfIndx] = min(reserrs); % Finding best fit
    strtY = strtYi + (bfIndx-1)*parstep; % Recover at t=0 of best fit
    reserr(1) = bf; % Residual error of best fit
    chngPar2 = abs(strtY - lastPar)/lastPar; % The proportion change of parameter from the last loop run

    maxYi = maxY-0.5*percRange*abs(maxY); % Beginning of range of max recovery values to sweep
    parstep = percStep*abs(maxY); % Step size of sweep for max recovery
    lastPar = maxY; % max recovery of best fit from the last run of the loop

    for aa = 1:nruns
        maxY0 = maxYi + (aa-1)*parstep; % Sweeping max recovery
        rcvryFit0 = (strtY-maxY0)*exp(tm/tau)+maxY0; % Fit equation
        edif0 = (rcvryFit0 - rcvry).*(rcvryFit0 - rcvry);
        reserrs(aa) = sum(edif0); % Residual error
        aa = aa+1;
    end
    [bf, bfIndx] = min(reserrs); % Finding best fit
    maxY = maxYi + (bfIndx-1)*parstep; % Max recovery of best fit
    reserr(1) = bf; % Residual error of best fit
    chngPar3 = abs(maxY - lastPar)/lastPar; % The proportion change of parameter from the last loop run

    if bb > 1 % This forces the code to run the loop at least twice
        chngPar = max([chngPar1,chngPar2,chngPar3]); % Here we find the maximum proportion change of the parameters
    end                                              % then check if it is below the threshold for breaking the loop

    bb = bb+1;
end
disp(['bb = ' num2str(bb) ', tau = ' num2str(tau) ', strtY = ' num2str(strtY) ', maxY = ' num2str(maxY)]);

tmFit = xmin:0.01:xmax;
rcvryFit = (strtY-maxY)*exp(tmFit/tau)+maxY; % Fit equation to be plotted

if VariableYrange
    ymin0 = min(rcvryFit);
    ymax0 = max(rcvryFit);
    yl0 = ymax0 - ymin0;
    ymin = ymin0 - 0.1*yl0;
    ymax = ymax0 + 0.1*yl0;
end
yl = ymax - ymin;

f = figure();
f.PaperPositionMode = 'manual';
f.PaperUnits = 'inches';
f.Units = 'inches';
f.OuterPosition = [1 1 6.0 5.4];
f.InnerPosition = [0.25 0.25 5.5 4.9];
f.PaperPosition = [0.25 0.25 5.0 4.4];
f.RendererMode = 'manual';

axes('position',[0.18 0.17 0.78 0.7]);
hold on;
if PlotFit
    plot(tmFit,rcvryFit,'color',[0.7 0.7 0.7],'linewidth',2);
end
plot(tm,rcvry,'.','color',fitColor,'markersize',18);
text(xmin+textXpostn*(xmax-xmin),ymin+textYpostn*yl,['\tau_{' boltzType 'NaP} = ' num2str(round(abs(tau),2)) ' s'],'fontsize',fntsz,'fontname',fntnm,'fontweight',fntwght);
ylabel([boltzType '_{NaP} Recovery']);
xlabel('Time (s)');
xlim([xmin xmax]);
ylim([ymin ymax]);
box off;
ax = gca;
ax.FontName = fntnm;
ax.FontSize = fntsz;
ax.FontWeight = fntwght;
ax.LineWidth = 3.0;

print(f,['plots/tau' boltzType 'NaPfitLeastSquares' dir3 msrRealVrsn '_' vrsn '.eps'],'-depsc','-r0');

if CloseFigures
    close(f);
end

dlmwrite([dataDir 'tau' boltzType 'NaPfit' dir3 msrRealVrsn '_' vrsn '.txt'],[tau,strtY,maxY]);


