% plotTraceVariableCurrentsFormal.m
% Jessica Parker, October 16, 2024
%
% This Matlab script produces Fig. 5C, plotting measured INaP, normalized gNaP as measured, Vclamp, [Na+]i, ENa, the actual 
% model INaPn, and IPump. The time scale plotted goes from 10 s before the step to -40 mV to 20 s after the step.

clear all;
close all;

vrsn = 'A';
run1 = 3;
run2 = 0;
run3 = 0;
run4 = 2;
run5i = 31;
run5f = 31;
run6i = 1;
run6f = 2;

dir1 = [num2str(run1) '_' num2str(run2) '_' num2str(run3)];
dir2 = [dir1 '_' num2str(run4)];
leakDir = [dir1 '_1']; % Data in 3_0_0_1 is used to measure leak current using P/4 leak subtraction

dataDir = 'data/';
figDir = 'plots/';
prfx = 'NaiMnap';
nvar = 2;
Naindx = 1;
mNaPindx = 2;
VclampStrt = -70; % The voltage set for run5 = 1 and run6 = 2.
VclampStp = 1; % The voltage increase between run5 instances
Vref = -90; % The holding potential used in 3_0_0_0
VclampInter = -80; % The starting potental when run6 = 1
tint = 0.0001;

% Figure properties and options
fntnm = 'Lucida Sans';
fntsz = 16;
fntwght = 'bold';
vsblty = 'on'; % Set to 'off' to make figures not pop up or set to 'on'
CloseFigs = 0; % Set to 1 to automatically close figures after printing them to a file, or set to 0 to leave them open.
xmin = 90;
xmax = 120;
xl = xmax - xmin;

readpars;
assmdENa = 45.0; % This is assumed Na reversal potential used to measure gNaP

VarN = ['../' dir1 '_0/' dataDir prfx dir1 '_0_0_0.dat'];
fl = fopen(VarN);
yyStrt = fread(fl,[nvar, 10000000],'double')';
fclose(fl);

for run5 = run5i:run5f
    yy = yyStrt;
    leakyy = yyStrt;
    indxs(1) = length(yy(:,1));
    lngths(1) = length(yy(:,1))-1;
    for run6=run6i:run6f   % Concatenating portions of time saved in separate files

        dir3 = [dir2 '_' num2str(run5)];

        VarN = [dataDir prfx dir3 '_' num2str(run6) '.dat'];
        fl = fopen(VarN);
        yy0 = fread(fl,[nvar, 10000000],'double')';
        fclose(fl);

        yy = [yy(1:end-1,:); yy0]; % Remove last point here, because otherwise this time point will repeat as the first time point of the next file
        indxs(run6+1) = length(yy(:,1));
        lngths(run6+1) = length(yy0(:,1));
        if run6 < run6f
            lngths(run6+1) = lngths(run6+1) - 1;
        end

        leakFlN = ['../' leakDir '/' dataDir prfx leakDir '_' num2str(run5) '_' num2str(run6) '.dat'];
        leakFl = fopen(leakFlN);
        leakyy0 = fread(leakFl,[nvar, 10000000],'double')';
        fclose(leakFl);

        leakyy = [leakyy(1:end-1,:); leakyy0];
    end

    Vclamp = VclampStrt + VclampStp*(run5-1);
    Vsteps = [Vref, VclampInter, Vclamp];

    Vm = [];
    leakVm = [];
    for aa = 1:length(Vsteps)
        Vm = [Vm; Vsteps(aa)*ones(lngths(aa),1)]; % Membrane potential steps for this run5
        leakVm = [leakVm; (Vref + 0.25*(Vsteps(aa) - Vref))*ones(lngths(aa),1)]; % The membrane potential used to measure Ileak in 3_0_0_1 for
    end                                                                        % P/4 leak subtraction

    Ipump = Ipmpmx./(1.0+(Naih*exp(alpha*Vm/rtf)./yy(:,Naindx)).^nhillNa); % Ipump in 3_0_0_2
    Inap = gNaP*yy(:,mNaPindx).*(Vm-rtf*log(Nae./yy(:,Naindx))); % INaP in 3_0_0_2
    IleakNa = gL*(EL-EK)/(ENa-EK)*(Vm-rtf*log(Nae./yy(:,Naindx))); % ILeakNa in 3_0_0_2
    IleakK = gL*(EL-ENa)/(EK-ENa)*(Vm-EK); % ILeakK in 3_0_0_2
    Ileak = IleakNa + IleakK;
    Itot = Ipump + Inap + Ileak;
    realENa = rtf*log(Nae./yy(:,Naindx));

    IleakPump = Ipmpmx./(1.0+(Naih*exp(alpha*leakVm/rtf)./leakyy(:,Naindx)).^nhillNa); % Ipump in 3_0_0_1
    IleakNap = gNaP*leakyy(:,mNaPindx).*(leakVm-rtf*log(Nae./leakyy(:,Naindx))); % INaP in 3_0_0_1
    IleakLNa = gL*(EL-EK)/(ENa-EK)*(leakVm-rtf*log(Nae./leakyy(:,Naindx))); % ILeakNa in 3_0_0_1
    IleakLK = gL*(EL-ENa)/(EK-ENa)*(leakVm-EK); % ILeakK in 3_0_0_1
    IleakL = IleakLNa + IleakLK;
    IleakMsrd0 = IleakPump + IleakNap + IleakL;
    IleakMsrd = IleakMsrd0;
    IleakIntl = mean(IleakMsrd0(round(indxs(1)/2):indxs(1)));
    IleakMsrd(indxs(1)+1:indxs(end)) = IleakIntl + 4.0*(IleakMsrd0(indxs(1)+1:indxs(end)) - IleakIntl); % Measured leak current using P/4 leak subtraction


    IleakRef = mean(IleakMsrd0(round(indxs(1)/2):indxs(1)-1));
    IleakMsrd = 4*(IleakMsrd0 - IleakRef) + IleakRef;
    IleakMsrd(1:indxs(1)) = IleakMsrd0(1:indxs(1));

    InapMsrd = Itot - IleakMsrd;
    gnapMsrd = InapMsrd./(Vm - assmdENa);

    tt = 0.0:tint:tint*(indxs(end)-1);
    xmini = round(xmin/tint) + 1;
    xmaxi = round(xmax/tint) + 1;

    f = figure('visible',vsblty);
    f.PaperPositionMode = 'manual';
    f.PaperUnits = 'inches';
    f.Units = 'inches';
    f.OuterPosition = [1.0 1.0 6.25 9];
    f.InnerPosition = [0.25 0.25 5.75 8.5];
    f.PaperPosition = [0.25 0.25 5.25 8];
    f.RendererMode = 'manual';

    titlPos = 0.12;

    axes('position',[0.07 0.93 0.9 0.07]);
    hold on;
    text(0.08,0.8,'Canonical Voltage-Clamp Model','fontname',fntnm,'fontsize',fntsz+1,'fontweight',fntwght);
    xlim([0 1]);
    ylim([0 1]);
    axis off;

    ymin = min(InapMsrd(xmini:xmaxi));
    ymax = max(InapMsrd(xmini:xmaxi));
    yl = ymax - ymin;
    minInapMsrd = ymin;
    maxInapMsrd = ymax;
    minInapLbl = -150;
    maxInapLbl = 0;

    shft = 0.01*xl;

    axes('position',[0.07 0.83 0.9 0.095]);
    hold on;
    plot(tt(xmini:xmaxi),InapMsrd(xmini:xmaxi),'k','linewidth',2.0);
    xticks([]);
    xlim([xmin-0.1*xl xmax]);
    text(xmin-0.9*titlPos*xl,ymin+0.6*yl,'I_{NaP}','fontname',fntnm,'fontsize',fntsz+1,'fontweight',fntwght);
    plot([tt(indxs(2))+0.04*xl tt(indxs(2))+0.075*xl],maxInapLbl*[1 1],'k','linewidth',3.5);
    text(tt(indxs(2))+0.095*xl,maxInapLbl,[num2str(maxInapLbl) ' pA'],'fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    plot([tt(indxs(2))-0.075*xl tt(indxs(2))-0.04*xl],minInapLbl*[1 1],'k','linewidth',3.5);
    text(tt(indxs(2))-0.305*xl-shft,minInapLbl,[num2str(minInapLbl) ' pA'],'fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    ylim([ymin-0.1*yl ymax+0.1*yl]);
    axis off;

    ymin = min(gnapMsrd(xmini:xmaxi));
    ymax = max(gnapMsrd(xmini:xmaxi));
    yl = ymax - ymin;
    minGnapLbl = 0;
    maxGnapLbl = 1.5;

    axes('position',[0.07 0.695 0.9 0.095]);
    hold on;
    plot(tt(xmini:xmaxi),gnapMsrd(xmini:xmaxi),'k','linewidth',2.0);
    xlim([xmin-0.1*xl xmax]);
    xticks([]);
    text(xmin-0.9*titlPos*xl,ymin+0.6*yl,'g_{NaP}','fontname',fntnm,'fontsize',fntsz+1,'fontweight',fntwght);
    plot([tt(indxs(2))-0.075*xl tt(indxs(2))-0.04*xl],maxGnapLbl*[1 1],'k','linewidth',3.5);
    text(tt(indxs(2))-0.25*xl-shft,maxGnapLbl,[num2str(maxGnapLbl) ' nS'],'fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    plot([tt(indxs(2))+0.04*xl tt(indxs(2))+0.075*xl],minGnapLbl*[1 1],'k','linewidth',3.5);
    text(tt(indxs(2))+0.095*xl,minGnapLbl,[num2str(minGnapLbl) ' nS'],'fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    ylim([ymin-0.1*yl ymax+0.1*yl]);
    axis off;

    ymin = min(Vsteps(2:end));
    ymax = max(Vsteps(2:end));
    yl = ymax - ymin;

    axes('position',[0.07 0.58 0.9 0.065]);
    hold on;
    plot(tt(xmini:xmaxi),Vm(xmini:xmaxi),'k','linewidth',2.0);
    xlim([xmin-0.1*xl xmax]);
    text(xmin-titlPos*xl,-55,'V_{clamp}','fontname',fntnm,'fontsize',fntsz+1,'fontweight',fntwght);
    plot([tt(indxs(2))-0.075*xl tt(indxs(2))-0.04*xl],[-40 -40],'k','linewidth',3.5);
    text(tt(indxs(2))-0.285*xl-shft,-40,'-40 mV','fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    plot([tt(indxs(2))+0.04*xl tt(indxs(2))+0.075*xl],[-80 -80],'k','linewidth',3.5);
    text(tt(indxs(2))+0.095*xl,-80,'-80 mV','fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    plot([xmax-8 xmax-3],[-70 -70],'k','linewidth',3.5);
    text(xmax-6.6,-83,'5 s','fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    ylim([ymin-0.1*yl ymax+0.1*yl]);
    axis off

    minNa = min(yy(xmini:xmaxi,Naindx));
    maxNa = max(yy(xmini:xmaxi,Naindx));
    hghtNa = maxNa-minNa;
    minNaiLbl = 10;
    maxNaiLbl = 90;

    axes('position',[0.07 0.425 0.9 0.095]);
    hold on;
    plot(tt(xmini:xmaxi),yy(xmini:xmaxi,Naindx),'k','linewidth',2.0);
    xticks([]);
    xlim([xmin-0.1*xl xmax]);
    text(xmin-titlPos*xl,minNa+0.5*hghtNa,'[Na^+]_i','fontname',fntnm,'fontsize',fntsz+1,'fontweight',fntwght);
    plot([tt(indxs(2))+0.04*xl tt(indxs(2))+0.075*xl],minNaiLbl*[1 1],'k','linewidth',3.5);
    text(tt(indxs(2))+0.095*xl,minNaiLbl,[num2str(minNaiLbl) ' mM'],'fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    plot([tt(indxs(2))-0.075*xl tt(indxs(2))-0.04*xl]+0.1*xl,maxNaiLbl*[1 1],'k','linewidth',3.5);
    text(tt(indxs(2))+0.1*xl-0.265*xl-shft,maxNaiLbl,[num2str(maxNaiLbl) ' mM'],'fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    ylim([minNa-0.3*hghtNa maxNa+0.1*hghtNa]);
    axis off;

    minEna = min(realENa(xmini:xmaxi));
    maxEna = max(realENa(xmini:xmaxi));
    hghtEna = maxEna-minEna;
    minEnaLbl = 10;
    maxEnaLbl = 50;

    axes('position',[0.07 0.29 0.9 0.095]);
    hold on;
    plot(tt(xmini:xmaxi),realENa(xmini:xmaxi),'k','linewidth',2.0);
    xticks([]);
    xlim([xmin-0.1*xl xmax]);
    text(xmin-0.9*titlPos*xl,minEna+0.5*hghtEna,'E_{Na}','fontname',fntnm,'fontsize',fntsz+1,'fontweight',fntwght);
    plot([tt(indxs(2))+0.04*xl tt(indxs(2))+0.075*xl],maxEnaLbl*[1 1],'k','linewidth',3.5);
    text(tt(indxs(2))+0.095*xl,maxEnaLbl,[num2str(maxEnaLbl) ' mV'],'fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    plot([tt(indxs(2))-0.075*xl tt(indxs(2))-0.04*xl]+0.1*xl,minEnaLbl*[1 1],'k','linewidth',3.5);
    text(tt(indxs(2))+0.1*xl-0.255*xl-shft,minEnaLbl,[num2str(minEnaLbl) ' mV'],'fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    ylim([minEna-0.1*hghtEna maxEna+0.1*hghtEna]);
    axis off;

    minInap = min(Inap(xmini:xmaxi));
    maxInap = max(Inap(xmini:xmaxi));
    hghtInap = maxInap-minInap;
    minInapnLbl = -150;
    maxInapnLbl = 0;

    axes('position',[0.07 0.155 0.9 0.095]);
    hold on;
    plot(tt(xmini:xmaxi),Inap(xmini:xmaxi),'k','linewidth',2.0);
    xticks([]);
    xlim([xmin-0.1*xl xmax]);
    text(xmin-titlPos*xl,minInap+0.6*hghtInap,'I_{NaPn}','fontname',fntnm,'fontsize',fntsz+1,'fontweight',fntwght);
    plot([tt(indxs(2))+0.04*xl tt(indxs(2))+0.075*xl],maxInapnLbl*[1 1],'k','linewidth',3.5);
    text(tt(indxs(2))+0.095*xl,maxInapnLbl,[num2str(maxInapnLbl) ' pA'],'fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    plot([tt(indxs(2))-0.075*xl tt(indxs(2))-0.04*xl],minInapnLbl*[1 1],'k','linewidth',3.5);
    text(tt(indxs(2))-0.305*xl-shft,minInapnLbl,[num2str(minInapnLbl) ' pA'],'fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    ylim([minInap-0.1*hghtInap maxInap+0.1*hghtInap]);
    axis off;

    minIpump = min(Ipump(xmini:xmaxi));
    maxIpump = max(Ipump(xmini:xmaxi));
    hghtIpump = maxIpump-minIpump;
    maxIpumpLbl = 30;
    minIpumpLbl = 10;

    axes('position',[0.07 0.02 0.9 0.095]);
    hold on;
    plot(tt(xmini:xmaxi),Ipump(xmini:xmaxi),'k','linewidth',2.0);
    xticks([]);
    xlim([xmin-0.1*xl xmax]);
    text(xmin-titlPos*xl,minIpump+0.5*hghtIpump,'I_{Pump}','fontname',fntnm,'fontsize',fntsz+1,'fontweight',fntwght);
    plot([tt(indxs(2))+0.04*xl tt(indxs(2))+0.075*xl],minIpumpLbl*[1 1],'k','linewidth',3.5);
    text(tt(indxs(2))+0.095*xl,minIpumpLbl,[num2str(minIpumpLbl) ' pA'],'fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    plot([tt(indxs(2))-0.075*xl tt(indxs(2))-0.04*xl]+0.05*xl,maxIpumpLbl*[1 1],'k','linewidth',3.5);
    text(tt(indxs(2))+0.05*xl-0.24*xl-shft,maxIpumpLbl,[num2str(maxIpumpLbl) ' pA'],'fontname',fntnm,'fontsize',fntsz,'fontweight',fntwght);
    ylim([minIpump-0.1*hghtIpump maxIpump+0.1*hghtIpump]);
    axis off;

    print(f,[figDir 'multipleVariablesCurrents' dir3 vrsn '.eps'],'-depsc','-r0');

    if strcmp(vsblty,'off')
        close(f);
    elseif CloseFigs
        close(f);
    end
end


  

