% plotInapGnapVclamp.m
% Jessica Parker, October 18, 2024
%
% This Matlab script measures and plots INaP and gNaP after subtracting leak current, but it does not write any 
% data to files. It is useful if you want to make changes to the figures without worrying about overwriting data files.

clear all;
close all;

vrsn = 'A';

run1 = 3;
run2 = 0;
run3 = 0;
run4 = 6;
run5i = 1;
run5f = 35;
run6i = 1;
run6f = 3;
skpPltTrc = 7; % Number skipped (+1) between traces plotted if PlotTraces is 1, set to 1 to plot all run5 traces between strtPltTrc and stpPltTrc

dir1 = [num2str(run1) '_' num2str(run2)];
dir2 = [dir1 '_' num2str(run3)];

leakDir = [dir2 '_5'];
mnapVrsn = 'A';
mnapDir = [dir2 '_2'];
mxmlGvrsn = 'A';
mxmlGdir = [dir2 '_2'];

dataDir = 'data/';
prfx = 'NaiMnap';
nvar = 2;
Naindx = 1;
mNaPindx = 2;
VclampPulse1 = -10; % Membrane potential for first step after holding at -90, which initially activates and inactivates the cell (run6 = 1)
VclampPulse2 = -80; % Membrane potential used for deinactivating step (run6 = 2)
VclampPulse3 = -40; % Final voltage step where the recovery is measured (run6 = 3)
VclampHold = -90; % Holding potential used in 3_0_0_0
holdDurStrt = 0.5; % Initial duration of deinactivating step where run6 = 2
holdDurStep = 0.5; % Increase in duration of deinactivating step between consecutive run5 instances
SetMeasureTime = 0; % Set to 1 to use a preset time (msrTm) after the step to -40 mV to measure the peak gNaP. Set to 0 to measue the actual peak gNaP.
msrTm = 0.031; % Used if SetMeasureTime is set to 1.
tint = 0.0001; % Time step used by integral. Time between consecutive simulation data points.

%% Plot Properties
fntnm = 'Lucida Sans';
fntsz = 14;
vsblty = 'on';
CloseFigs = 1;
ZoomInOnStepUp = 0; % Set to 1 to zoom in on last step or set to 0 and set xmin and xmax below to the axis range you want, if PlotTraces = 1
xmin = 45; % Start of x-axis range plotted when PlotTraces is set to 1.
xmax = 80; % End of x-axis range plotted when PlotTraces is set to 1.

readpars; % Reading parameter values using readpars.m
assmdENa = 45; % This is the assumed Na reversal potential used to measure gNaP

mnapMsrd = load(['../' mnapDir '/' dataDir 'mnapMsrd' mnapDir mnapVrsn '.txt']); % Reading steady-state activation data from 3_0_0_2
mnapIndx = find(mnapMsrd(2,:) == VclampPulse3); % index of mnapMsrd corresponding to Vclamp = -40 mV
mnapMsrdFnl = mnapMsrd(1,mnapIndx); % measured mNaP at Vclamp = -40 mV
mnapGFnl = mnapMsrd(3,mnapIndx); % measured gNaP at Vclamp = -40 mV

mxmlGnap = load(['../' mxmlGdir '/' dataDir 'maximalGnap' mxmlGdir mxmlGvrsn '.txt']); % Reading maximum gNaP as measured in 3_0_0_2

holdDurs = holdDurStrt+(run5i-1)*holdDurStep:holdDurStep:holdDurStrt+(run5f-1)*holdDurStep; % Durations of deinactivating step for all run5 instances


dir3 = [dir2 '_' num2str(run4)];

VarN = ['../' dir2 '_0/' dataDir prfx dir2 '_0_0_0.dat'];
fl = fopen(VarN);
yyStrt = fread(fl,[nvar, 10000000],'double')'; % Contains simulation data from 3_0_0_0 where it is held at the holding potential of -90 mV
fclose(fl);

for run5 = run5i:skpPltTrc:run5f
    yy = yyStrt; % starts at -90 mV with data from 3_0_0_0
    leakyy = yyStrt;
    indxs(1) = length(yy);
    for run6=run6i:run6f   % Concatenating portions of time saved in separate files

        dir4 = [dir3 '_' num2str(run5)];

        VarN = [dataDir prfx dir4 '_' num2str(run6) '.dat'];
        fl = fopen(VarN);
        yy0 = fread(fl,[nvar, 10000000],'double')'; % Reading simulation data from 3_0_0_6
        fclose(fl);

        lngth(run6) = length(yy0(:,1));
        yy = [yy(1:end-1,:); yy0];   % Remove last point here, because otherwise you will repeat the same time point
        indxs(run6+1) = length(yy);

        leakFlN = ['../' leakDir '/' dataDir prfx leakDir '_' num2str(run5) '_' num2str(run6) '.dat'];
        leakFl = fopen(leakFlN);
        leakyy0 = fread(leakFl,[nvar, 10000000],'double')'; % Reading data from 3_0_0_3 where leak current is simulated for P/4 leak subtraction
        fclose(leakFl);

        leakyy = [leakyy(1:end-1,:); leakyy0];
    end

    VclampLkPls1 = 0.25*(VclampPulse1 - VclampHold) + VclampHold;
    VclampLkPls2 = 0.25*(VclampPulse2 - VclampHold) + VclampHold;
    VclampLkPls3 = 0.25*(VclampPulse3 - VclampHold) + VclampHold;

    % Membrane potential data over time for 3_0_0_6
    Vm = [VclampHold*ones(indxs(1),1); VclampPulse1*ones(indxs(2)-indxs(1),1); VclampPulse2*ones(indxs(3)-indxs(2),1); ...
        VclampPulse3*ones(indxs(4)-indxs(3),1)];

    % Membrane potential data over time for 3_0_0_5 used to measure the leak current for P/4 leak subtraction for this run5 instance.
    leakVm = [VclampHold*ones(indxs(1),1); VclampLkPls1*ones(indxs(2)-indxs(1),1); VclampLkPls2*ones(indxs(3)-indxs(2),1); ...
        VclampLkPls3*ones(indxs(4)-indxs(3),1)];

    tt = 0.0:tint:tint*(indxs(end)-1);

    Ipump = Ipmpmx./(1.0+(Naih*exp(alpha*Vm/rtf)./yy(:,Naindx)).^nhillNa); % Ipump in 3_0_0_6
    Inap = gNaP*yy(:,mNaPindx).*(Vm-rtf*log(Nae./yy(:,Naindx))); % INaP in 3_0_0_6
    IleakNa = gL*(EL-EK)/(ENa-EK)*(Vm-rtf*log(Nae./yy(:,Naindx))); % IleakNa in 3_0_0_6
    IleakK = gL*(EL-ENa)/(EK-ENa)*(Vm-EK); % IleakK in 3_0_0_6
    Ileak = IleakNa + IleakK;
    Itot = Ipump + Inap + Ileak;

    IleakPump = Ipmpmx./(1.0+(Naih*exp(alpha*leakVm/rtf)./leakyy(:,Naindx)).^nhillNa); % Ipump in 3_0_0_5
    IleakNap = gNaP*leakyy(:,mNaPindx).*(leakVm-rtf*log(Nae./leakyy(:,Naindx))); % INaP in 3_0_0_5
    IleakLNa = gL*(EL-EK)/(ENa-EK)*(leakVm-rtf*log(Nae./leakyy(:,Naindx))); % IleakNa in 3_0_0_5
    IleakLK = gL*(EL-ENa)/(EK-ENa)*(leakVm-EK); % IleakK in 3_0_0_5
    IleakL = IleakLNa + IleakLK;
    IleakMsrd0 = IleakPump + IleakNap + IleakL;

    IleakRef = mean(IleakMsrd0(round(0.75*indxs(1)):indxs(1)-1)); % The steady-state leak current after the cell has been held at -90 mV for a while
    IleakMsrd = 4*(IleakMsrd0 - IleakRef) + IleakRef; % The estimated leak current in the actual simulation in 3_0_0_4 used for P/4 leak subtraction
    IleakMsrd(1:indxs(1)) = IleakMsrd0(1:indxs(1));

    InapMsrd = Itot - IleakMsrd; % Subtracting measured/estimated leak current
    gnapMsrd = InapMsrd./(Vm - assmdENa);  % Measuring gNaP using assumed Na reversal potential
    if SetMeasureTime
        ttindx = round(msrTm/tint) + indxs(3);  % Index used to measure peak gNaP using time of expected peak if SetMeasureTime is 1
        msrTmGnap = gnapMsrd(ttindx); % Estimated peak gNaP using msrTm set above
    else
        [maxtmp, ttindx0] = max(gnapMsrd(indxs(3)+1:indxs(3)+500)); % Finding peak gNaP
        ttindx = ttindx0 + indxs(3); % index of peak gNaP
        msrTmGnap = gnapMsrd(ttindx); % peak gNaP
    end
    allGnaps(run5-run5i+1) = msrTmGnap; % array of peak gNaPs for all run5 instances
    msrTms(run5-run5i+1) = ttindx - indxs(3);  % array of times of peak gNaP for all run5 instances

    if ZoomInOnStepUp
        xmin = tt(indxs(3)) - 0.01;
        xmax = tt(indxs(3)) + 0.1;
        mrk = 'zm';
    else
        mrk = '';
    end

    f = figure('visible',vsblty);
    f.PaperPositionMode = 'manual';
    f.PaperUnits = 'inches';
    f.Units = 'inches';
    f.OuterPosition = [1 1 8.0 5.5];
    f.InnerPosition = [0.25 0.25 7.5 5.0];
    f.PaperPosition = [0.25 0.25 7.0 4.5];
    f.RendererMode = 'manual';

    axes('position',[0.15 0.74 0.8 0.2]);
    hold on;
    plot(tt, InapMsrd, 'k', 'linewidth', 1.5);
    ylabel('I_{NaP} (pA)');
    xticks([]);
    xlim([xmin xmax]);
    box off;
    ax = gca;
    ax.FontName = fntnm;
    ax.FontSize = fntsz;
    ax.LineWidth = 2.0;

    axes('position',[0.15 0.45 0.8 0.2]);
    hold on;
    plot(tt, gnapMsrd, 'k', 'linewidth', 1.5);
    plot(tt(ttindx),msrTmGnap,'rx','linewidth',1.5,'markersize',10);
    ylabel('g_{NaP} (nS)');
    xticks([]);
    xlim([xmin xmax]);
    box off;
    ax = gca;
    ax.FontName = fntnm;
    ax.FontSize = fntsz;
    ax.LineWidth = 2.0;

    axes('position',[0.15 0.16 0.8 0.2]);
    hold on;
    plot(tt, Vm, 'k', 'linewidth', 1.5);
    ylabel('V_{Clamp} (mV)');
    xlabel('Time (s)');
    xlim([xmin xmax]);
    box off;
    ax = gca;
    ax.FontName = fntnm;
    ax.FontSize = fntsz;
    ax.LineWidth = 2.0;

    print(f,['plots/InapGnapVclamp' dir3 '_' num2str(run5) vrsn mrk '.eps'],'-depsc','-r0');

    if strcmp(vsblty,'off')
        close(f);
    elseif CloseFigs
        close(f);
    end
end


