% sweepFitBoltzmannMnap.m
% Jessica Parker, October 12, 2024
%
% This Matlab script fits the measured steady-state activation levels with a first order Boltzmann function, estimating the voltage of
% half-activation and steepness.

close all;
clear all;

vrsn = 'A';
msrRealVrsn = 'A';

run1 = 3;
run2 = 0;
run3 = 0;
run4 = 2;

boltzType = 'm'; %Set to m or h depending on whether it is an activation or inactivation curve.
PlotResidualError = 0;
DisplayParameters = 1;
percRange = 0.4; % Sets the range of parameter values you are testing as a proportion of the starting fit parameter value
percStep = 0.005; % Sets the resolution you use to find the optimal parameter values
chngParThresh = 0.0005; %When the greatest proportion change in the parameters of best fit is less than this, the code stops searching
minRuns = 3; %The minimum number of searches that the sweep has to perform before settling on a parameter set

skpPnts = 2; % Plotting every other point so that the final curve does not look to cluttered. Set to 1 to plot every point.
xmin = -80;
xmax = -10;
PlotFit = 1;
CloseFigures = 0;
fntnm = 'Lucida Sans';
fntsz = 14;
fntwght = 'bold';
textYpostn = 0.3;
textXpostn = 0.55;

% Starting fit parameters
VmNaP = -47;
kmNaP = -3; % Negative for activation, positive for inactivation

dir2 = [num2str(run1) '_' num2str(run2) '_' num2str(run3)];
dir3 = [dir2 '_' num2str(run4)];

realMNaP = load(['data/' boltzType 'napMsrd' dir3 msrRealVrsn '.txt']);

Vclamp = realMNaP(2,:);
mNaP = realMNaP(1,:);

Vm = Vclamp;
mNaPfit = 1.0./(1.0+exp((Vm-VmNaP)/kmNaP)); % Calculate steady-state mNaP curve using starting parameters guesses
edif = (mNaPfit - mNaP).*(mNaPfit - mNaP); % Used to calculate residual error
reserr = sum(edif); % (residual error)^2

nruns = round(percRange/percStep)+1; % Number of instances in each sweep

chngPar = 100; % Start it with just some big number that will definitely be bigger than the amount a fit parameter would change
bb = 1;
sfty = 100;
while chngPar > chngParThresh && bb < sfty
    VmNaPI = VmNaP-0.5*percRange*abs(VmNaP); % Start of VmNaP values to sweep
    kmNaPI = kmNaP-0.5*percRange*abs(kmNaP); % Start of kmNaP values to sweep

    lastPar = VmNaP;
    parstep = percStep*abs(VmNaP); % Step size of VmNaP sweep
    for aa = 1:nruns
        VmNaP0 = VmNaPI + (aa-1)*parstep; % Sweeping VmNaP
        mNaPfit0 = 1.0./(1.0+exp((Vm-VmNaP0)/kmNaP)); % Calculating mNaP curve at this VmNaP value
        edif0 = (mNaPfit0 - mNaP).*(mNaPfit0 - mNaP); % Calculating residual error
        reserrs(aa) = sum(edif0); % (Residual error)^2
        aa = aa+1;
    end
    [bf, bfIndx] = min(reserrs); % Finding best fit from sweep
    VmNaP = VmNaPI + (bfIndx-1)*parstep; % VmNaP of best fit
    reserr(1) = bf; % Squared residual error of best fit
    chngPar1 = abs(VmNaP - lastPar)/lastPar;

    if PlotResidualError
        allpars = VmNaPI:parstep:VmNaPI+(nruns-1)*parstep;
        f = figure();
        plot(allpars,reserrs);
        hold on;
        plot(VmNaP,bf,'rx');
        xlabel(['V_{' boltzType 'NaP}']);
        ylabel('(Residual Error)^2');
        if CloseFigures
            close(f);
        end
    end

    lastPar = kmNaP;
    parstep = percStep*abs(kmNaP); % Step size of kmNaP sweep
    for aa = 1:nruns
        kmNaP0 = kmNaPI + (aa-1)*parstep; % Sweeping kmNaP
        mNaPfit0 = 1.0./(1.0+exp((Vm-VmNaP)/kmNaP0)); % mNaP curve for this kmNaP value
        edif0 = (mNaPfit0 - mNaP).*(mNaPfit0 - mNaP);
        reserrs(aa) = sum(edif0); % (Residual error)^2
        aa = aa+1;
    end
    [bf, bfIndx] = min(reserrs); % Finding best fit from sweep
    kmNaP = kmNaPI + (bfIndx-1)*parstep; % kmNaP of best fit
    reserr(2) = bf; % Squared residual error of best fit
    chngPar2 = abs(kmNaP - lastPar)/lastPar;

    if PlotResidualError
        allpars = kmNaPI:parstep:kmNaPI+(nruns-1)*parstep;
        f = figure();
        plot(allpars,reserrs);
        hold on;
        plot(kmNaP,bf,'rx');
        xlabel(['k_{' boltzType 'NaP}']);
        ylabel('(Residual Error)^2');
        if CloseFigures
            close(f);
        end
    end

    lastChngPar = chngPar;
    if bb >= minRuns
        chngPar = max([chngPar1,chngPar2]);
    end

    disp(['bb = ' num2str(bb) ', max par change = ' num2str(chngPar) ', VmNaP = ' num2str(VmNaP) ', kmNaP = ' ... 
        num2str(kmNaP) ', res. error = ' num2str(sqrt(bf))])
    bb = bb+1;
end

dlmwrite(['data/residualError' dir3 msrRealVrsn '_' vrsn '.txt'],sqrt(bf));

Vm = xmin:0.01:xmax;
mNaPfit = 1.0./(1.0+exp((Vm-VmNaP)/kmNaP)); % Final mNaP curve

Vclmp = Vclamp(1:skpPnts:end);
mNaPplt = mNaP(1:skpPnts:end);

f = figure();
f.PaperPositionMode = 'manual';
f.PaperUnits = 'inches';
f.Units = 'inches';
f.OuterPosition = [1 1 6.5 5.5];
f.InnerPosition = [0.25 0.25 6.0 5.0];
f.PaperPosition = [0.25 0.25 5.5 4.5];
f.RendererMode = 'manual';

axes('position',[0.18 0.17 0.78 0.79]);
hold on;
if PlotFit
    plot(Vm,mNaPfit,'color',[0.7 0.7 0.7],'linewidth',2);
end
plot(Vclmp,mNaPplt,'r.','markersize',18);
if DisplayParameters
    text(xmin+textXpostn*(xmax-xmin),textYpostn+0.13,['V_{1/2' boltzType 'NaP} = ' num2str(round(VmNaP,1)) ' mV'],'fontsize',fntsz-1,'fontname',fntnm,'fontweight',fntwght);
    text(xmin+textXpostn*(xmax-xmin),textYpostn,['k_{' boltzType 'NaP} = ' num2str(round(kmNaP,2)) ' mV'],'fontsize',fntsz-1,'fontname',fntnm,'fontweight',fntwght);
end
ylabel([boltzType '_{NaP}']);
xlabel('Membrane Potential (mV)');
xlim([xmin xmax]);
ylim([0 1]);
box off;
ax = gca;
ax.FontName = fntnm;
ax.FontSize = fntsz;
ax.FontWeight = fntwght;
ax.LineWidth = 3.0;

print(f,['plots/' boltzType 'NaPfitLeastSquares' dir3 msrRealVrsn '_' vrsn '.eps'],'-depsc','-r0');

if CloseFigures
    close(f);
end

dlmwrite(['data/' boltzType 'NaPfitParameters' dir3 msrRealVrsn '_' vrsn '.txt'],[VmNaP,kmNaP]);
