% measureTailCurrent.m
% Jessica Parker, October 18, 2024
%
% This Matlab script measures and saves the tail current and tail conductance from the peak gNaP onwards during the last 
% membrane potential step to -40 mV for run5 = 31. It is only used for run5 = 31 in order to make Figure 3C and Figure 5C. 

clear all;
close all;

vrsn = 'A';

run1 = 3;
run2 = 0;
run3 = 0;
run4 = 2;
run5i = 31;
run5f = 31;
run6i = 1;
run6f = 2;
skpPltTrc = 10; % Number of run5 instances (+1) to skip between traces plotted. To skip none, set to 1.

dir1 = [num2str(run1) '_' num2str(run2) '_' num2str(run3)];
dir2 = [dir1 '_' num2str(run4)];
leakDir = [dir1 '_1']; % Data in 3_0_0_1 is used to measure leak current using P/4 leak subtraction

dataDir = 'data/';
prfx = 'NaiMnap'; % Prefix used in the name of the simulation data file
nvar = 2; % Number of variables in simulation data file
Naindx = 1; % Index of [Na+]i variable in simulation data file
mNaPindx = 2; % Index of mNaP variable in simulation data file

VclampStrt = -70; % The voltage set for run5 = 1 and run6 = 2.
VclampStp = 1; % The voltage increase between run5 instances
VclampIntl = -90; % The holding potential used in 3_0_0_0
VclampInter = -80; % The starting potental when run6 = 1
tint = 0.0001; % Time step used by integrator

fntnm = 'Lucida Sans';
fntsz = 12;
vsblty = 'on'; % Set to 'off' to make figures not pop up or set to 'on'
CloseFigs = 0; % Set to 1 to automatically close figures after printing them to a file, or set to 0 to leave them open.
ZoomInOnStepUp = 0; % Set to 1 to zoom in on the depolarization step or set to zero and set xmin and xmax below to the axis range you want
xmin = 95;
xmax = 120;
SetMeasureTime = 0; % Set to 0 in order to measure the actual peak current after depolarization step, set to 1 to estimate this peak current by
                    % measuring at a specified time (msrTm) after the step
msrTm = 0.034; % Used if SetMeasureTime is set to 1.
msrTm0 = 0.034; % Only used if max gNaP does not occur shortly after voltage step and SetMeasureTime is set to 0

readpars; % Model Parameters, not all of these are actually used by this code
assmdENa = 45.0; % This is assumed Na reversal potential used to measure gNaP

VarN = ['../' dir1 '_0/' dataDir prfx dir1 '_0_0_0.dat'];
fl = fopen(VarN);
yyStrt = fread(fl,[nvar, 10000000],'double')';
fclose(fl);

for run5 = run5i:skpPltTrc:run5f
    yy = yyStrt;
    leakyy = yyStrt;
    indxs(1) = length(yy);
    leakIndxs(1) = length(leakyy);
    for run6=run6i:run6f   % Concatenating portions of time saved in separate files

        dir3 = [dir2 '_' num2str(run5)];

        VarN = [dataDir prfx dir3 '_' num2str(run6) '.dat'];
        fl = fopen(VarN);
        yy0 = fread(fl,[nvar, 10000000],'double')';
        fclose(fl);

        yy = [yy(1:end-1,:); yy0]; % Remove last point here, because otherwise this time point will be repeating as the first time point of the next file
        indxs(run6+1) = length(yy);

        leakFlN = ['../' leakDir '/' dataDir prfx leakDir '_' num2str(run5) '_' num2str(run6) '.dat'];
        leakFl = fopen(leakFlN);
        leakyy0 = fread(leakFl,[nvar, 10000000],'double')';
        fclose(leakFl);

        leakyy = [leakyy(1:end-1,:); leakyy0];
        leakIndxs(run6+1) = length(leakyy);
    end

    Vclamp = VclampStrt + VclampStp*(run5-1);
    VclampL = 0.25*(Vclamp - VclampIntl) + VclampIntl;
    VclampInterL = 0.25*(VclampInter - VclampIntl) + VclampIntl;
    Vclamps(run5-run5i+1) = Vclamp;
    Vm = [VclampIntl*ones(indxs(1),1); VclampInter*ones(indxs(2)-indxs(1),1); Vclamp*ones(indxs(3)-indxs(2),1)]; % Membrane potential steps for this run5 instance
    leakVm = [VclampIntl*ones(indxs(1),1); VclampInterL*ones(indxs(2)-indxs(1),1); VclampL*ones(indxs(3)-indxs(2),1)]; % The membrane potential used to
                                                                                                                       % measure the leak current in 3_0_0_1
                                                                                                                       % with P/4 leak subtraction

    Ipump = Ipmpmx./(1.0+(Naih*exp(alpha*Vm/rtf)./yy(:,Naindx)).^nhillNa); % Ipump in 3_0_0_2
    Inap = gNaP*yy(:,mNaPindx).*(Vm-rtf*log(Nae./yy(:,Naindx))); % INaP in 3_0_0_2
    IleakNa = gL*(EL-EK)/(ENa-EK)*(Vm-rtf*log(Nae./yy(:,Naindx))); % ILeakNa in 3_0_0_2
    IleakK = gL*(EL-ENa)/(EK-ENa)*(Vm-EK); % ILeakK in 3_0_0_2
    Ileak = IleakNa + IleakK;
    Itot = Ipump + Inap + Ileak;

    IleakPump = Ipmpmx./(1.0+(Naih*exp(alpha*leakVm/rtf)./leakyy(:,Naindx)).^nhillNa); % Ipump in 3_0_0_1
    IleakNap = gNaP*leakyy(:,mNaPindx).*(leakVm-rtf*log(Nae./leakyy(:,Naindx))); % INaP in 3_0_0_1
    IleakLNa = gL*(EL-EK)/(ENa-EK)*(leakVm-rtf*log(Nae./leakyy(:,Naindx))); % ILeakNa in 3_0_0_1
    IleakLK = gL*(EL-ENa)/(EK-ENa)*(leakVm-EK); % ILeakK in 3_0_0_1
    IleakL = IleakLNa + IleakLK;
    IleakMsrd0 = IleakPump + IleakNap + IleakL;
    IleakMsrd = IleakMsrd0;
    IleakIntl = mean(IleakMsrd0(round(indxs(1)/2):indxs(1)));
    IleakMsrd(indxs(1)+1:indxs(end)) = IleakIntl + 4.0*(IleakMsrd0(indxs(1)+1:indxs(end)) - IleakIntl); % Measured leak current for P/4 leak subtraction

    tt = 0.0:tint:tint*(indxs(end)-1);

    iMsr = length(indxs)-1;
    InapMsrd = Itot - IleakMsrd; % Subtracting measured leak current
    gnapMsrd = InapMsrd./(Vm - assmdENa); % Measuring gNaP using assumed Na reversal potential
    if SetMeasureTime
        maxGnapIndx = round(msrTm/tint) + indxs(iMsr);
        maxGnap = gnapMsrd(maxGnapIndx); % Estimating peak gNaP using time of expected peak
    else
        [maxGnap, maxGnapIndx0] = max(gnapMsrd(indxs(iMsr)+1:indxs(iMsr)+round(1.8*msrTm0/tint))); % Finding peak gNaP
        maxGnapIndx = maxGnapIndx0 + indxs(iMsr); % index of peak gNaP
        if maxGnapIndx-indxs(iMsr) > 1.7*round(msrTm0/tint) || maxGnapIndx-indxs(iMsr) < round(0.1*msrTm0/tint) % If the maximum current does not occur
                                                                                                                % shortly after the depolarization step
            maxGnapIndx = round(msrTm0/tint)+indxs(iMsr); % Use expected time of peak (msrTm0)
            maxGnap = gnapMsrd(maxGnapIndx); % gNaP at expected time of peak
            disp(['Setting measurement time to ' num2str(msrTm0) ' ms for run5 = ' num2str(run5)]);
        end
    end
    normGnapMsrd = gnapMsrd/maxGnap;

    tailcurrent = InapMsrd(maxGnapIndx:end)';
    tailcondctnc = gnapMsrd(maxGnapIndx:end)';
    tailnormcond = normGnapMsrd(maxGnapIndx:end)';
    ttail = tt(maxGnapIndx:end)-tt(maxGnapIndx);

    dlmwrite([dataDir 'tailCurrent' dir3 vrsn '.txt'],[tailcurrent; ttail]);
    dlmwrite([dataDir 'tailConductance' dir3 vrsn '.txt'],[tailcondctnc; ttail]);
    dlmwrite([dataDir 'tailNormalizedConductance' dir3 vrsn '.txt'],[tailnormcond; ttail]);
    dlmwrite([dataDir 'tailNaConcentration' dir3 vrsn '.txt'],[yy(maxGnapIndx:end,Naindx)'; ttail]);

    if ZoomInOnStepUp
        xmin = tt(indxs(iMsr)+1)-0.1;
        xmax = tt(indxs(iMsr)+1) + 0.2;
        mrk = 'zm';
    else
        mrk = '';
    end

    f = figure('visible',vsblty);
    f.PaperPositionMode = 'manual';
    f.PaperUnits = 'inches';
    f.Units = 'inches';
    f.OuterPosition = [1 1 8.0 5.0];
    f.InnerPosition = [0.25 0.25 7.5 4.5];
    f.PaperPosition = [0.25 0.25 7.0 4.0];
    f.RendererMode = 'manual';

    axes('position',[0.15 0.75 0.8 0.2]);
    hold on;
    plot(tt, InapMsrd, 'k', 'linewidth', 1.5);
    ylabel('I_{NaP} (pA)');
    xticks([]);
    xlim([xmin xmax]);
    box off;
    ax = gca;
    ax.FontName = fntnm;
    ax.FontSize = fntsz;
    ax.LineWidth = 2.0;

    axes('position',[0.15 0.46 0.8 0.2]);
    hold on;
    plot(tt, gnapMsrd, 'k', 'linewidth', 1.5);
    plot(tt(maxGnapIndx),maxGnap,'rx','linewidth',1.5,'markersize',10);
    ylabel('g_{NaP} (nS)');
    xticks([]);
    xlim([xmin xmax]);
    box off;
    ax = gca;
    ax.FontName = fntnm;
    ax.FontSize = fntsz;
    ax.LineWidth = 2.0;

    axes('position',[0.15 0.17 0.8 0.2]);
    hold on;
    plot(tt, Vm, 'k', 'linewidth', 1.5);
    ylabel('V_{clamp} (mV)');
    xlabel('Time (s)');
    xlim([xmin xmax]);
    box off;
    ax = gca;
    ax.FontName = fntnm;
    ax.FontSize = fntsz;
    ax.LineWidth = 2.0;

    print(f,['plots/InapGnapTail' dir2 '_' num2str(run5) vrsn mrk '.eps'],'-depsc','-r0');

    if strcmp(vsblty,'off')
        close(f);
    elseif CloseFigs
        close(f);
    end
end

