% measureNaPinactivation.m
% Jessica Parker, October 16, 2024
%
% This Matlab script loads the simulation data made by the C code and measures the steady-state inactivation level 
% at an array of different membrane potentials, which the cell is clamped at for run6 = 1 and which are indexed by run5.

clear all;
close all;

vrsn = 'A';

run1 = 3;
run2 = 0;
run3 = 0;

dir1 = [num2str(run1) '_' num2str(run2) '_' num2str(run3)];

leakDir = [dir1 '_3'];
mnapVrsn = 'A';
mnapDir = [dir1 '_2'];

run4 = 4;
run5i = 1;
run5f = 55;
run6i = 1;
run6f = 2;

dataDir = 'data/';
prfx = 'NaiMnap';
nvar = 2;
Naindx = 1;
mNaPindx = 2;
VclampStrt = -80; % Membrane potential for run5 = 1 and run6 = 1
VclampStep = 1; % Increase in clamped membrane potential for each consecutive run5 instance
VclampFnl = -40; % Membrane potential for run6 = 2
VclampIntl = -90; % Holding potential from 3_0_0_0
VclampLeakFnl = 0.25*(VclampFnl - VclampIntl) + VclampIntl; % Final membrane potential for leak current simulation in 3_0_0_3 when run6 = 2 (-77.5 mV)
tint = 0.0001; % Time step used by integrator. Time between consecutive simulation data points.
SetMeasureTime = 0; % Set to 1 to use a preset time (msrTm) after the step to -40 mV to measure the peak gNaP. Set to 0 to measue the actual peak gNaP.
msrTm = 0.032; % Used if SetMeasureTime is set to 1.

%% Plot Properties
fntnm = 'Lucida Sans';
fntsz = 14;
vsblty = 'on';
CloseFigs = 0;
PlotTraces = 1;
strtPltTrc = 1; % First run5 instance plotted if PlotTraces is set to 1
stpPltTrc = 55; % Last run5 instance plotted if PlotTraces is set to 1
skpPltTrc = 10; % Number of run5 instances (+1) to skip between traces plotted. To skip none, set to 1.
ZoomInOnFinalStep = 1; % Set to 1 to zoom in on the step to -40 mV (run6 = 2) if PlotTraces is set 1. Otherwise, set to 0 and set xmin and xmax below.
xmin = 50; % Start of x-axis range plotted when PlotTraces is set to 1.
xmax = 75; % End of x-axis range plotted when PlotTraces is set to 1.

readpars; % Reading parameter values using readpars.m
assmdENa = 45.0; % This is the assumed Na reversal potential used to measure gNaP
dir2 = [dir1 '_' num2str(run4)];

mnapMsrd = load(['../' mnapDir '/' dataDir 'mnapMsrd' mnapDir mnapVrsn '.txt']); % measured steady-state mNaP from 3_0_0_2
mnapIndx = find(mnapMsrd(2,:) == VclampFnl); % index of mnapMsrd corresponding to Vclamp = -40 mV
mnapMsrdFnl = mnapMsrd(1,mnapIndx); % measured mNaP at Vclamp = -40 mV
mnapGFnl = mnapMsrd(3,mnapIndx); % measured gNaP at Vclamp = -40 mV

VarN = ['../' dir1 '_0/' dataDir prfx dir1 '_0_0_0.dat'];
fl = fopen(VarN);
yyStrt = fread(fl,[nvar, 10000000],'double')'; % Contains simulation data from 3_0_0_0 where it is held at the holding potential of -90 mV
fclose(fl);

lastindx = 400;
for run5 = run5i:run5f
    yy = yyStrt; % starts at -90 mV with data from 3_0_0_0
    leakyy = yyStrt;
    indxs(1) = length(yy);
    leakIndxs(1) = length(leakyy);
    for run6=run6i:run6f  % Concatenating portions of time saved in separate files

        dir3 = [dir2 '_' num2str(run5)];

        VarN = [dataDir prfx dir3 '_' num2str(run6) '.dat'];
        fl = fopen(VarN);
        yy0 = fread(fl,[nvar, 10000000],'double')'; % Reading simulation data from 3_0_0_4
        fclose(fl);

        yy = [yy(1:end-1,:); yy0];   % Remove last point here, because otherwise you will repeat the same time point
        indxs(run6+1) = length(yy);

        leakFlN = ['../' leakDir '/' dataDir prfx leakDir '_' num2str(run5) '_' num2str(run6) '.dat'];
        leakFl = fopen(leakFlN);
        leakyy0 = fread(leakFl,[nvar, 10000000],'double')'; % Reading data from 3_0_0_3 where leak current is simulated for P/4 leak subtraction
        fclose(leakFl);

        leakyy = [leakyy(1:end-1,:); leakyy0];
        leakIndxs(run6+1) = length(leakyy);
    end

    Vclamp = VclampStrt + VclampStep*(run5-1); % First membrane potential step (run6 = 1) after holding potential for this run5 instance
    VclampL = 0.25*(Vclamp - VclampIntl) + VclampIntl; % First membrane potential step (run6 = 1) for this run5 instance in 3_0_0_3 for leak current simulation
    Vclamps(run5-run5i+1) = Vclamp; % Array of first Vclamp steps for all run5 instances
    Vm = [VclampIntl*ones(indxs(1),1); Vclamp*ones(indxs(2)-indxs(1),1); VclampFnl*ones(indxs(3)-indxs(2),1)]; % All membrane potential steps for this run5
    leakVm = [VclampIntl*ones(indxs(1),1); VclampL*ones(indxs(2)-indxs(1),1); VclampLeakFnl*ones(indxs(3)-indxs(2),1)]; % The membrane potential protocol 
                                                                                        % used to measure the leak current in 3_0_0_3 for P/4 leak 
                                                                                        % subtraction for this run5 instance.

    Ipump = Ipmpmx./(1.0+(Naih*exp(alpha*Vm/rtf)./yy(:,Naindx)).^nhillNa); % Ipump in 3_0_0_4
    Inap = gNaP*yy(:,mNaPindx).*(Vm-rtf*log(Nae./yy(:,Naindx))); % INaP in 3_0_0_4
    IleakNa = gL*(EL-EK)/(ENa-EK)*(Vm-rtf*log(Nae./yy(:,Naindx))); % ILeakNa in 3_0_0_4
    IleakK = gL*(EL-ENa)/(EK-ENa)*(Vm-EK); % ILeakK in 3_0_0_4
    Ileak = IleakNa + IleakK;
    Itot = Ipump + Inap + Ileak;

    IleakPump = Ipmpmx./(1.0+(Naih*exp(alpha*leakVm/rtf)./leakyy(:,Naindx)).^nhillNa); % Ipump in 3_0_0_3
    IleakNap = gNaP*leakyy(:,mNaPindx).*(leakVm-rtf*log(Nae./leakyy(:,Naindx))); % INaP in 3_0_0_3
    IleakLNa = gL*(EL-EK)/(ENa-EK)*(leakVm-rtf*log(Nae./leakyy(:,Naindx))); % ILeakNa in 3_0_0_3
    IleakLK = gL*(EL-ENa)/(EK-ENa)*(leakVm-EK); % ILeakK in 3_0_0_3
    IleakL = IleakLNa + IleakLK;
    IleakMsrd0 = IleakPump + IleakNap + IleakL;

    IleakRef = mean(IleakMsrd0(round(indxs(1)/2):indxs(1)-1)); % The steady-state leak current after the potential has been held at -90 mV for a while
    IleakMsrd = 4*(IleakMsrd0 - IleakRef) + IleakRef; % The estimated leak current in the actual simulation in 3_0_0_4 used for P/4 leak subtraction
    IleakMsrd(1:indxs(1)) = IleakMsrd0(1:indxs(1));

    InapMsrd = Itot - IleakMsrd; % Subtracting measured/estimated leak current
    gnapMsrd = InapMsrd./(Vm - assmdENa); % Measuring gNaP using assumed Na reversal potential
    if SetMeasureTime
        ttindx = round(msrTm/tint) + indxs(2); % Index used to measure peak gNaP using time of expected peak if SetMeasureTime is 1
    else
        [maxtmp, ttindx0] = max(gnapMsrd(indxs(2)+1:indxs(2)+800)); % Finding peak gNaP
        if ttindx0 < 25 || ttindx0 > 750 % If the maximum gNaP does not occur shortly after the depolarization step
            ttindx0 = lastindx;            % then use the time of the peak gNaP in the last run5 instance
            disp(['Using last measurement time for run5 = ' num2str(run5)]);
        end
        ttindx = ttindx0 + indxs(2); % index of peak gNaP
        lastindx = ttindx0;
    end
    msrTmGnap = gnapMsrd(ttindx); % peak gNaP or gNaP at time of measurement
    allGnaps(run5-run5i+1) = msrTmGnap; % array of peak gNaPs for all run5 instances

    tt = 0.0:tint:tint*(indxs(end)-1);
    msrTms(run5-run5i+1) = tt(ttindx)-tt(indxs(2)); % array of times of peak gNaP for all run5 instances

    %% Plot example traces if PlotTraces is set to 1
    if (PlotTraces && run5 >= strtPltTrc) && (run5 <= stpPltTrc && mod(run5,skpPltTrc) == 0)
        if ZoomInOnFinalStep
            xmin = tt(indxs(2)) - 0.01;
            xmax = tt(indxs(2)) + 0.1;
        end

        f = figure('visible',vsblty);
        f.PaperPositionMode = 'manual';
        f.PaperUnits = 'inches';
        f.Units = 'inches';
        f.OuterPosition = [1 1 8.0 7.0];
        f.InnerPosition = [0.25 0.25 7.5 6.5];
        f.PaperPosition = [0.25 0.25 7.0 6.0];
        f.RendererMode = 'manual';

        axes('position',[0.15, 0.72, 0.8, 0.2]);
        hold on;
        plot(tt, InapMsrd, 'k', 'linewidth', 1.5);
        ylabel('I_{NaP} (pA)');
        xticks([]);
        xlim([xmin xmax]);
        box off;
        ax = gca;
        ax.FontName = fntnm;
        ax.FontSize = fntsz;
        ax.LineWidth = 2.0;

        axes('position',[0.15, 0.44, 0.8, 0.2]);
        hold on;
        plot(tt, gnapMsrd, 'k', 'linewidth', 1.5);
        plot(tt(ttindx),msrTmGnap,'rx','linewidth',1.5,'markersize',10);
        ylabel('g_{NaP} (nS)');
        xticks([]);
        xlim([xmin xmax]);
        box off;
        ax = gca;
        ax.FontName = fntnm;
        ax.FontSize = fntsz;
        ax.LineWidth = 2.0;

        axes('position',[0.15, 0.16, 0.8, 0.2]);
        hold on;
        plot(tt, Vm, 'k', 'linewidth', 1.5);
        ylabel('V_{clamp} (mV)');
        xlabel('Time (s)');
        xlim([xmin xmax]);
        box off;
        ax = gca;
        ax.FontName = fntnm;
        ax.FontSize = fntsz;
        ax.LineWidth = 2.0;

        print(f,['plots/Vclamp' dir2 '_' num2str(run5) vrsn '.eps'],'-depsc','-r0');

        if strcmp(vsblty,'off')
            close(f);
        elseif CloseFigs
            close(f);
        end
    end
end

mxmlGnap = max(mnapMsrd(3,:)); % Maximum measured gNaP
if max(allGnaps) > mxmlGnap % Check if the maximum gNaP measured here is greater than the max gNaP measured in 3_0_0_2
    disp(['new maximal gnap = ' num2str(max(allGnaps)/mxmlGnap) ' x old maximal gnap']);
    mxmlGnap = max(allGnaps);
    dlmwrite([dataDir 'maximalGnap' dir2 vrsn '.txt'], mxmlGnap);
end
normGnap= allGnaps/mxmlGnap; % Normalized gNaP
hnapMsrd = normGnap/mnapMsrdFnl; % Measure hNaP by dividing by mNaP measured at -40 mV in 3_0_0_2
dlmwrite([dataDir 'originalMaxHnap' dir2 vrsn '.txt'], max(hnapMsrd));
if max(hnapMsrd) > 1 && max(hnapMsrd) <= 1.05 % Check if max measured hNaP is greater than 1 but not abnormally large
    disp(['max hnap = ' num2str(max(hnapMsrd))]);
    hnapMsrd = hnapMsrd/max(hnapMsrd);
elseif max(hnapMsrd) > 1.05 % If max measured hnaP is greater than 1.05 than check your code. The culprit is likely 
    disp('Error! Maximum hNaP is unusually large.'); % the value you use for the assumed Na reversal potential
end

f = figure('visible',vsblty);
f.PaperPositionMode = 'manual';
f.PaperUnits = 'inches';
f.Units = 'inches';
f.OuterPosition = [1 1 5.0 5.0];
f.InnerPosition = [0.25 0.25 4.5 4.5];
f.PaperPosition = [0.25 0.25 4.0 4.0];
f.RendererMode = 'manual';

axes('position',[0.2 0.2 0.75 0.65]);
hold on;
plot(Vclamps,hnapMsrd,'k','linewidth',1.5);
xlabel('V_{m} (mV)');
ylabel('h_{NaP}');
title(['Assumed E_{Na} = ' num2str(assmdENa) ' mV']);
ylim([0 1]);
box off;
ax = gca;
ax.FontName = fntnm;
ax.FontSize = fntsz;
ax.LineWidth = 2.0;

print(f,['plots/hnapMsrd' dir2 vrsn '.eps'],'-depsc','-r0');

if strcmp(vsblty,'off')
    close(f);
elseif CloseFigs
    close(f);
end

dlmwrite([dataDir 'hnapMsrd' dir2 vrsn '.txt'], [hnapMsrd; Vclamps]);

