"""
netParams.py 

NetPyNE high-level specifications for cortical network model to study epilepsy 

Contributors: salvadordura@gmail.com
"""

from netpyne import specs
import pickle, json
#import math

netParams = specs.NetParams()   # object of class NetParams to store the network parameters

try:
	from __main__ import cfg  # import SimConfig object with params from parent module
except:
	from cfg import cfg


#------------------------------------------------------------------------------
# VERSION 
#------------------------------------------------------------------------------
netParams.version = 1

#------------------------------------------------------------------------------
#
# NETWORK PARAMETERS
#
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# General network parameters
#------------------------------------------------------------------------------

netParams.scale = cfg.scale # Scale factor for number of cells # NOT DEFINED YET! 3/11/19 # How is this different than scaleDensity? 
netParams.sizeX = cfg.sizeX # x-dimension (horizontal length) size in um
netParams.sizeY = cfg.sizeY # y-dimension (vertical height or cortical depth) size in um
netParams.sizeZ = cfg.sizeZ # z-dimension (horizontal depth) size in um
netParams.shape = 'cylinder'  # cylindrical (column-like) volume

netParams.defaultDelay = [DT] # 2.0 # default conn delay (ms)
netParams.propVelocity = 5e6 # 500.0 # propagation velocity (um/ms)
#netParams.defaultThreshold = 0.0	# 0 mV for threshold detection
netParams.defaultThreshold = -5.0 #-20.0 #-5.0

#------------------------------------------------------------------------------
# Cell parameters
#------------------------------------------------------------------------------

## Import Pyramidal cell params from py file 
netParams.importCellParams(label='Pyr_simple', conds={'cellType': 'Pyr'}, fileName='PY.hoc', cellName='PY', cellInstance = True)
# netParams.addCellParamsWeightNorm('Pyr_simple', 'cells/CSTR6_weightNorm.pkl', threshold=cfg.weightNormThreshold)  # converts weight (PSP in mv) to syn conduct (uS) 

## Import PV cell params from hoc file 
netParams.importCellParams(label='FS_simple', conds={'cellType': 'FS'}, fileName='FS.hoc', cellName='FS', cellInstance = True)
# netParams.addCellParamsWeightNorm('FS_simple', 'cells/PV_simple_weightNorm.pkl', threshold=cfg.weightNormThreshold)  # converts weight (PSP in mV) to syn conduct (uS) 

#------------------------------------------------------------------------------
# Population parameters
#------------------------------------------------------------------------------

## Set layer boundaries and densities
#layer = {'2': [0.12, 0.38], '4': [0.38, 0.51], '5': [0.51,0.76], '6': [0.76,1.0]}  
#density = {'L2': 1.4e5, 'L4': 2.0e5, 'L5': 0.8e5, 'L6': 1.5e5}
#layer = {'2': [0.12, 0.38], '4': [0.38, 0.51], '5': [0.51,1.0]}  
#layer = {'2': [0.55, 0.88], '4': [0.49, 0.55], '5': [0,0.49]}
#layer = {'2': [0.12, 0.38], '4': [0.38, 0.55], '5': [0.55,1.0]}
layer = {'2': [0.12, 0.38], '5':[0.38,1.0]}

# density = {'L2': 1.4e4, 'L4': 2.5e4, 'L5': 2.5e4} # cells per mm2
density = {'L2': 1.0e4, 'L4': 1.0e4, 'L5': 1.0e4} # cells per mm2

# define E and I pops to use later in conn
Epops = ['E2', 'E5']  
# Epops = ['E2']  
Ipops = ['I2']  

# specify population parameter 
## L2
#netParams.popParams['E2'] = {'cellType': 'Pyr', 'ynormRange': layer['2'],   'density': 0.3*density['L2']*cfg.scaleDensity}     
#netParams.popParams['I2'] = {'cellType': 'FS',  'ynormRange': layer['2'],   'density': 0.15*density['L2']*cfg.scaleDensity}      
## L4
#netParams.popParams['E4'] = {'cellType': 'Pyr', 'ynormRange': layer['4'],   'density': 0.85*density['L4']*cfg.scaleDensity}      
# netParams.popParams['E4'] = {'cellType': 'Pyr',  'ynormRange': layer['4'],   'density': 0.3*density['L4']*cfg.scaleDensity}      
## L5
#netParams.popParams['E5'] = {'cellType': 'Pyr', 'ynormRange': layer['5'],   'density': 0.4*density['L5']*cfg.scaleDensity}      
#netParams.popParams['I5'] = {'cellType': 'FS',  'ynormRange': layer['5'],   'density': 0.15*density['L5']*cfg.scaleDensity}      
## L6
#netParams.popParams['E6'] = {'cellType': 'Pyr', 'ynormRange': layer['6'],   'density': 0.85*density['L6']*cfg.scaleDensity}      
#netParams.popParams['I6'] = {'cellType': 'FS',  'ynormRange': layer['6'],   'density': 0.15*density['L6']*cfg.scaleDensity}      

# specify population parameter 
## L2
netParams.popParams['E2'] = {'cellType': 'Pyr', 'ynormRange': layer['2'],  'numCells':65}     
netParams.popParams['I2'] = {'cellType': 'FS',  'ynormRange': layer['2'],  'numCells':32}      
 
## L5
netParams.popParams['E5'] = {'cellType': 'Pyr', 'ynormRange': layer['5'],   'numCells':206}      



# option to create populations with a single cell (for debugging)
if cfg.singleCellPops:
    for pop in netParams.popParams.values(): pop['numCells'] = 1


#------------------------------------------------------------------------------
# Synaptic mechanism parameters
#------------------------------------------------------------------------------

#netParams.synMechParams['NMDA'] = {'mod': 'MyExp2SynNMDABB', 'tau1NMDA': 15, 'tau2NMDA': 150, 'e': 0}
#netParams.synMechParams['AMPA'] = {'mod': 'MyExp2SynBB', 'tau1': 0.05, 'tau2': 5.3, 'e': 0}
#netParams.synMechParams['GABAA'] = {'mod': 'MyExp2SynBB', 'tau1': 0.07, 'tau2': 18.2, 'e': -80}
#netParams.synMechParams['GABAB'] = {'mod': 'MyExp2SynBB', 'tau1': 3.5, 'tau2': 260.9, 'e': -93} 
netParams.synMechParams['AMPA'] = {'mod': 'Exp2Syn', 'tau1': 0.02, 'tau2': 3.0, 'e': 0}
netParams.synMechParams['GABA'] = {'mod': 'Exp2Syn', 'tau1': 0.02, 'tau2': 6.0, 'e': -73}




#------------------------------------------------------------------------------
# Local connectivity parameters
#------------------------------------------------------------------------------
if cfg.addConn:

# EE
    netParams.connParams['E2->E2'] ={
    'preConds':{'pop':'E2'},
    'postConds':{'pop':'E2'},
    'probability':'0.15', # 5% up to 300um
    'weight':1*cfg.EEGain,    
		'delay':'defaultDelay+dist_3D/propVelocity',
#'delay':defaultDelay,
    'synMech':['AMPA'],
    'sec':'soma'
    }

    netParams.connParams['E2->E5'] ={
    'preConds':{'pop':'E2'},
    'postConds':{'pop':'E5'},
    'probability':'0.15', # 5% up to 300um
    'weight':1*cfg.EEGain,
		'delay':'defaultDelay+dist_3D/propVelocity',
#'delay':defaultDelay,
    'synMech':['AMPA'],
    'sec':'soma'
    }

    netParams.connParams['E5->E2'] ={
    'preConds':{'pop':'E5'},
    'postConds':{'pop':'E2'},
    'probability':'0.15', # 1% up to 300um
    'weight':1*cfg.EEGain,
		'delay':'defaultDelay+dist_3D/propVelocity',
#'delay':defaultDelay,
    'synMech':['AMPA'],
    'sec':'soma'
    }

    netParams.connParams['E5->E5'] ={
    'preConds':{'pop':'E5'},
    'postConds':{'pop':'E5'},
    'probability':'0.15', # 2.5% up to 300um
    'weight':1*cfg.EEGain,
		'delay':'defaultDelay+dist_3D/propVelocity',
#'delay':defaultDelay,
    'synMech':['AMPA'],
    'sec':'soma'
    }

## EI

    netParams.connParams['E2->I2'] ={
    'preConds':{'pop':'E2'},
    'postConds':{'pop':'I2'},
    'probability':'0.15', # 5% of E connections up to 300 um
    'weight':1*cfg.EIGain,
		'delay':'defaultDelay+dist_3D/propVelocity',
#'delay':defaultDelay,
    'synMech':['AMPA'],
    'sec':'soma'
    }

    netParams.connParams['E5->I2'] ={
    'preConds':{'pop':'E5'},
    'postConds':{'pop':'I2'},
    'probability':'0.15', # 5% of E connections up to 300 um
    'weight':1*cfg.EIGain,
		'delay':'defaultDelay+dist_3D/propVelocity',
#'delay':defaultDelay,
    'synMech':['AMPA'],
    'sec':'soma'
    }

### II
    netParams.connParams['I2->I2'] ={
    'preConds':{'pop':'I2'},
    'postConds':{'pop':'I2'},
    'probability':'0.40', # 5% of E connections up to 300 um
    #'divergence':50,
    #'weight':0.012*cfg.IIGain,
    'weight':1*cfg.IIGain,

		'delay':'defaultDelay+dist_3D/propVelocity',
#'delay':defaultDelay,
    'synMech':['GABA'],
    'sec':'soma',
    # 'synsPerConn':1
    }



##### IE
    netParams.connParams['I2->E2'] ={
    'preConds':{'pop':'I2'},
    'postConds':{'pop':'E2'},
    'probability':'0.35', # 5% of E connections up to 300 um
    #'divergence':500,
    #'weight':0.018*cfg.IEGain,
    'weight':1*cfg.IEGain,

		'delay':'defaultDelay+dist_3D/propVelocity',
#'delay':defaultDelay,
    'synMech':['GABA'],
    'sec':'soma',
    # 'synsPerConn':1

    }

    netParams.connParams['I2->E5'] ={
    'preConds':{'pop':'I2'},
    'postConds':{'pop':'E5'},
    'probability':'0.35', # 5% of E connections up to 300 um
    #'divergence':500,
    #'weight':0.018*cfg.IEGain,
    'weight':1*cfg.IEGain,

		'delay':'defaultDelay+dist_3D/propVelocity',
#'delay':defaultDelay,
    'synMech':['GABA'],
    'sec':'soma',
    # 'synsPerConn':1
    }


  #------------------------------------------------------------------------------
# Reaction-diffusion (RxD) parameters
#------------------------------------------------------------------------------  

# RxD params

### constants
constants = {}
netParams.rxdParams['constants'] = constants



### regions
regions = {}
# regions['cyts'] = {'cells': 'all', 'secs': 'all', 'geometry':{'class':'Shell','args':{'lo':0,'hi':1}},'nrn_region':'i'} # cyts space
#regions['ints'] = {'cells': 'all', 'secs': 'all', 'nrn_region':None} # interstitial space
# regions['ints'] = {'cells': 'all', 'secs': 'all', 'geometry':{'class':'FractionalVolume','args':{'surface_fraction':0.15}},'nrn_region':None} # interstitial space
#regions['ints'] = {'cells': 'all', 'secs': 'all', 'geometry':{'class':'FractionalVolume','args':{'surface_fraction':0.15}},'nrn_region':'o'} 
#regions['cyts'] = {'cells': 'all', 'secs': 'all', 'geometry':{'class':'FractionalVolume','args':{'volume_fraction':1}},'nrn_region':'i'} 

regions['cyt'] = {'cells': 'all', 'secs': 'all', 'nrn_region':'i'} 
# regions['fhs'] = {'cells': 'all', 'secs': 'all', 'geometry':{'class':'Shell','args':{'lo':1,'hi':1.002247}},'nrn_region':'o'} # interstitial space
regions['fhs_i'] = {'cells': ['I2'], 'secs': 'all', 'geometry':{'class':'Shell','args':{'lo':1,'hi':1.00727}},'nrn_region':'o'} # interstitial space
regions['fhs_e'] = {'cells': ['E2','E5'], 'secs': 'all', 'geometry':{'class':'Shell','args':{'lo':1,'hi':1.00889}},'nrn_region':'o'} # interstitial space

# regions['cyt_fhs_membrane'] = {'cells': 'all', 'secs': 'all', 'geometry':{'class':'membrane'},'args':{}}
# regions['fhs_ecs_membrane'] = {'cells': 'all', 'secs': 'all', 'geometry': {'class': 'ScalableBorder', 'args': {'scale': 1.002247*3.1416, 'on_cell_surface': True}}}

regions['cyt_fhs_i_membrane'] = {'cells': ['I2'], 'secs': 'all', 'geometry':{'class':'membrane','args':{}}}
regions['cyt_fhs_e_membrane'] = {'cells': ['E2','E5'], 'secs': 'all', 'geometry':{'class':'membrane','args':{}}}

#regions['fhs_i_ecs_membrane'] = {'cells': ['I2'], 'secs': 'all', 'geometry': {'class': 'ScalableBorder', 'args': {'scale': 1.002247*3.1416, 'on_cell_surface': True}}}
#regions['fhs_e_ecs_membrane'] = {'cells': ['E2','E5'], 'secs': 'all', 'geometry': {'class': 'ScalableBorder', 'args': {'scale': 1.002247*3.1416, 'on_cell_surface': True}}}
regions['fhs_i_ecs_membrane'] = {'cells': ['I2'], 'secs': 'all', 'geometry': {'class': 'ScalableBorder', 'args': {'diam_scale': 1.00727, 'on_cell_surface': True}}}
regions['fhs_e_ecs_membrane'] = {'cells': ['E2','E5'], 'secs': 'all', 'geometry': {'class': 'ScalableBorder', 'args': {'diam_scale': 1.00889, 'on_cell_surface': True}}}



# regions['fhs_ecs_membrane'] = {'cells': 'all', 'secs': 'all', 'geometry': {'class': 'membrane', 'args': {'scale': 1.002247*3.1416, 'on_cell_surface': True}}}


fraction = 1.0
margin = 10  # extracellular volume additional margin 
x = [0 - margin, cfg.sizeX*fraction + margin]
y = [-cfg.sizeY*fraction - margin, 0 + margin]
z = [0 - margin, cfg.sizeZ*fraction + margin]

alpha = 0.1
dx = 50 #um
regions['ecs'] = {'extracellular': True, 'xlo': x[0], 'ylo': y[0], 'zlo': z[0], 'xhi': x[1], 'yhi': y[1], 'zhi': z[1], 'dx': dx, 'volume_fraction': alpha, 'tortuosity': 1.5}  
netParams.rxdParams['regions'] = regions


### species 
species = {}
diff_const = [MCRATE] #1.0 um^2/ms
# species['k'] = {'regions': ['fhs','cyt'], 'd': 0.0, 'charge': 1, 'initial': 'lambda nd: 140.0*mM if nd.region == cyt else 3.0*mM'}
# species['k'] = {'regions': ['fhs','cyt'], 'd': 0.0, 'charge': 1, 'initial': lambda nd: 140.0 if nd.region.name == 'cyt' else 3.0}
species['k'] = {'regions': ['fhs_i','fhs_e','cyt'], 'd': 0.0, 'charge': 1, 'initial': lambda nd: 138 if nd.region.name == 'cyt' else [KOINIT]}


# species['k'] = {'regions': ['fhs','cyt'], 'd': 0.0, 'charge': 1, 'initial': '140 if rxd.region == cyt else 3.0'}

# species['k'] = {'regions': ['fhs','cyt'], 'd': 0.0, 'charge': 1, 'initial': 2}

#species['buf'] = {'regions':['fhs'], 'd':0.0, 'charge':0, 'initial': 400}
#species['kbuf'] = {'regions':['fhs'], 'd':0.0, 'charge':0, 'initial': 100}

species['buf_i'] = {'regions':['fhs_i'], 'd':0.0, 'charge':0, 'initial': [BINITI]}
species['kbuf_i'] = {'regions':['fhs_i'], 'd':0.0, 'charge':0, 'initial': 500-[BINITI]}

species['buf_e'] = {'regions':['fhs_e'], 'd':0.0, 'charge':0, 'initial': [BINITE]}
species['kbuf_e'] = {'regions':['fhs_e'], 'd':0.0, 'charge':0, 'initial': 500-[BINITE]}

species['garbage_i'] = {'regions':['fhs_i'], 'd':0.0, 'charge':0, 'initial': 0}
species['garbage_e'] = {'regions':['fhs_e'], 'd':0.0, 'charge':0, 'initial': 0}

species['extP'] = {'regions': ['ecs'], 'd': diff_const, 'charge': 1, 'initial': [KOINIT]}


netParams.rxdParams['species'] = species


### reactions
reactions = {}
# reactions['buffering'] = {'reactant':'k+buf','product':'kbuf','rate_f': '0.0008 / (1 + rxdmath.exp(((15.0 - 1 * k) / 1.15)))','rate_b':0.0008}
# reactions['buffering'] = {'reactant':'k+buf','product':'kbuf','rate_f': '0.0008 / (1 + rxdmath.exp(((10.0 - 1 * k) / 0.02)))','rate_b':0.0008}
reactions['buffering_i'] = {'reactant':'k+buf_i','product':'kbuf_i','rate_f': '0.0008*k*buf_i / (1 + rxd.rxdmath.exp(((k - [KOTHI]) / [SLOPEI])))','rate_b':'0.0008*(500.0-buf_i)', 'custom_dynamics':True}
reactions['buffering_e'] = {'reactant':'k+buf_e','product':'kbuf_e','rate_f': '0.0008*k*buf_e / (1 + rxd.rxdmath.exp(((k - [KOTHE]) / [SLOPEE])))','rate_b':'0.0008*(500.0-buf_e)', 'custom_dynamics':True}
reactions['potassium_garbage_i'] = {'reactant':'k','product':'garbage_i','rate_f': '0.0909*0.0008*(500.0-buf_i)','rate_b': 0, 'custom_dynamics':True}
reactions['potassium_garbage_e'] = {'reactant':'k','product':'garbage_e','rate_f': '0.0909*0.0008*(500.0-buf_e)','rate_b':0, 'custom_dynamics':True}

#reactions['potassium_garbage_i'] = {'reactant':'k','product':'garbage_i','rate_f': '0.0*0.0008*(500.0-buf_i)','rate_b': 0, 'custom_dynamics':True}
#reactions['potassium_garbage_e'] = {'reactant':'k','product':'garbage_e','rate_f': '0.0*0.0008*(500.0-buf_e)','rate_b':0, 'custom_dynamics':True}



#reactions['buffering_i'] = {'reactant':'k+buf_i','product':'kbuf_i','rate_f': '0.0008 / (1 + rxd.rxdmath.exp(((k - [KOTHI]) / [SLOPEI])))','rate_b':0.0008}
#reactions['buffering_e'] = {'reactant':'k+buf_e','product':'kbuf_e','rate_f': '0.0008 / (1 + rxd.rxdmath.exp(((k - [KOTHE]) / [SLOPEE])))','rate_b':0.0008}
netParams.rxdParams['reactions'] = reactions

mcReactions = {}
mM_to_mol_per_um = 6.0221409e+23 * 1e-18

# mcReactions['exchangeK'] = {'reactant': 'k[fhs]', 'product': 'extP[ecs]', 'rate_f':mM_to_mol_per_um*diff_const*alpha/(0.5*dx) , 'rate_b':mM_to_mol_per_um*diff_const*alpha/(0.5*dx)  ,'membrane': 'fhs_ecs_membrane'}
mcReactions['exchangeK_i'] = {'reactant': 'k[fhs_i]', 'product': 'extP[ecs]', 'rate_f':mM_to_mol_per_um*diff_const*alpha/(0.5*dx) , 'rate_b':mM_to_mol_per_um*diff_const*alpha/(0.5*dx)  ,'membrane': 'fhs_i_ecs_membrane'}
mcReactions['exchangeK_e'] = {'reactant': 'k[fhs_e]', 'product': 'extP[ecs]', 'rate_f':mM_to_mol_per_um*diff_const*alpha/(0.5*dx) , 'rate_b':mM_to_mol_per_um*diff_const*alpha/(0.5*dx)  ,'membrane': 'fhs_e_ecs_membrane'}




# mcReactions['exchangeK'] = {'reactant': 'k[fhs]', 'product': 'extP[ecs]', 'rate_f':mM_to_mol_per_um*0.05/(0.5*dx) , 'rate_b':mM_to_mol_per_um*0.05/(0.5*dx)  ,'membrane': 'fhs_ecs_membrane'}

# mM_to_mol_per_um*(Dkeff*alpha/(dx/2)),mM_to_mol_per_um*(Dkeff*alpha/(dx/2)),membrane=bor)
netParams.rxdParams['multicompartmentReactions'] = mcReactions


# Rates

rates = {}
rates['buffer_extra_contri_i'] = {'species': 'k[fhs_i]', 'rate': '-10*0.0909*0.0008*(500.0-buf_i[fhs_i])'}
rates['buffer_extra_contri_e'] = {'species': 'k[fhs_e]', 'rate': '-10*0.0909*0.0008*(500.0-buf_e[fhs_e])'}
#netParams.rxdParams['rates'] = rates # rates don't want with netpyne

# Stimulation parameters


netParams.stimSourceParams['bkg_e'] = {'type': 'NetStim', 'rate': [BCKRATEE], 'noise': [BCKNOISEE], 'seed':[EXSEED]} #SET1 12 SET2 75 SET3 15 SET4 80 SET5 6 SET6 125 SET7 236 SET8 5675
netParams.stimSourceParams['bkg_i'] = {'type': 'NetStim', 'rate': [BCKRATEI], 'noise': [BCKNOISEI], 'seed':[INSEED]} #SET1 24 SET2 57 SET3 2 SET4 10 SET5 25 SET6 98 SET7 750 SET8 2373
#netParams.stimSourceParams['bkg_e'] = {'type': 'NetStim', 'rate': [BCKRATEE], 'noise': [BCKNOISEE], 'seed':12}
#netParams.stimSourceParams['bkg_i'] = {'type': 'NetStim', 'rate': [BCKRATEI], 'noise': [BCKNOISEI], 'seed':24}


#netParams.stimTargetParams['bkg_e->Pyr'] = {'source': 'bkg_e', 'conds': {'cellType': 'Pyr'}, 'weight': 0.00025, 'delay': 0, 'synMech': 'AMPA'} # 0.00025*1e5/2500
#netParams.stimTargetParams['bkg_e->FS'] = {'source': 'bkg_e', 'conds': {'cellType': 'FS'}, 'weight': 0.00025, 'delay': 0, 'synMech': 'AMPA'}



netParams.stimTargetParams['bkg_e->Pyr'] = {'source': 'bkg_e', 'conds': {'cellType': 'Pyr'}, 'weight': [BCKWTEE], 'delay': 0, 'synMech': 'AMPA'} # 0.00025*1e5/2500
netParams.stimTargetParams['bkg_e->FS'] = {'source': 'bkg_e', 'conds': {'cellType': 'FS'}, 'weight': [BCKWTEI], 'delay': 0, 'synMech': 'AMPA'}

netParams.stimTargetParams['bkg_i->Pyr'] = {'source': 'bkg_i', 'conds': {'cellType': 'Pyr'}, 'weight': [BCKWTIE], 'delay': 0, 'synMech': 'GABA'}
netParams.stimTargetParams['bkg_i->FS'] = {'source': 'bkg_i', 'conds': {'cellType': 'FS'}, 'weight': [BCKWTII], 'delay': 0, 'synMech': 'GABA'}
# critical 0.0006183 gKCa=0.1
# 0.0006130 gKCa=0.15


#------------------------------------------------------------------------------
# Description
#------------------------------------------------------------------------------

netParams.description = """
v1 - Created simple 4-layer cortical model based Allen Brain V1 params
"""

