COMMENT
	HCN channels - ih current
ENDCOMMENT

: ----------------------------------------------------------- 

UNITS {
    (mA) = (milliamp)
    (mV) = (millivolt)
	(S)  = (siemens)
}

: ----------------------------------------------------------- 

: The NEURON block was introduced to make possible insert different values 
: of states and parameters, in each segment (compartment) of a model cell by 
: defining what the model of the mechanism looks like from the "outside".
NEURON 	{
	
	: it identifies this to be a distributed mechanism, which can be 
	: incorporated into a NEURON cable section by an insert statement 
	: it tells the NEURON interpreter that the names for variables and
	: parameters that belong to this mechanism will include the suffix 
	: "_SUFFIX", so there will be no conflict with similar names 
	SUFFIX hcn

	NONSPECIFIC_CURRENT ihcn

	: The RANGE statement asserts that i, e, and g are range variables:
	: each of these variables is a function of position, and can have a 
	: different value in each of the segments that make up a section 
	: Each variable mentioned in a RANGE statement should also be 
	: declared in a PARAMETER or ASSIGNED block. 
	RANGE ghcn, V12, ihcn, ehcn, Ft

	: If a PARAMETER does not appear in a NEURON block's RANGE statement,
	: it will have GLOBAL scope, which means that changing its value will 
	: affect every instance of that mechanism throughout an entire 
	: GLOBAL 

}

: ----------------------------------------------------------- 

: Variables whose values are normally specified by the user are parameters,
: and are declared in a PARAMETER block 
PARAMETER 	{

        ehcn    = -30 		(mV)
        Ft      = 1			(ms)
    	ghcn	= 0.000085	(S/cm2)
    	V12		= -90		(mV)
}

: ----------------------------------------------------------- 

: The ASSIGNED block is used to declare two kinds of variables: those that
: are given values outside the mod file, and those that appear on the left 
: hand side of assignment statements within the mod file. 
: By default, a mechanism-specific ASSIGNED variable is a range variable, 
: in that it can have a different value for each instance of the mechanism. 
: However, it will not be visible at the hoc level unless it is declared in 
: a RANGE or GLOBAL statement in the NEURON block 
: Mechanisms that produce fluxes of specific ions, such as K+, may cause 
: the ionic equilibrium potential to change in the course of a simulation. 
: However, the NONSPECIFIC_CURRENT generated by the leak mechanism was not 
: linked to any particular ionic species, so e_leak remains fixedunless 
: explicitly altered by hoc statements or the GUI. 
ASSIGNED {
	: Always here 
    v 		(mV)
    celsius (degC)
    : Variables 
    ihcn	(mA/cm2)
    sinf
    stau
}

: ----------------------------------------------------------- 

: STATE variables are automatically RANGE variables and do not need 
: to be declared in the NEURON block. 	
: If a model involves differential equations, families of algebraic 
: equations, or kinetic reaction schemes, their dependent variables or 
: unknowns are to be listed in the STATE block 
STATE { shcn }

: ----------------------------------------------------------- 

: Main computation block of the mechanism. 
: Remark: its aim it that by the end of the BREAKPOINT block, 
: all variables are consistent with the new time. 
: If a mechanism has STATEs, this block must contain one SOLVE 
: statement that tells how the values of the STATEs will be computed 
: over each time step. 
: Currents are set with assignment statements at the end of the 
: BREAKPOINT block. 
BREAKPOINT 	{
	: Calculate variables of the STATE block 
	SOLVE states METHOD cnexp
	: Current ih 
	ihcn = ghcn * shcn * ( v-ehcn )
}

: ----------------------------------------------------------- 

: Proper initialization of all STATEs is as important as correctly 
: computing their temporal evolution. 
INITIAL 	{
	rates()
	shcn = sinf
}

: ----------------------------------------------------------- 

: This is used to assign values to the derivatives of those STATEs 
: that are described by differential equations. The statements in this 
: block are of the form y' = expr 
: In fixed time step simulations, these equations are integrated 
: using the numerical method specified by the SOLVE statement in the 
: BREAKPOINT block. The cnexp method, which combines second order 
: accuracy with computational efficiency, and it is appropriate when 
: the right hand side of y' = f(v,y) is linear in y and involves no 
: other states 
: The procedure rate() assigns values to the voltage sensitive 
: parameters of this equation. The procedure rate() will be executed 
: individually for each segment in the model that has the HCN mechanism 
DERIVATIVE states	{
	rates()
	shcn' = (sinf-shcn)/stau
}

: ----------------------------------------------------------- 

: The functions and the procedures implement the mathematical 
: expressions that describe the resto of the variables. To facilitate 
: units checking, their arguments can be tagged with the units that 
: they use. 
PROCEDURE rates(){
	UNITSOFF 
		sinf = 1.0000/( 1.0000 + exp((v- V12)/8.0000) ) 
		stau = Ft*exp( 0.033*(v+75) )/( 0.013*( 1+exp(0.083*(v+75)) ) )
	UNITSON
}